/*
 * netlib.c
 *
 * Christopher Adam Telfer
 *
 * Copyright 1999 -- All rights reserved.
 *
 * Many of the ideas for this code come from Stevans.  I have tweaked them here
 * and there for my purposes, but I make no claim that the ideas here are 
 * original.
 */

#include "cat_aux.h"
#include "cat_net.h"
#include "netinc.h"

/* Changed to remove warnings */
int cat_resolve_socket_addr(const char *hostname, const char *service, 
                            const char *proto, struct sockaddr_in *sin);



int
cat_resolve_socket_addr(
   const char *hostname,
   const char *service,
   const char *proto,
   struct sockaddr_in *sin)
{ 
struct hostent *hp; 
struct servent *sp; 
unsigned long portnum, ia; 
char *cp; 

  memset((void *)sin, 0, sizeof(*sin)); 
  sin->sin_family = AF_INET; 

  /* Look up the address */
  if ( ! hostname ) 
  { 
    sin->sin_addr.s_addr = INADDR_ANY; 
  } 
  else if ( (ia = inet_addr(hostname)) != INADDR_NONE  ) 
  { 
    memcpy((void *)&sin->sin_addr, (void *)&ia, sizeof(ia)); 
  } 
  else if ( (hp = gethostbyname(hostname)) ) 
  { 
    memcpy((void *)&sin->sin_addr, (void *)hp->h_addr, hp->h_length);
  } 
  else
  { 
    return -1; 
  } 


  /* now look up the service */
  if ( ! service || ( *service == '\0' ) ) 
  { 
    portnum = htons(0); 
    goto skiplookup;
  } 
  else
    portnum = strtoul(service, &cp, 10); 


  if ( cp != service ) 
  {
    portnum = htons((unsigned short)portnum); 
  } 
  else if ( (sp = getservbyname(service, proto)) )
  { 
    portnum = sp->s_port; 
  } 
  else
  { 
      return -1; 
  } 

skiplookup:
  sin->sin_port = portnum; 

  return 0;
}

int 
cat_tcp_passive(
   const char *hostname, 
   const char *service,
   int *addrlen) 
/*
 * This function is used to open a passive tcp socket.  It uses hostname and
 * service to declare explicity which address and port to bind to if specified.
 * The length of the address is returned in addrlen if addrlen != 0;  To wait
 * for a connection from here do an Accept()... 
 */
{ 
int sock, setraddr = 1; 
struct sockaddr_in sin;


  if ( cat_resolve_socket_addr(hostname, service, "tcp", &sin) < 0 ) 
    return -1; 


  /* now we have the address to use */
  if ((sock = socket(AF_INET, SOCK_STREAM, 0)) < 0)
    return -2; 

  if (setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, 
      (void *)&setraddr, sizeof(setraddr))<0)
  { 
    close(sock); 
    return -3; 
  } 

  if ( bind(sock, (struct sockaddr *)&sin, sizeof(sin)) < 0)
  { 
    close(sock); 
    return -4; 
  } 

  if ( listen(sock, LISTENQ) < 0 ) 
  { 
    close(sock); 
    return -5; 
  } 

  if (addrlen) *addrlen = sizeof(sin); 

  return sock; 
} 


int 
cat_tcp_connect(
   const char *hostname, 
   const char *service) 
/*
 * This function opens a tcp connection to the address specified.  It is 
 * (theoretically) protocol independant.  You can start reading and writing
 * from here. 
 */
{ 
int sock; 
struct sockaddr_in sin;

  if ( cat_resolve_socket_addr(hostname, service, "tcp", &sin) < 0 ) 
    return -1; 

  if ( (sock = socket(AF_INET, SOCK_STREAM, 0)) < 0 )
    return -2; 

  if (connect(sock, (struct sockaddr *)&sin, sizeof(sin)) < 0)
  { 
    close(sock); 
    return -3; 
  } 

  return sock; 
} 



int 
readn(
   int fd, 
   void *buf, 
   int nbytes) 
{ 
char *p = buf; 
int n, numread = 0;

  while (numread < nbytes) { 

    if ( (n = read(fd, p, nbytes)) < 0)
#ifndef SA_RESTART
      if (errno == EINTR) 
        continue; 
      else 
#endif /* SA_RESTART */
        return -1;

    else if (n == 0)   /* Hit an EOF */
      return numread; 

    numread += n;
    p += n; 
  } 

  return numread;
} 


int 
writen(
   int fd, 
   void *buf, 
   int nbytes) 
{ 
char *p = buf; 
int n, numwritten = 0;

  while (numwritten < nbytes) { 

    if ( (n = write(fd, p, nbytes)) <= 0)
#ifndef SA_RESTART
      if (errno == EINTR) 
        continue; 
      else 
#endif /* SA_RESTART */
        return -1;

    numwritten += n;
    p += n; 
  } 

  return numwritten;
} 
