/* ds_police_eof_0.asl - policing functions for DiffServ example	*/

/*
 * Policing script implementing dual token bucket algorithm
 */

#include "policeNp5.h"

#define MAX_RTC_TIME 0xffffffff


/* 
 * Dual token bucket parameters
 * (these are initialized during configuration)
 * Bit rates are measured in RTC ticks per byte.
 * Burst sizes are measured in RTC ticks, i.e.
 * BurstSize_in_ticks = BurstSize_in_bytes x ticks_per_byte 
 */
unsigned PBR param_block[0:1] input;   /* peak bit rate        */
unsigned SBR param_block[2:3] input;   /* sustained bit rate   */
unsigned PBS param_block[4:11] input;  /* peak burst size      */
unsigned SBS param_block[12:19] input; /* sustained burst size */

/*
 * Previous pdu arrival time
 * (initialized to zero during configuration)
 */
unsigned last_pdu_arrival param_block[20:23] inout;

/*
 * token counter for peak bucket
 * (initialized to PBS during configuration)
 */
unsigned p_tokens param_block[24:31]  inout;

/* token counter for sustained bucket
 * (initialized to SBS during configuration)
 */
unsigned s_tokens param_block[32:39] inout;

/* PDU length in bytes (passed by FPP) */
unsigned pdu_length fpp_args[0:1] input;

/* DSCP_CU field (passed by FPP) */
unsigned DSCP_CU    fpp_args[2];

/* RTC ticks since last packet arrived, temporary variable */
unsigned(4) delta_t;

/* PDU length in RTC ticks for sustained rate, temporary variable */
unsigned(4) pdu_peak_len_t;

/* PDU length in RTC ticks for peak rate, temporary variable */
unsigned(4) pdu_sust_len_t;

/* resulting predicate bits */
boolean PeakBucketFailed $[15] output;
boolean SustBucketFailed $[14] output;

script dual_tbucket {

  delta_t = (MAX_RTC_TIME - last_pdu_arrival) + current_time;
  if (current_time >= last_pdu_arrival) {
    delta_t = current_time - last_pdu_arrival;
  }

  pdu_peak_len_t = pdu_length*PBR;
  pdu_sust_len_t = pdu_length*SBR;


  /* update first bucket */
  p_tokens = p_tokens + delta_t;
  if (p_tokens > PBS) {
    p_tokens = PBS;
  }

  /* update second bucket */
  s_tokens = s_tokens + delta_t;
  if (s_tokens > SBS) {
    s_tokens = SBS;
  }

  /* assign color bits (account for pre-coloring) */
  SustBucketFailed = (s_tokens < pdu_sust_len_t)||((DSCP_CU&0x10)==0x10);
  PeakBucketFailed = (p_tokens < pdu_peak_len_t)||((DSCP_CU&0x18)==0x18);

  if (!PeakBucketFailed) {
    /* update peak bucket */
    p_tokens = p_tokens - pdu_peak_len_t;
  }

  if (!SustBucketFailed) {
    /* update sustained bucket */
    s_tokens = s_tokens - pdu_sust_len_t;
  }

  /* timestamp */
  last_pdu_arrival = current_time;
}
