/* ds_host_iface.c - external host interface for DiffServ example	*/

/*
 * Load configuration file and provide command interface for host
 */

#include <agere_np5.h>
#include <stdio.h>

#define MAXQUEUE 15        /* maximal queue ID number     */
#define MAX_FRAMESIZE 1514 /* maximal frame size          */
#define MAX_LINE_LEN  16   /* maximal command line length */

/* locations of limits and expense in parameters block */
#define LIMIT1_NUM  0
#define LIMIT2_NUM  1
#define LIMIT3_NUM  2
#define EXPENSE_NUM 3

/* read config function prototype */
int cfg_read_func(void *fp, char *buf, int len);

/* chipset handle definitions */
ag_chipset_t chipsetHdl;
ag_np5_t app550Hdl;
ag_rsp_t rspHdl;

#ifdef VXWORKS
static int DiffServStart(int argc, char ** argv)
#else
int main(int argc, char ** argv)
#endif
{
   int weight, queue, rc;
   char cmd_buf[MAX_LINE_LEN];
   ag_uint32_t parameters[4];
   ag_uint8_t param_nums[]={LIMIT1_NUM,LIMIT2_NUM,LIMIT3_NUM,EXPENSE_NUM};
   char * filename;
   FILE * cfg_fp;
   ag_chipset_chip_error_t chipError;

   /* get and check arguments */
   if (argc != 2) {
     fprintf(stderr,"Usage: DiffServStart <cfg file name>\n");
     exit(-1);
   }
   filename = argv[1];
   if ((cfg_fp = fopen(filename, "r")) == NULL) {
          fprintf(stderr,"Error: can not open config file %s\n",filename);
          exit(-1);
   }

   /* initialize chipset */
   rc = ag_chipset_init(0, &chipsetHdl);
   if (rc != AG_CHIPSET_ST_SUCCESS) {
          fprintf(stderr,"Error: ag_chipset_init failed\n");
          fclose(cfg_fp);
          exit(-1);
   }

   /* configure chipset */
   rc = ag_chipset_config(chipsetHdl, (ag_read_fn_t) cfg_read_func,
                           (void *) cfg_fp, NULL, NULL, &chipError);
   if (rc != AG_CHIPSET_ST_SUCCESS) {
          fprintf(stderr,"Error: ag_chipset_config failed\n");
          fclose(cfg_fp);
          exit(-1);
   }
   /* close config file */
   fclose(cfg_fp);

   /* get object-level APP550 handle */
   rc = ag_chipset_hdl_get(chipsetHdl, (char*) "APP550", 0, &app550Hdl);
   if (rc != AG_CHIPSET_ST_SUCCESS) {
       fprintf(stderr,"Error: ag_chipset_hdl_get failed\n");
       exit(-1);
   }

   /* get object-level RSP handle */
   rc = ag_rsp_hdl_get(app550Hdl, &rspHdl);
   if (rc != AG_ST_SUCCESS) {
       fprintf(stderr, "Error: ag_rsp_hdl_get failed\n");
       exit(-1);
   }

   /* start our own "shell" */
   while (1) { /* do forever */
      /* read queue ID and weight from standard input */
      fprintf(stdout,"Enter Queue ID and Weight (separated by space): ");
      if ( fgets(cmd_buf,MAX_LINE_LEN,stdin) == NULL )
            exit(0); /* EOF encountered */
      queue  = atoi(strtok(cmd_buf," \n"));
      weight = atoi(strtok(NULL," \n"));
      if (queue < 1 || queue > MAXQUEUE) {
        fprintf(stderr,"Error: invalid queue number: %i\n", queue);
        continue;
      }
      if (weight < 1) {
         fprintf(stderr,"Error: invalid weight value: %i\n", weight);
         continue;
      }

      /* compute new limits */
      parameters[0] = MAX_FRAMESIZE*weight/3; /* limit1      */
      parameters[1] = parameters[0]*2;        /* limit2      */
      parameters[2] = parameters[0]*3;        /* limit3      */
      parameters[3] = 0;                      /* expense = 0 */

      /* update parameters */
      rc=ag_rsp_qid_unatomic_replace_ts_parms_words(rspHdl, queue,
                                          parameters, param_nums,4);
      switch (rc) {
        case AG_ST_SUCCESS:
          /* success */
          fprintf(stdout,"Weight for queue %i set to %i\n",queue,weight);
          break;
        case AG_ST_RSP_QID_INVALID:
          fprintf(stderr, "Error: QID %i is invalid\n",queue);
          continue;
        case AG_ST_RSP_QID_NOT_EXIST:
          fprintf(stderr, "Error: QID %i does not exist\n",queue);
          continue;
        case AG_ST_RSP_INVALID_HANDLE:
          fprintf(stderr, "Error: invalid RSP handle\n");
          exit(-1);
        default:
          fprintf(stderr, "Error: invalid return code %i\n",rc);
          exit(-1);
      }
   }
   exit(0);
}

/* function to read configuration file */
int cfg_read_func(void *fp, char *buf, int len) {
    int nread;
    if ((nread=fread(buf, 1, len, (FILE *)fp)) < len) {
      if (ferror((FILE *)fp)) {
           fprintf(stderr,"Error: cfg_read_func failed reading\n");
           return -1;
      }
    }
    return nread;
}
