/*
 * ==============================================================
 * matlab_bgl_sp_mex.c The mex interface to the matlab bgl wrapper.
 *
 * David Gleich
 * 20 April 20020
 * =============================================================
 */

#include "mex.h"

#include "matlab_bgl.h"

#include <math.h>
#include <stdlib.h>

void expand_to_double(int* src, double* dst, int len, double offset)
{
    int i;
    for (i = len-1; i>=0; i--)
    {
        dst[i] = (double)src[i] + offset;
    }
}

/*
 * The mex function runs a shortest path problem.
 */
void mexFunction(int nlhs, mxArray *plhs[],
                 int nrhs, const mxArray *prhs[])
{
    int i,j,k;
    
    int mrows, ncols;
    
    int n,nz;
    
    /* sparse matrix */
    int *ia, *ja;
    double *a;
    
    /* source/sink */
    int u;
    
    double dinf;
    
    /* output data */
    double *d, *pred;
    int *int_pred;
    
    /* sp type string */
    int buflen;
    char *algname;
    int status;
     
    
    if (nrhs != 4) 
    {
        mexErrMsgTxt("4 inputs required.");
    }

    /* The first input must be a sparse matrix. */
    mrows = mxGetM(prhs[0]);
    ncols = mxGetN(prhs[0]);
    if (mrows != ncols ||
        !mxIsSparse(prhs[0]) ||
        !mxIsDouble(prhs[0]) || 
        mxIsComplex(prhs[0])) 
    {
        mexErrMsgTxt("Input must be a noncomplex square sparse matrix.");
    }
    
    n = mrows;
         
    
    /* Get the sparse matrix */
    
    /* recall that we've transposed the matrix */
    ja = mxGetIr(prhs[0]);
    ia = mxGetJc(prhs[0]);
    a = mxGetPr(prhs[0]);
    
    nz = ia[n];
    
    /* Get the scalar */
    u = (int)mxGetScalar(prhs[1]);
    u = u-1;
    
    /* Get the uninitialized value */
    dinf = mxGetScalar(prhs[3]);
    
    /* Get the algorithm type */
    
    if (mxIsChar(prhs[2]) != 1)
        mexErrMsgTxt("Input 3 must be a string (algname).");
    
    /* Input must be a row vector. */
    if (mxGetM(prhs[2]) != 1)
        mexErrMsgTxt("Input 3 must be a row vector.");
    
    /* Get the length of the input string. */
    buflen = (mxGetM(prhs[2]) * mxGetN(prhs[2])) + 1;

    /* Allocate memory for input and output strings. */
    algname = mxCalloc(buflen, sizeof(char));

    status = mxGetString(prhs[2], algname, buflen);
    if (status != 0) 
        mexErrMsgTxt("Not enough space for algname input.");
    
    plhs[0] = mxCreateDoubleMatrix(n,1,mxREAL);
    plhs[1] = mxCreateDoubleMatrix(1,n,mxREAL);
    
    /* create the output vectors */
    
    d = mxGetPr(plhs[0]);
    pred = mxGetPr(plhs[1]);
    
    
    if (strcmp(algname, "dijkstra") == 0)
    {
        dijkstra_sp(n, ja, ia, a,
            u,
            d, (int*)pred, dinf);
    }
    else if (strcmp(algname, "bellman_ford") == 0)
    {
        bellman_ford_sp(n, ja, ia, a,
            u,
            d, (int*)pred, dinf);
    }
    else if (strcmp(algname, "dag") == 0)
    {
        dag_sp(n, ja, ia, a,
            u,
            d, (int*)pred, dinf);
    }
    else
    {
        mexErrMsgTxt("Unknown algname.");
    }
    
    int_pred = (int*)pred;
    for (i=0; i<n;i++)
    {
        if (int_pred[i] == i) { int_pred[i] = -1; }
    }
    
    expand_to_double((int*)pred, pred, n, 1.0);
   
}



