#ifndef MATLAB_BGL_H
#define MATLAB_BGL_H

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */


/**
 * Wrap the boost graph library class for a push relabel max flow computation.
 *
 * the ja and ia arrays specify the connectivity of the underlying graph,
 * ia is a length (nverts+1) array with the indices in ja that start the
 * nonzeros in each row.  ja is a length (ia(nverts)) array with the 
 * columns of the connectivity.  
 *
 * the connectivity graph must be symmetric, that is, for each edge (i,j)
 * the edge (j,i) must exist in the connectivity graph.  
 *
 * @param nverts the number of vertices in the graph
 * @param ja the connectivity for each vertex
 * @param ia the row connectivity points into ja
 * @param src the source vertex for the flow
 * @param sink the sink vertex for the flow
 * @param cap the array of capacities for each edge
 * @param res the array of residual capacities for each edge
 * @param rev_edge_index an array indicating the index of the reverse edge 
 * for the edge with the current index
 * @param flow the maximum flow in the graph
 * @return an error code if possible
 */                  
int push_relabel_max_flow(
    int nverts, int *ja, int *ia,  /* connectivity params */
    int src, int sink, /* flow data */
    int* cap, int* res, /* capacity and residual capacity */
    int* rev_edge_index, int *flow);

/**
 * Wrap a boost graph library call to bfs.
 *
 * the ja and ia arrays specify the connectivity of the underlying graph,
 * ia is a length (nverts+1) array with the indices in ja that start the
 * nonzeros in each row.  ja is a length (ia(nverts)) array with the 
 * columns of the connectivity.  
 *
 * if d, dt, or pred is NULL, then that parameter is not computed.
 *   (currently not implemented)
 *
 * @param nverts the number of vertices in the graph
 * @param ja the connectivity for each vertex
 * @param ia the row connectivity points into ja
 * @param src the source vertex for the flow
 * @param d the distance array
 * @param dt the discover time array
 * @param pred the predecessor array
 * @return an error code if possible
 */
int breadth_first_search(
    int nverts, int *ja, int *ia, /* connectivity params */
    int src, /* problem data */
    int* d, int* dt, int* pred /* output data: distance, discover time, predecessor */
    );

/**
 * Wrap a boost graph library call to dfs.
 *
 * the ja and ia arrays specify the connectivity of the underlying graph,
 * ia is a length (nverts+1) array with the indices in ja that start the
 * nonzeros in each row.  ja is a length (ia(nverts)) array with the 
 * columns of the connectivity.  
 *
 * if dt, ft, or pred is NULL, then that parameter is not computed.
 *   (currently not implemented)
 *
 * @param nverts the number of vertices in the graph
 * @param ja the connectivity for each vertex
 * @param ia the row connectivity points into ja
 * @param src the source vertex for the flow
 * @param full if full is non-zero, then compute the full dfs over all
 *    vertices, not just the connected component.
 * @param d
 * @param dt the discover time array
 * @param ft the finish time array
 * @param pred the predecessor array
 * @return an error code if possible
 */
int depth_first_search(
    int nverts, int *ja, int *ia, /* connectivity params */
    int src, int full, /* problem data */
    int* d, int* dt, int* ft, int *pred /* output data: discover time, finish time, predecessor */
    );

/**
 * Wrap a boost graph library call to biconnected_components.
 *
 * the ja and ia arrays specify the connectivity of the underlying graph,
 * ia is a length (nverts+1) array with the indices in ja that start the
 * nonzeros in each row.  ja is a length (ia(nverts)) array with the 
 * columns of the connectivity.  
 *
 * if a or ci is NULL, then that parameter is not computed.
 *
 * @param nverts the number of vertices in the graph
 * @param ja the connectivity for each vertex
 * @param ia the row connectivity points into ja
 * @param a an array which will store the articulaion points of the graph 
 *     the array length should be n
 * @param ci the component index array which is length (nnz)
 * @return an error code if possible
 */

int strong_components(
    int nverts, int *ja, int *ia, /* connectivity params */
    int* ci);

int biconnected_components(
    int nverts, int *ja, int *ia, /* connectivity params */
    int* a, int* ci);

int dijkstra_sp(
    int nverts, int *ja, int *ia, double *weight, /* connectivity params */
    int src, /* problem data */
    double* d, int *pred, double dinf);

int bellman_ford_sp(
    int nverts, int *ja, int *ia, double *weight, /* connectivity params */
    int src, /* problem data */
    double* d, int *pred, double dinf);

int dag_sp(
    int nverts, int *ja, int *ia, double *weight, /* connectivity params */
    int src, /* problem data */
    double* d, int *pred, double dinf);

int johnson_all_sp(
    int nverts, int *ja, int *ia, double *weight, /* connectivity params */
    double* D, double dinf);

int floyd_warshall_all_sp(
    int nverts, int *ja, int *ia, double *weight, /* connectivity params */
    double* D, double dinf);

int kruskal_mst(
    int nverts, int *ja, int *ia, double *weight, /* connectivity params */
    int* i, int* j, double* val, int* nedges /* tree output */);

int prim_mst(
    int nverts, int *ja, int *ia, double *weight, /* connectivity params */
    int* i, int* j, double* val, int* nedges /* tree output */);

/**
 * If weight == NULL, then the algorithm is used is slightly more
 * efficient, so if you don't want the weighted version, explicitly
 * set weight to NULL
 */
int betweenness_centrality(
    int nverts, int *ja, int *ia, double *weight, /* connectivity params */
    double *centrality);

int clustering_coefficients(
    int nverts, int *ja, int *ia, /* connectivity params */
    double *ccoeffs);

int reverse_cuthill_mckee_order(
    int nverts, int *ja, int *ia, /* connectivity params */
    int start, /* input */
    int *perm /* permutation output */);

int king_order(
    int nverts, int *ja, int *ia, /* connectivity params */
    int *perm /* permutation output */);

int minimum_degree_order(
    int nverts, int *ja, int *ia, /* connectivity params */
    int *perm /* permutation output */);

int sloan_order(
    int nverts, int *ja, int *ia, /* connectivity params */
    int *perm /* permutation output */);


#ifdef __cplusplus
}
#endif /* __cplusplus */



#endif /* MATLAB_BGL_H */


