%% Results for top-k in the Katz measure

%% Evaluate the accuracy of our push-approximation to the F-measure

addpath('../../matlab');
addpath('../../tools');
addpath('../');

%% load a graph
if ~exist('graphname','var') || ~exist('A','var')
    error('need to load a graph first another function');
end
if ~exist('katz_alpha','var') || ~exist('katz_alpha_name','var')
    error('need to set Katz alpha values');
end
%% normalize the graph
A = A|A';
A = A - diag(diag(A));
deg = sum(A,1)'; 
n = size(A,1);
B = speye(n) - katz_alpha*A;

%% Pick a vertex
vert = pairs_conv(2);

%% Generate a result filename
rfilename = sprintf('accuracy-results-%s-katz-%s-%i',graphname,katz_alpha_name,vert);

%% Solve the system with minres
ei = zeros(n,1);
ei(vert) = 1;
[katz,flag,relres,iter,resvec] = minres(B,ei,1e-8,5000);
if flag ~= 0,
    error('minres did not converge');
end
katz(vert) = 0;
[ignore korder] = sort(-katz);
%% Solve the system with our push operator
nsteps = [100,1000,2500,5000,10000,0.1*n, 0.5*n n,2*n,5*n,10*n];
ks = [10,25,100,1000];
nsteps = sort(nsteps);
results = [];
for ni=1:length(nsteps)
    %%
    nstep = nsteps(ni);
    [kp,r,hist,nverts] = katz_push_mex(A,katz_alpha,vert,0,nstep);
    kp(vert) = 0;
    [ignore kporder] = sort(-kp);
    results(ni).diff = norm(katz-kp);
    results(ni).ktau = ktau(katz,kp);
    ktauk = zeros(length(ks),1);
    overlapk = ktauk; % copy the zero values
    for ki=1:length(ks);
        k = ks(ki);
        ktauk(ki) = ktau(kp(korder(1:k)),katz(korder(1:k)));
        overlapk(ki) = length(intersect(korder(1:k),kporder(1:k)))./k;
    end
    results(ni).ktauk = ktauk;
    results(ni).overlapk = overlapk;
    
    results(ni).kp = kp;
    results(ni).r = r;
    results(ni).hist = hist;
    results(ni).nverts = nverts;
    results(ni).nedges = hist(1,end);
    clf; hold on; 
	allktau = [results(1:ni).ktauk];
    alloverlap = [results(1:ni).overlapk];
    for ki=1:length(ks);
        semilogx([results(1:ni).nedges]/nnz(A),allktau(ki,:),'.-');
        semilogx([results(1:ni).nedges]/nnz(A),alloverlap(ki,:),'r.-');
    end
    set(gca,'XScale','log');
    drawnow;
end

save(rfilename,'results','ks','nsteps','vert');

%% Use CG for baseline
nmults=[2 5 10 15 25 50];
cgresults =[];
for ni=1:length(nmults)
    nmult=nmults(ni);
    kcg = pcg(B,ei,0,nmult);
    kcg(vert) = 0;
    [ignore kcgorder] = sort(-kcg);
    cgresult.diff = norm(katz-kcg);
    cgresult.ktau = ktau(katz,kcg);
    % only show results for k=25
    k = 25;
    ktauk = ktau(kcg(korder(1:k)),katz(korder(1:k)));
    overlapk = length(intersect(korder(1:k),kcgorder(1:k)))./k;
    cgresult.ktauk = ktauk;
    cgresult.overlapk = overlapk;

    cgresult.kcg = kcg;
    cgresult.nverts = n;
    cgresult.nedges = nmult*nnz(A);
    if ni==1
        cgresults=cgresult;
    else
        cgresults(ni)=cgresult;
    end
end
    
%%
save(rfilename,...
    'results','cgresults','ks','nsteps','nmults','vert');

%% Load the data we just saved
load(rfilename);

%% Plot the convergence in the ordering

setup_plot;

clf; hold on;
allktau = [results(1:length(results)).ktauk];
alloverlap = [results(1:length(results)).overlapk];

xs = [results(1:length(results)).nedges]/nnz(A);
legtexts = {};
hs=[];
styles = {'.-','.--','.-.','.:'};
for ki=1:length(ks);
    hs(ki) = semilogx(xs,allktau(ki,:),styles{ki},'MarkerSize',14,...
        'Color',0.5*[1,1,1],'linewidth',1.1);
    % againth, the space is important'
    %text(xs(1),alloverlap(ki,1),sprintf('k=%i ',ks(ki)),'HorizontalAlignment','right');
    legtexts{ki} = sprintf('k=%i ',ks(ki));
end
% double-print k=25
hs(end+1) = semilogx([cgresults.nedges]/nnz(A),[cgresults.ktauk],'ko-');
legtexts{end+1} = 'cg k=25 ';
ki=2;
hs(end+1) = semilogx(xs,allktau(ki,:),styles{ki},'MarkerSize',14,'Color','r');
legtexts{end+1} = sprintf('k=%i ',ks(ki));
legtexts(ki)=[];
hs(ki)=[];

ylim([-0.1,1.1]);
xlim([1e-3,1e3]);
lh = legend(hs,legtexts{:},'Location','SE');
legend boxoff;
set(lh,'FontSize',18);
set(gca,'XScale','log');
xlabel('Equivalent matrix-vector products');
ylabel('Kendtall-\tau ordering vs. exact');

%%
box on;
greygrid;
set(gca,'Color','none'), set(gcf,'Color','none');
set(gcf, 'InvertHardCopy','off');

print(gcf,sprintf('katz-%s-%s-topk-ktau-%i.eps',katz_alpha_name,graphname,vert),'-depsc2');

%% Plot the convergence in the sets
clf; hold on;
alloverlap = [results(1:length(results)).overlapk];
xs = [results(1:length(results)).nedges]/nnz(A);
legtexts = {};
hs=[];
styles = {'.-','.--','.-.','.:'};
for ki=1:length(ks);
    hs(ki) = semilogx(xs,alloverlap(ki,:),styles{ki},'MarkerSize',14,...
        'Color',0.5*[1,1,1],'linewidth',1.1);
    % againth, the space is important'
    %text(xs(1),alloverlap(ki,1),sprintf('k=%i ',ks(ki)),'HorizontalAlignment','right');
    legtexts{ki} = sprintf('k=%i ',ks(ki));
end
% double-print k=25
hs(end+1) = semilogx([cgresults.nedges]/nnz(A),[cgresults.overlapk],'ko-');
legtexts{end+1} = 'cg k=25 ';
ki=2;
hs(end+1) = semilogx(xs,alloverlap(ki,:),styles{ki},'MarkerSize',14,'Color','r');
legtexts{end+1} = sprintf('k=%i ',ks(ki));
legtexts(ki)=[];
hs(ki)=[];
set(gca,'XScale','log');
ylim([-0.1,1.1]);
xlim([1e-3,1e3]);
lh = legend(hs,legtexts{:},'Location','SE');
legend boxoff;
set(lh,'FontSize',18);

xlabel('Equivalent matrix-vector products');
ylabel('Precision@k for exact top-k sets');

%%
box on;
greygrid;
set(gca,'Color','none'), set(gcf,'Color','none');
set(gcf, 'InvertHardCopy','off');
print(gcf,sprintf('katz-%s-%s-topk-overlap-%i.eps',katz_alpha_name,graphname,vert),'-depsc2');