%% Experimental sweeps for top-k algorithms and Katz's rankings
% We have a local-graph based algorithm to approximate the largest Katz
% scores for a particular row/column of the Katz matrix.  In this
% experiment, we investigate the accuracy and scalability of the algorithm
% on the datasets for our paper.

%% Basic setup
% First add information to the paths
addpath('../../matlab'); % add the directory with the katz_push algorithm

%%
% Set the graphs
graphs = {'flickr2','dblp','arxiv'};
datadir = '../../data';

%%
% Set the values of alpha for Katz
alphas = [0.0005, 0.005, 0.01];

%%
% Set the number of trials
ntrials = 50;

%% 
% Set a random seed
seed = 0;

%%
% Set a tolerance
tol = 1e-4;
maxsteps = 50000;

%% Run the experimental sweep

results = [];
for gi=1:length(graphs)
    graph = graphs{gi};
    graphfile = fullfile(datadir, [graph '.smat']);
    A = readSMAT(graphfile);
    n = size(A,1);
    % precompute the csr form of A
    [rp,ci] = sparse_to_csr(A);
    As.rp = rp; As.ci = ci;
    for ai=1:length(alphas)
        alpha = alphas(ai);
        
        % reset the random generator
        rand('state',seed);
        verts = randperm(n);
        
        for ti=1:ntrials
            % pick a vertex 
            vertex = verts(ti);
            result = [];
            result.graph = graph;
            result.alpha = alpha;
            result.trial = ti;
            result.vertex = vertex;
            result.tol = tol;
            st=tic;
            [x,r,hist,nverts]=katz_push(As,alpha,vertex,tol,maxsteps);
            dt=toc(st);
            result.time = dt;
            result.nverts = nverts;
            result.hist = hist;
            
            fprintf('%10s  %6f  %5i  %6.1f (s)  %5i  %8.2e\n',...
                graph, alpha, vertex, dt, nverts, norm(r));
            
            if isempty(results), results = result; 
            else  results(end+1) = result;
            end
        end
    end
end
