%% Examine the number of vertices explored by the Katz push algorithm
% While the local-push algorithm for Katz is running, it only accesses a
% small number of vertices.  In this experiment, we explore just how many
% vertices it needs to access, to see if there are any interesting
% patterns.

%% Basic setup
% First add information to the paths
addpath('../../matlab'); % add the directory with the katz_push algorithm

%%
% Set the graphs
graph = 'dblp';
datadir = '../../data';
%%
% Set the values of alpha for Katz
alpha = 0.01;

%%
% Set the number of trials
ntrials = 5000;

%% 
% Set a random seed
seed = 0;

%%
% Set a tolerance
tol = 1e-7;
maxsteps = 500000;

%% Run the experimental sweep

results = [];
graphfile = fullfile(datadir, [graph '.smat']);
A = readSMAT(graphfile);
A = spones(A);
n = size(A,1);
% precompute the csr form of A
[rp,ci] = sparse_to_csr(A);
As.rp = rp; As.ci = ci;
rand('state',seed);
verts = randperm(n);
for ti=1:ntrials
    % pick a vertex 
    vertex = verts(ti);
    result = [];
    result.graph = graph;
    result.alpha = alpha;
    result.trial = ti;
    result.vertex = vertex;
    result.tol = tol;
    st=tic;
    [x,r,hist,nverts]=katz_push(As,alpha,vertex,tol,maxsteps);
    dt=toc(st);
    result.time = dt;
    result.nverts = nverts;
    result.hist = hist;

    fprintf('%10s  %6f  %5i  %6.1f (s)  %5i  %8.2e\n',...
        graph, alpha, vertex, dt, nverts, norm(r));

    if isempty(results), results = result; 
    else  results(end+1) = result;
    end
end

%%
% Show a histogram of number of vertices explored
clear hist
hist([results.nverts])

%% 
% Show a histogram of the number of steps
nsteps = [];
for i=1:length(results)
    nsteps(i) = length(results(i).hist);
end
hist(nsteps);
