%% Experimental sweeps for top-k algorithms and pinv(L) rankings
% We have a local-graph based algorithm to approximate the largest Katz
% scores for a particular row/column of the pinv(L) matrix.  In this
% experiment, we investigate the accuracy and scalability of the algorithm
% on the datasets for our paper.

%% Basic setup
% First add information to the paths
addpath('../../matlab'); % add the directory with the katz_push algorithm

%%
% Set the graphs
graphs = {'dblp','arxiv','flickr2'};
datadir = '../../data';

%%
% Set the number of trials
ntrials = 50;

%% 
% Set a random seed
seed = 0;

%%
% Set a tolerance
tol = 1e-4;
maxsteps = 250000;

%% Run the experimental sweep

results = [];
for gi=1:length(graphs)
    graph = graphs{gi};
    graphfile = fullfile(datadir, [graph '.smat']);
    A = readSMAT(graphfile);
    n = size(A,1);
    L = diag(sum(A)) - A;
    % precompute the csr form of A
    [rp,ci,ai] = sparse_to_csr(L);
    As.rp = rp; As.ci = ci; As.ai=ai;

    % reset the random generator
    rand('state',seed);
    verts = randperm(n);
        
    for ti=1:ntrials
        % pick a vertex 
        vertex = verts(ti);
        result = [];
        result.graph = graph;
        result.trial = ti;
        result.vertex = vertex;
        result.tol = tol;
        st=tic;
        [x,r,hist,nverts]=Lp_push(As,vertex,[],tol,maxsteps);
        dt=toc(st);
        result.time = dt;
        result.nverts = nverts;
        result.hist = hist;

        fprintf('%10s  %5i  %6.1f (s)  %5i  %6i %8.2e\n',...
            graph, vertex, dt, nverts, length(hist), norm(r));

        if isempty(results), results = result; 
        else  results(end+1) = result;
        end
    end
end
