%% Poisson's equation with periodic boundary conditions
% Consider the Poisson problem with periodic boundary conditions
% 
% $$ -\Delta u(x,y) = f(x,y) ; u(x,0) = u(x,1), u(0,y) = u(1,y) = 0 $$
%
% We approximate $u$ on a periodic mesh:
%  $$ U(i,j) \approx u(x_i, y_j) $$
% where $x_1 = 0, x_1 = h, \ldots, x_{n} = 1 - h$, (and the same for $y$)
% thus, we end up with the matrix equation
%   -U(i-1,j) -U(i,j-1) + 4*U(i,j) - U(i+1,j) - U(i,j+1) = h^2 f(x_i,y_j),
% which can be written 
%   $A \vec(U) = \vec(f)$
% where $A$ is a matrix of circulant blocks.
%
%

n = 50;
h = 1/n;

x = h:h:1-h;
y = 0:h:1-h;

N = length(x)*length(y);


%% Construct A
% construct A by arranging in groups with equal values of x.
% this will product a matrix of circulant blocks

A = zeros(N,N);

%f = @(x,y) (1-x)*(1-y);
f = @(x,y) 0;

b = zeros(N,1);

pos = @(i,j) (n)*(i-1)+(mod(j,n)+1);
for i=1:n-1
    for j=0:n-1
        % at position i,j in the grid, we depend on all adjacency positions
        % first compute the row number
        row = pos(i,j);
        if (i>1)
            A(row,pos(i-1,j)) = -1;
        end
        if (i<n-1) 
            A(row,pos(i+1,j)) = -1;
        end
        A(row,row) = 4;
        A(row,pos(i,j+1)) = -1;
        A(row,pos(i,j-1)) = -1;
        % lookup xy coordinates
        xi = x(i); 
        yj = y(j+1); 
        b(row) = f(xi,yj);
    end
end

b(pos(ceil(n/2),2)) = 1;

b = b*h^2;
u = A\b;
mesh(x,y,reshape(u,n,n-1))

%% 
CA = uncirc(A,n);
Cb = unvec(b,n);

%% Construct this directly as a circulant
h = 1/n;

x = h:h:1-h;
y = 0:h:1-h;

N = length(x)*length(y);
%%
%
A = cazeros(n-1,n-1,n);
cv = zeros(n,1); cv(1) = -1; cone = cascalar(cv);
cv = zeros(n,1); cv(1) = 4; cv(2) = -1; cv(end) = -1; c = cascalar(cv);
for i=1:size(A,1)
    A(i,i) = c;
    if i>1, A(i,i-1) = cone; end
    if i<size(A,1), A(i,i+1) = cone; end
end
b = cazeros(n-1,1,n);
bvec = zeros(n,1); bvec(3)=1*h^2;
b(floor(n/2)) = cascalar(bvec);

%%
% Check that we got it right
mag(norm(CA-A))
mag(norm(Cb-b))

%% Run the power method
randn('state',0);
[x1,l1,hist,histl] = cpowermtd(A,[],'maxiter',10000);
l1
%%
% Double check with exact diagonalization
[V,D] = eig(A);
mag(norm(A*V - V*D))
d1 = D(1,1)
%%
% Plot convergence
figure(1); semilogy(hist,'.-'); xlabel('iteration'); ylabel('residual');
%%
% plot error
histe = zeros(length(hist),size(A,3));
for i=1:length(hist)
    histe(i,:) = abs(fft(data(histl(i)-d1)));
end
figure(2); semilogy(histe,'k-'); xlabel('iteration'); ylabel('component error');
ylim([1e-15,1]);

%%
% get the convergence ratios
[V,D] = eig(A);
d1 = data(D(1,1));
d2 = data(D(2,2));
id1 = fft(d1(:));
id2 = fft(d2(:));
ratios = abs(id2)./abs(id1);
slowest = (6+2*cos(pi/n))/(6+2*cos(2*pi/n));
fastest = (2+2*cos(pi/n))/(2+2*cos(2*pi/n));

%%
% nice error plot with convergence ratios

figure(2); clf; 
% only plot 1/2 the components because the others mirror them
h1=semilogy(histe(:,1:ceil(n/2)),'-','LineWidth',1,'Color',0.55*[1,1,1]); 
hold on;
xx = 1:length(hist);
semilogy(0.2*slowest.^(2*xx),'-','LineWidth',1.4);
semilogy(0.0005*fastest.^(2*xx),'-','LineWidth',1.4);
text(500,1e-12,...
    '$$\left(\frac{6+2\cos(\pi/n)}{6+2\cos(\pi/n)}\right)^{2i}$$',...
    'interpreter','latex');
text(500,1e-12,...
    '$$\left(\frac{2+2\cos(\pi/n)}{2+2\cos(\pi/n)}\right)^{2i}$$',...
    'interpreter','latex','color','b');
%text('\sim\left(\frac{2+2cos(\pi/n)}{2+2cos(\pi/n)}\right)^{2i}');
%text('\sim\left(\frac{6+2cos(\pi/n)}{6+2cos(\pi/n)}\right)^{i}');

h2=semilogy(hist,'r-','LineWidth',2.5);
set(gca,'LineWidth',1.4,'FontSize',18);
axis square;
xlabel('iteration'); ylabel('magnitude');
xlim([1,length(hist)])
ylim([1e-15,1]); box off;
legend([h1(1),h2],'Eigenvalue Error','Eigenvector Change');
legend boxoff;




%axes('position',[0.65 0.65 0.2 0.2]);set(gca,'LineWidth',1.2,'FontSize',15);
%semilogy(hist,'k-'); ylim([1e-10,1]);
%box off; ylabel('convergence criteria'); xlabel('iteration');

%% 
% save output
print(gcf,'eig-components.eps','-depsc2');
%% Run the Arnoldi method
ns = 50;
[Q,H] = carnoldi(A,b,ns);
x = A\b;
%% 
% Check it's accuracy
hist = cazeros(ns,1,size(A,3));
for i=3:ns
    eone = cazeros(i+1,1,size(A,3));
    eone(1) = 1;
    Hi = H(1:i+1,1:i);
    xa = norm(b)*Q(:,1:i)*(Hi\eone);
    hist(i) = norm(x-xa);
end
%%
histm = zeros(size(hist,1),1);
histc = zeros(size(hist,1),size(A,3));
for i=1:ns
    histm(i) = mag(hist(i));
    histc(i,:) = abs(fft(data(hist(i))));
end
figure(1); semilogy(3:ns, histm(3:ns),'.-'); xlabel('arnoldi iteration'); ylabel('error');
print(gcf,sprintf('total-error-%i.png',n),'-dpng');
figure(2); semilogy(3:ns, histc(3:ns,:),'.-'); xlabel('arnoldi iteration'); ylabel('component error');
print(gcf,sprintf('indiv-error-%i.png',n),'-dpng');
