/*
 * circmult_mex.c
 * ==============
 *
 * Implement sparse circulant matrix-by-vector multiplication.
 *
 * To compile
 * ----------
 *
 * mex -O -largeArrayDims CFLAGS="\$CFLAGS -Wall -std=c99" circmult_mex.c
 */

#include <mex.h>

/* Compute the product y=circ(a)*x where A is a sparse circulant
 *
 * for this function, the circ is defined by the first row.
 *
 * @param k the size of the circulant and x and y
 * @param nz the number of entries in the sparse circulant circ(a), which is 
 *   defined by a sparse vector a=sparse(ai(1:nz), 1, av(1:nz),k,1);
 * @param ai the non-zero entries of the sparse circulant
 * @param av the values of the non-zeros entries 
 * @param x the right hand side vector
 * @param y the output vector
 */
void sparse_circ_mult_1(
        mwSize k, 
        mwSize nz, mwIndex* ai, double* av,
        const double* x, double* y)
{
    /* for each row */
    for (mwIndex i=0; i<k; i++) {
        double yi=0.;
        /* for each non-zero */
        for (mwIndex nzi=0; nzi<nz; nzi++) {
            yi += av[nzi]*x[(ai[nzi]+i)%k];
        }
        y[i] = yi;
    }
}

void sparse_circ_mult_2(
        mwSize k, 
        mwSize nz, mwIndex* ai, double* av,
        const double* x, double* y)
{
    /* for each non-zero */
    for (mwIndex nzi=0; nzi<nz; nzi++) {
        mwIndex j = ai[nzi];
        double a = av[nzi];
        for (mwIndex i=0; i<k; i++) {
            y[i] += a*x[(i+j)%k];
        }
    }
}

void mexFunction(int nargout, mxArray* pargout[],
                 int nargin, const mxArray* pargin[])
{
    const mxArray *sparse_a = pargin[0];
    const mxArray *x = pargin[1];
    mxArray *y;
    int type=1;
    mwIndex *ai;
    double *av;
    mwSize k = mxGetNumberOfElements(x);
    mwSize nz;

    if (nargin>2) {
        type = (int)mxGetScalar(pargin[2]);
    }

    /* parameter checking */
    if (!mxIsSparse(sparse_a)) {
        mexErrMsgTxt("Input 1 must be a sparse vector");
    }

    if (!mxIsDouble(sparse_a) || !mxIsDouble(x)) {
        mexErrMsgTxt("Inputs 1 and 2 must be real");
    } 

    if (mxGetM(sparse_a) > k) {
        mexErrMsgTxt("size(a,1) must be <= numel(x)");
    }

    if (mxGetN(sparse_a) != 1) {
        mexErrMsgTxt("size(a,2) must be 1");
    }

    av = mxGetPr(sparse_a);
    ai = mxGetIr(sparse_a);
    nz = (mwSize)(mxGetJc(sparse_a)[1]);

    /*mexPrintf("k=%Zi, nz=%Zi\n", k, nz);*/

    pargout[0] = mxCreateDoubleMatrix(k,1,mxREAL);
    y = pargout[0];

    switch (type) {
    case 1:
        sparse_circ_mult_1(k, nz, ai, av, mxGetPr(x), mxGetPr(y));
        break;

    case 2:
        sparse_circ_mult_2(k, nz, ai, av, mxGetPr(x), mxGetPr(y));
        break;

    default:
        mexErrMsgTxt("type is unknown");
        break;
    }
}
