function [V,H,f] = arnoldi(A,V,k,varargin);
% ARNOLDI Compute a k-step arnoldi factorization of a matrix A.
%
% [V,H] = arnoldi(A,V,k) computes an n x k+1 factorization of the matrix A
%   such that A*V(:,1:k) = V*H, where V is n x k+1, and H is k x k+1.
% [V,H,f] = arnoldi(A,V,k,'r') computes a n x k factorization of the 
%   matrix A such that A*V = V*H + e_k*f'
%
% The matrix A can be a dense, sparse, or an operator.
%
% There are multiple driver modes for this call.  These are called by
% providing optional parameters.  
%
% arnoldi(A,v,k)          -- generate a new Arnoldi sequence from v
% arnoldi(A,V,k,H)        -- extend a compact Arnoldi factorization V,H
% arnoldi(A,v,k,mode)     -- generate a new Arnoldi sequence from v
% arnoldi(A,V,k,H,mode)   -- extend a compact fatorization V,H (see note 1) 
% arnoldi(A,V,k,H,f,mode) -- extend a residual factorization V,H,f 
%
% mode just specifies the way results are returned, i.e., if H is returned 
% in compact mode (the default) or in residual mode.
%
% Result Choice
%   'c' : compact mode (default) (H is k x k+1, V is n x k+1)
%   'r' : residual mode, return the residual vector f at termination
%         instead of extending the factorization as in compact mode.
%
% Note 1: In the call arnoldi(A,V,k,H), if H is a square matrix, then we
% can still update the factorization, but at significant cost because we
% must compute the residual vector A*V - V*H explicitly.  While this is
% possible and implemented, this mode should be avoided.  Calling this
% driver throws a performance warning.
%
% The matrix A must be an input that is a valid singleton input to the
% matrixop class or a matrixop class.  See matrixop for more details.
%
% Example:
%   % compute the basis for one step of GMRES10
%   [V H] = arnoldi(A,b,10);
%
% See also LANCZOS, LANCZOSUB, LANCZOSLB

%
% David Gleich
% Stanford University
%

%
% 11 December 2006
% Updated the code to use the matrixop library.
%

% A must be an input that creates a matrix operator
if (isnumeric(A) || islogical(A))
    Aop = A;
else
    Aop = matrixop(A);
end

n = size(V,1);

extend = 0;

if (length(varargin) == 0)
    mode = 'c';
elseif (length(varargin) == 1)
    if (ischar(varargin{1}))
        mode = varargin{1};
    else
        mode = 'c';
        H = varargin{1};
        extend = 1;
    end;
elseif (length(varargin) == 2)
    H = varargin{1};
    if (ischar(varargin{2}))
        mode = varargin{2};
    else
        f = varargin{2};
        mode = 'c';
    end;
    extend = 1;
elseif (length(varargin) == 3)
    H = varargin{1};
    f = varargin{2};
    mode = varargin{3};
    extend = 1;
else
    error('invalidArgument', 'Incorrect number of parameters');
end;

if (~extend)
    v = V(:,1);
    V = zeros(n,k+1);
    
    f = v;
    beta = norm(f);
    
    ii = 0;
    start = 0;
    
    H = [];
    
    H = zeros(k+1,k);
else
    Hstart = H;
    H = Hstart;
    if (size(H,1) ~= size(H,2))
        % they provided f in V
        f = V(:,end);
        beta = Hstart(end,end);
        f = f*beta;
        start = size(V,2) - 1;
        ii = start;
        
    elseif (~exist('f','var'))
        % recall that H is square, so we need to compute the vector f.  But
        % then H has to be the same size as V.
        if (size(H,1) ~= size(V,2))
            error('arnoldi:invalidArgument',...
                'the matrices H and V must have compatible sizes (size(H,1) == size(V,2))');
        end
        
        f = Aop*(V(:,end)) - V*H(:,end);
        beta = norm(f);
        
        start = size(V,2);
        ii = start;
        
    else 
        % f exists, we dont' need to do anything!
        beta = norm(f);
        start = size(V,2);
        ii = start;
    end;
    
    H(start+k+1,start+k) = 0;
    H(start+1,start) = beta;
end

% compute the factorizations
while (ii < start + k)
    % generate the "new" start vector
    v = f/beta;
    
    % step the iteration
    ii = ii + 1;
    
    % store the output
    V(:,ii) = v;
    
    % perform most of the work
    w=Aop*(V(:,ii));  
    h=V(:,1:ii)'*w;
    f=w-V(:,1:ii)*h;
    
    % update H
    H(1:ii,ii) = h;
    
    beta=norm(f);
    H(ii+1,ii) = beta;
end

if (~isempty(strfind(mode,'c')))
    % build the extended Arnoldi factorization
    beta=norm(f);
    V(:,start+k+1) = f/beta;
else
    if (~extend)
        V = V(:,1:end-1);
    end
    H = H(1:end-1,1:end);
end;