/** 
 * Replicate a product between two dense arrays, similar to what is done
 * in R.  In other words, is array a has length p and array b has length q,
 * (q < p), then compute a(1)b(1) a(2)b(2) ... a(q)b(q) a(q+1)b(1) 
 *
 * @author David F. Gleich
 * :2009-11-25: Initial coding
 * :2009-11-30: Added support for real and complex entries
 */

#include <mex.h>

/**
 * Run simple argument checking 
 */
void mex_arg_check(int nargin, int nargout, int minin, int maxin, int maxout)
{
    if (nargin < minin) {
        mexErrMsgIdAndTxt("MATLAB:nargchk:notEnoughInputs", 
            "Not enough input arguments.");
    } else if (nargin > maxin) {
        mexErrMsgIdAndTxt("MATLAB:nargchk:tooManyInputs", 
            "Too many input arguments.");
    }
    if (maxout < nargout) {
        mexErrMsgIdAndTxt("MATLAB:nargoutchk:tooManyOutputs", 
            "Too many output arguments.");
    }
}

/**
 * Get data on a complex vector 
 */
void mex_get_complex(const mxArray* X, double **pr, double **pi) 
{
    *pr = mxGetPr(X); *pi = mxGetPi(X);
}
 

/**
 * m must be larger than n, we wrap yr around as necessary
 */
void repprod(double *xr, double *xi, mwSize m,
             double *yr, double *yi, mwSize n,
             double *zr, double *zi)
{
    mwIndex i, j;
    mxAssert(n < m, "repprod received invalid arrays");
    if (xi == NULL && yi == NULL) {
        mxAssert(zi == NULL,"complex output from real input");
        for (i=0, j=0; i<m; i++, j++) {
            if (j >= n) { j = 0; } /* reset j */
            zr[i] = xr[i]*yr[j];
        }
    } else if (xi == NULL && yi != NULL) {
        for (i=0, j=0; i<m; i++, j++) {
            if (j >= n) { j = 0; } /* reset j */
            zr[i] = xr[i]*yr[j];
            zi[i] = xr[i]*yi[j];
        }
    } else if (xi != NULL && yi == NULL) {
        for (i=0, j=0; i<m; i++, j++) {
            if (j >= n) { j = 0; } /* reset j */
            zr[i] = xr[i]*yr[j];
            zi[i] = xi[i]*yr[j];
        }
    } else {
        for (i=0, j=0; i<m; i++, j++) {
            if (j >= n) { j = 0; } /* reset j */
            zr[i] = xr[i]*yr[j] - xi[i]*yi[j];
            zi[i] = xi[i]*yr[j] + xr[i]*yi[j];
        }
    }
}

void mexFunction(int nargout, mxArray* pargout[],
                 int nargin, const mxArray* pargin[])
{
    const mxArray *x, *y;
    mxArray *z;
    mwSize m, n;
    double *xr, *xi, *yr, *yi, *zr, *zi;
    
    mex_arg_check(nargin, nargout, 2, 2, 1);
    m = mxGetNumberOfElements(pargin[0]);
    n = mxGetNumberOfElements(pargin[1]);
    if (m < n) {
        x = pargin[1]; y = pargin[0]; /* set x to be the longer array */
    } else {
        x = pargin[0]; y = pargin[1]; 
    }
    m = mxGetNumberOfElements(x);
    n = mxGetNumberOfElements(y);
    
    if (mxIsComplex(x) == mxIsComplex(y)) {
        z = mxDuplicateArray(x); /* copy the array */
    } else {
        mwSize ndim = mxGetNumberOfDimensions(x);
        z = mxCreateNumericArray(ndim, mxGetDimensions(x),
                mxDOUBLE_CLASS, mxCOMPLEX);
    }
    pargout[0] = z;
    
    mex_get_complex(x, &xr, &xi);
    mex_get_complex(y, &yr, &yi);
    mex_get_complex(z, &zr, &zi);
    
    repprod(xr, xi, m, yr, yi, n, zr, zi);
}
