#include "ppc.h"
#include "m33.h"

PPC::PPC(float hfov, int _w, int _h) {

  w = _w;
  h = _h;
  a = V3(1.0f, 0.0f, 0.0f);
  b = V3(0.0f, -1.0f, 0.0f);
  float hfovRadians = hfov / 180.0f * 3.1415f;
  c = V3((float)-w/2.0f, (float)h/2.0f, -(float)w/(2.0f*tan(hfovRadians/2.0f)));
  C = V3(0.0f, 0.0f, 0.0f);

}

bool PPC::Project(V3 P, V3& projP) {

  M33 cam;
  cam.SetColumn(0, a);
  cam.SetColumn(1, b);
  cam.SetColumn(2, c);
  V3 q = cam.Inverted() * (P-C);
  if (q[2] < 0.0f)
    return false;

  projP[0] = q[0] / q[2];
  projP[1] = q[1] / q[2];
  projP[2] = 1.0f / q[2];
  return true;

}

void PPC::Draw(float actualFocalLength, FrameBuffer *fb, PPC *ppc) {

  V3 P[4];
  P[0] = Unproject(0.0f, 0.0f, actualFocalLength);
  P[1] = Unproject((float)w, 0.0f, actualFocalLength);
  P[2] = Unproject((float)w, (float)h, actualFocalLength);
  P[3] = Unproject(0.0f, (float)h, actualFocalLength);

  V3 black(0.0f, 0.0f, 0.0f);
  V3 red(1.0f, 0.0f, 0.0f);

  for (int i = 0; i < 4; i++) {
    fb->Draw3DSegment(P[i], P[(i+1)%4], black, black, ppc);
    fb->Draw3DSegment(P[i], C, black, red, ppc);
  }

}

V3 PPC::GetVD() {

  V3 ret = (a^b).UnitVector();
  return ret;

}

float PPC::Getf() {

  return c*GetVD();

}

V3 PPC::Unproject(float uf, float vf, float distance) {

  float f = Getf();
  V3 ret = C + (c + a*uf + b*vf) * distance / f;
  return ret;

}

void PPC::Pan(float theta) {

  a = a.RotateThisVectorAboutDirection((b*-1.0f).UnitVector(), theta);
  c = c.RotateThisVectorAboutDirection((b*-1.0f).UnitVector(), theta);

}

void PPC::Roll(float theta) {

  V3 aDir = GetVD();
  a = a.RotateThisVectorAboutDirection(aDir, theta);
  b = b.RotateThisVectorAboutDirection(aDir, theta);
  c = c.RotateThisVectorAboutDirection(aDir, theta);

}

void PPC::SetByInterpolationOLD(PPC *ppc0, PPC *ppc1, float frac) {

  // assumes that ppc0 and ppc1 are the same camera except for 
  //      position and orientation

  C = ppc0->C + (ppc1->C-ppc0->C)*frac;
  a = ppc0->a + (ppc1->a-ppc0->a)*frac;
  b = ppc0->b + (ppc1->b-ppc0->b)*frac;
  c = ppc0->c + (ppc1->c-ppc0->c)*frac;

}

void PPC::SetByInterpolation(PPC *ppc0, PPC *ppc1, float frac) {

  // assumes that ppc0 and ppc1 are the same camera except for 
  //      position and orientation

  V3 newC = ppc0->C + (ppc1->C-ppc0->C)*frac;
  V3 newVD = ppc0->GetVD()+ (ppc1->GetVD()-ppc0->GetVD())*frac;
  V3 lookAtPoint = newC + newVD*100.0f;
  V3 vectorInVerticalPlane = (ppc0->b + (ppc1->b-ppc0->b)*frac)*-1.0f;

  PositionAndOrient(newC, lookAtPoint, vectorInVerticalPlane);

}


void PPC::PositionAndOrient(V3 newC, V3 lookAtPoint, V3 vectorInVerticalPlane) {

  // new a, b, c, C
  
  V3 newVD = (lookAtPoint-newC).UnitVector();
  float f = Getf();
  V3 P1 = newC + newVD*f;
  V3 newa = (newVD^vectorInVerticalPlane).UnitVector() * a.Length();
  V3 newb = (newVD^newa).UnitVector()*b.Length();
  V3 newc = (P1 - newa*(float)w/2.0f - newb*(float)h/2.0f) - newC;

  a = newa;
  b = newb;
  c = newc;
  C = newC;

}

