#include "framebuffer.h"
#include <iostream>
#include "scene.h"
#include <math.h>
#include "m33.h"

#include "tiffio.h"

using namespace std;


// makes an OpenGL window that supports SW, HW rendering, that can be displayed on screen
//        and that receives UI events, i.e. keyboard, mouse, etc.
FrameBuffer::FrameBuffer(int u0, int v0, 
  int _w, int _h) : Fl_Gl_Window(u0, v0, _w, _h, 0) {

  w = _w;
  h = _h;
  pix = new unsigned int[w*h];
  zb = new float[w*h];
}

// rendering callback; see header file comment
void FrameBuffer::draw() {

  // SW window, just transfer computed pixels from pix to HW for display
  glDrawPixels(w, h, GL_RGBA, GL_UNSIGNED_BYTE, pix);

}


// function called automatically on event within window (callback)
int FrameBuffer::handle(int event)  {  

  switch(event) 
  {   
  case FL_KEYBOARD: {
    KeyboardHandle();
    return 0;
  }
  default:
    break;
  }
  return 0;

}


void FrameBuffer::KeyboardHandle() {

  int key = Fl::event_key();
  switch (key) {
    case FL_Left: {
      cerr << "pressed left arrow" << endl;
      break;
    }
    case 'a': {
      cerr << "pressed a" << endl;
      break;
    }
    default:
    cerr << "INFO: do not understand keypress" << endl;
  }
}

// load a tiff image to pixel buffer
void FrameBuffer::LoadTiff(char* fname) {
  TIFF* in = TIFFOpen(fname, "r");
  if (in == NULL) {
	cerr << fname << " could not be opened" << endl;
	return;
  }

  int width, height;
  TIFFGetField(in, TIFFTAG_IMAGEWIDTH, &width);
  TIFFGetField(in, TIFFTAG_IMAGELENGTH, &height);
  if (w != width || h != height) {
	w = width;
	h = height;
	delete[] pix;
	pix = new unsigned int[w*h];
	size(w, h);
	glFlush();
	glFlush();
  }

  if (TIFFReadRGBAImage(in, w, h, pix, 0) == 0) {
	cerr << "failed to load " << fname << endl;
  }

  TIFFClose(in);
}

// save as tiff image
void FrameBuffer::SaveAsTiff(char *fname) {

  TIFF* out = TIFFOpen(fname, "w");

  if(out == NULL) {
    cerr << fname << " could not be opened" << endl;
    return;
  }

  TIFFSetField(out, TIFFTAG_IMAGEWIDTH, w);
  TIFFSetField(out, TIFFTAG_IMAGELENGTH, h);
  TIFFSetField(out, TIFFTAG_SAMPLESPERPIXEL, 4);
  TIFFSetField(out, TIFFTAG_BITSPERSAMPLE, 8);
  TIFFSetField(out, TIFFTAG_ORIENTATION, ORIENTATION_TOPLEFT);
  TIFFSetField(out, TIFFTAG_PLANARCONFIG, PLANARCONFIG_CONTIG);
  TIFFSetField(out, TIFFTAG_PHOTOMETRIC, PHOTOMETRIC_RGB);

  for (uint32 row = 0; row < (unsigned int) h; row++) {
	TIFFWriteScanline(out, &pix[(h - row - 1) * w], row);
  }
  
  TIFFClose(out);
}

void FrameBuffer::Clear(unsigned int bgr, float z0) {

  for (int v = 0; v < h; v++) {
    for (int u = 0; u < w; u++) {
      int uv = (h-1-v)*w + u;
      pix[uv] = bgr;
      zb[uv] = z0;
    }
  }

}

void FrameBuffer::Draw3DSegment(V3 P0, V3 P1, V3 C0, V3 C1, PPC *ppc) {

    V3 projP0, projP1;
    if (!ppc->Project(P0, projP0) || !ppc->Project(P1, projP1))
      return;

    Draw2DSegment(projP0, projP1, C0, C1);

}

void FrameBuffer::Draw2DSegment(V3 P0, V3 P1, V3 C0, V3 C1) {

  V3 dP = P1-P0;
  int stepsN;
  if (fabsf(dP[0]) < fabsf(dP[1])) {
    stepsN = fabsf(dP[1]) + 1;
  }
  else {
    stepsN = fabsf(dP[0]) + 1;
  }

  // one step along segment
  dP = dP / (float) stepsN;
  V3 dC = (C1-C0) / (float) stepsN;

  V3 currP, currC;
  int i;
  for (i = 0, currP = P0, currC = C0; 
      i < stepsN; 
      i++, currP = currP + dP, currC = currC + dC) {
    if (IsFarther(currP)) {
      continue;
    }
    Setv(currP, currC);
  }

}

bool FrameBuffer::IsFarther(V3 projP) {

  int u = (int) projP[0];
  int v = (int) projP[1];
  if (u < 0 || u >= w || v < 0 || v >= h)
    return true;

  int uv = (h-1-v)*w+u;
  if (zb[uv] >= projP[2])
    return true;

  return false;

}

void FrameBuffer::Setv(V3 projP, V3 cv) {

  int u = (int) projP[0];
  int v = (int) projP[1];
  if (u < 0 || u >= w || v < 0 || v >= h)
    return;
  Set(u, v, cv.UIntColor());
  SetZB(u, v, projP[2]);

}

void FrameBuffer::Setf(float uf, float vf, unsigned int color) {


  int u = (int) uf;
  int v = (int) vf;
  if (u < 0 || u >= w || v < 0 || v >= h)
    return;
  Set(u, v, color);

}

void FrameBuffer::Set(int u, int v, unsigned int color) {

  pix[(h-1-v)*w+u] = color;

}

void FrameBuffer::SetZB(int u, int v, float currz) {

  zb[(h-1-v)*w+u] = currz;

}

void FrameBuffer::Draw3DPpoint(V3 P, V3 color, int pointSize, PPC *ppc) {

  V3 projP;
  if (!ppc->Project(P, projP))
    return;

  V3 currP;
  for (int i = 0; i < pointSize; i++) {
    for (int j = 0; j < pointSize; j++) {
      currP = projP + V3(j-pointSize/2, i-i/pointSize, 0.0f);
      if (IsFarther(currP))
        continue;
      Setv(currP, color);
    }
  }
 
}

