#include <math.h>
#include "tmesh.h"
#include "triangle.h"
#include "linesegment.h"
#include <float.h>
#include <fstream>
#include <iostream>

TMesh::TMesh(V3 C, V3 dims) {

  vertsN = 8;
  trisN = 12;
  
  verts = new V3[vertsN];
  tris = new unsigned int[trisN*3];

  int i = 0;
  verts[i] = C + V3(-dims[0]/2.0f, dims[1]/2.0f, dims[2]/2.0f);
  i++;
  verts[i] = C + V3(-dims[0]/2.0f, -dims[1]/2.0f, dims[2]/2.0f);
  i++;
  verts[i] = C + V3(+dims[0]/2.0f, -dims[1]/2.0f, dims[2]/2.0f);
  i++;
  verts[i] = C + V3(+dims[0]/2.0f, +dims[1]/2.0f, dims[2]/2.0f);
  i++;
  verts[i] = C + V3(-dims[0]/2.0f, dims[1]/2.0f, -dims[2]/2.0f);
  i++;
  verts[i] = C + V3(-dims[0]/2.0f, -dims[1]/2.0f, -dims[2]/2.0f);
  i++;
  verts[i] = C + V3(+dims[0]/2.0f, -dims[1]/2.0f, -dims[2]/2.0f);
  i++;
  verts[i] = C + V3(+dims[0]/2.0f, +dims[1]/2.0f, -dims[2]/2.0f);
  i++;

  i = 0;
  tris[i*3+0] = 0;
  tris[i*3+1] = 1;
  tris[i*3+2] = 2;
  i++;
  tris[i*3+0] = 0;
  tris[i*3+1] = 2;
  tris[i*3+2] = 3;
  i++;
  tris[i*3+0] = 3;
  tris[i*3+1] = 2;
  tris[i*3+2] = 6;
  i++;
  tris[i*3+0] = 3;
  tris[i*3+1] = 6;
  tris[i*3+2] = 7;
  i++;
  tris[i*3+0] = 7;
  tris[i*3+1] = 6;
  tris[i*3+2] = 5;
  i++;
  tris[i*3+0] = 7;
  tris[i*3+1] = 5;
  tris[i*3+2] = 4;
  i++;
  tris[i*3+0] = 4;
  tris[i*3+1] = 5;
  tris[i*3+2] = 1;
  i++;
  tris[i*3+0] = 4;
  tris[i*3+1] = 1;
  tris[i*3+2] = 0;
  i++;
  tris[i*3+0] = 0;
  tris[i*3+1] = 3;
  tris[i*3+2] = 7;
  i++;
  tris[i*3+0] = 0;
  tris[i*3+1] = 7;
  tris[i*3+2] = 4;
  i++;
  tris[i*3+0] = 1;
  tris[i*3+1] = 5;
  tris[i*3+2] = 6;
  i++;
  tris[i*3+0] = 1;
  tris[i*3+1] = 6;
  tris[i*3+2] = 2;
  i++;

}

void TMesh::Project(PPC *ppc, V3 *&projVerts) {

  projVerts = new V3[vertsN];
  for (int i = 0; i < vertsN; i++) {
    ppc->Project(verts[i], projVerts[i]);
  }

}

void TMesh::RenderFilled(PPC *ppc, FrameBuffer *fb) {

  V3 *projVerts;
  Project(ppc, projVerts);

  for (int i = 0; i < trisN; i++) {
    int inds[3];
    V3 triVerts[3];
    bool discardTriangle = false;
    for (int j = 0; j < 3; j++) {
      inds[j] = tris[3*i+j];
      triVerts[j] = projVerts[inds[j]];
      discardTriangle = (triVerts[j][0] == FLT_MAX);
    }
    if (discardTriangle) {
      continue;
    }
    Triangle tri(triVerts[0], triVerts[1], triVerts[2]);
    tri.colors[0] = colors[inds[0]];
    tri.colors[1] = colors[inds[1]];
    tri.colors[2] = colors[inds[2]];
    V3 v0color = colors[inds[0]];
    tri.Rasterize(fb, (v0color*255.0f).Color());
  }

  delete projVerts;


}


void TMesh::RenderPoints(int ptSize, unsigned int color, 
  PPC *ppc, FrameBuffer *fb) {

  V3 *projVerts;
  Project(ppc, projVerts);

  for (int i = 0; i < vertsN; i++) {
    if (projVerts[i][0] == FLT_MAX)
      continue;
    unsigned int currColor = color;
    if (colors) {
      V3 colorv = colors[i] * 255.0f; 
      currColor = colorv.Color();
    }
    fb->DrawPoint(projVerts[i], ptSize, currColor);
  }

  delete projVerts;

}

void TMesh::RenderWF(unsigned int color, 
  PPC *ppc, FrameBuffer *fb) {

  V3 *projVerts;
  Project(ppc, projVerts);

  for (int i = 0; i < trisN; i++) {
    int inds[3];
    inds[0] = tris[3*i+0];
    inds[1] = tris[3*i+1];
    inds[2] = tris[3*i+2];
    for (int j = 0; j < 3; j++) {
      int _j = (j+1)%3;
      LineSegment ls(projVerts[inds[j]], projVerts[inds[_j]]);
      ls.Draw2D(fb, color);
    }
  }

  delete projVerts;

}


void TMesh::Load(char *fname) {

  ifstream ifs(fname, ios::binary);
  if (ifs.fail()) {
    cerr << "INFO: cannot open file: " << fname << endl;
    return;
  }

  ifs.read((char*)&vertsN, sizeof(int));
  char yn;
  ifs.read(&yn, 1); // always xyz
  if (yn != 'y') {
    cerr << "INTERNAL ERROR: there should always be vertex xyz data" << endl;
    return;
  }
  if (verts)
    delete verts;
  verts = new V3[vertsN];

  ifs.read(&yn, 1); // colors 3 floats
  if (colors)
    delete colors;
  colors = 0;
  if (yn == 'y') {
    colors = new V3[vertsN];
  }

  ifs.read(&yn, 1); // normals 3 floats
  if (normals)
    delete normals;
  normals = 0;
  if (yn == 'y') {
    normals = new V3[vertsN];
  }

  ifs.read(&yn, 1); // texture coordinates 2 floats
  float *tcs = 0; // don't have texture coordinates for now
  if (tcs)
    delete tcs;
  tcs = 0;
  if (yn == 'y') {
    tcs = new float[vertsN*2];
  }

  ifs.read((char*)verts, vertsN*3*sizeof(float)); // load verts

  if (colors) {
    ifs.read((char*)colors, vertsN*3*sizeof(float)); // load colors
  }

  if (normals)
    ifs.read((char*)normals, vertsN*3*sizeof(float)); // load normals

  if (tcs)
    ifs.read((char*)tcs, vertsN*2*sizeof(float)); // load texture coordinates

  ifs.read((char*)&trisN, sizeof(int));
  if (tris)
    delete tris;
  tris = new unsigned int[trisN*3];
  ifs.read((char*)tris, trisN*3*sizeof(unsigned int)); // read tiangles

  ifs.close();

  cerr << "INFO: loaded " << vertsN << " verts, " << trisN << " tris from " << endl << "      " << fname << endl;
  cerr << "      xyz " << ((colors) ? "rgb " : "") << ((normals) ? "nxnynz " : "") << ((tcs) ? "tcstct " : "") << endl;

}

void TMesh::Translate(V3 transv) {

  for (int i = 0; i < vertsN; i++) {
    verts[i] = verts[i] + transv;
  }

}

V3 TMesh::GetCenter() {

  V3 ret(0.0f, 0.0f, 0.0f);
  for (int i = 0; i < vertsN; i++) {
    ret = ret + verts[i];
  }

  ret = ret / (float) vertsN;
  return ret;

}

void TMesh::BoundingBox(V3 bb[2]) {

  bb[0] = bb[1] = verts[0];
  for (int i = 0; i < vertsN; i++) {
    for (int j = 0; j < 3; j++) {
      if (bb[0][j] > verts[i][j])
        bb[0][j] = verts[i][j];
      if (bb[1][j] < verts[i][j])
        bb[1][j] = verts[i][j];
    }
  }


}

void TMesh::Scale(float toSize) {

  // find current aabb
  V3 bb[2];
  BoundingBox(bb);
  // find scale factor
  float currd = (bb[1]-bb[0]).Length();
  float scf = toSize / currd;
  // translate C to O
  V3 C = GetCenter();
  Translate(C*-1.0f);
  // scale
  for (int i = 0; i < vertsN; i++) {
    verts[i] = verts[i]*scf; 
  }
  // translate to old C
  Translate(C);

}

void TMesh::TranslateTo(V3 newCenter) {

  V3 C = GetCenter();
  Translate(newCenter-C);

}


TMesh::TMesh(FrameBuffer *fb, PPC *ppc, float _1ow) {

  vertsN = fb->w*fb->h;
  verts = new V3[vertsN];
  trisN = 0;
  tris = 0;
  colors = new V3[vertsN];
  for (int v = 0; v < fb->h; v++) {
    for (int u = 0; u < fb->w; u++) {
      int uv = (fb->h-1-v)*fb->w+u;
      float z = (_1ow == -1.0f) ? fb->zb[uv] : _1ow;
      verts[uv] = ppc->GetPoint(u, v, z);
      colors[uv] = V3(fb->pix[uv])/255.0f;
    }
  }

}

void TMesh::Light(V3 pointLightSource, V3 Eye) {


  float ka = 0.2f;
  for (int i = 0; i < vertsN; i++) {
    V3 lv = (pointLightSource - verts[i]).Normalized();
    V3 n = (normals[i]).Normalized();
    float kd = lv*n;
    kd = (kd < 0.0f) ? 0.0f : kd;

    V3 r = n.Reflect(lv);
    V3 ev = (Eye-verts[i]).Normalized();
    float ks = r*ev;
    ks = (ks < 0.0f) ? 0.0f : ks;
    float exp = 90.0f;
    ks = powf(ks, exp);
    ks = (kd == 0.0f) ? 0.0f : ks;

    V3 highlightColor(1.0f, 1.0f, 1.0f);
//    highlightColor = colors[i];
    colors[i] = colors[i]*(ka+(1.0f-ka)*kd) + highlightColor*ks;
    colors[i][0] = (colors[i][0]>1.0f) ? 1.0f:colors[i][0];
    colors[i][1] = (colors[i][1]>1.0f) ? 1.0f:colors[i][1];
    colors[i][2] = (colors[i][2]>1.0f) ? 1.0f:colors[i][2];

  }

}

void TMesh::Explode() {

  V3 *newVerts = new V3[3*trisN];
  V3 *newColors = new V3[3*trisN];
  V3 *newNormals = new V3[3*trisN];
  for (int i = 0; i < trisN; i++) {
    for (int j = 0; j < 3; j++) {
      newVerts[3*i+j] = verts[tris[3*i+j]];
      newColors[3*i+j] = colors[tris[3*i+j]];
    }
    V3 n = ((newVerts[3*i+1]-newVerts[3*i+0])^
      (newVerts[3*i+2]-newVerts[3*i+0])).Normalized();
    for (int j = 0; j < 3; j++) {
      newNormals[3*i+j] = n;
      tris[3*i+j] = 3*i+j;
    }
  }

  delete verts;
  verts = newVerts;
  vertsN = trisN*3;
  delete colors;
  colors = newColors;
  delete normals;
  normals = newNormals;

}