#include "ppc.h"
#include "m3x3.h"
#include "linesegment.h"

#include <math.h>
#include <float.h>

PPC::PPC(float hfov, int _w, int _h) : 
  w(_w), h(_h) {

  C = V3(0.0f, 0.0f, 0.0f);
  a = V3(1.0f, 0.0f, 0.0f);
  b = V3(0.0f, -1.0f, 0.0f);
  float f = (float)w/(2.0f*tanf(hfov/2.0f/180.0f*3.1415f));
  c = V3(-(float)w/2.0f, (float)h/2.0f, -f);

}

bool PPC::Project(V3 P, V3 &projP) {

  projP = V3(FLT_MAX, FLT_MAX, FLT_MAX);
  M3x3 cam;
  cam.SetColumn(0, a);
  cam.SetColumn(1, b);
  cam.SetColumn(2, c);
  M3x3 invcam = cam.Invert();
  V3 q = invcam*(P-C);
  if (q[2] < 0.0f)
    return false;
  projP[0] = q[0]/q[2];
  projP[1] = q[1]/q[2];
  projP[2] = 1.0f/q[2];
  return true;

}

void PPC::TranslateFB(float step) {

  V3 vd = (a^b);
  vd = vd / vd.Length();
  C = C + vd*step;

}


void PPC::Pan(float rstep) {

  V3 axis = b.Normalized()*-1.0f;
  a = a.RotateThisVectorAboutAxis(axis, rstep);
  c = c.RotateThisVectorAboutAxis(axis, rstep);

}

void PPC::TranslateLR(float step) {

  V3 vd = a.Normalized();
  C = C + vd*step;

}

void PPC::TranslateUD(float step) {

  V3 vd = b.Normalized()*-1.0f;
  C = C + vd*step;

}


void PPC::Tilt(float rstep) {

  V3 axis = a.Normalized();
  b = b.RotateThisVectorAboutAxis(axis, rstep);
  c = c.RotateThisVectorAboutAxis(axis, rstep);

}

float PPC::FocalLength() {

  V3 vd = (a^b).Normalized();
  float f = c*vd;
  return f;

}

void PPC::Visualize(PPC *ppc, FrameBuffer *fb, 
                    float f, unsigned int color) {

  float scf = f / FocalLength();
  V3 P[4], projP[4];
  P[0] = C + c*scf;
  P[1] = C + (c + b*(float)h)*scf;
  P[2] = C + (c + b*(float)h + a*(float)w)*scf;
  P[3] = C + (c + a*(float)w)*scf;
  V3 projC;
  ppc->Project(C, projC);
  for (int i = 0; i < 4; i++) {
    ppc->Project(P[i], projP[i]);
  }
  for (int i = 0; i < 4; i++) {
    LineSegment ls(projC, projP[i]);
    ls.Draw2D(fb, color);
    LineSegment ls2(projP[i], projP[(i+1)%4]);
    ls2.Draw2D(fb, color);
  }

  int duv = 48;
  for (int v = 0; v < h; v += duv) {
    V3 P[2];
    P[0] = C + (c+b*(float)v)*scf;
    P[1] = C + (c+b*(float)v+a*(float)w)*scf;
    V3 projP[2];
    ppc->Project(P[0], projP[0]);
    ppc->Project(P[1], projP[1]);
    LineSegment ls(projP[0], projP[1]);
    ls.Draw2D(fb, color);
  }
  for (int u = 0; u < w; u += duv) {
    V3 P[2];
    P[0] = C + (c+a*(float)u)*scf;
    P[1] = C + (c+a*(float)u+b*(float)h)*scf;
    V3 projP[2];
    ppc->Project(P[0], projP[0]);
    ppc->Project(P[1], projP[1]);
    LineSegment ls(projP[0], projP[1]);
    ls.Draw2D(fb, color);
  }

}


void PPC::Zoom(float focalLengthScale) {

  float step = FocalLength()*(focalLengthScale - 1.0f);
  V3 vd = (a^b).Normalized();
  c = c + vd*step;

}


V3 PPC::GetPoint(int u, int v, float _1ow) {

  if (_1ow == 0.0f)
    return V3(FLT_MAX, FLT_MAX, FLT_MAX);

  float uf = .5f+(float)u;
  float vf = .5f+(float)v;
  V3 ret = C + (c+a*uf+b*vf)/_1ow;
  return ret;

}

ostream& operator<<(ostream &ostr, PPC &ppc) {

  ostr << ppc.w << " " << ppc.h << endl;
  ostr << ppc.a << endl << ppc.b << endl << ppc.c << endl << ppc.C << endl;
  return ostr;

}


istream& operator>>(istream &istr, PPC &ppc) {

  istr >> ppc.w >> ppc.h;
  istr >> ppc.a >> ppc.b >> ppc.c >> ppc.C;
  return istr;

}
