#include <iostream>
#include <fstream>
#include "framebuffer.h"
#include "m3x3.h"
#include <math.h>

#include <tiffio.h>

using namespace std;

#include "scene.h"

FrameBuffer::FrameBuffer(int u0, int v0, 
  int _w, int _h) : Fl_Gl_Window(u0, v0, _w, _h, 0) {

  w = _w;
  h = _h;
  pix = new unsigned int[w*h];
  zb = new float[w*h];
  Set(0xFFFF0000);

}

void FrameBuffer::Set(unsigned int color) {

  for (int i = 0; i < w*h; i++)
    pix[i] = color;

}

void FrameBuffer::draw() {

  glDrawPixels(w, h, GL_RGBA, 
    GL_UNSIGNED_BYTE, pix);

}


void FrameBuffer::Set(int u, int v, unsigned int color) {

  pix[(h-1-v)*w+u] = color;

}

unsigned int FrameBuffer::Get(int u, int v) {

  return pix[(h-1-v)*w+u];

}

void FrameBuffer::SetChecker(int cw, unsigned int colors[2]) {

  for (int v = 0; v < h; v++) {
    int cv = v / cw;
    for (int u = 0; u < w; u++) {
      int cu = u / cw;
      unsigned int color = ((cu+cv)%2) ? colors[1] : colors[0];
      Set(u, v, color);
    }
  }

}

void FrameBuffer::SetGuarded(float uf, 
  float vf, unsigned int color, float z) {

  if (uf < 0.0f || uf >= (float)w 
    || vf < 0.0f || vf >= (float)h)
    return;

  int u = (int)uf;
  int v = (int)vf;
  Set(u, v, color);

  if (z < 0.0f)
    return;
  zb[(h-1-v)*w+u] = z;

}

void FrameBuffer::EdgeMap(FrameBuffer *&edgeMap) {

  edgeMap = new FrameBuffer(w, 0, w, h);
  edgeMap->label("Edge map");

  edgeMap->Set(0xFF000000);

  M3x3 krn;
  krn[0] = V3(-1.0f, -2.0f, -1.0f);
  krn[1] = V3(-2.0f, 12.0f, -2.0f);
  krn[2] = V3(-1.0f, -2.0f, -1.0f);

  for (int v = 1; v < h-1; v++) {
    for (int u = 1; u < w-1; u++) {
      float ef = 0.0f;
      int vv, uu, vr, ur;
      for (vv = v-1, vr = 0; vv <= v+1; vv++, vr++) {
        for (uu = u-1, ur = 0; uu <= u+1; uu++, ur++) {
          V3 pixv(Get(uu, vv));
          float pixInt = pixv.Length();
          ef += krn[vr][ur]*pixInt;
        }
      }
      ef = fabsf(ef);
      if (ef > 255.0f)
        ef = 255.0f;
      V3 ev(ef, ef, ef);
      edgeMap->Set(u, v, ev.Color());
    }
  }

}

int FrameBuffer::Load(char *tiffFileName) {
  
  if (!strcmp(tiffFileName + strlen(tiffFileName)-4, ".tif")) {
    
    ifstream tst(tiffFileName);
    if (tst.fail()) {
      cerr << "INFO: cannot open: " << tiffFileName << endl;
      return 0;
    }
    tst.close();
    
    TIFF* tif = TIFFOpen(tiffFileName, "r");
    
    if (tif) {
      if (pix)
        delete pix;
      
      cerr << "INFO: started loading: " << tiffFileName << endl;
      
      TIFFGetField(tif, TIFFTAG_IMAGEWIDTH, &w);
      TIFFGetField(tif, TIFFTAG_IMAGELENGTH, &h);
      pix = new unsigned int[w*h];
      
      if (!TIFFReadRGBAImage(tif, w, h, (uint32*) pix, 0)) {
        delete pix;
        pix = 0;
        w = h = 0;
        return 0;
      }
      
      TIFFClose(tif);
      label(tiffFileName);
      resize(0, 0, w, h);
      cerr << "INFO: done loading: " << tiffFileName << endl;
      return 1;
    }
    return 0;
  }
  cerr << "INFO: do not know how to open *" << tiffFileName + strlen(tiffFileName)-4 << endl;
  return 0;
}

FrameBuffer::FrameBuffer() : w(0), h(0), pix(0), zb(0), Fl_Gl_Window(0, 0, 0, 0, 0)  {}

bool FrameBuffer::Closer(int u, int v, float z) {

  if (u < 0 || u >= w 
    || v < 0.0f || v >= h)
    return false;

  return (zb[(h-1-v)*w+u] < z);

}

void FrameBuffer::DrawPoint(V3 pt, int ptSize, unsigned int color) {

  int u0 = (int)pt[0];
  int v0 = (int)pt[1];
  for (int v = v0-ptSize/2; v <= v0+ptSize/2; v++) {
    for (int u = u0-ptSize/2; u <= u0+ptSize/2; u++) {
      if (!Closer(u, v, pt[2]))
        continue;
      SetGuarded(.5f+(float)u, .5f+(float)v, color, pt[2]);
    }
  }

}

void FrameBuffer::ClearBuffers() {

//  Set(0xFFFFFFFF);
  Set(0xFF000000);
  for (int i = 0; i < w*h; i++)
    zb[i] = 0.0f;

}

int FrameBuffer::handle(int event)  {  

  switch(event) 
  {   
  case FL_KEYBOARD: {
    KeyboardHandle();
    return 0;
  }
  default:
    break;
  }
  return 0;

}

void FrameBuffer::KeyboardHandle() {

  float rstep = 5.0f;
  float tstep = 5.0f;
  int key = Fl::event_key();
  switch (key) {
    case FL_Left: {
      scene->ppc->Pan(rstep);
      break;
    }
    case FL_Right: {
      scene->ppc->Pan(-rstep);
      break;
    }
    case FL_Up: {
      scene->ppc->TranslateFB(tstep);
      break;
    }
    case FL_Down: {
      scene->ppc->TranslateFB(-tstep);
      break;
    }
    case ',': {
      scene->ppc->TranslateLR(-tstep);
      break;
    }
    case '.': {
      scene->ppc->TranslateLR(+tstep);
      break;
    }
    case '1': {
      scene->ppc->TranslateUD(tstep);
      break;
    }
    case '2': {
      scene->ppc->TranslateUD(-tstep);
      break;
    }
    case 'q': {
      scene->ppc->Tilt(rstep);
      break;
    }
    case 'w': {
      scene->ppc->Tilt(-rstep);
      break;
    }
    case 'z': {
      scene->ppc->Zoom(1.1f);
      break;
    }
    case 'x': {
      scene->ppc->Zoom(0.9f);
      break;
    }
    default:
      cerr << "INFO: pressed a key" << endl;
  }

  scene->Render();

}
