/*

Copyright (C) Purdue University 2002

This program, which is part of the DPF software tool kit for
static distributed denial-of-service (DDoS) performance evaluation
developed at the Network Systems Lab, Purdue University, may be
redistributed and/or modified under the terms of the GNU General
Public License published by the Free Software Foundation.


Technical Contact at Purdue University:

Ali Selcuk (selcuk@cs.purdue.edu) or Kihong Park (park@cs.purdue.edu)

For future version updates and related information, check the website
http://www.cs.purdue.edu/nsl

*/


/*
 *                 route.c
 */


#include "types.h"
#include "extern.h"
#include "functions.h"
#include "edge.h"


static struct {
    NODE_ID  first, last;
}  dList[MAXHOP+2];


typedef struct {
    NODE_ID  next, prev, next_hop;
    DIST  d;
    NODE_TYPE  type;
}  DNODE;


DNODE  *dNode;
NODELIST  *PrSrcHops;
char  *prerouted;



void  ComputeRoutes()
/* Top-level route computation function for single-path routing */
{
    NODE_ID  src, dest;

    AllocTmpRtStrct();
    if ((ROUTE == 1) || (ROUTE == 2))
	UploadRoutes();
    if ((ROUTE == 0) || (ROUTE == 1))
	for (dest = 0; dest < N; dest++)
	    Dijkstra(dest);
    ConvertRouteTables();
    FreeTmpRtStrct();
    if ((ROUTE == 1) || (ROUTE == 2))
	for (src = 0; src < N; src++)
	    for (dest = 0; dest < N; dest++)
		CycleDetect(src, dest, src, 0);
}


void  AllocTmpRtStrct()
/* Allocate the data structures used during the route computation stage */
{
    int i, j;

    /* Allocate & initialize the InitialRouteTable */
    assert ((InitialRouteTable = calloc(N,sizeof(NODE_ID*))) != NULL);
    for (i=0; i<N; i++) {
	assert ((InitialRouteTable[i] = calloc(N,sizeof(NODE_ID))) != NULL);
	for (j=0; j<N; j++)
	    InitialRouteTable[i][j] = nNIL;
    }
    if ((ROUTE == 1) || (ROUTE == 2))
	UploadRoutes();

    assert((dNode = calloc(N, sizeof(DNODE))) != NULL);
    assert((PrSrcHops = calloc(N, sizeof(NODELIST))) != NULL);
    assert((prerouted = calloc(N, sizeof(char))) != NULL);
}


void  FreeTmpRtStrct()
/* Free the data structures used during the route computation stage */
{
    int i;

    free(dNode);
    free(prerouted);
    for (i=0; i<N; i++) 
	free(PrSrcHops[i].list);
    free(PrSrcHops);
    /* InitialRouteTable is freed inside ConvertRouteTables() */
}



void  Dijkstra(NODE_ID  dest)
    /* Computes shortest paths to the given destination. 
       Assumes all edges has a unit weight of 1 (hop) */
{
    NODE_ID  n, m;
    DIST  d;
    int  i;

    /* Initialize */
    InitList(dest);
    if (ROUTE == 1)
	ReadPR(dest);

    /* Run the Dijkstra's */
    for(d=0; d <= MAXHOP; d++) {
	/* Process the next node in the priority queue */
	for(n = dList[d].first; n != nNIL; n = dNode[n].next) {
	    if (ROUTE == 1) {
		/* "Relax" the prerouted neighbors */
		for (i=0; i < PrSrcHops[n].n; i++) {
		    m = PrSrcHops[n].list[i];
		    UpdateNode(m, d+1, n);
		}
	    }
	    if (dNode[n].type == TRANSIT) {
		/* General neighbor "relaxation" */
		for (i=0; i < Degree[n]; i++) {
		    m = node[n].edge[i].to;   
		    if ((d + 1 < dNode[m].d) &&
			((ROUTE==0) || ((ROUTE==1) && prerouted[m]==0))){
			UpdateNode(m, d+1, n);
		    }
		}
	    }
	    /* Record routing info */
	    InitialRouteTable[n][dest] = dNode[n].next_hop;
	}
    }
}
    

void  InitList(NODE_ID  dest)
/* Initialize the distance list for the Dijkstra's algorithm */
{
    NODE_ID  n;
    DIST  d;

    /* Put all nodes in the list MAXHOP+1; 
       i.e., practically at infinite distance */
    for( d=0; d <= MAXHOP; d++)
	dList[d].first = dList[d].last = nNIL;
    dList[MAXHOP+1].first = 1;
    dList[MAXHOP+1].last = N-1;
    for( n=0; n<N; n++) {
	dNode[n].next = n+1;
	dNode[n].prev = n-1;
	dNode[n].next_hop = nNIL;
	dNode[n].d = MAXHOP+1; /* ie. infinity */
	if (Transit[n])
	    dNode[n].type = TRANSIT;
	else
	    dNode[n].type = STUB;
    }
    dNode[0].prev = nNIL;
    dNode[N-1].next = nNIL;

    /* Now move the destination node to list 0 */
    dNode[dest].type = TRANSIT;
    UpdateNode(dest, 0, nNIL);
}


void  UpdateNode( NODE_ID x, DIST d, NODE_ID p)
/* The function for "Relaxation" of a node in the Dijkstra's */
{    
    /* Remove x from its current list */
    if( dNode[x].prev == nNIL) 
	dList[ dNode[x].d ].first = dNode[x].next;
    else
	dNode[ dNode[x].prev ].next = dNode[x].next;
    if( dNode[x].next == nNIL)
	dList[ dNode[x].d ].last = dNode[x].prev;
    else
	dNode[ dNode[x].next ].prev = dNode[x].prev;

    /* Insert x at the end of its new list */
    dNode[x].d = d;
    dNode[x].next = nNIL;
    if( dList[d].first == nNIL) {
	dList[d].first = dList[d].last = x;
	dNode[x].prev = nNIL;
    }
    else {
	dNode[ dList[d].last ].next = x;
	dNode[x].prev =  dList[d].last;
	dList[d].last = x;
    }

    /* Update the routing info on x */
    if(d == 0) 
	dNode[x].next_hop = nNIL;
    else
	dNode[x].next_hop = p;

}



void ConvertRouteTables()
/* The central routing table is distributed to the nodes and next hop 
   information is replaced by the edge index to that neighbor.
*/
{
    NODE_ID  src, dest, nxt;

    for (src = 0; src < N; src++) {
        for (dest = 0; dest < N; dest++) {
	    nxt = InitialRouteTable[src][dest];
	    if (nxt == nNIL) 
		node[src].rtable[dest] = eNIL;
	    else
		GetEdge(src, nxt, node[src].rtable[dest]);
	}
	free( InitialRouteTable[src] );
    }
    free( InitialRouteTable );
}	


void ReadPR(NODE_ID dest)
/* Read the pre-routed entries for the given destination */
{
    int i;
    NODE_ID  src, hop;

    for (i=0; i<N; i++) {
	PrSrcHops[i].n = 0;
	prerouted[i] = 0;
    }

    for (src=0; src<N; src++) {
	if ((hop = InitialRouteTable[src][dest]) != nNIL) {
	    prerouted[src] = 1;
	    PrSrcHops[hop].n++;
	    assert((PrSrcHops[hop].list = realloc(PrSrcHops[hop].list, 
		PrSrcHops[hop].n * sizeof(NODE_ID))) != NULL);
	    PrSrcHops[hop].list[ PrSrcHops[hop].n - 1 ] = src;
	}
    } 
}


void  CycleDetect(NODE_ID src, NODE_ID dest, NODE_ID x, int hop_cnt)
/* Check the route from src to dest for cycles */
{
    static NODE_ID  hop_list[65536];
    EDGE_ID  e;
    int i;

    if (x == dest)
	return;

    /* Check for cycle: */
    for (i = 0; i < hop_cnt; i++) {
	if (x == hop_list[i]) {
	    fprintf(stderr, "ERROR: Cycle detected on route from %d to %d at %d.\n", id2as[src], id2as[dest], id2as[x]);
	    fprintf(stderr, "hop_cnt = %d\n", hop_cnt);
	    for (i = 0; i < hop_cnt; i++)
		fprintf(stderr, "%d ", id2as[hop_list[i]]);
	    fprintf(stderr, "\n");
	    exit(-1);
	}
    }    

    /* Continue, if there is a path to the destination: */
    if ((e = node[x].rtable[dest]) == eNIL)
	return;
    hop_list[hop_cnt] = x;
    CycleDetect(src, dest, node[x].edge[e].to, hop_cnt + 1);
}
