/*

Copyright (C) Purdue University 2002

This program, which is part of the DPF software tool kit for
static distributed denial-of-service (DDoS) performance evaluation
developed at the Network Systems Lab, Purdue University, may be
redistributed and/or modified under the terms of the GNU General
Public License published by the Free Software Foundation.


Technical Contact at Purdue University:

Ali Selcuk (selcuk@cs.purdue.edu) or Kihong Park (park@cs.purdue.edu)

For future version updates and related information, check the website
http://www.cs.purdue.edu/nsl

*/


/*
 *  preroute.c
 *  Functions for route uploading in the multi-path case.
 */


#include "types.h"
#include "extern.h"
#include "functions.h"
#include "edge.h"

void  ComputeRoutes_MP()
/* Upload the pre-computed routes */
{
    char *rtfilename;
    FILE *rtfile;
    shNODELIST  **InitialRouteTable_MP;
    NODE_ID  src, dest, hop, src_as, dest_as, hop_as;
    int i, j, n;


    /*** Initialization: ***/

    /* Allocate & initialize the InitialRouteTable */
    assert ((InitialRouteTable_MP = calloc(N,sizeof(shNODELIST*))) != NULL);
    for (i = 0; i < N; i++) {
        assert ((InitialRouteTable_MP[i] = calloc(N,sizeof(shNODELIST))) 
		!= NULL);
        for (j = 0; j < N; j++)
            InitialRouteTable_MP[i][j].n = 0;
    }


    /*** Route read-in: ***/

    /* Open the pre-route input file */
    assert(( rtfilename = malloc(strlen(graphfilename)+5)) != NULL);
    strcpy(rtfilename, graphfilename);
    strcat(rtfilename, ".rt");
    assert ((rtfile = fopen(rtfilename, "r")) != NULL); 

    /* Read the pre-route input file and fill the data structure */
    while( (fread(&src_as,  sizeof(NODE_ID), 1, rtfile) == 1) && 
	   (fread(&dest_as, sizeof(NODE_ID), 1, rtfile) == 1) && 
	   (fread(&hop_as,  sizeof(NODE_ID), 1, rtfile) == 1) ) {
	src = as2id[src_as]; dest = as2id[dest_as]; hop = as2id[hop_as];

	/* Check for error conditions */
	if (src == nNIL) {
	    fprintf(stderr, "ERROR: Source does not exist\n");
	    fprintf(stderr, "Src = %d, Dest = %d, Hop = %d.\n", 
		    src_as, dest_as, hop_as);
	    exit(-1);
	}
	if (dest == nNIL) {
	    fprintf(stderr, "ERROR: Destination does not exist\n");
	    fprintf(stderr, "Src = %d, Dest = %d, Hop = %d.\n", 
		    src_as, dest_as, hop_as);
	    exit(-1);
	}
	if (dest == src)
	    continue;
	if (hop == nNIL) {
	    fprintf(stderr, "ERROR: Hop does not exist\n");
	    fprintf(stderr, "Src = %d, Dest = %d, Hop = %d.\n", 
		    src_as, dest_as, hop_as);
	    exit(-1);
	}
	if (hop == src) {
	    fprintf(stderr, "ERROR: Hop = Source is not allowed\n");
	    fprintf(stderr, "Src = %d, Dest = %d, Hop = %d.\n", 
		    src_as, dest_as, hop_as);
	    exit(-1);
	}

	/* Check if "hop" is already in the list of (src,dest): */
	for (i = 0; (i < InitialRouteTable_MP[src][dest].n) &&
		 InitialRouteTable_MP[src][dest].list[i] != hop; i++);
	/* if "hop" is found in the list, continue: */
	if (i < InitialRouteTable_MP[src][dest].n)
	    continue;

	n = ++InitialRouteTable_MP[src][dest].n;
	assert((InitialRouteTable_MP[src][dest].list = realloc(
	    InitialRouteTable_MP[src][dest].list, 
	    n * sizeof(NODE_ID))) != NULL);
	InitialRouteTable_MP[src][dest].list[n-1] = hop;

    }
    fclose(rtfile);
    /* Route read-in complete */


    /*** ConvertRouteTables_MP: ***/

    for (src = 0; src < N; src++) {
        for (dest = 0; dest < N; dest++) {
	    n = node[src].rtable_MP[dest].n= InitialRouteTable_MP[src][dest].n;
	    assert((node[src].rtable_MP[dest].list = 
		    calloc(n, sizeof(EDGE_ID))) != NULL);
	    for (i = 0; i < n; i++) {
		hop = InitialRouteTable_MP[src][dest].list[i];
		GetEdge(src, hop, node[src].rtable_MP[dest].list[i]);
	    }
	}
	free( InitialRouteTable_MP[src] );
    }
    free( InitialRouteTable_MP );


    /*** Check for cycles: ***/

    for (src = 0; src < N; src++)
	for (dest = 0; dest < N; dest++)
	    CycleDetect_MP(src, dest, src, 0);


}



void  CycleDetect_MP(NODE_ID src, NODE_ID dest, NODE_ID x, int hop_cnt)
/* Check the route from src to dest for cycles */
{
    static NODE_ID  hop_list[65536];
    EDGE_ID  e;
    int i;

    if (x == dest)
	return;

    /* Check for cycle: */
    for (i = 0; i < hop_cnt; i++) {
	if (x == hop_list[i]) {
	    fprintf(stderr, "ERROR: Cycle detected on route from %d to %d.\n",
		    id2as[src], id2as[dest]);
	    exit(-1);
	}
    }    

    /* Continue, if there is a path to the destination: */
    hop_list[hop_cnt] = x;
    for (i = 0; i < node[x].rtable_MP[dest].n; i++) {
	e = node[x].rtable_MP[dest].list[i];
	CycleDetect_MP(src, dest, node[x].edge[e].to, hop_cnt + 1);
    }
}

