/*

Copyright (C) Purdue University 2002

This program, which is part of the DPF software tool kit for
static distributed denial-of-service (DDoS) performance evaluation
developed at the Network Systems Lab, Purdue University, may be
redistributed and/or modified under the terms of the GNU General
Public License published by the Free Software Foundation.


Technical Contact at Purdue University:

Ali Selcuk (selcuk@cs.purdue.edu) or Kihong Park (park@cs.purdue.edu)

For future version updates and related information, check the website
http://www.cs.purdue.edu/nsl

*/


#include "types.h"
#include "extern.h"
#include "functions.h"
#include "filter.h"


/*** Functions for filter & set computations in the multi-path case ***/



void TraverseRoute_SetFilters_MP(NODE_ID s /* source node */, 
				 NODE_ID d /* destination */, 
				 NODE_ID x /* current node */, 
				 FTR_ID  f /* filter on x */)
/* Traversing the route from s to d;  updating the filter at x */
{
    EDGE_ID  e;
    int  i;

    /* Update the filter on x, if x is a filter node */
    if ( f != fNIL ) { 
	if ( ! GetBit( Filter1[f], s)) {
	    SetBit( Filter1[f], s);
	    AppendFilter2( Filter2[f], s);
	}
    }


    /* Continue traversing the path as long as x is not the destination and
       there is a path from x to the destination from x. 
    */
    if (x == d) 
	return;
    for (i = 0; i < node[x].rtable_MP[d].n; i++) {
	e = node[x].rtable_MP[d].list[i];
	TraverseRoute_SetFilters_MP(s, d, node[x].edge[e].to, node[x].edge[e].ftr);
    }
}



void ComputeSets(NODE_ID t /* target destination */)
/* Main function for computing the reachability sets.
   Works with one target destination at a time. */
{

    NODE_ID a;
    char Done[MAXN] = {0};
    int i;

    /* Initialize all S_ta's to all 0's*/ 
    UnsetStas();

    /* Initialize the S_ta[t] to include all source addresses. 
       Needed for recursive Sta computation.
       This setting is unset at the end of this function. */
    SetBitArray( Sta1[t] );
    Sta2[t].n = N;
    assert((Sta2[t].list = realloc(Sta2[t].list, N * sizeof(NODE_ID))) != 0);
    for (i=0; i<N; i++)
        Sta2[t].list[i] = i;
    Done[t] = 1;

    /* Compute S_ta sets */
    for (a=0; a<N; a++) 
	if (!Done[a])
	    ComputeSta_MP(t, a, Done);


    /* If egress filtering is on, unset spoofing at the filter nodes */
    if(EGRESS) 
	for (a=0; a<N; a++) 
	    if (Filter[a]) {
		UnsetSta(a);
		SetBit( Sta1[a], a); /* except for s=a */
		Sta2[a].n = 1;
		assert((Sta2[a].list = realloc(
		    Sta2[a].list, sizeof(NODE_ID))) != NULL);
		Sta2[a].list[0] = a;
	    }

    UnsetSta(t);
    /* Assumes that t is not counted as an attack node for itself */
    

}



void ComputeSta_MP(NODE_ID t, NODE_ID a, char Done[])
/* Analogous to ComputeSta_SP; but the computation runs over all entries 
   in the routing table for the given target destination.
   Moreover, an entry is checked in Sta1[a] before being added to the buffer. 
*/
{
    EDGE_ID  e;
    FTR_ID  f;
    NODE_ID next_hop;
    static NODE_ID buffer[MAXN];
    int j;

    Done[a] = 1;
    for (j = 0; j < node[a].rtable_MP[t].n; j++) {
	e = node[a].rtable_MP[t].list[j];
	next_hop = node[a].edge[e].to;
	f = node[a].edge[e].ftr;
	if (!Done[next_hop])
	    ComputeSta_MP(t, next_hop, Done);
	

	/***   THE MAIN PART   ***/
	/* If the next hop is not a filter node, just merge its S_ta set */
	if (!Filter[next_hop]) {
	    NODE_CNT i, nsta;
	    NODE_ID  x, *sta;
	    sta = Sta2[next_hop].list;
	    nsta = Sta2[next_hop].n;
            for (i = 0; i < nsta; i++) {
                x = sta[i];
                if ( !GetBit( Sta1[a], x) ) {
                    SetBit( Sta1[a], x);
                    buffer[ Sta2[a].n++ ] = x;
                }
            }
	}
	/* Otherwise, take the intersection of next hop's S_ta set with
	   the filter in-between, and then merge. */
	else {
	    NODE_CNT i, nftr;
	    NODE_ID  x, *ftr;
	    Sta2[a].n = 0;
	    ftr = Filter2[f].list;
	    nftr = Filter2[f].n;
	    for (i = 0; i < nftr; i++) {
		x = ftr[i];
		if ( GetBit( Sta1[next_hop], x) && !GetBit( Sta1[a], x)) {
		    SetBit( Sta1[a], x);
		    buffer[ Sta2[a].n++ ] = x;
		}
	    }
	}
    }
    assert((Sta2[a].list = realloc(
	Sta2[a].list, Sta2[a].n*sizeof(NODE_ID))) != NULL);
    memcpy(Sta2[a].list, buffer, Sta2[a].n*sizeof(NODE_ID));
}



void AllocFilters()
/*  Allocate the filter tables  */
{
    AllocFilters1();
    AllocFilters2();
}

void AllocFilters1()
/*  Allocate the Filter1 array  */
{
    int i;
    assert((Filter1 = calloc(N_ftr, sizeof(BITARRAY))) != NULL);
    for (i=0; i < N_ftr; i++) 
        assert((Filter1[i] = calloc(numF, sizeof(FILTER_CELL))) != NULL);
}

void AllocFilters2()
/*  Allocate the Filter2 array  */
{
    assert((Filter2 = calloc(N_ftr, sizeof(NODELIST))) != NULL);
}

void FreeFilters1()
/*  Free the Filter1 array once it's no more needed  */
{
    int i;
    for (i=0; i < N_ftr; i++) 
        free(Filter1[i]);
    free(Filter1);
}

void AllocSets()
/*  Allocate the reachability sets  */
{
    int i;
    assert((Sta2 = calloc(N, sizeof(NODELIST))) != NULL);
    if(SETS)
	assert((Cts2 = calloc(N, sizeof(NODELIST))) != NULL);
    assert((Sta1 = calloc(N, sizeof(BITARRAY))) != NULL);
    for (i=0; i<N; i++) 
        assert((Sta1[i] = calloc(numF, sizeof(FILTER_CELL))) !=NULL);
}

void FreeSets()
/*  Free the Sta & Cts arrays when they are not needed  */
{
    int i;
    for (i=0; i<N; i++) 
        free(Sta1[i]);
    free(Sta1);
    for (i=0; i<N; i++) 
        free(Sta2[i].list);
    free(Sta2);
    if(SETS) {
	for (i=0; i<N; i++) 
	    free(Cts2[i].list);
	free(Cts2);
    }
}




void UnsetFilters()
/* function to initialize all filters to zero */
{    
    int i;
    for (i=0; i < N_ftr; i++) {
	UnsetFilter(i);
    }
}


void UnsetStas()
/* function to initialize all S_ta sets to zero */
{    
    int i;
    for (i=0; i < N; i++) {
	UnsetSta(i);
    }
}

