/* symbolic-range.c */

#include "config.h"
#include "system.h"
#include "coretypes.h"
#include "tree.h"
#include "target.h"
#include "gimple.h"
#include "cgraph.h"
#include "flags.h"
#include "timevar.h"
#include "diagnostic.h"
#include "params.h"
#include "cfgloop.h"
#include "pointer-set.h"
#include "bitmap.h"
#include "tree-flow.h"
#include "tree-pass.h"
#include "tree-chrec.h"
#include "tree-scalar-evolution.h"
#include "tree-data-ref.h"
#include "tree-pretty-print.h"
#include "tree-dump.h"
#include <string.h>

#include "tree-vrp.h"
#include "symbolic-expr-eval.h"


/* symbolic range analysis based on VRP and SYMEE */


extern void
symee_set_state_sym_value_range (symee_context ctx, symee_const_expr sym,
                                 value_range_ptr vr)
{
  pointer_map_ptr vr_map = ctx->value_range;
  void** slot = pointer_map_contains (vr_map, sym);

  gcc_assert (ctx);

  if (!slot)
    slot = pointer_map_insert (vr_map, sym);

  gcc_assert (slot);
  *slot = vr;
}


extern symee_expr
symee_get_state_sym_vr_min (symee_context ctx, symee_const_expr sym)
{
  pointer_map_ptr vr_map = ctx->value_range;
  void** slot = NULL;

  gcc_assert (ctx && ctx->value_range);
  
  slot = pointer_map_contains (vr_map, sym);

  if (slot) {
    value_range_ptr vr = *slot;
    gcc_assert (vr);
    if (vr->min)
      return vr->min;
  }

  /* if no value_range, just return [expr, expr] */
  return symee_get_state_sym (ctx, sym);
}


extern symee_expr
symee_get_state_sym_vr_max (symee_context ctx, symee_const_expr sym)
{
  pointer_map_ptr vr_map = ctx->value_range;
  void** slot = NULL;

  gcc_assert (ctx && ctx->value_range);
  
  slot = pointer_map_contains (vr_map, sym);

  if (slot) {
    value_range_ptr vr = *slot;
    gcc_assert (vr);
    if (vr->max)
      return vr->max;
  }

  /* if no value_range, just return [expr, expr] */
  return symee_get_state_sym (ctx, sym);
}


static symee_expr
substitute_min (symee_context ctx, symee_expr expr)
{
  const tree t = expr;
  enum tree_code code = TREE_CODE (expr);
  enum tree_code_class kind = TREE_CODE_CLASS (code);
  tree tem = NULL_TREE;
  location_t loc = EXPR_LOCATION (expr);
  tree type = TREE_TYPE (t);
  tree op0 = NULL_TREE, op1 = NULL_TREE, op2 = NULL_TREE;

  if (DECL_P (t)) {
    return symee_get_state_sym_vr_min (ctx, expr);
  }

  if (eval_reference &&
      REFERENCE_CLASS_P (expr)) {
    return symee_get_state_sym_by_ref (ctx, expr);
  }
  
  if (TREE_CODE (t) == ASSERT_EXPR)
    return TREE_OPERAND (t, 0);
  
#if 0
  if (TREE_CODE (t) == ADDR_EXPR
      || TREE_CODE (t) == NOP_EXPR) {
    return t;
  }
#endif
  
  /* TODO: */
  if (code == SSA_NAME) {
    if (TREE_CODE (SSA_NAME_VAR (expr)) == PARM_DECL)
      return symee_get_state_sym_vr_min (ctx, expr); // SSA_NAME_VAR (expr));
    else
      return symee_get_state_sym_vr_min (ctx, expr);
  }

  if (IS_EXPR_CODE_CLASS (TREE_CODE_CLASS (code))) {
    switch (TREE_CODE_LENGTH (code)) {
    case 1:
      op0 = TREE_OPERAND (t, 0);
      op0 = substitute_min (ctx, op0);
      tem = build1_loc (loc, code, type, op0);
      return tem ? tem : expr;
    case 2:
      op0 = TREE_OPERAND (t, 0);
      op1 = TREE_OPERAND (t, 1);
      op0 = substitute_min (ctx, op0);
      op1 = substitute_min (ctx, op1);
      tem = build2_loc (loc, code, type, op0, op1);
      return tem ? tem : expr;
    case 3:
      op0 = TREE_OPERAND (t, 0);
      op1 = TREE_OPERAND (t, 1);
      op2 = TREE_OPERAND (t, 2);
      op0 = substitute_min (ctx, op0);
      op1 = substitute_min (ctx, op1);
      /* TODO: op2 of COMPONENT_REF maybe NULL */
      if (op2)
        op2 = substitute_min (ctx, op2);
      tem = build3_loc (loc, code, type, op0, op1, op2);
      return tem ? tem : expr;
    case 4:
      if (code == ARRAY_REF
          || code == SYMEE_DIRECT_SUM)
        break;
    default:
      gcc_unreachable ();
      break;
    }
  }

  if (eval_reference &&
      REFERENCE_CLASS_P (expr)) {
    return symee_get_state_sym_by_ref (ctx, expr);
  }

  switch (code) {
  case INDIRECT_REF:
  case ARRAY_REF:
    return symee_get_state_sym_1 (ctx, t);

  case CONST_DECL:
    return (DECL_INITIAL (t));

  default:
    return t;
  }

  gcc_unreachable ();

  return expr;
}


static symee_expr
substitute_max (symee_context ctx, symee_expr expr)
{
  const tree t = expr;
  enum tree_code code = TREE_CODE (expr);
  enum tree_code_class kind = TREE_CODE_CLASS (code);
  tree tem = NULL_TREE;
  location_t loc = EXPR_LOCATION (expr);
  tree type = TREE_TYPE (t);
  tree op0 = NULL_TREE, op1 = NULL_TREE, op2 = NULL_TREE;

  if (DECL_P (t)) {
    return symee_get_state_sym_vr_max (ctx, expr);
  }

  if (eval_reference &&
      REFERENCE_CLASS_P (expr)) {
    return symee_get_state_sym_by_ref (ctx, expr);
  }
  
  if (TREE_CODE (t) == ASSERT_EXPR)
    return TREE_OPERAND (t, 0);
  
#if 0
  if (TREE_CODE (t) == ADDR_EXPR
      || TREE_CODE (t) == NOP_EXPR) {
    return t;
  }
#endif
  
  /* TODO: */
  if (code == SSA_NAME) {
    if (TREE_CODE (SSA_NAME_VAR (expr)) == PARM_DECL)
      return symee_get_state_sym_vr_max (ctx, expr); // SSA_NAME_VAR (expr));
    else
      return symee_get_state_sym_vr_max (ctx, expr);
  }

  if (IS_EXPR_CODE_CLASS (TREE_CODE_CLASS (code))) {
    switch (TREE_CODE_LENGTH (code)) {
    case 1:
      op0 = TREE_OPERAND (t, 0);
      op0 = substitute_max (ctx, op0);
      tem = build1_loc (loc, code, type, op0);
      return tem ? tem : expr;
    case 2:
      op0 = TREE_OPERAND (t, 0);
      op1 = TREE_OPERAND (t, 1);
      op0 = substitute_max (ctx, op0);
      op1 = substitute_max (ctx, op1);
      tem = build2_loc (loc, code, type, op0, op1);
      return tem ? tem : expr;
    case 3:
      op0 = TREE_OPERAND (t, 0);
      op1 = TREE_OPERAND (t, 1);
      op2 = TREE_OPERAND (t, 2);
      op0 = substitute_max (ctx, op0);
      op1 = substitute_max (ctx, op1);
      /* TODO: op2 of COMPONENT_REF maybe NULL */
      if (op2)
        op2 = substitute_max (ctx, op2);
      tem = build3_loc (loc, code, type, op0, op1, op2);
      return tem ? tem : expr;
    case 4:
      if (code == ARRAY_REF
          || code == SYMEE_DIRECT_SUM)
        break;
    default:
      gcc_unreachable ();
      break;
    }
  }

  if (eval_reference &&
      REFERENCE_CLASS_P (expr)) {
    return symee_get_state_sym_by_ref (ctx, expr);
  }

  switch (code) {
  case INDIRECT_REF:
  case ARRAY_REF:
    return symee_get_state_sym_1 (ctx, t);

  case CONST_DECL:
    return (DECL_INITIAL (t));

  default:
    return t;
  }

  gcc_unreachable ();

  return expr;
}


extern value_range_ptr
eval_expr_tree_value_range (symee_context ctx, symee_expr expr)
{
  value_range_t *expr_vr = XCNEW (value_range_t);

  expr_vr->type = VR_RANGE;
  expr_vr->min = substitute_min (ctx, expr);
  expr_vr->max = substitute_max (ctx, expr);

  /* at least one boundary */
  if (!expr_vr->min && !expr_vr->max)
    {
      XDELETE (expr_vr);
      expr_vr = NULL;
    }

  return expr_vr;
}


value_range_t*
symee_eval_value_range (symee_context ctx, value_range_t *vr)
{
  value_range_t *sym_vr;
  tree min, max;

  gcc_assert (vr);
  
  sym_vr = XCNEW (value_range_t);

  min = symee_eval (ctx, vr->min);
  max = symee_eval (ctx, vr->max);
  
  sym_vr->type = vr->type;
  sym_vr->min = min ? min : vr->min;
  sym_vr->max = max ? max : vr->max;
  sym_vr->equiv = vr->equiv;

  return sym_vr;
}


value_range_t*
symee_get_symbolic_index_range (symee_context ctx, symee_expr sym)
{
}


value_range_t*
symee_get_symbolic_value_range (symee_context ctx, symee_expr sym)
{
  value_range_t *vr = NULL, *sym_vr = NULL;
  enum tree_code code = TREE_CODE (sym);
  
  switch (code) {
  case SSA_NAME:
    vr = symee_vrp_get_value_range (sym);
    if (vr)
      sym_vr = symee_eval_value_range (ctx, vr);
    break;
    
  case ARRAY_REF:
  default:
    break;
  }

  return sym_vr;
}


void
symee_dump_symbolic_range (symee_context ctx, symee_expr sym)
{
  value_range_t *vr = NULL;
  
  if (TREE_CODE (sym) == SSA_NAME)
    vr = symee_vrp_get_value_range (sym);

  if (dump_details)
    {
      fprintf (dump_file, "symee_range for ");
      print_generic_expr (dump_file, sym, dump_flags);

      /* { min --> max } */
      fprintf (dump_file, " --> { ");
      if (vr)
        {
          if (vr->min)
            print_generic_expr (dump_file, symee_get_state_sym (ctx, vr->min),
                                dump_flags);
          fprintf (dump_file, "  --> ");
          if (vr->max)
            print_generic_expr (dump_file, symee_get_state_sym (ctx, vr->max),
                                dump_flags);
        }
      else
        fprintf (dump_file, "no value_range_t");
      fprintf (dump_file, " }\n");
    }

  return;
}

