/* symbolic-loop.c -*- c-*- */

#include "config.h"
#include "system.h"
#include "coretypes.h"
#include "tree.h"
#include "target.h"
#include "gimple.h"
#include "cgraph.h"
#include "flags.h"
#include "timevar.h"
#include "diagnostic.h"
#include "params.h"
#include "cfgloop.h"
#include "pointer-set.h"
#include "bitmap.h"
#include "tree-flow.h"
#include "tree-pass.h"
#include "tree-chrec.h"
#include "tree-scalar-evolution.h"
#include "tree-data-ref.h"
#include "tree-pretty-print.h"
#include "tree-dump.h"
#include <string.h>

#include "symbolic-analysis.h"


/* The worklist for symbolic execution of loops. */
static VEC(symee_context,heap) *worklist;

/* Map from LOOP_P (loop) to UNSIGNED (number of loop iterations
   having executed). */
static struct pointer_map_t *loop_iter_cnt;

/* map <cgraph_node_ptr, symee_cgraph_node_info> */
static pointer_map_ptr cgraph_node_info_map = NULL;

/**************************************/
/* Section of loop symbolic execution */
/**************************************/

basic_block
symee_context_get_prev (symee_context ctx)
{
  gcc_assert (ctx);
  return ctx->prev;
}

void
symee_context_set_prev (symee_context ctx, basic_block bb)
{
  gcc_assert (ctx);
  ctx->prev = bb;
}

gimple
symee_context_get_next (symee_context ctx)
{
  gcc_assert (ctx);
  return ctx->next;
}

void
symee_context_set_next (symee_context ctx, gimple stmt)
{
  gcc_assert (ctx);
  ctx->next = stmt;
}

static void
dump_symbolic_expr (FILE* fp, symee_const_expr sym, symee_const_expr expr)
{
  fprintf (fp, "sym (%p) -> expr (%p)\n", sym, expr);
  print_generic_expr (fp, sym, 0);
  fprintf (fp, "\n\t----->\t");
  print_generic_expr (fp, expr, 0);
  fprintf (fp, "\n");
}

void
dump_vec_context (FILE* fp, VEC(symee_context,heap) *vec)
{
  unsigned ix;
  symee_context elt;

  fprintf (fp, "===vec_context begin===\n");
  FOR_EACH_VEC_ELT (symee_context, vec, ix, elt)
    {
      dump_context_brief (fp, elt);
    }
  fprintf (fp, "===vec_context end===\n");
}

static struct pointer_map_t *
init_looop_iter_cnt (void)
{
  loop_iter_cnt = pointer_map_create ();
}

static unsigned
get_number_of_iterations (struct loop *l)
{
  unsigned *cnt = XCNEW (unsigned);
  void **slot = pointer_map_contains (loop_iter_cnt, l);

  if (!slot)
    {
      slot = pointer_map_insert (loop_iter_cnt, l);
      *slot = cnt;
    }

  return **((unsigned **) slot);
}

static void
increase_number_of_iterations (struct loop *l)
{
  unsigned **cnt = (unsigned **) pointer_map_contains (loop_iter_cnt, l);

  gcc_assert (cnt);

  (**cnt)++;
}

static bool
free_values (const void *key, void **slot, void *data)
{
  XDELETE ((unsigned *) *slot);

  return true;
}

static void
destroy_number_of_iterations (void)
{
  pointer_map_traverse (loop_iter_cnt, free_values, NULL);
  pointer_map_destroy (loop_iter_cnt);
}

static bool
dump_mapping (const void *key, void **slot, void *data)
{
  FILE *fp = data;

  fprintf (fp, "loop(%d)---->iter: %d\n",
	   ((struct loop *) key)->num,
	   *((unsigned *) *slot));

  return true;
}

void
dump_number_of_iterations (FILE *fp)
{
  fprintf (fp, "===number of iterations begin===\n");
  pointer_map_traverse (loop_iter_cnt, dump_mapping, fp);
  fprintf (fp, "===number of iterations end===\n");
}

/* dup of get_base_var */

static tree
get_base_sym (const_tree ct)
{
  tree t = ct;

  while (!SSA_VAR_P (t)
	 && (!CONSTANT_CLASS_P (t))
	 && TREE_CODE (t) != LABEL_DECL
	 && TREE_CODE (t) != FUNCTION_DECL
	 && TREE_CODE (t) != CONST_DECL
	 && TREE_CODE (t) != CONSTRUCTOR)
    {
      t = TREE_OPERAND (t, 0);
    }
  
  return t;
}

static void
execute_stmt_assign_ref (symee_context ctx, tree ref, tree value)
{
  gimple stmt = symee_context_get_next (ctx);
  loop_p l = loop_containing_stmt (stmt);
  tree ref_sym = get_base_sym (ref);
  symee_ref_list ref_list = symee_get_state_sym_ref_list_create (ctx, ref_sym);
  symee_ref_node ref_node = symee_ref_list_append_ref (ref_list, ref);

  ref_node->stmt = stmt;
  ref_node->is_read = false;
  ref_node->dr = create_data_ref (l, l, ref, stmt, false);
  ref_node->value = value;

#if 0
  if (TREE_CODE (ref) == ARRAY_REF) {
    tree index = NULL_TREE;
    tree chrec = analyze_array_evolution (l, ref_sym, &index);
  }
#endif
  
  symee_ref_list_kill (ref_list, ref_node);

  /* gen_context_ref_range (ctx, stmt, ref, value); */
}

static void
execute_stmt_assign (symee_context ctx)
{
  gimple stmt = symee_context_get_next (ctx);
  tree lhs = gimple_assign_lhs (stmt);
  tree rhs = gimple_assign_rhs_to_tree (stmt);
  symee_expr expr = eval_expr_tree (ctx, rhs);

  if (dump_file)
    dump_symbolic_expr (dump_file, lhs, expr);

  if (eval_reference &&
      REFERENCE_CLASS_P (lhs))
    {
      execute_stmt_assign_ref (ctx, lhs, expr);
    }
  else
    {
      symee_set_state_sym (ctx, lhs, expr);
    }
}

extern symee_context
symee_context_cpy (symee_context to_ctx, symee_context from_ctx)
{
  if (from_ctx->state)
    {
      if (to_ctx->state)
	pointer_map_destroy (to_ctx->state);

      to_ctx->state = pointer_map_copy (from_ctx->state);
    }

  if (from_ctx->ref_state)
    {
      if (to_ctx->ref_state)
	pointer_map_destroy (to_ctx->ref_state);

      to_ctx->ref_state = pointer_map_copy (from_ctx->ref_state);
    }

  /* TODO: deep copy */
  to_ctx->state_cond = from_ctx->state_cond;
  to_ctx->path_cond = from_ctx->path_cond;
}

struct merge_callee_context_data {
  gimple call_stmt;
  symee_context ctx;
  symee_context in_ctx;
  symee_context out_ctx;
  symee_context ret_ctx;
  symee_context actual_out_ctx;
};

static bool
eval_context_1 (const void* key, void** slot, void* data)
{
  symee_const_expr sym = (symee_const_expr) key;
  symee_expr expr = *(symee_expr*)slot;
  symee_context ctx = (symee_context) data;
  symee_expr ret_expr = NULL_TREE;

  ret_expr = eval_expr_tree (ctx, expr);
  gcc_assert (ret_expr);
  if (ret_expr)
    *slot = ret_expr;

  return true;
}

static symee_context
eval_context (symee_context ctx, symee_context eval_ctx)
{
  symee_context ret_ctx = symee_context_copy (eval_ctx);

  pointer_map_traverse (ret_ctx->state, eval_context_1, ctx);

  if (dump_details) {
    fprintf (dump_file, "SYMEE: evaluate ctx\n");
    fprintf (dump_file, "input ctx =\n");
    dump_context_brief (dump_file, ctx);
    fprintf (dump_file, "evaluated ctx =\n");
    dump_context_brief (dump_file, eval_ctx);
    fprintf (dump_file, "result ctx =\n");
    dump_context_brief (dump_file, ret_ctx);
  }

  return ret_ctx;
}

static bool
merge_callee_context_1 (const void *key, void **slot, void *data)
{
  struct merge_callee_context_data* mdata =
    (struct merge_callee_context_data*) data;
  symee_const_expr sym = (symee_const_expr) key;
  symee_expr expr = *(symee_expr*)slot;
  symee_expr basesym = get_base_sym (sym);
  symee_expr retsym = NULL;

  /* global out */
  if (DECL_P (basesym) && is_global_var (basesym)) {
    retsym = sym;
  }

  /* formal out */
  if (REFERENCE_CLASS_P (sym) &&
      TREE_CODE (basesym) == SSA_NAME) {
    symee_expr actual_sym = NULL;
    if (eval_formal_outsym) {
      symee_context formal_out_ctx = symee_context_create ();
      formal_out_ctx->state = pointer_map_copy (current_node_info->formal_out);
      eval_context (mdata->actual_out_ctx, formal_out_ctx);
      actual_sym = eval_expr_tree (formal_out_ctx, sym);
    }
    else
      actual_sym = eval_expr_tree (mdata->out_ctx, sym);
    gcc_assert (actual_sym);
    symee_set_state_sym (mdata->ret_ctx, actual_sym, expr);
  }

  if (retsym)
    symee_set_state_sym (mdata->ret_ctx, sym, expr);

  return true;
}


/* ctx : context before call_stmt
   in_ctx : input ctx of callee
   out_ctx : output ctx of callee

   return context of after call_stmt */

static symee_context
merge_callee_context (symee_context ctx, gimple call_stmt,
                      symee_context in_ctx, symee_context out_ctx,
                      symee_context actual_out_ctx)
{
  symee_context ret_ctx = symee_context_copy (ctx);
  struct merge_callee_context_data mdata =
    {call_stmt, ctx, in_ctx, out_ctx, ret_ctx, actual_out_ctx};

  pointer_map_traverse (out_ctx->state, merge_callee_context_1, &mdata);

  return ret_ctx;
}


/* 'call_stmt'' has no callee cgraph_node */

static symee_context
gen_context_call_indirect (symee_context ctx, gimple call_stmt)
{
  struct cgraph_node* node = current_cgraph_node;
  struct cgraph_edge* edge = cgraph_edge (node, call_stmt);
  tree lhs = gimple_call_lhs (call_stmt);
  tree ret_type = gimple_call_return_type (call_stmt);
  tree tmp_var = NULL_TREE;
  if (lhs) {
    char var_name [256] = "icall";
    tree fn = gimple_call_fn (call_stmt);
    if (TREE_CODE (fn) == SSA_NAME)
      fn = SSA_NAME_VAR (fn);
    if (DECL_P (fn)) {
      strcat (var_name, ".");
      if (DECL_NAME (fn))
        strcat (var_name, IDENTIFIER_POINTER (DECL_NAME (fn)));
      else {
        strcat (var_name, "D.");
        sprintf (var_name + strlen (var_name), "%d", DECL_UID (fn));
      }
    }
    strcat (var_name, ".suffix");
    tmp_var = add_new_static_var (ret_type);
    DECL_NAME (tmp_var) = create_tmp_var_name (var_name);
  }

  symee_set_state_sym (ctx, lhs, tmp_var);
  if (dump_details)
    dump_symbolic_expr (dump_file, lhs, tmp_var);

  return ctx;
}


/* callee cgraph_node of 'call_stmt' has no symee_context */

static symee_context
gen_context_call_no_body (symee_context ctx, gimple call_stmt)
{
  struct cgraph_node* node = current_cgraph_node;
  struct cgraph_edge* edge = cgraph_edge (node, call_stmt);
  struct cgraph_node* callee_node = edge->callee;
  tree lhs = gimple_call_lhs (call_stmt);
  tree ret_type = gimple_call_return_type (call_stmt);
  const char* fname = IDENTIFIER_POINTER (DECL_NAME (gimple_call_fndecl (call_stmt)));
  tree tmp_var = NULL_TREE;

  if (strcmp (fname, "assert") == 0) {
    tree expr = gimple_call_arg (call_stmt, 0);
    gcc_assert (expr);
    if (ctx->path_cond) {
      symee_expr and_expr = build2 (TRUTH_AND_EXPR, boolean_type_node,
                                    ctx->path_cond, expr);
      ctx->path_cond = eval_expr_tree (ctx, and_expr);
    }
    else
      ctx->path_cond = eval_expr_tree (ctx, expr);

    if (dump_details) {
      fprintf (dump_file, "assert change path_cond:\n");
      dump_symbolic_expr (dump_file, expr, ctx->path_cond);
    }
    return ctx;
  }

  if (lhs) {
    char var_name [256] = "call.";
    strcat (var_name, fname);
    strcat (var_name, ".suffix");
    tmp_var = add_new_static_var (ret_type);
    DECL_NAME (tmp_var) = create_tmp_var_name (var_name);
  }

  symee_set_state_sym (ctx, lhs, tmp_var);
  if (dump_details)
    dump_symbolic_expr (dump_file, lhs, tmp_var);

  if (lhs && strcmp (fname, "malloc") == 0)
    {
      symee_set_range_base (ctx, lhs, tmp_var);
      tree size = gimple_call_arg (call_stmt, 0);
      tree min = tmp_var;
      tree max = build2 (PLUS_EXPR, TREE_TYPE (lhs), min, size);
      symee_range range = symee_range_create (0, min, max);
      symee_set_sym_index_range (ctx, lhs, range);
    }
  
  return ctx;
}

static symee_cgraph_node_info
get_cgraph_node_info (cgraph_node_ptr node)
{
  symee_cgraph_node_info node_info = NULL;
  void** slot = pointer_map_contains (cgraph_node_info_map, node);

  if (slot)
    node_info = *slot;

  return node_info;
}

static symee_expr
cgraph_node_get_retsym (cgraph_node_ptr node)
{
  symee_expr retsym = NULL;
  symee_cgraph_node_info node_info = get_cgraph_node_info (node);

  if (node_info)
    retsym = node_info->result_decl;

  return retsym;
}


static symee_expr
cgraph_node_get_insym (cgraph_node_ptr node, tree parm)
{
  symee_expr insym = NULL;
  symee_cgraph_node_info node_info = get_cgraph_node_info (node);

  if (node_info) {
    void** slot = pointer_map_contains (node_info->formal_in, parm);
    if (slot)
      insym = *slot;
  }
  
  return insym;
}


static symee_expr
cgraph_node_get_outsym (cgraph_node_ptr node, tree parm)
{
  symee_expr outsym = NULL;
  symee_cgraph_node_info node_info = get_cgraph_node_info (node);

  if (node_info) {
    void** slot = pointer_map_contains (node_info->formal_out, parm);
    if (slot)
      outsym = *slot;
  }
  
  return outsym;
}

static symee_context
gen_context_call_2 (symee_context ctx, gimple call_stmt)
{
  struct cgraph_node* caller_node = current_cgraph_node;
  struct cgraph_edge* edge = cgraph_edge (caller_node, call_stmt);
  struct cgraph_node* callee_node = edge->callee;
  symee_context in_ctx = symee_context_copy (ctx);
  symee_context callee_ctx = cgraph_node_get_context (callee_node);
  symee_context out_ctx = NULL;
  symee_context actual_out_ctx = symee_context_create ();
  symee_context ret_ctx = symee_context_copy (ctx);
  tree fndecl = gimple_call_fndecl (call_stmt);
  tree parm;
  size_t i = 0;
  symee_expr lhs = gimple_call_lhs (call_stmt);
  
  /* parameter in */
  for (parm = DECL_ARGUMENTS (fndecl); parm; parm = DECL_CHAIN (parm)) {
    if (i < gimple_call_num_args (call_stmt)) {
      tree argi = gimple_call_arg (call_stmt, i);
      symee_expr actual_expr = eval_expr_tree (ctx, argi);
      if (eval_formal_insym) {
        symee_expr insym = cgraph_node_get_insym (callee_node, parm);
        symee_set_state_sym (in_ctx, insym, actual_expr);
      }
      else
        symee_set_state_sym (in_ctx, parm, actual_expr);

      if (eval_formal_outsym) {
        symee_expr outsym = cgraph_node_get_outsym (callee_node, parm);
        symee_set_state_sym (actual_out_ctx, outsym, actual_expr);
      }
    }
    else {
      symee_assert (0, (stderr, "SYMEE: can not process variable call arguments\n"));
    }
    
    i++;
  }

  /* re-eval callee_ctx under in_ctx */
  out_ctx = eval_context (in_ctx, callee_ctx);

  /* return value */
  symee_expr retsym = cgraph_node_get_retsym (callee_node);
  symee_expr retval = symee_delete_state_sym (out_ctx, retsym);
  if (lhs) {
    symee_set_state_sym (ret_ctx, lhs, retval);
    if (dump_details)
      dump_symbolic_expr (dump_file, lhs, retval);
  }


/* parameter out */
ret_ctx = merge_callee_context (ret_ctx, call_stmt, in_ctx, out_ctx, actual_out_ctx);

#if 0
  /* parameter out */
  for (parm = DECL_ARGUMENTS (fndecl); parm; parm = DECL_CHAIN (parm)) {
    if (i < gimple_call_num_args (call_stmt)) {
      tree argi = gimple_call_arg (call_stmt, i);
      symee_expr actual_expr = eval_expr_tree (ctx, argi);
      if (eval_formal_insym) {
        symee_expr insym = cgraph_node_get_insym (callee_node, parm);
        symee_set_state_sym (in_ctx, insym, actual_expr);
      }
      else
        symee_set_state_sym (in_ctx, parm, actual_expr);
    }
    else {
      symee_assert (0, (stderr, "SYMEE: can not process variable call arguments\n"));
    }
    
    i++;
  }
#endif
  
  if (dump_details) {
    fprintf (dump_file, "result context of merge_callee_context\n");
    dump_context_brief (dump_file, ret_ctx);
  }

  return ret_ctx;
}

static symee_context
gen_context_call (symee_context ctx, gimple call_stmt)
{
  struct cgraph_node* node = current_cgraph_node;
  struct cgraph_edge* edge = cgraph_edge (node, call_stmt);
  struct cgraph_node* callee_node = edge->callee;
  symee_context callee_ctx = NULL;

  if (!callee_node)
    return gen_context_call_indirect (ctx, call_stmt);

  callee_ctx = cgraph_node_get_context (callee_node);

  /* no body or no ctx for callee */
  if (!callee_ctx)
    return gen_context_call_no_body (ctx, call_stmt);

  return gen_context_call_2 (ctx, call_stmt);
}

static void
execute_stmt_call (symee_context ctx)
{
  gimple stmt = symee_context_get_next (ctx);
  symee_context call_ctx = gen_context_call (ctx, stmt);

  symee_context_cpy (ctx, call_ctx);
}

static void
process_cond_edge (symee_context ctx, struct loop *loop,
		   edge e, tree branch_cond,
		   VEC(symee_context,heap) **out_ctx)
{
  basic_block target_bb = e->dest;
  symee_context target_ctx;
  tree path_cond = symee_get_path_cond (ctx);
  tree fold_path_cond;

  if (symee_is_const_false (path_cond))
    return;

  if (symee_is_const_true (path_cond))
    {
      path_cond = branch_cond;
      if (symee_is_const_false (path_cond))
	return;

      if (symee_is_const_true (path_cond))
	path_cond = boolean_true_node;
    }
  else
    {
      if (symee_is_const_false (branch_cond))
	return;

      if (!symee_is_const_true (branch_cond)
	  && !symee_implies (path_cond, branch_cond))
	path_cond = build2 (TRUTH_AND_EXPR, boolean_type_node,
			    path_cond, branch_cond);
    }

  fold_path_cond = eval_expr_tree (ctx, path_cond);
  path_cond = fold_path_cond ? fold_path_cond : path_cond;

  if (!symee_is_const_false (path_cond))
    {
      if (symee_is_const_true (path_cond))
	path_cond = boolean_true_node;
      
      target_ctx = symee_context_copy (ctx);
      basic_block prev;
      /* bypass empty basic blocks */
      while (!first_stmt (target_bb))
	{
	  prev = target_bb;
	  target_bb = single_succ (target_bb);
	}
      symee_context_set_prev (target_ctx, prev);
      symee_context_set_next (target_ctx, first_stmt (target_bb));
      symee_set_path_cond (target_ctx, path_cond);

      if (loop_exit_edge_p (loop, e))
	VEC_safe_push (symee_context, heap, *out_ctx, target_ctx);
      else
	VEC_safe_push (symee_context, heap, worklist, target_ctx);
    }
}

static void
execute_stmt_cond (symee_context ctx, struct loop *loop,
		   VEC(symee_context,heap) **out_ctx)
{
  gimple stmt = symee_context_get_next (ctx);
  enum tree_code cond_expr_code = gimple_cond_code (stmt);
  tree cond_expr_lhs = gimple_cond_lhs (stmt);
  tree cond_expr_rhs = gimple_cond_rhs (stmt);
  tree cond_expr = build2 (cond_expr_code, boolean_type_node,
                           cond_expr_lhs, cond_expr_rhs);
  tree not_cond_expr = build1 (TRUTH_NOT_EXPR, boolean_type_node, cond_expr);
  tree fold_cond_expr = eval_expr_tree (ctx, cond_expr);
  tree true_cond_expr = fold_cond_expr ? fold_cond_expr : cond_expr;
  tree fold_not_cond_expr = eval_expr_tree (ctx, not_cond_expr);
  tree false_cond_expr = fold_not_cond_expr ? fold_not_cond_expr : not_cond_expr;
  edge true_edge;
  edge false_edge;

  extract_true_false_edges_from_block (gimple_bb (stmt),
				       &true_edge, &false_edge);
  process_cond_edge (ctx, loop, true_edge, true_cond_expr, out_ctx);
  process_cond_edge (ctx, loop, false_edge, false_cond_expr, out_ctx);
}

static void
process_switch_edge (symee_context ctx, struct loop *loop,
		     edge e, tree index, tree branch_cond,
		     tree case_high, tree case_low,
		     VEC(symee_context,heap) **out_ctx)
{
  basic_block target_bb = e->dest;
  symee_context target_ctx;
  tree path_cond = symee_get_path_cond (ctx);
  tree fold_path_cond;

  if (symee_is_const_false (path_cond))
    return;

  if (symee_is_const_true (path_cond))
    {
      path_cond = branch_cond;
      if (symee_is_const_false (path_cond))
	return;

      if (symee_is_const_true (path_cond))
	path_cond = boolean_true_node;
    }
  else
    {
      if (symee_is_const_false (branch_cond))
	return;

      if (!symee_is_const_true (branch_cond)
	  && !symee_implies (path_cond, branch_cond))
	path_cond = build2 (TRUTH_AND_EXPR, boolean_type_node,
			    path_cond, branch_cond);
    }

  fold_path_cond = eval_expr_tree (ctx, path_cond);
  path_cond = fold_path_cond ? fold_path_cond : path_cond;

  if (!symee_is_const_false (path_cond))
    {
      if (symee_is_const_true (path_cond))
	path_cond = boolean_true_node;
      
      target_ctx = symee_context_copy (ctx);
      basic_block prev;
      while (!first_stmt (target_bb))
	{
	  prev = target_bb;
	  target_bb = single_succ (target_bb);
	}
      symee_context_set_prev (target_ctx, prev);
      symee_context_set_next (target_ctx, first_stmt (target_bb));

      if (!case_high)
	symee_set_state_sym (target_ctx, index, case_low);
      else
	symee_set_path_cond (target_ctx, path_cond);

      if (loop_exit_edge_p (loop, e))
	VEC_safe_push (symee_context, heap, *out_ctx, target_ctx);
      else
	VEC_safe_push (symee_context, heap, worklist, target_ctx);
    }
}

static void
process_switch_default_edge (symee_context ctx, struct loop *loop,
			     edge e, tree index, tree branch_cond,
			     VEC(symee_context,heap) **out_ctx)
{
  basic_block target_bb = e->dest;
  symee_context target_ctx;
  tree path_cond = symee_get_path_cond (ctx);
  tree fold_path_cond;

  if (symee_is_const_false (path_cond))
    return;

  if (symee_is_const_true (path_cond))
    {
      path_cond = branch_cond;
      if (symee_is_const_false (path_cond))
	return;

      if (symee_is_const_true (path_cond))
	path_cond = boolean_true_node;
    }
  else
    {
      if (symee_is_const_false (branch_cond))
	return;

      if (!symee_is_const_true (branch_cond)
	  && !symee_implies (path_cond, branch_cond))
	path_cond = build2 (TRUTH_AND_EXPR, boolean_type_node,
			    path_cond, branch_cond);
    }

  fold_path_cond = eval_expr_tree (ctx, path_cond);
  path_cond = fold_path_cond ? fold_path_cond : path_cond;

  if (!symee_is_const_false (path_cond))
    {
      if (symee_is_const_true (path_cond))
	path_cond = boolean_true_node;
      
      target_ctx = symee_context_copy (ctx);
      basic_block prev;
      while (!first_stmt (target_bb))
	{
	  prev = target_bb;
	  target_bb = single_succ (target_bb);
	}
      symee_context_set_prev (target_ctx, prev);
      symee_context_set_next (target_ctx, first_stmt (target_bb));
      symee_set_path_cond (target_ctx, path_cond);

      if (loop_exit_edge_p (loop, e))
	VEC_safe_push (symee_context, heap, *out_ctx, target_ctx);
      else
	VEC_safe_push (symee_context, heap, worklist, target_ctx);
    }
}

static void
execute_stmt_switch (symee_context ctx, struct loop *loop,
		     VEC(symee_context,heap) **out_ctx)
{
  gimple stmt = symee_context_get_next (ctx);
  tree index = gimple_switch_index (stmt);
  symee_expr sym = eval_expr_tree (ctx, index);
  size_t n = gimple_switch_num_labels (stmt);
  tree default_label = gimple_switch_default_label (stmt);
  tree default_cond_expr = NULL_TREE;
  tree folded_default_cond = NULL_TREE;
  size_t i;

  /* default_label must be label 0 */
  for (i = 1; i < n; i++)
    {
      tree case_label_expr = gimple_switch_label (stmt, i);
      tree case_label = CASE_LABEL (case_label_expr);
      basic_block case_bb = label_to_block (case_label);
      edge case_edge = find_edge (gimple_bb (stmt), case_bb);
      tree case_low = CASE_LOW (case_label_expr);
      tree case_high = CASE_HIGH (case_label_expr);
      tree case_cond_expr = NULL_TREE;
      tree not_case_cond_expr = NULL_TREE;
      tree folded_case_cond = NULL_TREE;
      tree folded_not_case_cond = NULL_TREE;
      symee_expr path_cond = NULL_TREE;
      symee_context case_ctx = symee_context_copy (ctx);
    
      /* index >= low && index <= high */
      if (case_high)
	{
	  tree ge_expr = build2 (GE_EXPR, boolean_type_node, index, case_low);
	  tree le_expr = build2 (LE_EXPR, boolean_type_node, index, case_high);
	  case_cond_expr = build2 (TRUTH_AND_EXPR, boolean_type_node,
				   ge_expr, le_expr);
	}
      else
	case_cond_expr = build2 (EQ_EXPR, boolean_type_node, index, case_low);

      folded_case_cond = eval_expr_tree (ctx, case_cond_expr);
      if (folded_case_cond)
	case_cond_expr = folded_case_cond;

      not_case_cond_expr = build1 (TRUTH_NOT_EXPR, boolean_type_node,
				   case_cond_expr);
      folded_not_case_cond = eval_expr_tree (ctx, not_case_cond_expr);
      if (folded_not_case_cond)
	not_case_cond_expr = folded_not_case_cond;

      if (default_cond_expr)
	  default_cond_expr = build2 (TRUTH_AND_EXPR, boolean_type_node,
				      default_cond_expr, not_case_cond_expr);
      else
	default_cond_expr = not_case_cond_expr;

      folded_default_cond = eval_expr_tree (ctx, default_cond_expr);
      if (folded_default_cond)
	default_cond_expr = folded_default_cond;
    
      process_switch_edge (ctx, loop, case_edge, index, case_cond_expr,
			   case_high, case_low, out_ctx);

      if (dump_details)
	{
	  fprintf (dump_file, "switch: case %zd path_cond = ", i);
	  print_generic_expr (dump_file, case_ctx->path_cond, 0);
	  fprintf (dump_file, "\n");
	}
    }

  /* default_label, 0'th label */
  symee_context default_ctx = symee_context_copy (ctx);
  basic_block default_bb = label_to_block (CASE_LABEL (default_label));
  edge default_edge = find_edge (gimple_bb (stmt), default_bb);

  process_switch_default_edge (ctx, loop, default_edge, index,
			       default_cond_expr, out_ctx);

  if (dump_details)
    {
      fprintf (dump_file, "switch: default path_cond = ");
      print_generic_expr (dump_file, default_ctx->path_cond, 0);
      fprintf (dump_file, "\n");
    }
}

static void
execute_stmt_phi (symee_context ctx)
{
  gimple phi = symee_context_get_next (ctx);
  basic_block last = symee_context_get_prev (ctx);
  size_t n = gimple_phi_num_args (phi);
  size_t i;
  tree lhs = gimple_phi_result (phi);
  tree rhs = NULL;

  // gcc_assert (is_gimple_reg (gimple_phi_result (phi)));
  gcc_assert (n > 0);

  if (dump_file)
    {
      fprintf (dump_file, "executing gimple phi (%p):\n", phi);
      print_gimple_stmt (dump_file, phi, 0, TDF_VOPS|TDF_MEMSYMS);
    }

  for (i = 0; i < n; i++)
    {
      tree op = gimple_phi_arg_def (phi, i);
      edge e = gimple_phi_arg_edge (phi, i);
      /* lhs = op */
      if (e->src == last)
	{
	  rhs = symee_get_state_sym (ctx, op);
	  break;
	}
    }

  gcc_assert (rhs);
  symee_set_state_sym (ctx, lhs, rhs);

  if (dump_file)
    {
      dump_symbolic_expr (dump_file, lhs, rhs);
    }
}

static void
execute_stmt_ret (symee_context ctx)
{
  gcc_unreachable ();
}

static void
execute_stmt_goto (symee_context ctx)
{
}

static void
execute_stmt_label (symee_context ctx)
{
}

static void
execute_stmt_debug (symee_context ctx)
{
  gimple stmt = symee_context_get_next (ctx);

  gcc_assert (stmt && gimple_code (stmt) == GIMPLE_DEBUG);
  symee_expr var = gimple_debug_bind_get_var (stmt);
  symee_expr value = gimple_debug_bind_get_value (stmt);
  symee_expr expr = symee_get_state_sym (ctx, var);

  /* TODO: compare value and expr.
     can we change state var = value here ? */
}

static void
execute_stmt_ignore (symee_context ctx)
{
  gimple stmt = symee_context_get_next (ctx);

  /* gimple code for eh and omp will be not processed. */
  if (dump_file)
    fprintf (dump_file, "NYI: %s statement will be not processed!\n",
             gimple_code_name [(int) gimple_code (stmt)]);

  fprintf (stderr, "NYI: %s statement will not be processed!\n",
           gimple_code_name [(int) gimple_code (stmt)]);
}


static void
execute_stmt_1 (symee_context ctx)
{
  gimple stmt = symee_context_get_next (ctx);

  if (dump_file)
    fprintf (dump_file, "NYI: %s statement can not be processed now!\n",
             gimple_code_name [(int) gimple_code (stmt)]);

  fprintf (stderr, "NYI: %s statement can not be processed now!\n",
           gimple_code_name [(int) gimple_code (stmt)]);

  gcc_unreachable ();
}

/* BB has a single successor?? */

static void
push_bb_succ (symee_context ctx, basic_block bb)
{
  edge e;
  edge_iterator ei;
  unsigned count = 0;
  symee_context target_ctx;

  gcc_assert (single_succ_p (bb));

  FOR_EACH_EDGE (e, ei, bb->succs)
    {
      gcc_assert (last_stmt (e->src) == symee_context_get_next (ctx));

      target_ctx = symee_context_copy (ctx);
      basic_block target = e->dest;
      basic_block prev;
      /* bypass empty basic blocks */
      while (!first_stmt (target))
	{
	  prev = target;
	  target = single_succ (target);
	}
      symee_context_set_prev (target_ctx, prev);
      symee_context_set_next (target_ctx, first_stmt (target));
      VEC_safe_push (symee_context, heap, worklist, target_ctx);
      count++;
    }

  gcc_assert (count == 1);
}

static void
execute_stmt_vops (symee_context ctx)
{
  gimple stmt = symee_context_get_next (ctx);
  tree lhs = gimple_vdef (stmt);
  tree rhs = gimple_vuse (stmt);

  if (!rhs)
    return;
#if 0
  if (!lhs)
    lhs = gimple_get_lhs (stmt);
#endif
  if (!lhs)
    return;
  
  if (dump_details)
    {
      dump_symbolic_expr (dump_file, lhs, rhs);
    }
  
  if (eval_reference &&
      REFERENCE_CLASS_P (lhs))
    execute_stmt_assign_ref (ctx, lhs, rhs);
  else
    symee_set_state_sym (ctx, lhs, rhs);
}

/* Symbolic execution of a gimple statement. CTX is the context before
   the statement. OUT_CTX is the set of loop exit contexts. */

static void
execute_stmt (symee_context ctx, struct loop *loop,
	      VEC(symee_context,heap) **out_ctx)
{
  gimple stmt = symee_context_get_next (ctx);
  
  if (dump_file)
    {
      fprintf (dump_file, "\nexecuting gimple stmt (%p):\n", stmt);
      print_gimple_stmt (dump_file, stmt, 0, dump_flags);
    }

  /* TODO: vdef/vuse */
  if (1) {
    ssa_op_iter iter;
    use_operand_p use_p;
    tree op;

    FOR_EACH_SSA_USE_OPERAND (use_p, stmt, iter, SSA_OP_USE|SSA_OP_VUSE)
      {
        op = USE_FROM_PTR (use_p);
        gcc_assert (!REFERENCE_CLASS_P (op));
      }
  }
  
  switch (gimple_code (stmt))
    {
    case GIMPLE_ASSIGN:
      execute_stmt_assign (ctx);
      break;
    
    case GIMPLE_CALL:
      execute_stmt_call (ctx);
      break;
    
    case GIMPLE_COND:
      execute_stmt_cond (ctx, loop, out_ctx);
      break;
    
    case GIMPLE_SWITCH:
      execute_stmt_switch (ctx, loop, out_ctx);
      break;
    
    case GIMPLE_GOTO:
      execute_stmt_goto (ctx);
      break;
    
    case GIMPLE_LABEL:
      execute_stmt_label (ctx);
      break;
    
    case GIMPLE_RETURN:
      execute_stmt_ret (ctx);
      break;
    
    case GIMPLE_DEBUG:
      execute_stmt_debug (ctx);
      break;
    
    case GIMPLE_PHI:
      execute_stmt_phi (ctx);
      break;
    
    case GIMPLE_ASM:
    case GIMPLE_BIND:
    case GIMPLE_ERROR_MARK:
      execute_stmt_1 (ctx);
      break;
    
    case GIMPLE_NOP:
    case GIMPLE_PREDICT:
    default:
      execute_stmt_ignore (ctx);
      break;
    }

  if (stmt == last_stmt (gimple_bb (stmt))
      && gimple_code (stmt) != GIMPLE_COND
      && gimple_code (stmt) != GIMPLE_SWITCH)
    push_bb_succ (ctx, gimple_bb (stmt));
  /* if (eval_memsyms) */
  /*   ret_ctx = gen_context_stmt_vops (ret_ctx, stmt); */
}

/* Symbolic execution of the basic block BB. CTX is the context at
   the entry of the basic block. OUT_CTX is the set of loop exit contexts.
   This routine adds items into OUT_CTX through calling execute_stmt. */

static void
execute_bb (symee_context bb_ctx, struct loop *loop,
	    VEC(symee_context,heap) **out_ctx)
{
  gimple stmt = symee_context_get_next (bb_ctx);
  basic_block bb = gimple_bb (stmt);
  gimple_stmt_iterator gsi;

  for (gsi = gsi_start_phis (bb); !gsi_end_p (gsi); gsi_next (&gsi))
    {
      stmt = gsi_stmt (gsi);
      symee_context_set_next (bb_ctx, stmt);
      execute_stmt (bb_ctx, loop, out_ctx);
    }

  for (gsi = gsi_start_bb (bb); !gsi_end_p (gsi); gsi_next (&gsi))
    {
      stmt = gsi_stmt (gsi);
      symee_context_set_next (bb_ctx, stmt);
      execute_stmt (bb_ctx, loop, out_ctx);
    }

  //  symee_context_set_prev (bb_ctx, bb);
}

/* Return true if BB->loop_father has reached the maximum number of
   iterations flag_ipa_symee_loop_bound.
   TODO: Increase corresponding counter when BB is header
   of some loop. */

static bool
reached_max_iteration (basic_block bb)
{
  unsigned cnt;
  
  if (bb != bb->loop_father->header)
    return false;

  cnt = get_number_of_iterations (bb->loop_father);
 
  if (cnt >= flag_ipa_symee_loop_bound)
    return true;
  else
    return false;
}

static void
increase_iteration (basic_block bb)
{
  if (bb == bb->loop_father->header)
    increase_number_of_iterations (bb->loop_father);
}

/* Symbolic execution of the loop LOOP using explict state exploration.
   This routine is meant to be used for cases when the IV analysis
   fails to return meaningful result (chrec_dont_know).

   CTX is the context at the entry of the loop. This routine returns
   the set (VEC) of contexts at the exit of the loop. Each exit
   context is the result of the exercising one execution path through
   the loop. The PREV and NEXT fields of the symee_context_def structure
   tells the program point out of the loop. Return NULL is the set
   is empty.

   This routine is assumed to be called at the point right before the
   loop header (LOOP->HEADER). The maximum number of loop iterations is
   set by the comand line option fipa_symee_loop_bound=*. */

VEC(symee_context,heap) *
execute_loop (symee_context ctx, struct loop *loop)
{
  /* The set of contexts at loop exit. */
  VEC(symee_context,heap) *loop_exit_context = NULL;
  symee_context init_bb_ctx;

  if (dump_file && (dump_flags & TDF_DETAILS))
    {
      fprintf (dump_file, "(execute_loop\n");
      fprintf (dump_file, "  (loop_nb = %d)\n", loop->num);
    }

  /* Return NULL is current path condition is false. */
  if (symee_is_const_false (symee_get_path_cond (ctx)))
    return NULL;

  loop_exit_context = VEC_alloc (symee_context, heap, 5);
  worklist = VEC_alloc (symee_context, heap, 5);
  init_looop_iter_cnt ();
  init_bb_ctx = symee_context_copy (ctx);
  /* The next two lines can be removed if execute_loop is called with
   a proper ctx s.t. a direct copy above suffices. */
  symee_context_set_prev (init_bb_ctx,
			  loop_preheader_edge (loop)->src);
  symee_context_set_next (init_bb_ctx,
			  first_stmt (loop->header));
  /* Initialize the worklist with the header of the loop. */
  VEC_safe_push (symee_context, heap, worklist, init_bb_ctx);

  /* Extract items from the worklist until it is empty. */
  while (!VEC_empty (symee_context, worklist))
    {
      symee_context bb_ctx = VEC_pop (symee_context, worklist);
      basic_block bb = gimple_bb (symee_context_get_next (bb_ctx));

      if (dump_file && (dump_flags & TDF_DETAILS))
	{
	  fprintf (dump_file, "\n(LOOP: starting to process <bb %d>\n",
		   bb->index);
	  fprintf (dump_file, "bb_ctx:\n");
	  dump_context_brief (dump_file, bb_ctx);
	  dump_number_of_iterations (dump_file);
	  fprintf (dump_file, ")\n");
	}
      
      /* Bound loop iteration. */
      if (!reached_max_iteration (bb))
	{
	  increase_iteration (bb);
	  execute_bb (bb_ctx, loop, &loop_exit_context);
	}

      if (dump_file && (dump_flags & TDF_DETAILS))
	{
	  fprintf (dump_file, "(\nworklist:\n");
	  dump_vec_context (dump_file, worklist);
	  fprintf (dump_file, "loop_exit_context:\n");
	  dump_vec_context (dump_file, loop_exit_context);
	  dump_number_of_iterations (dump_file);
	  fprintf (dump_file, ")\n");
	}

      symee_context_destory (bb_ctx);
    }
  
  /* Free the worklist at the end. */
  VEC_free (symee_context, heap, worklist);
  destroy_number_of_iterations ();

  if (dump_file && (dump_flags & TDF_DETAILS))
    {
      fprintf (dump_file, "LOOP END:\nloop_exit_context:\n");
      dump_vec_context (dump_file, loop_exit_context);
      fprintf (dump_file, ")\n");
    }

  return loop_exit_context;
}
