/* symbolic-expr-eval.h  -*- c -*- */

#ifndef GCC_SYMBOLIC_EXPR_EVAL_H
#define GCC_SYMBOLIC_EXPR_EVAL_H

#ifdef __cplusplus
extern "C" {
#endif

#include "config.h"
#include "system.h"
#include "coretypes.h"
#include "ggc.h"
#include "tree.h"
#include "target.h"
#include "gimple.h"
#include "cgraph.h"
#include "flags.h"
#include "timevar.h"
#include "diagnostic.h"
#include "params.h"
#include "cfgloop.h"
#include "pointer-set.h"
#include "bitmap.h"
#include "tree-flow.h"
#include "tree-pass.h"
#include "tree-chrec.h"
#include "tree-scalar-evolution.h"
#include "tree-data-ref.h"
#include "tree-pretty-print.h"
#include "tree-dump.h"
#include <string.h>

#include "tree-vrp.h"

#include "symbolic-pointer-set.h"

/* Fahringer's symbolic expression evaluation framework (SYMEE) */

typedef tree symee_expr;
typedef const_tree symee_const_expr;
typedef struct symee_context_def* symee_context;
typedef const struct symee_context_def* symee_const_context;

DEF_VEC_P(symee_context);
DEF_VEC_ALLOC_P(symee_context,heap);

struct symee_range_def {
  struct symee_range_def* prev;
  struct symee_range_def* next;

  tree min;
  tree max;

  int type;
};

typedef struct symee_range_def *symee_range;

struct symee_range_list_def {
  symee_range head;
  symee_range tail;

  tree sym;
  tree base;
};

typedef struct symee_range_list_def* symee_range_list;


struct symee_ref_node_def {
  struct symee_ref_node_def* prev;
  struct symee_ref_node_def* next;

  gimple stmt;
  tree ref;
  struct data_reference* dr;

  tree base;                    /* see dr */
  tree index;                   /* see dr */

  tree evolution;               /* colletion of reference with */
  struct loop* l;
  tree n_iters;
  
  tree value;                   /* symee_expr */

  bool is_read;
  
  /* base flags */
  unsigned int ref_base_is_normal_array : 1;
  unsigned int ref_base_is_pointer_to_array : 1;
  unsigned int ref_base_is_unique_ptr : 1;

};

typedef struct symee_ref_node_def* symee_ref_node;

struct symee_ref_list_def {
  symee_ref_node head;
  symee_ref_node tail;

  /* all nodes have same base ? */
  tree base;
};

typedef struct symee_ref_list_def* symee_ref_list;


struct GTY(()) symee_context_def {
  int kind;
  unsigned int flags;

  /* program point after 'stmt' */
  gimple stmt;

  basic_block prev;
  gimple next;

  symbolic_pointer_map_ptr GTY((param1_is (tree), param2_is (tree))) state;
  symee_expr state_cond;
  symee_expr path_cond;

  symbolic_pointer_map_ptr GTY((param1_is (tree), param2_is (tree))) property;

  /* array -> range */
  symbolic_pointer_map_ptr GTY((param1_is (tree), param2_is (tree))) value_range;
  symbolic_pointer_map_ptr GTY((param1_is (tree), param2_is (tree))) index_range;

  /* non-scalar -> symee_ref_list */
  symbolic_pointer_map_ptr GTY((param1_is (tree), param2_is (tree))) ref_state;
  
};


struct GTY(()) symee_cgraph_node_info_def {
  cgraph_node_ptr node;

  int flags;
  VEC(tree,heap) * GTY((skip)) parm_decls;
  tree formal_chain;
  tree result_decl;
  tree heap_pointer;
  
  /* map from pointer before gimple statement 'stmt' to ctx */
  symbolic_pointer_map_ptr GTY((param1_is (gimple), param2_is (symee_context))) entrycontext;
  symbolic_pointer_map_ptr GTY((param1_is (gimple), param2_is (symee_context))) exitcontext;
  /* map from cfg bb and edge to ctx */
  symbolic_pointer_map_ptr GTY((param1_is (basic_block), param2_is (symee_context))) bbentrycontext;
  symbolic_pointer_map_ptr GTY((param1_is (basic_block), param2_is (symee_context))) bbexitcontext;
  symbolic_pointer_map_ptr GTY((param1_is (edge), param2_is (symee_context))) edgecontext;
  
  symbolic_pointer_map_ptr GTY((param1_is (tree), param2_is (tree))) formals;
  symbolic_pointer_map_ptr GTY((param1_is (tree), param2_is (tree))) formal_in;
  symbolic_pointer_map_ptr GTY((param1_is (tree), param2_is (tree))) formal_out;
  symbolic_pointer_map_ptr GTY((param1_is (tree), param2_is (tree))) exit_state;

  symee_context insym_context;
  symee_context outsym_context;
  symee_context exit_context;

  symee_context working_context;
  gimple processing_stmt;
  
  bitmap GTY((skip)) visited_bbs;

  int* GTY((skip)) rpo;
  int rpo_cnt;
};

typedef struct symee_cgraph_node_info_def* symee_cgraph_node_info;
typedef const struct symee_cgraph_node_info_def* symee_const_cgraph_node_info;


typedef enum SYMEE_PHASE_ENUM {
  SYMEE_PHASE_LOCAL_SUMMARY,
  SYMEE_PHASE_FUNCTION_SUMMARY,
  SYMEE_PHASE_GLOBAL,
} SYMEE_PHASE;

struct GTY(()) symee_global_info_def {
  SYMEE_PHASE phase;
  
  /* symee_cgraph_node_info for each cgraph_node */
  splay_tree GTY((param1_is (cgraph_node_ptr), param2_is (symee_cgraph_node_info))) node_info;

  /* working */
  struct symee_cgraph_node_info_def * GTY((skip)) current_node_info;

  /* eval flags */
  unsigned int symee_eval_flags;

  int sym_exec_loop;
  int use_ref_list;
  int run_vrp;
  int forward_r;
};

typedef struct symee_global_info_def *symee_global_info;


extern GTY(()) symee_global_info global_info;

/* symee eval flags */

#define SYMEE_EVAL_SCALAR               (1 << 0)

#define SYMEE_EVAL_ARRAY_REF            (1 << 4)
#define SYMEE_EVAL_COMPONENT_REF        (1 << 5)
#define SYMEE_EVAL_MEM_REF              (1 << 6)
#define SYMEE_EVAL_MORE_REFERENCE       (1 << 7)

#define SYMEE_EVAL_VOPS                 (1 << 8)
#define SYMEE_EVAL_MEMSYMS              (1 << 9)

#define SYMEE_EVAL_INSYM                (1 << 12)
#define SYMEE_EVAL_OUTSYM               (1 << 13)

#define SYMEE_EVAL_NOTHING              0x0
#define SYMEE_EVAL_NULL                 0x0
#define SYMEE_EVAL_ALL                  0x0fffffff

#define SYMEE_EVAL_REFERENCE (SYMEE_EVAL_ARRAY_REF                      \
                              | SYMEE_EVAL_COMPONENT_REF                \
                              | SYMEE_EVAL_MEM_REF                      \
                              | SYMEE_EVAL_MORE_REFERENCE)

#define SYMEE_EVAL_INOUTSYM (SYMEE_EVAL_INSYM           \
                             | SYMEE_EVAL_OUTSYM)

#define symee_eval_flags (global_info->symee_eval_flags)

#define eval_reference (symee_eval_flags & (SYMEE_EVAL_REFERENCE))
#define eval_memsyms (symee_eval_flags & (SYMEE_EVAL_VOPS | SYMEE_EVAL_MEMSYMS))
#define eval_formal_insym (symee_eval_flags & (SYMEE_EVAL_INSYM))
#define eval_formal_outsym (symee_eval_flags & (SYMEE_EVAL_OUTSYM))
#define eval_global_insym (symee_eval_flags & (SYMEE_EVAL_INSYM))
#define eval_global_outsym (symee_eval_flags & (SYMEE_EVAL_OUTSYM))


#define dump_details (dump_file && (dump_flags & TDF_DETAILS))

#define symee_assert(expr,fprintf_args) \
  ((void)(expr ? 0: (fprintf fprintf_args, gcc_assert (expr), 0)))

#define symee_warn(expr,fprintf_args)           \
  ((void)(expr ? 0: (fprintf fprintf_args, 0)))


#define current_node_info (global_info->current_node_info)
#define current_cgraph_node (current_node_info->node)
#define current_context (current_node_info->working_context)
#define current_formals (current_node_info->formals)
#define current_stmt (current_node_info->processing_stmt)

#define symee_sym_exec_loop (global_info->sym_exec_loop)
#define symee_use_ref_list (global_info->use_ref_list)
#define symee_run_vrp (global_info->run_vrp)
#define symee_forward_recursive (global_info->forward_r)


symee_context symee_context_create (void);
symee_context symee_context_copy (symee_context ctx);

symee_context symee_get_context (struct cgraph_node* node);
symee_context symee_get_entrycontext (struct cgraph_node* node, gimple stmt);

symee_expr symee_get_state_sym (symee_context ctx, symee_expr sym);
void symee_set_state_sym (symee_context ctx, symee_const_expr sym, symee_expr expr);

symee_expr symee_get_state_cond (symee_context ctx);
void symee_set_state_cond (symee_context ctx, symee_expr state_cond);

symee_expr symee_get_path_cond (symee_context ctx);
void symee_set_path_cond (symee_context ctx, symee_expr path_cond);

extern symee_expr symee_get_state_sym_by_ref (symee_context ctx, tree ref);

extern basic_block symee_context_get_prev (symee_context);
extern void symee_context_set_prev (symee_context, basic_block);
extern gimple symee_context_get_next (symee_context);
extern void symee_context_set_next (symee_context, gimple);

extern void dump_context_brief (FILE *, symee_context);

extern symee_ref_list symee_get_state_sym_ref_list_create (symee_context, symee_expr);
extern symee_ref_node symee_ref_list_append_ref (symee_ref_list, tree);
extern symee_ref_list symee_ref_list_kill (symee_ref_list, symee_ref_node);

/* symee will use these evaluation tools provided by ... */
symee_expr symee_eval (symee_context ctx, symee_expr expr);
symee_expr symee_loopeval (symee_context ctx, symee_expr expr);

symee_context symee_gen_context_loop (symee_context ctx, struct loop* l);
extern VEC(symee_context,heap) *execute_loop (symee_context, struct loop *);

void symee_dump_context (FILE* fp, symee_context ctx);
extern symee_expr eval_expr_tree (symee_context ctx, symee_expr expr);

extern symee_cgraph_node_info symee_get_cgraph_node_info (cgraph_node_ptr node);

/**/
typedef bool (*symee_context_visit_fn) (symee_context, void*);
void symee_propagate_context (symee_context, symee_context_visit_fn, void*);


/* Call symee_initialize before generate context,
   Call symee_finalize to free memory when you do not need context. */

void symee_initialize (unsigned int flags);
void symee_generate_context (void);
void symee_finalize (void);


/* symee client */


/* alias */
extern unsigned int symee_init_alias (void);
extern void symee_fini_alias (void);
extern bool symee_refs_may_alias_p (tree ref1, tree ref2);
extern bool symee_refs_must_alias_p (tree ref1, tree ref2);
extern bool symee_dr_may_alias_p (struct data_reference *ref1, struct data_reference *ref2);


extern void symee_flatten (struct cgraph_node *node);
extern unsigned int symee_inline_transform (struct cgraph_node *node);

/* debug function */
DEBUG_FUNCTION bool symee_debug (void);
DEBUG_FUNCTION bool symee_debug_name (const char* name1, const char* name2);
DEBUG_FUNCTION bool symee_debug_function (const char* fname);

/* alias */
extern unsigned int symee_init_alias (void);
extern void symee_fini_alias (void);
extern bool symee_refs_may_alias_p (tree ref1, tree ref2);
extern bool symee_refs_must_alias_p (tree ref1, tree ref2);
extern bool symee_dr_may_alias_p (struct data_reference *ref1, struct data_reference *ref2);

#ifdef __cplusplus
} /* extern "C" */
#endif

#endif /* GCC_SYMBOLIC_EXPR_EVAL_H */

