/* symbolic-expr-eval.c  -*- C -*- */

#include "config.h"
#include "system.h"
#include "coretypes.h"
#include "ggc.h"
#include "tree.h"
#include "target.h"
#include "gimple.h"
#include "cgraph.h"
#include "flags.h"
#include "timevar.h"
#include "diagnostic.h"
#include "params.h"
#include "cfgloop.h"
#include "bitmap.h"
#include "tree-flow.h"
#include "tree-pass.h"
#include "tree-chrec.h"
#include "tree-scalar-evolution.h"
#include "tree-data-ref.h"
#include "tree-pretty-print.h"
#include "tree-dump.h"
#include <string.h>

#include "symbolic-expr-eval.h"

/* symbolic expression evaluation (symee) framework

 1. multiple representations
   * gimple
   * yices
   * other CAS, such as Mathematica or Sage

 2. multiple CASs
   * yices (not CAS?)
   * sage
   * mathematica

 3. two pass propagation
   * summary
   ** local summary
        statement, block, edge
   ** function summary
        cgraph_node

   * propagation
   ** propagate_context_forward
   ** propagate_context_backward

 */


symee_global_info global_info = NULL;

static tree get_base_sym (const_tree t);
static bool is_interesting_sym (symee_const_expr sym, symee_expr expr);
static tree add_new_sym (tree t);

static void dump_symbolic_expr (FILE* fp, symee_const_expr sym,
                                symee_const_expr expr);
static bool same_lvalue_array (const_tree expr1, const_tree expr2);
static bool same_lvalue_deref (const_tree expr1, const_tree expr2);
static bool same_lvalue (const_tree expr1, const_tree expr2);
static bool same_array_ref_sym (const void* key, void** slot, void* data);
static bool same_indirect_ref_sym (const void* key, void** slot, void* data);
static symee_context gen_context_ref (symee_context ctx, gimple stmt,
                                      tree ref, tree value);
extern symee_expr symee_get_state_sym_by_ref (symee_context ctx, tree ref);

static bool is_loop_preheader (basic_block bb);
static bool is_loop_header (basic_block bb);
static bool is_loop_postbody (basic_block bb);
static bool is_loop_postexit (basic_block bb);

extern bool symee_is_const_true (tree a);
extern bool symee_is_const_false (tree a);
extern bool symee_value_equal_p (tree a, tree b);


static symee_context gen_context_bb (symee_context ctx, basic_block bb);
static symee_context gen_context_cgraph_node (symee_context ctx, cgraph_node_ptr node);

static symee_expr cgraph_node_get_insym (cgraph_node_ptr node, tree parm);
static void init_start_node (basic_block bb);
static symee_expr subst_formal_outsym (symee_expr sym);


//#define MEM_AS_ARRAY

void
fprintf_loc (FILE* fp, location_t loc, const char *cmsgid, ...)
{
  va_list ap;
  expanded_location xloc = expand_location (loc);

#if 1
  fprintf (fp, "%s:%d:", xloc.file, xloc.line);
#else
  fprintf (fp, "%s:%d:%d:", xloc.file, xloc.line, xloc.column);
#endif

  va_start (ap, cmsgid);
  vfprintf (fp, cmsgid, ap);
  va_end (ap);
}


DEBUG_FUNCTION bool
symee_debug (void)
{
  return true;
}


DEBUG_FUNCTION bool
symee_debug_prefix (const char* name1, const char* prefix)
{
  if (strncmp (name1, prefix, strlen (prefix)) == 0)
    return symee_debug ();
  return false;
}


DEBUG_FUNCTION bool
symee_debug_name (const char* name1, const char* name2)
{
  if (strcmp (name1, name2) == 0)
    return symee_debug ();
  return false;
}


DEBUG_FUNCTION bool
symee_debug_function (const char* fname)
{
  const char* decl_name =
    IDENTIFIER_POINTER (DECL_ASSEMBLER_NAME (current_function_decl));
  return symee_debug_name (decl_name, fname);
}


/* dump_file stack */

static FILE* dump_file_stack [1024];
static int dump_file_stack_sp = 0;

static void
push_dump_file (void)
{
#if SYMEE_DUMP_CONTROL
  gcc_assert (dump_file_stack_sp >= 0 && dump_file_stack_sp <= 1000);
  dump_file_stack [dump_file_stack_sp++] = dump_file;
  dump_file = NULL;
#endif
}


static void
pop_dump_file (void)
{
#if SYMEE_DUMP_CONTROL
  dump_file = dump_file_stack [--dump_file_stack_sp];
  gcc_assert (dump_file_stack_sp >= 0 && dump_file_stack_sp <= 1000);
  dump_file_stack [dump_file_stack_sp] = NULL;
#endif
}


extern symee_context
symee_context_create (void)
{
  symee_context ctx = ggc_alloc_cleared_symee_context_def ();
  ctx->state = symbolic_pointer_map_create ();
  ctx->index_range = symbolic_pointer_map_create ();
  ctx->ref_state = symbolic_pointer_map_create ();
  ctx->state_cond = NULL;
  ctx->path_cond = boolean_true_node;

  gcc_assert (ctx->state);
  
  return ctx;
}


extern void
symee_context_destory (symee_context ctx)
{
  gcc_assert (ctx);
  if (ctx->state)
    symbolic_pointer_map_destroy (ctx->state);
  if (ctx->property)
    symbolic_pointer_map_destroy (ctx->property);
  if (ctx->ref_state)
    symbolic_pointer_map_destroy (ctx->ref_state);

  ggc_free (ctx);
}


extern symee_context
symee_context_copy (symee_context ctx)
{
  symee_context ret_ctx = symee_context_create ();

  gcc_assert (ctx);

  if (ctx->state)
    ret_ctx->state = symbolic_pointer_map_copy (ctx->state);
  if (ctx->ref_state)
    ret_ctx->ref_state = symbolic_pointer_map_copy (ctx->ref_state);
  /* TODO: deep copy */
  ret_ctx->state_cond = ctx->state_cond;
  ret_ctx->path_cond = ctx->path_cond;

  return ret_ctx;
}


extern symee_context
symee_context_copy_init (symee_context ctx)
{
  symee_context ret_ctx = symee_context_create ();

  if (!ctx)
    return ret_ctx;

  if (ctx->state)
    ret_ctx->state = symbolic_pointer_map_copy (ctx->state);
  if (ctx->ref_state)
    ret_ctx->ref_state = symbolic_pointer_map_copy (ctx->ref_state);
  /* TODO: deep copy */
  ret_ctx->state_cond = ctx->state_cond;
  ret_ctx->path_cond = ctx->path_cond;

  return ret_ctx;
}


/* return the value expr of under ctx, return NULL, if ctx does not contains
   definition of sym */

extern symee_expr
symee_get_state_sym_1_1 (symee_context ctx, symee_expr sym)
{
  symee_expr expr = NULL;
  void** slot = symbolic_pointer_map_contains (ctx->state, sym);
  if (slot)
    expr = *slot;
  return expr;
}


/* if sym -> NULL, return sym? */

extern symee_expr
symee_get_state_sym (symee_context ctx, symee_expr sym)
{
  symee_expr expr = NULL;
  void** slot = NULL;

  symee_assert (ctx && ctx->state,
                (stderr, "SYMEE: (%s) input ctx can not be NULL\n", __func__));

  slot = symbolic_pointer_map_contains (ctx->state, sym);
  
  if (!slot /*&& global_info->phase == SYMEE_PHASE_FUNCTION_SUMMARY*/) {
    if (TREE_CODE (sym) == SSA_NAME
        && SSA_NAME_IS_DEFAULT_DEF (sym)) {
      tree var = SSA_NAME_VAR (sym);
      slot = symbolic_pointer_map_contains (ctx->state, var);
      if (!slot) {
        if (eval_formal_insym) {
          expr = cgraph_node_get_insym (current_cgraph_node, var);
        }
        else {
          slot = symbolic_pointer_map_insert (ctx->state, sym);
          *slot = var;
          if (dump_details)
            dump_symbolic_expr (dump_file, sym, var);
        }
      }
    }
    else if (1) {
#if 0
      symee_expr insym = cgraph_node_get_insym (current_cgraph_node, sym);
      if (insym)
        *slot = insym;
#endif
    }
  }

  if (slot)
    expr = *slot;

  if (!expr)
    expr = sym;
  
  return expr;
}


/* delete sym and return corresponding value */

extern symee_expr
symee_delete_state_sym (symee_context ctx, symee_expr sym)
{
  symee_expr expr = NULL;
  void** slot = NULL;

  symee_assert (ctx && ctx->state,
                (stderr, "SYMEE: (%s) input ctx can not be NULL\n", __func__));

  slot = symbolic_pointer_map_contains (ctx->state, sym);
  
  if (slot) {
    expr = *slot;
    symbolic_pointer_map_delete (ctx->state, sym);
  }
  
  return expr;
}


extern void
symee_set_state_sym (symee_context ctx, symee_const_expr sym, symee_expr expr)
{
  symbolic_pointer_map_ptr state = ctx->state;
  void** slot = symbolic_pointer_map_contains (state, sym);

  gcc_assert (ctx);
  gcc_assert (expr);
  
  if (!slot)
    slot = symbolic_pointer_map_insert (state, sym);

  gcc_assert (slot);
  *slot = expr;
}


extern symee_range
symee_range_create (int type, tree min, tree max)
{
  symee_range range = XCNEW (struct symee_range_def);
  range->type = type;
  range->min = min;
  range->max = max;

  return range;
}


extern void
symee_set_range_base (symee_context ctx, symee_const_expr sym, symee_expr base)
{
  symbolic_pointer_map_ptr index_range = ctx->index_range;
  void **slot = symbolic_pointer_map_insert (index_range, sym);
  symee_range_list range_list = XCNEW (struct symee_range_list_def);
  range_list->base = base;

  *slot = range_list;
}


extern void
symee_range_list_push_back (symee_range_list range_list, symee_range range)
{
  if (range_list->tail) {
    range_list->tail->next = range;
    range->prev = range_list->tail;
  }
  else
    range_list->head = range;
  range_list->tail = range;
}


extern void
dump_range (FILE* fp, symee_range range)
{
  print_generic_expr (fp, range->min, dump_flags);
  fprintf (fp, " : ");
  print_generic_expr (fp, range->max, dump_flags);
}


extern void
dump_range_list (FILE* fp, symee_range_list range_list)
{
  symee_range range = range_list->head;
  print_generic_expr (fp, range_list->base, dump_flags);
  fprintf (fp, " :");
  while (range) {
    fprintf (fp, " ");
    dump_range (fp, range);
    fprintf (fp, ";");

    range = range->next;
  }
}


extern void
symee_set_sym_index_range (symee_context ctx, symee_const_expr sym, symee_range range)
{
  symbolic_pointer_map_ptr index_range = ctx->index_range;
  symee_range_list range_list;
  void **slot = symbolic_pointer_map_contains (index_range, sym);

  if (!slot) {
    slot = symbolic_pointer_map_insert (index_range, sym);
  }

  if (*slot)
    range_list = *slot;    
  else {
    range_list = XCNEW (struct symee_range_list_def);
    range_list->sym = sym;
    *slot = range_list;
  }

  symee_range_list_push_back (range_list, range);
}


extern symee_ref_node
symee_ref_node_create (void)
{
  symee_ref_node ref_node = XCNEW (struct symee_ref_node_def);
  return ref_node;
}


extern symee_ref_node
symee_ref_node_create_ref (tree ref)
{
  symee_ref_node ref_node = XCNEW (struct symee_ref_node_def);
  ref_node->ref = ref;
  return ref_node;
}


extern symee_ref_list
symee_ref_list_create (void)
{
  symee_ref_list ref_list = XCNEW (struct symee_ref_list_def);

  return ref_list;
}


extern symee_ref_list
symee_ref_list_create_ref (tree ref)
{
  symee_ref_list ref_list = symee_ref_list_create ();
  symee_ref_node ref_node = symee_ref_node_create_ref (ref);
  
  ref_list->head = ref_node;
  ref_list->tail = ref_node;

  return ref_list;
}


static bool
symee_ref_list_empty (symee_ref_list ref_list)
{
  if (ref_list->head = NULL) {
    gcc_assert (ref_list->tail == NULL);
    return true;
  }

  return false;
}


extern void
symee_ref_list_push_front (symee_ref_list ref_list, symee_ref_node ref_node)
{
  gcc_assert (ref_list);
  gcc_assert (ref_node && ref_node->prev == NULL && ref_node->next == NULL);

  symee_ref_node head = ref_list->head;

  if (head) {
    head->prev = ref_node;
    ref_node->next = head;
  }
  /* if head is NULL, tail must be NULL. */
  else {
    ref_list->tail = ref_node;
  }

  ref_list->head = ref_node;
}


extern void
symee_ref_list_push_back (symee_ref_list ref_list, symee_ref_node ref_node)
{
  gcc_assert (ref_list);
  gcc_assert (ref_node && ref_node->prev == NULL && ref_node->next == NULL);

  symee_ref_node tail = ref_list->tail;

  if (tail) {
    tail->next = ref_node;
    ref_node->prev = tail;
  }
  /* if tail is NULL, head must be NULL. */
  else {
    ref_list->head = ref_node;
  }

  ref_list->tail = ref_node;
}


extern symee_ref_node
symee_ref_list_append_ref (symee_ref_list ref_list, tree ref)
{
  gcc_assert (ref_list);
  symee_ref_node ref_node = symee_ref_node_create_ref (ref);
  symee_ref_list_push_back (ref_list, ref_node);

  return ref_node;
}


extern symee_ref_node
symee_ref_list_delete (symee_ref_list ref_list, symee_ref_node ref_node)
{
  if (ref_list->head == ref_node)
    ref_list->head = ref_node->next;

  if (ref_list->tail == ref_node)
    ref_list->tail = ref_node->prev;

  if (ref_node->prev)
    ref_node->prev->next = ref_node->next;

  if (ref_node->next)
    ref_node->next->prev = ref_node->prev;
  
  ref_node->prev = NULL;
  ref_node->next = NULL;

  return ref_node;
}


extern bool
symee_ref_node_kill_p (symee_context ctx,
                       symee_ref_node killee, symee_ref_node killer)
{
  if (same_lvalue (killee->ref, killer->ref))
    return true;
  return false;
}


/* if 'ref_node' kill some node in 'ref_list', return it. */

extern symee_ref_node
symee_ref_list_kill_p (symee_context ctx,
                       symee_ref_list ref_list, symee_ref_node ref_node)
{
  symee_ref_node rn = ref_list->tail;
  while (rn) {
    if (symee_ref_node_kill_p (ctx, rn, ref_node))
      return rn;
    rn = rn->prev;
  }

  return NULL;
}


extern symee_ref_list
symee_ref_list_kill (symee_ref_list ref_list, symee_ref_node ref_node)
{
  symee_ref_list ret_ref_list = symee_ref_list_create ();
  symee_ref_node rn = ref_list->tail;
  
  gcc_assert (rn && rn == ref_node);

  rn = rn->prev;
  while (rn) {
    symee_ref_node prev = rn->prev;

    if (stmt_kills_ref_p (ref_node->stmt, rn->ref)
        || same_lvalue (rn->ref, ref_node->ref)) {
      symee_ref_list_delete (ref_list, rn);
      symee_ref_list_push_front (ret_ref_list, rn);

      if (dump_details) {
        fprintf (dump_file, "kill ");
        print_generic_expr (dump_file, rn->ref, dump_flags);
        fprintf (dump_file, " \t\tin ");
        print_gimple_stmt (dump_file, rn->stmt, 0, dump_flags);
        fprintf (dump_file, "\n");
      }
    }

    rn = prev;
  }

  return ret_ref_list;
}


extern symee_ref_list
symee_get_state_sym_ref_list (symee_context ctx, symee_expr sym)
{
  symee_ref_list ref_list = NULL;
  void** slot = NULL;

  symee_assert (ctx && ctx->ref_state,
                (stderr, "SYMEE: (%s) input ctx can not be NULL\n", __func__));

  slot = symbolic_pointer_map_contains (ctx->ref_state, sym);
  
  if (slot)
    ref_list = *slot;

  return ref_list;
}


extern symee_ref_list
symee_get_state_sym_ref_list_create (symee_context ctx, symee_expr sym)
{
  symee_ref_list ref_list = NULL;
  void** slot = NULL;

  symee_assert (ctx && ctx->ref_state,
                (stderr, "SYMEE: (%s) input ctx can not be NULL\n", __func__));

  slot = symbolic_pointer_map_contains (ctx->ref_state, sym);
  
  if (!slot) {
    slot = symbolic_pointer_map_insert (ctx->ref_state, sym);
    *slot = symee_ref_list_create ();
  }

  if (slot)
    ref_list = *slot;

  return ref_list;
}


extern void
symee_set_state_sym_ref_list (symee_context ctx,
                              symee_expr sym, symee_ref_list ref_list)
{
  symbolic_pointer_map_ptr ref_state = ctx->ref_state;
  void** slot = symbolic_pointer_map_contains (ref_state, sym);

  gcc_assert (ctx);

  if (!slot)
    slot = symbolic_pointer_map_insert (ref_state, sym);

  gcc_assert (slot);
  *slot = ref_list;
}


static bool
update_context_1 (const void* key, void** slot, void* data)
{
  symee_context ctx = (symee_context) data;
  symee_const_expr sym = (symee_const_expr) key;
  symee_expr expr = *(symee_expr*)slot;

  symee_set_state_sym (ctx, sym, expr);

  return true;
}


extern symee_context
symee_context_update (symee_context ctx, symee_context delta_ctx)
{
  size_t i;
  symee_const_expr sym;
  void **slot;

  SYMBOLIC_POINTER_MAP_FOR_EACH_SLOT (delta_ctx->state, sym, slot, i)
    {
      symee_set_state_sym (ctx, sym, *slot);
    }

  //symbolic_pointer_map_traverse (delta_ctx->state, update_context_1, ctx);

  return ctx;
}


extern symee_context
symee_context_update_path_cond (symee_context ctx)
{
  /* TODO: after set ctx->path_cond, eval ctx */
  return ctx;
}


struct array_ref_1_1 {
  symee_expr sym;
  symee_expr* expr;
};

static bool
same_array_ref_sym (const void* key, void** slot, void* data)
{
  symee_const_expr sym = (symee_const_expr) key;
  struct array_ref_1_1* tmp = (struct array_ref_1_1*) data;
  if (same_lvalue_array (sym, tmp->sym)) {
    *(tmp->expr) = *slot;
    return false;
  }

  return true;
}


static bool
same_indirect_ref_sym (const void* key, void** slot, void* data)
{
  symee_const_expr sym = (symee_const_expr) key;
  struct array_ref_1_1* tmp = (struct array_ref_1_1*) data;
  if (same_lvalue_deref (sym, tmp->sym)) {
    *(tmp->expr) = *slot;
    return false;
  }

  return true;
}


extern symee_expr
symee_get_state_sym_1 (symee_context ctx, symee_expr sym)
{
  symee_expr expr = NULL_TREE;
  struct array_ref_1_1 data = {sym, &expr};
  
  if (TREE_CODE (sym) == ARRAY_REF) {
    symbolic_pointer_map_traverse (ctx->state, same_array_ref_sym, &data);
  }
  else if (TREE_CODE (sym) == MEM_REF
           || TREE_CODE (sym) == INDIRECT_REF) {
    symbolic_pointer_map_traverse (ctx->state, same_indirect_ref_sym, &data);
  }

  if (!expr)
    expr = symee_get_state_sym (ctx, sym);

  if (!expr)
    expr = sym;
  
  return expr;
}


extern symee_expr
symee_get_state_cond (symee_context ctx)
{
  gcc_assert (ctx);
  return ctx->state_cond;
}


extern void
symee_set_state_cond (symee_context ctx, symee_expr state_cond)
{
  gcc_assert (ctx);
  ctx->state_cond = state_cond;
}


extern symee_expr
symee_get_path_cond (symee_context ctx)
{
  gcc_assert (ctx);
  return ctx->path_cond;
}


extern void
symee_set_path_cond (symee_context ctx, symee_expr path_cond)
{
  gcc_assert (ctx);
  ctx->path_cond = path_cond;
}


static void
dump_symbolic_expr (FILE* fp, symee_const_expr sym, symee_const_expr expr)
{
  fprintf (fp, "sym (%p) -> expr (%p)\n", sym, expr);
  print_generic_expr (fp, sym, 0);
  fprintf (fp, "\n\t----->\t");
  print_generic_expr (fp, expr, 0);
  fprintf (fp, "\n");

  return;
}


static void
dump_ref_node (FILE* fp, symee_ref_node ref_node)
{
  print_generic_expr (fp, ref_node->ref, 0);
  fprintf (fp, " = ");
  print_generic_expr (fp, ref_node->value, 0);
}


static void
dump_ref_list (FILE* fp, symee_ref_list ref_list)
{
  symee_ref_node ref_node = ref_list->head;
  print_generic_expr (fp, ref_list->base, 0);
  fprintf (fp, " :");
  while (ref_node) {
    fprintf (fp, " ");
    dump_ref_node (fp, ref_node);
    fprintf (fp, ";");

    ref_node = ref_node->next;
  }
}


DEBUG_FUNCTION void
debug_ref_list (symee_ref_list ref_list)
{
  dump_ref_list (stderr, ref_list);
}


static bool
dump_state_1 (const void* key, void** slot, void* data)
{
  symee_const_expr sym = (symee_const_expr) key;
  symee_expr expr = *((symee_expr*)slot);
  FILE* fp = (FILE*) data;

  dump_symbolic_expr (fp, sym, expr);

  /* continue to dump ... */
  return true;
}


static void
dump_context (FILE* fp, symee_context ctx)
{
  size_t i;
  symee_const_expr sym;
  symee_expr expr;
  
  fprintf (fp, "==== context %p ====\n", ctx);
  //symbolic_pointer_map_traverse (ctx->state, dump_state_1, fp);
  SYMBOLIC_POINTER_MAP_FOR_EACH (ctx->state, sym, expr, i)
    {
      dump_symbolic_expr (fp, sym, expr);
    }
  fprintf (fp, "path condition of context %p\n", ctx);
  print_generic_expr (fp, ctx->path_cond, 0);
  fprintf (fp, "\n");
}

void
symee_dump_context (FILE* fp, symee_context ctx)
  __attribute__ ((weak, alias("dump_context")));


DEBUG_FUNCTION void
debug_context (symee_context ctx)
{
  dump_context (stdout, ctx);
}


static bool
dump_state_brief_1 (const void* key, void** slot, void* data)
{
  symee_const_expr sym = (symee_const_expr) key;
  symee_expr expr = *((symee_expr*)slot);
  FILE* fp = (FILE*) data;

  print_generic_expr (fp, sym, 0);
  fprintf (fp, " = ");
  print_generic_expr (fp, expr, 0);
  fprintf (fp, "\n");

  return true;
}


static bool
dump_ref_state_brief_1 (const void* key, void** slot, void* data)
{
  symee_const_expr sym = (symee_const_expr) key;
  symee_ref_list ref_list = *((symee_ref_list*)slot);
  FILE* fp = (FILE*) data;

  print_generic_expr (fp, sym, 0);
  fprintf (fp, " = ");
  dump_ref_list (fp, ref_list);
  fprintf (fp, "\n");

  return true;
}


static bool
dump_range_brief_1 (const void* key, void** slot, void* data)
{
  symee_const_expr sym = (symee_const_expr) key;
  symee_range_list range_list = *((symee_range_list*)slot);
  FILE* fp = (FILE*) data;

  print_generic_expr (fp, sym, 0);
  fprintf (fp, " = ");
  dump_range_list (fp, range_list);
  fprintf (fp, "\n");

  return true;
}


static void
dump_state_brief (FILE* fp, symbolic_pointer_map_ptr state)
{
  symbolic_pointer_map_traverse (state, dump_state_brief_1, fp);
}


void
dump_context_brief (FILE* fp, symee_context ctx)
{
  if (!ctx) {
    fprintf (fp, "void context\n");
    return;
  }
  
  symbolic_pointer_map_traverse (ctx->state, dump_state_brief_1, fp);
  if (symee_context_get_prev (ctx))
    fprintf (fp, "prev bb: %d\n", symee_context_get_prev (ctx)->index);
  else
    fprintf (fp, "prev bb: NULL\n");
  fprintf (fp, "next stmt: ");
  print_gimple_stmt (fp, symee_context_get_next (ctx), 0, dump_flags);
  fprintf (fp, "\npath_cond = ", ctx);
  print_generic_expr (fp, ctx->path_cond, 0);
  fprintf (fp, "\n");
  symbolic_pointer_map_traverse (ctx->ref_state, dump_ref_state_brief_1, fp);
  fprintf (fp, "\n");
  fprintf (fp, "symee range:\n");
  symbolic_pointer_map_traverse (ctx->index_range, dump_range_brief_1, fp);
  fprintf (fp, "\n");
}


DEBUG_FUNCTION void
debug_ctx (symee_context ctx)
{
  dump_context_brief (stdout, ctx);
}


static symee_expr
build_symee_gamma_expr (symee_expr cond, symee_expr left, symee_expr right)
{
  tree ltype = TREE_TYPE (left);
  tree rtype = TREE_TYPE (right);

  // TODO: check more
  gcc_assert (ltype == rtype);

  return build3 (SYMEE_GAMMA, TREE_TYPE (left), cond, left, right);
}


static symee_expr
eval (symee_context context, symee_expr expr)
{
  return expr;
}

extern symee_expr
symee_eval (symee_context context, symee_expr expr)
  __attribute__ ((weak, alias("eval_expr_tree")));


static symee_expr
loopeval (symee_context context, symee_expr expr)
{
  return expr;
}


extern symee_expr
symee_loopeval (symee_context context, symee_expr expr)
  __attribute__ ((weak, alias("loopeval")));


static symee_context
symee_context_merge (symee_context c1, symee_context c2, symee_expr path_cond)
{
  // symee_expr gama = path_cond ? symee_get_expr (c1) : symee_get_expr (c2);
  // return symee_context (htab_t, gamma, c1.path_cond || c2.path_cond)
}


static void
insert_formal_map (symee_cgraph_node_info node_info, tree parm, int num)
{
  void** slot = symbolic_pointer_map_insert (node_info->formals, parm);
  gcc_assert (slot);
  if (slot)
    *slot = (void*)(unsigned long)num;
  
  return;
}


static int
get_formal_map (symee_cgraph_node_info node_info, tree parm)
{
  void** slot = symbolic_pointer_map_contains (node_info->formals, parm);
  if (slot)
    return (int)(unsigned long)(*slot);

  return -1;
}


static void
init_formal (symee_cgraph_node_info node_info)
{
  tree parm;
  int num = 0;
  
  for (parm = DECL_ARGUMENTS (current_function_decl); parm;
       parm = DECL_CHAIN (parm)) {
    insert_formal_map (node_info, parm, num);
  }

  return;
}


static bool
is_interesting_sym (symee_const_expr sym, symee_expr expr)
{
  tree var = sym;

  if (REFERENCE_CLASS_P (sym))
    var = get_base_sym (sym);

  if (TREE_CODE (var) == SSA_NAME)
    var = SSA_NAME_VAR (var);

  if (TREE_CODE (var) == PARM_DECL ||
      TREE_CODE (var) == RESULT_DECL)
    return true;

  if (DECL_P (var) && is_global_var (var))
    return true;

  return false;
}


static tree
has_uninteresting_sym_1 (tree *tp, int *walk_subtrees, void *data ATTRIBUTE_UNUSED)
{
  if (!is_interesting_sym (*tp, NULL)) {
    *walk_subtrees = 0;
    return *tp;
  }

  return NULL_TREE;
}


static bool
has_uninteresting_sym (tree t)
{
#if SYMEE_CLEANUP
  tree ret_tree = walk_tree (&t, has_uninteresting_sym_1, NULL, NULL);
  gcc_assert (ret_tree == NULL_TREE
              || !is_interesting_sym (ret_tree, NULL));

  return (ret_tree != NULL_TREE);
#endif
  return false;
}


static bool
copy_and_cleanup_1 (const void* key, void** slot, void* data)
{
  symee_const_expr sym = (symee_const_expr) key;
  symee_expr expr = *(symee_expr*)slot;
  symee_context ctx = (symee_context) data;

  if (is_interesting_sym (sym, expr)) {
    symee_expr subst_sym = subst_formal_outsym (sym);
    if (subst_sym)
      sym = subst_sym;
    void** slot = symbolic_pointer_map_insert (ctx->state, sym);
    gcc_assert (slot);

    gcc_assert (expr);
    
    if (!has_uninteresting_sym (expr)) {
      *slot = expr;
    }
    else
      *slot = add_new_sym (sym);
  }

  return true;
}


static symee_context
symee_context_copy_and_cleanup (struct cgraph_node* node, symee_context ctx)
{
  symee_context ret_ctx = symee_context_create ();

  symbolic_pointer_map_traverse (ctx->state, copy_and_cleanup_1, ret_ctx);

  return ret_ctx;
}


static symee_cgraph_node_info
cgraph_node_info_create (void)
{
  symee_cgraph_node_info node_info =
    ggc_alloc_cleared_symee_cgraph_node_info_def ();

  return node_info;
}


static symee_cgraph_node_info
get_cgraph_node_info (cgraph_node_ptr node)
{
  symee_cgraph_node_info node_info = NULL;
  splay_tree_node st_node = splay_tree_lookup (global_info->node_info, node);

  if (st_node)
    node_info = st_node->value;

  return node_info;
}


extern symee_cgraph_node_info
symee_get_cgraph_node_info (cgraph_node_ptr node)
  __attribute__ ((weak, alias ("get_cgraph_node_info")));


static symee_cgraph_node_info
get_cgraph_node_info_create (cgraph_node_ptr node)
{
  symee_cgraph_node_info node_info = NULL;
  splay_tree_node st_node = splay_tree_lookup (global_info->node_info, node);

  if (st_node) {
    node_info = st_node->value;
  }
  else {
    node_info = cgraph_node_info_create ();
    st_node = splay_tree_insert (global_info->node_info, node, node_info);
  }

  gcc_assert (node_info);

  return node_info;
}


static symee_cgraph_node_info
init_cgraph_node_info (cgraph_node_ptr node)
{
  symee_cgraph_node_info node_info = get_cgraph_node_info_create (node);
  tree t;
  int i;

  node_info->node = node;
  node_info->formal_chain = DECL_ARGUMENTS (node->decl);

  node_info->insym_context = symee_context_create ();
  node_info->outsym_context = symee_context_create ();
  node_info->working_context = symee_context_create ();
  node_info->visited_bbs = BITMAP_ALLOC (NULL);
  bitmap_clear (node_info->visited_bbs);
  
  node_info->entrycontext = symbolic_pointer_map_create ();
  node_info->exitcontext = symbolic_pointer_map_create ();
  node_info->bbentrycontext = symbolic_pointer_map_create ();
  node_info->bbexitcontext = symbolic_pointer_map_create ();
  node_info->edgecontext = symbolic_pointer_map_create ();
  node_info->formals = symbolic_pointer_map_create ();
  node_info->formal_in = node_info->insym_context->state;
  node_info->formal_out = node_info->outsym_context->state;
  node_info->exit_state = symbolic_pointer_map_create ();

  return node_info;
}


static symee_expr
cgraph_node_get_retsym (cgraph_node_ptr node)
{
  symee_expr retsym = NULL;
  symee_cgraph_node_info node_info = get_cgraph_node_info (node);

  if (node_info)
    retsym = node_info->result_decl;

  return retsym;
}


static symee_expr
cgraph_node_get_insym (cgraph_node_ptr node, tree parm)
{
  symee_expr insym = NULL;
  symee_cgraph_node_info node_info = get_cgraph_node_info (node);

  if (node_info) {
    void** slot = symbolic_pointer_map_contains (node_info->formal_in, parm);
    if (slot)
      insym = *slot;
  }
  
  return insym;
}


static symee_expr
cgraph_node_get_outsym (cgraph_node_ptr node, tree parm)
{
  symee_expr outsym = NULL;
  symee_cgraph_node_info node_info = get_cgraph_node_info (node);

  if (node_info) {
    void** slot = symbolic_pointer_map_contains (node_info->formal_out, parm);
    if (slot)
      outsym = *slot;
  }
  
  return outsym;
}


static symee_context
cgraph_node_set_context (struct cgraph_node* node, symee_context ctx)
{
  symee_cgraph_node_info node_info = get_cgraph_node_info (node);

  ctx = symee_context_copy_and_cleanup (node, ctx);
  
  if (node_info)
    node_info->exit_context = ctx;

  return ctx;
}


extern symee_context
cgraph_node_get_context (struct cgraph_node* node)
{
  symee_cgraph_node_info node_info = get_cgraph_node_info (node);

  if (node_info)
    return node_info->exit_context;

  if (node->clone_of) {
    node_info = get_cgraph_node_info (node->clone_of);
    if (node_info) {
      fprintf (stderr, "Warning: %s has not context, use the context of %s\n",
               cgraph_node_name (node), cgraph_node_name (node->clone_of));
      return node_info->exit_context;
    }
  }
  
  if (dump_file && dump_details) {
    fprintf (dump_file, "SYMEE: %s has not context!\n",
             IDENTIFIER_POINTER (DECL_ASSEMBLER_NAME (node->decl)));
  }

  return NULL;
}

extern symee_context
symee_get_context (struct cgraph_node* node)
  __attribute__ ((weak, alias ("cgraph_node_get_context")));


static symee_context
get_entrycontext (gimple stmt)
{
  symee_context ctx = NULL;
  symbolic_pointer_map_ptr entrycontext_map = current_node_info->entrycontext;
  void** slot = symbolic_pointer_map_contains (entrycontext_map, stmt);

  if (slot)
    ctx = *slot;

  return ctx;
}


static symee_context
get_entrycontext_in_node (cgraph_node_ptr node, gimple stmt)
{
  symee_cgraph_node_info node_info = get_cgraph_node_info (node);
  symbolic_pointer_map_ptr entrycontext_map = node_info->entrycontext;
  void **slot = symbolic_pointer_map_contains (entrycontext_map, stmt);
  if (slot)
    return *slot;
  return NULL;
}


extern symee_context symee_get_entrycontext (cgraph_node_ptr node, gimple stmt)
  __attribute__ ((weak, alias ("get_entrycontext_in_node")));


static void
set_entrycontext (gimple stmt, symee_context ctx)
{
  gcc_assert (ctx);
  symbolic_pointer_map_ptr entrycontext_map = current_node_info->entrycontext;
  void** slot = symbolic_pointer_map_insert (entrycontext_map, stmt);
  gcc_assert (slot);
  *slot = ctx;
}


static symee_context
get_exitcontext (gimple stmt)
{
  symee_context ctx = NULL;
  symbolic_pointer_map_ptr exitcontext_map = current_node_info->exitcontext;
  void** slot = symbolic_pointer_map_contains (exitcontext_map, stmt);

  if (slot)
    ctx = *slot;

  return ctx;
}


static void
set_exitcontext (gimple stmt, symee_context ctx)
{
  symbolic_pointer_map_ptr exitcontext_map = current_node_info->exitcontext;
  void** slot = symbolic_pointer_map_insert (exitcontext_map, stmt);
  gcc_assert (slot);
  *slot = ctx;
}


static symee_context
get_bb_entry_context (basic_block bb)
{
  symee_context ctx = NULL;
  symbolic_pointer_map_ptr bbentrycontext_map = current_node_info->bbentrycontext;
  void** slot = symbolic_pointer_map_contains (bbentrycontext_map, bb);

  if (slot)
    ctx = *slot;

  return ctx;
}


static void
set_bb_entry_context (basic_block bb, symee_context ctx)
{
  symbolic_pointer_map_ptr bbentrycontext_map = current_node_info->bbentrycontext;
  void** slot = symbolic_pointer_map_insert (bbentrycontext_map, bb);
  gcc_assert (slot);
  *slot = ctx;
}


static symee_context
get_edge_context (edge e)
{
  symee_context ctx = NULL;
  symbolic_pointer_map_ptr edgecontext_map = current_node_info->edgecontext;
  void** slot = symbolic_pointer_map_contains (edgecontext_map, e);

  if (slot)
    ctx = *slot;

  return ctx;
}


static void
set_edge_context (edge e, symee_context ctx)
{
  symbolic_pointer_map_ptr edgecontext_map = current_node_info->edgecontext;
  void** slot = symbolic_pointer_map_insert (edgecontext_map, e);
  gcc_assert (slot);
  *slot = ctx;
}


static bool
bb_is_visited (basic_block bb)
{
  return bitmap_bit_p (current_node_info->visited_bbs, bb->index);
}


static bool
bb_set_visited (basic_block bb)
{
  return bitmap_set_bit (current_node_info->visited_bbs, bb->index);
}


static int
get_expr_string (tree expr, char *name)
{
  extern const char* print_generic_expr_to_string (tree t, int flags);
  char* expr_text = print_generic_expr_to_string (expr, 0);
  char* p = expr_text;
  char* q = name;

  while (*p != '\0') {
    if (*p == '$' || *p == '.') {
      *q = '_'; q++;
    }
    else if (*p == '\n') {
      ;
    }
    else {
      *q = *p; q++;
    }

    p++;
  }

  *q = '\0';
  
  XDELETEVEC (expr_text);

  return strlen (name);
}


static tree
add_new_static_var_1 (tree type, tree name)
{
  tree var = add_new_static_var (type);
  DECL_NAME (var) = name;
  return var;
}


static tree
add_new_static_var_2 (tree type, char* name)
{
  tree var = add_new_static_var (type);
  DECL_NAME (var) = get_identifier (name);
  return var;
}


static tree
add_new_static_var_3 (tree type, char* prefix)
{
  tree var = add_new_static_var (type);
  DECL_NAME (var) = create_tmp_var_name (prefix);
  return var;
}


static tree
add_new_sym (tree t)
{
  tree name;
  tree var;
  char name_str[1024] = "newsym.";
  char *p = name_str + 7;
  tree base_sym = get_base_sym (t);

  if (base_sym && VAR_OR_FUNCTION_DECL_P (base_sym))
    strcat (p, IDENTIFIER_POINTER (DECL_ASSEMBLER_NAME (base_sym)));
  else if (VAR_OR_FUNCTION_DECL_P (t))
    strcat (p, IDENTIFIER_POINTER (DECL_ASSEMBLER_NAME (t)));
  else {
    get_expr_string (t, p);
  }
  var = add_new_static_var_3 (TREE_TYPE (t), name_str);

  return var;
}


static tree
create_in_out_name (tree t, tree fndecl, bool in)
{
  char tmp_name [1024] = {'\0',};

  if (fndecl) {
    strcat (tmp_name, IDENTIFIER_POINTER (DECL_ASSEMBLER_NAME (fndecl)));
    strcat (tmp_name, ".");
  }
  else
    strcat (tmp_name, "global.");

  if (DECL_P (t))
    strcat (tmp_name, IDENTIFIER_POINTER (DECL_NAME (t)));
  else {
    char expr_name [1024] = {'\0',};
    get_expr_string (t, expr_name);
    strcat (tmp_name, expr_name);
  }
    
  if (in)
    strcat (tmp_name, "_in.suffix");
  else
    strcat (tmp_name, "_out.suffix");

  //symee_debug_prefix (tmp_name, "FreeFancyAli.ali_out");

  return create_tmp_var_name (tmp_name);
}


static tree
add_new_parm_sym (tree parm, tree fndecl, bool in)
{
  tree tmp_var = add_new_static_var (TREE_TYPE (parm));
  DECL_NAME (tmp_var) = create_in_out_name (parm, fndecl, in);

  return tmp_var;
}


static void
add_in_out_sym (tree parm)
{
  if (eval_formal_insym) {
    tree parm_in = add_new_parm_sym (parm, current_function_decl, true);
    void **slot = symbolic_pointer_map_insert (current_node_info->formal_in, parm);
    *slot = parm_in;
  }
  if (eval_formal_outsym) {
    tree parm_out = add_new_parm_sym (parm, current_function_decl, false);
    void **slot = symbolic_pointer_map_insert (current_node_info->formal_out, parm);
    *slot = parm_out;
  }

  return;
}


static symee_context
gen_context_bb_cfg_entry (symee_context ctx, basic_block bb)
{
  ctx = symee_context_create ();
  set_bb_entry_context (bb, ctx);

  gcc_assert (single_succ_p (bb));
  set_edge_context (single_succ_edge (bb), ctx);

  int formal_id = 0;
  tree parm = DECL_ARGUMENTS (current_function_decl);
  for (; parm; parm = TREE_CHAIN (parm)) {
    VEC_safe_push (tree, heap, current_node_info->parm_decls, parm);

    if (eval_formal_insym) {
      tree parm_in = add_new_parm_sym (parm, current_function_decl, true);
#if 0
      symee_set_state_sym (ctx, parm, parm_in);
#endif
      void** slot = symbolic_pointer_map_insert (current_node_info->formal_in, parm);
      *slot = parm_in;
    }

    if (eval_formal_outsym) {
      tree parm_out = add_new_parm_sym (parm, current_function_decl, false);
#if 0 /* it's not necessary ? */
      symee_set_state_sym (ctx, parm_out, parm);
#endif
      void** slot = symbolic_pointer_map_insert (current_node_info->formal_out, parm);
      *slot = parm_out;
    }
    
    formal_id++;
  }

  /* TODO: */
  if (!current_node_info->heap_pointer) {
    tree heap_pointer = add_new_static_var_3 (ptr_type_node, "heap_pointer");
    current_node_info->heap_pointer = heap_pointer;
    
    if (eval_formal_insym) {
      tree parm_in = add_new_parm_sym (heap_pointer, current_function_decl, true);
      void **slot = symbolic_pointer_map_insert (current_node_info->formal_in, heap_pointer);
      *slot = parm_in;
    }
    if (eval_formal_outsym) {
      tree parm_out = add_new_parm_sym (heap_pointer, current_function_decl, false);
      void **slot = symbolic_pointer_map_insert (current_node_info->formal_out, heap_pointer);
      *slot = parm_out;
    }

  }

  return NULL;
}


static symee_context
gen_context_bb_cfg_exit (symee_context ctx, basic_block bb)
{
  cgraph_node_ptr node = current_cgraph_node;
  symee_context ret_ctx = gen_context_bb (ctx, EXIT_BLOCK_PTR);
  symee_cgraph_node_info node_info = get_cgraph_node_info (node);

  ret_ctx = symee_context_copy_and_cleanup (node, ret_ctx);
  
  if (node_info)
    node_info->exit_context = ret_ctx;

  return ret_ctx;

  return NULL;
}


static void
init_start_node (basic_block bb)
{
}


static void
init_preheader (basic_block bb)
{
}


static void
update_preheader (basic_block bb)
{
}


static void
update_postexit (basic_block bb)
{
}


static void
merge_context (basic_block bb)
{
  return;
}


static bool
merge_1 (const void* key, void** slot, void* data)
{
  symee_const_expr sym = (symee_const_expr) key;
  symee_expr expr = *(symee_expr*)slot;
  symee_context ctx = (symee_context) data;

  void **ctx_slot = symbolic_pointer_map_contains (ctx->state, sym);
  if (ctx_slot) {
    /* just ignore sym, leave it to gen_context_phi */
  }
  else {
    ctx_slot = symbolic_pointer_map_insert (ctx->state, sym);
    gcc_assert (ctx_slot);
    *ctx_slot = expr;
  }

  return true;
}


/* merge ctx1 and ctx2 into new one */

static symee_context
gen_context_binary_merge (symee_context ctx1, symee_context ctx2)
{
  tree path_cond_or = NULL_TREE;
  symee_context ret_ctx = NULL;
  symee_expr expr = NULL;
  
  if (!ctx1) return ctx2;
  if (!ctx2) return ctx1;

  gcc_assert (ctx1 && ctx2);
  
  path_cond_or =
    build2 (TRUTH_OR_EXPR, TREE_TYPE (ctx1->path_cond),
            ctx1->path_cond, ctx2->path_cond);

  ret_ctx = symee_context_copy (ctx1);

  ret_ctx->path_cond = ctx1->path_cond;
  ret_ctx->state_cond = ctx1->state_cond;
  symbolic_pointer_map_traverse (ctx2->state, merge_1, ret_ctx);

  expr = eval_expr_tree (ret_ctx, path_cond_or);
  if (!expr)
    expr = path_cond_or;
  if (symee_is_const_true (expr))
    ret_ctx->path_cond = boolean_true_node;
  else if (symee_is_const_false (expr))
    ret_ctx->path_cond = boolean_false_node;
  else
    ret_ctx->path_cond = expr;

  return ret_ctx;
}


static symee_expr
symee_fold_binary_loc (location_t loc, enum tree_code code,
                       tree type, tree op0, tree op1)
{
  tree result = fold_binary_loc (loc, code, type, op0, op1);

  if (!result)
    result = build2_loc (loc, code, type, op0, op1);

  gcc_assert (result);

  return result;
}


static symee_expr
eval_cond_expr (symee_context ctx, symee_expr cond_expr)
{
  return cond_expr;
}


static symee_expr
eval_expr_tree_under_cond (symee_context ctx, symee_expr expr, symee_expr cond_expr)
{
  return expr;
}


static symee_expr
subst_formal_outsym (symee_expr sym)
{
  symee_expr ret_sym = eval_expr_tree (current_node_info->outsym_context, sym);

  gcc_assert (ret_sym);

  return ret_sym ? ret_sym : sym;
}


static symee_expr
substitute (symee_context ctx, symee_expr expr)
{
  const tree t = expr;
  enum tree_code code = TREE_CODE (expr);
  enum tree_code_class kind = TREE_CODE_CLASS (code);
  tree tem = NULL_TREE;
  location_t loc = EXPR_LOCATION (expr);
  tree type = TREE_TYPE (t);
  tree op0 = NULL_TREE, op1 = NULL_TREE, op2 = NULL_TREE;

  if (DECL_P (t)) {
    return symee_get_state_sym (ctx, expr);
  }

  if (eval_reference &&
      REFERENCE_CLASS_P (expr)) {
    return symee_get_state_sym_by_ref (ctx, expr);
  }
  
  if (TREE_CODE (t) == ASSERT_EXPR)
    return TREE_OPERAND (t, 0);
  
#if 0
  if (TREE_CODE (t) == ADDR_EXPR
      || TREE_CODE (t) == NOP_EXPR) {
    return t;
  }
#endif
  
  /* TODO: */
  if (code == SSA_NAME) {
    if (TREE_CODE (SSA_NAME_VAR (expr)) == PARM_DECL)
      return symee_get_state_sym (ctx, expr); // SSA_NAME_VAR (expr));
    else
      return symee_get_state_sym (ctx, expr);
  }

  if (IS_EXPR_CODE_CLASS (TREE_CODE_CLASS (code))) {
    switch (TREE_CODE_LENGTH (code)) {
    case 1:
      op0 = TREE_OPERAND (t, 0);
      op0 = substitute (ctx, op0);
      tem = build1_loc (loc, code, type, op0);
      return tem ? tem : expr;
    case 2:
      op0 = TREE_OPERAND (t, 0);
      op1 = TREE_OPERAND (t, 1);
      op0 = substitute (ctx, op0);
      op1 = substitute (ctx, op1);
      tem = build2_loc (loc, code, type, op0, op1);
      return tem ? tem : expr;
    case 3:
      op0 = TREE_OPERAND (t, 0);
      op1 = TREE_OPERAND (t, 1);
      op2 = TREE_OPERAND (t, 2);
      op0 = substitute (ctx, op0);
      op1 = substitute (ctx, op1);
      /* TODO: op2 of COMPONENT_REF maybe NULL */
      if (op2)
        op2 = substitute (ctx, op2);
      tem = build3_loc (loc, code, type, op0, op1, op2);
      return tem ? tem : expr;
    case 4:
      if (code == ARRAY_REF
          || code == SYMEE_DIRECT_SUM)
        break;
    default:
      gcc_unreachable ();
      break;
    }
  }

  if (eval_reference &&
      REFERENCE_CLASS_P (expr)) {
    return symee_get_state_sym_by_ref (ctx, expr);
  }

  switch (code) {
  case INDIRECT_REF:
  case ARRAY_REF:
    return symee_get_state_sym_1 (ctx, t);

  case CONST_DECL:
    return (DECL_INITIAL (t));

  default:
    return t;
  }

  gcc_unreachable ();

  return expr;
}


/* given 'ctx' and 'expr', replace symbols appeared in 'expr'
   and defined by 'ctx' with corresponding right value.
 */

static symee_expr
substitute_backward (symee_context ctx, symee_expr expr)
{
  const tree t = expr;
  enum tree_code code = TREE_CODE (expr);
  enum tree_code_class kind = TREE_CODE_CLASS (code);
  tree tem = NULL_TREE;
  location_t loc = EXPR_LOCATION (expr);
  tree type = TREE_TYPE (t);
  tree op0 = NULL_TREE, op1 = NULL_TREE, op2 = NULL_TREE;

  if (DECL_P (t)) {
    return symee_get_state_sym (ctx, expr);
  }

  if (eval_reference &&
      REFERENCE_CLASS_P (expr)) {
    return symee_get_state_sym_by_ref (ctx, expr);
  }
  
  if (TREE_CODE (t) == ASSERT_EXPR)
    return TREE_OPERAND (t, 0);
  
#if 0
  if (TREE_CODE (t) == ADDR_EXPR
      || TREE_CODE (t) == NOP_EXPR) {
    return t;
  }
#endif
  
  /* TODO: */
  if (code == SSA_NAME) {
    if (TREE_CODE (SSA_NAME_VAR (expr)) == PARM_DECL)
      return symee_get_state_sym (ctx, expr); // SSA_NAME_VAR (expr));
    else
      return symee_get_state_sym (ctx, expr);
  }

  if (IS_EXPR_CODE_CLASS (TREE_CODE_CLASS (code))) {
    switch (TREE_CODE_LENGTH (code)) {
    case 1:
      op0 = TREE_OPERAND (t, 0);
      op0 = substitute (ctx, op0);
      tem = build1_loc (loc, code, type, op0);
      return tem ? tem : expr;
    case 2:
      op0 = TREE_OPERAND (t, 0);
      op1 = TREE_OPERAND (t, 1);
      op0 = substitute (ctx, op0);
      op1 = substitute (ctx, op1);
      tem = build2_loc (loc, code, type, op0, op1);
      return tem ? tem : expr;
    case 3:
      op0 = TREE_OPERAND (t, 0);
      op1 = TREE_OPERAND (t, 1);
      op2 = TREE_OPERAND (t, 2);
      op0 = substitute (ctx, op0);
      op1 = substitute (ctx, op1);
      /* TODO: op2 of COMPONENT_REF maybe NULL */
      if (op2)
        op2 = substitute (ctx, op2);
      tem = build3_loc (loc, code, type, op0, op1, op2);
      return tem ? tem : expr;
    case 4:
      if (code == ARRAY_REF
          || code == SYMEE_DIRECT_SUM)
        break;
    default:
      gcc_unreachable ();
      break;
    }
  }

  if (eval_reference &&
      REFERENCE_CLASS_P (expr)) {
    return symee_get_state_sym_by_ref (ctx, expr);
  }

  switch (code) {
  case INDIRECT_REF:
  case ARRAY_REF:
    return symee_get_state_sym_1 (ctx, t);

  case CONST_DECL:
    return (DECL_INITIAL (t));

  default:
    return t;
  }

  gcc_unreachable ();

  return expr;
}


/* traverse the expr tree 'expr', substitute all referenced symbols */

static symee_expr
subst_and_fold (symee_context ctx, symee_expr expr)
{
  const tree t = expr;
  enum tree_code code = TREE_CODE (expr);
  enum tree_code_class kind = TREE_CODE_CLASS (code);
  tree tem = NULL_TREE;
  location_t loc = EXPR_LOCATION (expr);
  tree type = TREE_TYPE (t);
  tree op0 = NULL_TREE, op1 = NULL_TREE, op2 = NULL_TREE;

  if (DECL_P (t)) {
    return symee_get_state_sym (ctx, expr);
  }

  if (eval_reference &&
      REFERENCE_CLASS_P (expr)) {
    return symee_get_state_sym_by_ref (ctx, expr);
  }
  
#if 0
  if (TREE_CODE (t) == ADDR_EXPR
      || TREE_CODE (t) == NOP_EXPR) {
    return t;
  }
#endif
  
  /* TODO: */
  if (code == SSA_NAME) {
    if (TREE_CODE (SSA_NAME_VAR (expr)) == PARM_DECL)
      return symee_get_state_sym (ctx, expr); //SSA_NAME_VAR (expr));
    else
      return symee_get_state_sym (ctx, expr);
  }

  if (IS_EXPR_CODE_CLASS (TREE_CODE_CLASS (code))) {
    switch (TREE_CODE_LENGTH (code)) {
    case 1:
      op0 = TREE_OPERAND (t, 0);
      op0 = subst_and_fold (ctx, op0);
      tem = fold_unary_loc (loc, code, type, op0);
      if (!tem)
        tem = fold_build1_loc (loc, code, type, op0);
      return tem ? tem : expr;
    case 2:
      op0 = TREE_OPERAND (t, 0);
      op1 = TREE_OPERAND (t, 1);
      op0 = subst_and_fold (ctx, op0);
      op1 = subst_and_fold (ctx, op1);
      tem = fold_binary_loc (loc, code, type, op0, op1);
      if (!tem)
        tem = fold_build2_loc (loc, code, type, op0, op1);
      return tem ? tem : expr;
    case 3:
      op0 = TREE_OPERAND (t, 0);
      op1 = TREE_OPERAND (t, 1);
      op2 = TREE_OPERAND (t, 2);
      op0 = subst_and_fold (ctx, op0);
      op1 = subst_and_fold (ctx, op1);
      /* TODO: op2 of COMPONENT_REF maybe NULL */
      if (op2)
        op2 = subst_and_fold (ctx, op2);
      tem = fold_ternary_loc (loc, code, type, op0, op1, op2);
      return tem ? tem : expr;
    case 4:
      if (code == ARRAY_REF)
        break;
    default:
      gcc_unreachable ();
      break;
    }
  }

  if (eval_reference &&
      REFERENCE_CLASS_P (expr)) {
    return symee_get_state_sym_by_ref (ctx, expr);
  }

  switch (code) {
  case INDIRECT_REF:
  case ARRAY_REF:
    return symee_get_state_sym_1 (ctx, t);

  case CONST_DECL:
    return fold (DECL_INITIAL (t));

  default:
    return t;
  }

  gcc_unreachable ();

  return expr;
}


extern symee_expr
eval_expr_tree (symee_context ctx, symee_expr expr)
{
  tree ret = NULL_TREE;
  tree fold_expr = fold (expr);

  ret = fold_expr ? fold_expr : expr;
  ret = substitute (ctx, ret);
  if (ret)
    ret = fold (ret);
  if (!ret)
    ret = fold (expr);
  if (!ret)
    ret = expr;

  return ret;
}


static bool
eval_context_1 (const void* key, void** slot, void* data)
{
  symee_const_expr sym = (symee_const_expr) key;
  symee_expr expr = *(symee_expr*)slot;
  symee_context ctx = (symee_context) data;
  symee_expr ret_expr = NULL_TREE;

  ret_expr = eval_expr_tree (ctx, expr);
  gcc_assert (ret_expr);
  if (ret_expr)
    *slot = ret_expr;

  return true;
}


static symee_context
eval_context (symee_context ctx, symee_context eval_ctx)
{
  symee_context ret_ctx = symee_context_copy (eval_ctx);

  symbolic_pointer_map_traverse (ret_ctx->state, eval_context_1, ctx);

  if (dump_details) {
    fprintf (dump_file, "SYMEE: evaluate ctx\n");
    fprintf (dump_file, "input ctx =\n");
    dump_context_brief (dump_file, ctx);
    fprintf (dump_file, "evaluated ctx =\n");
    dump_context_brief (dump_file, eval_ctx);
    fprintf (dump_file, "result ctx =\n");
    dump_context_brief (dump_file, ret_ctx);
  }

  return ret_ctx;
}


static bool
call_eval_context_1 (const void* key, void** slot, void* data)
{
  symee_const_expr sym = (symee_const_expr) key;
  symee_expr expr = *(symee_expr*)slot;
  symee_context ctx = (symee_context) data;
  symee_expr ret_expr = NULL_TREE;

  /* TODO: */
  if (!expr)
    return;

  ret_expr = eval_expr_tree (ctx, expr);
  gcc_assert (ret_expr);
  if (ret_expr)
    *slot = ret_expr;

  return true;
}


static bool
replace_ssa_with_parm_decl (void* key, void** slot, void* data)
{
  symee_const_expr sym = (symee_const_expr) key;
  symee_expr expr = *(symee_expr*)slot;
  symee_context ctx = (symee_context) data;
  symee_expr ret_expr = NULL_TREE;

  if (TREE_CODE (sym) == SSA_NAME &&
      *slot == NULL) {
    tree parm = SSA_NAME_VAR (sym);
    *slot = parm;
  }
}


static symee_context
call_eval_context (symee_context ctx, symee_context eval_ctx)
{
  symee_context ret_ctx = symee_context_copy (eval_ctx);

  //symbolic_pointer_map_traverse (ret_ctx, replace_ssa_with_parm_decl, ctx);
  
  symbolic_pointer_map_traverse (ret_ctx->state, call_eval_context_1, ctx);

  return ret_ctx;
}


static int count_result = 0;

static bool
find_result_decl_1 (const void* key, void** slot, void* data)
{
  symee_const_expr sym = (symee_const_expr) key;

  if (TREE_CODE (sym) == RESULT_DECL ||
      (TREE_CODE (sym) == SSA_NAME &&
       TREE_CODE (SSA_NAME_VAR (sym)) == RESULT_DECL)) {
    *(symee_expr*)data = *slot;
    count_result++;
  }

  return true;
}

  
static symee_expr
find_result_decl (symee_context ctx)
{
  symee_expr expr = NULL_TREE;

  count_result = 0;
  symbolic_pointer_map_traverse (ctx->state, find_result_decl_1, &expr);

  if (dump_details) {
    if (count_result == 0)
      fprintf (dump_file, "SYMEE: %s not find result decl\n", __func__);
    if (count_result > 1)
      fprintf (dump_file,
               "SYMEE: %s not find more than one result decls\n",
               __func__);
  }
    
  return expr;
}


static bool
same_lvalue_array (const_tree expr1, const_tree expr2)
{
  if (TREE_CODE (expr1) == ARRAY_REF
      && TREE_CODE (expr2) == ARRAY_REF) {
    if (TREE_OPERAND (expr1, 0) == TREE_OPERAND (expr2, 0)
        && TREE_OPERAND (expr1, 1) == TREE_OPERAND (expr2, 1))
      return true;
  }

  return false;
}


static bool
same_lvalue_deref (const_tree expr1, const_tree expr2)
{
  if (TREE_CODE (expr1) == MEM_REF
      && TREE_CODE (expr2) == MEM_REF) {
    if (TREE_OPERAND (expr1, 0) == TREE_OPERAND (expr2, 0))
      return true;

    tree addr1 = eval_expr_tree (current_context, TREE_OPERAND (expr1, 0));
    tree addr2 = eval_expr_tree (current_context, TREE_OPERAND (expr2, 1));
    if (addr1 && addr2)
      return symee_value_equal_p (addr1, addr2);
  }

  if (TREE_CODE (expr1) == INDIRECT_REF
      && TREE_CODE (expr2) == INDIRECT_REF) {
    if (TREE_OPERAND (expr1, 0) == TREE_OPERAND (expr2, 0))
      return true;
  }

  return false;
}


static bool
same_lvalue (const_tree expr1, const_tree expr2)
{
  enum tree_code code = TREE_CODE (expr1);
  
  if (TREE_CODE (expr1) != TREE_CODE (expr2))
    return false;
  
  switch (code) {
  case ARRAY_REF:
    return same_lvalue_array (expr1, expr2);
    
  case MEM_REF:
  case INDIRECT_REF:
    return same_lvalue_deref (expr1, expr2);

  default:
    break;
  }
  
  return false;
}


static tree
same_ref_node (symee_ref_node ref1, symee_ref_node ref2)
{
  VEC (loop_p, heap) *loop_nest = NULL;
  struct data_dependence_relation *ddr = NULL;
  loop_p l1 = loop_containing_stmt (ref1->dr->stmt);
  loop_p l2 = NULL;
  loop_p l = NULL;

  if (ref2->dr && ref2->dr->stmt)
    l2 = loop_containing_stmt (ref2->dr->stmt);
  else
    l2 = l1;

  if (l1 == l2)
    l = l1;
  else
    l = find_common_loop (l1, l2);

  if (l->num == 0)
    return chrec_dont_know;
  
  if (!find_loop_nest (l, &loop_nest))
    {
      VEC_safe_push (loop_p, heap, loop_nest, l);
    }
  ddr = initialize_data_dependence_relation (ref1->dr, ref2->dr, loop_nest);
  compute_affine_dependence (ddr, l);

  return DDR_ARE_DEPENDENT (ddr);
}


static symee_expr
search_backward (symee_ref_list ref_list, symee_ref_node ref_node)
{
  symee_expr ret_expr = NULL;
  symee_ref_node node = ref_list->tail;
  while (node) {
    tree dep_r = chrec_dont_know; //same_ref_node (node, ref_node);
    if (dep_r == NULL_TREE) {
      ret_expr = node->value;
      break;
    }
    else if (dep_r == chrec_known) {
      node = node->prev;
      continue;
    }
    
    if (same_lvalue (node->ref, ref_node->ref)) {
      ret_expr = node->value;
      break;
    }
    node = node->prev;
  }

  return ret_expr;
}


/* dup of get_base_var */

static tree
get_base_sym (const_tree ct)
{
  tree t = ct;
  while (!SSA_VAR_P (t)
	 && (!CONSTANT_CLASS_P (t))
	 && TREE_CODE (t) != LABEL_DECL
	 && TREE_CODE (t) != FUNCTION_DECL
	 && TREE_CODE (t) != CONST_DECL
	 && TREE_CODE (t) != CONSTRUCTOR)
    {
      t = TREE_OPERAND (t, 0);
    }
  return t;
}


static symee_expr
symee_get_state_sym_by_ref_1 (symee_context ctx, tree ref)
{
  symee_expr expr = NULL;
  tree ref_sym = get_base_sym (ref);
  symee_expr ref_sym_value = symee_get_state_sym (ctx, ref_sym);

  switch (TREE_CODE (ref)) {
  case MEM_REF:
    {
      tree addr = TREE_OPERAND (ref, 0);
      if (ref_sym == addr
          && ref_sym_value) {
        expr = build2 (TREE_CODE (ref), TREE_TYPE (ref),
                       ref_sym_value, TREE_OPERAND (ref, 1));
      }
    }
    break;

  case INDIRECT_REF:
    {
      tree addr = TREE_OPERAND (ref, 0);
      if (ref_sym == addr
          && ref_sym_value) {
        expr = build1 (TREE_CODE (ref), TREE_TYPE (ref), ref_sym_value);
      }
    }
    break;

  default:
    break;
  }

  if (!expr)
    expr = ref;

  return expr;
}


static symee_expr
symee_get_state_sym_by_ref_from_ref_list (symee_context ctx, tree ref)
{
  symee_expr expr = NULL;
  symee_expr eval_expr = NULL;
  tree ref_sym = get_base_sym (ref);
  symee_ref_list ref_list = symee_get_state_sym_ref_list (ctx, ref_sym);
  if (ref_list) {
    symee_ref_node ref_node = symee_ref_node_create_ref (ref);
    gimple stmt = current_stmt;
    loop_p l = loop_containing_stmt (stmt);
    if (l) {
      ref_node->dr = create_data_ref (l, l, ref_node->ref,
                                      stmt, ref_node->is_read);
      if (l->num)
        expr = search_backward (ref_list, ref_node);
    }
  }
  else {
    expr = symee_get_state_sym_by_ref_1 (ctx, ref);
  }
  
  if (!expr)
    expr = ref;
  
  return expr;
}


#if 1
extern symee_expr
symee_get_state_sym_by_ref (symee_context ctx, tree ref)
{
  tree expr = NULL;
  
  switch (TREE_CODE (ref)) {
  case ARRAY_REF:
#ifdef MEM_AS_ARRAY
  case MEM_REF:
#endif
    {
      tree op0 = TREE_OPERAND (ref, 0);
      tree op1 = TREE_OPERAND (ref, 1);
      tree eval_op0 = eval_expr_tree (ctx, op0);
      tree eval_op1 = eval_expr_tree (ctx, op1);
      tree array_decl = eval_op0 ? eval_op0 : op0; // TODO:
      tree array_index = eval_op1 ? eval_op1 : op1;
      tree base_dsum = symee_get_state_sym (ctx, array_decl);

      tree rho = build2 (SYMEE_RHO, TREE_TYPE (ref), base_dsum, array_index);

      return rho;
    }
    break;
#ifndef MEM_AS_ARRAY
  case MEM_REF:
    {
      tree op0 = TREE_OPERAND (ref, 0);
      tree op1 = TREE_OPERAND (ref, 1);
      tree eval_op0 = eval_expr_tree (ctx, op0);
      tree addr_op0 = eval_op0 ? eval_op0 : op0;
      tree ref_new = build2 (MEM_REF, TREE_TYPE (ref), addr_op0, op1);

      return ref_new;
    }
    break;
#endif
  case INDIRECT_REF:
  default:
    break;
  }

  return ref;
}

#else

extern symee_expr
symee_get_state_sym_by_ref (symee_context ctx, tree ref)
{
  gimple stmt = current_stmt;
  
  struct loop *inner_loop = loop_containing_stmt (stmt);
  struct data_reference *dr = create_data_ref (inner_loop, inner_loop, ref, stmt, true);
  tree base_address = unshare_expr (DR_BASE_ADDRESS (dr));
  tree base = base_address;

  if (TREE_CODE (base_address) == ADDR_EXPR)
    base = TREE_OPERAND (base_address, 0);

  /* <base, */
  if (base) {
    tree dsum = NULL;
    tree array_decl = base;
    tree array_index = size_binop (PLUS_EXPR, DR_INIT (dr), DR_OFFSET (dr)); // TODO:
    tree base_dsum = symee_get_state_sym (ctx, array_decl);

    if (base_dsum == NULL)
      base_dsum = array_decl;

    tree rho = build2 (SYMEE_RHO, TREE_TYPE (ref), base_dsum, array_index);

    return rho;
  }

  return ref;
}

#endif

static symee_global_info
symee_initialize_global_info (void)
{
  symee_global_info ginfo = ggc_alloc_cleared_symee_global_info_def ();

  ginfo->node_info =
    splay_tree_new_ggc (splay_tree_compare_pointers,
                        ggc_alloc_splay_tree_cgraph_node_symee_cgraph_node_info_def_splay_tree_s,
                        ggc_alloc_splay_tree_cgraph_node_symee_cgraph_node_info_def_splay_tree_node_s);
  
  ginfo->phase = SYMEE_PHASE_FUNCTION_SUMMARY;
  
  return ginfo;
}


void
symee_initialize (unsigned int flags)
{
  if (mem_report) {
    fprintf (stderr, "symee: memory report before symee\n\n");
    dump_memory_report (false);
  }
  
  global_info = symee_initialize_global_info ();
  
  symee_eval_flags = flags;
  symee_eval_flags = symee_eval_flags & (~SYMEE_EVAL_VOPS);
  symee_eval_flags = symee_eval_flags & (~SYMEE_EVAL_MEMSYMS);

  symee_run_vrp = 0;

  flag_evaluation_order = 1;

  return;
}


void
symee_finalize (void)
{
  if (mem_report) {
    fprintf (stderr, "symee: memory report after symee\n\n");
    dump_memory_report (false);
  }

  return;
}


static bool
is_cfg_entry_bb (basic_block bb)
{
  return (bb == ENTRY_BLOCK_PTR);
}


/* TODO: if 'bb' is PREHEADER of 'some' loop, return true */

static bool
is_loop_preheader (basic_block bb)
{
  bool ret = false;
  loop_iterator li;
  struct loop* l;
  int i = 0;
  
  if (single_succ_p (bb)) {
    basic_block header = single_succ_edge (bb)->dest;
    if (is_loop_header (header)) {
      ret = true;
      edge e = loop_preheader_edge (header->loop_father);
      gcc_assert (bb == e->src);
    }
  }
  
  return ret;
  
  /* TOOD: record and query */
  FOR_EACH_LOOP (li, l, 0) {
    edge e = loop_preheader_edge (l);
    gcc_assert (e);
    if (e->src == bb)
      i++;
  }

  symee_assert (i < 2, (stderr, "SYMEE: bb is preheader of %d loops\n", i));

  return i > 0;
}


static bool
is_loop_header (basic_block bb)
{
  return (bb
          && bb != ENTRY_BLOCK_PTR
          && bb->loop_father
          && bb->loop_father->header
          && bb == bb->loop_father->header);
}


/* TODO: if 'bb' is POSTBODY of 'some' loop, return true */

static bool
is_loop_postbody (basic_block bb)
{
  /* when LOOPS_HAVE_SIMPLE_LATCHES, postbody is loop latch */
  return (bb
          && bb->loop_father
          && bb->loop_father->latch
          && bb == bb->loop_father->latch);
}


/* TODO: if 'bb' is POSTEXIT of 'some' loop, return true */

static bool
is_loop_postexit (basic_block bb)
{
  loop_iterator li;
  struct loop* current_loop;

  VEC (edge, heap) * exits;
  unsigned i;
  edge exit_edge;

  int count = 0;

  if (!bb)
    return false;

  FOR_EACH_LOOP (li, current_loop, 0) {
    exits = get_loop_exit_edges (current_loop);
    FOR_EACH_VEC_ELT (edge, exits, i, exit_edge) {
      if (exit_edge->dest == bb)
        count++;
    }
  }

  // At least one preceding edge is loop exit edge
  return count > 0;
}


/* TODO: if 'bb' is POSTEXIT of 'some' loop, return true */

static struct loop*
get_loop_of_postexit (basic_block bb)
{
  loop_iterator li;
  struct loop* current_loop = NULL;
  struct loop* ret_loop = NULL;
  VEC (edge, heap) * exits;
  unsigned i;
  edge exit_edge;

  int count = 0;

  if (!bb)
    return ret_loop;

  FOR_EACH_LOOP (li, current_loop, 0) {
    exits = get_loop_exit_edges (current_loop);
    FOR_EACH_VEC_ELT (edge, exits, i, exit_edge) {
      if (exit_edge->dest == bb) {
        if (ret_loop == NULL)
          ret_loop = current_loop;
        count++;
      }
    }
  }

  gcc_assert (count <= 1);
  
  return ret_loop;
}


/* ASSERT_EXPR */

static symee_context
gen_context_assert (symee_context ctx, gimple stmt)
{
  return ctx;
}


static symee_context
gen_context_ref_range (symee_context ctx, gimple stmt, tree ref, tree value)
{
  tree sym = get_base_sym (ref);
  if (TREE_CODE (ref) == MEM_REF) {
    tree addr_expr = symee_get_state_sym (ctx, TREE_OPERAND (ref, 0));
    symee_range range = symee_range_create (0, addr_expr, NULL);
    symee_set_sym_index_range (ctx, addr_expr, range);
  }

  return ctx;
}


static symee_context
gen_context_ref (symee_context ctx, gimple stmt, tree ref, tree value)
{
  loop_p l = loop_containing_stmt (stmt);
  tree ref_sym = get_base_sym (ref);
  symee_ref_list ref_list = symee_get_state_sym_ref_list_create (ctx, ref_sym);
  symee_ref_node ref_node = symee_ref_list_append_ref (ref_list, ref);
  ref_node->stmt = stmt;
  ref_node->is_read = false;
  ref_node->dr = create_data_ref (l, l, ref, stmt, false);
  ref_node->value = value;

#if 0
  if (TREE_CODE (ref) == ARRAY_REF) {
    tree index = NULL_TREE;
    tree chrec = analyze_array_evolution (l, ref_sym, &index);
  }
#endif
  
  symee_ref_list_kill (ref_list, ref_node);

  gen_context_ref_range (ctx, stmt, ref, value);

  return ctx;
}


static tree
fold_build_dsum (enum tree_code code, tree ret_type,
                 tree base_dsum, tree rhs_expr,
                 tree array_index, tree array_decl)
{
  tree dsum = NULL_TREE;
  
  dsum = build4 (code, ret_type, base_dsum, rhs_expr, array_index, array_decl);

  return dsum;
}

#if 1
static symee_context
gen_context_assign_ref (symee_context ctx, gimple stmt)
{
  tree lhs = gimple_assign_lhs (stmt);
  tree rhs = gimple_assign_rhs_to_tree (stmt);
  symee_expr rhs_expr = eval_expr_tree (ctx, rhs);
  symee_context ret_ctx = symee_context_copy_init (ctx);
  
  gcc_assert (REFERENCE_CLASS_P (lhs));

  switch (TREE_CODE (lhs)) {
  case ARRAY_REF:
#ifdef MEM_AS_ARRAY
  case MEM_REF:
#endif
    {
      tree dsum = NULL;
      tree op0 = TREE_OPERAND (lhs, 0);
      tree op1 = TREE_OPERAND (lhs, 1);
      tree eval_op0 = eval_expr_tree (ctx, op0);
      tree eval_op1 = eval_expr_tree (ctx, op1);
      tree array_decl = op0; // how about pointer to array ?
      tree array_index = eval_op1 ? eval_op1 : op1;
      tree base_dsum = symee_get_state_sym (ctx, array_decl);

      if (base_dsum == NULL)
        base_dsum = array_decl;

#if 0
      dsum = build4 (SYMEE_DIRECT_SUM, void_type_node,
                     base_dsum, rhs_expr, array_index, array_decl);
#else
      dsum = fold_build_dsum (SYMEE_DIRECT_SUM, void_type_node,
                              base_dsum, rhs_expr, array_index, array_decl);
#endif
      symee_set_state_sym (ret_ctx, array_decl, dsum);

      if (dump_details)
        dump_symbolic_expr (dump_file, array_decl, dsum);
    }
    break;

#ifndef MEM_AS_ARRAY
  case MEM_REF:
#endif
  case INDIRECT_REF:
  default:
    symee_set_state_sym (ret_ctx, lhs, rhs_expr);
    if (dump_details)
      dump_symbolic_expr (dump_file, lhs, rhs_expr);

    break;
  }

  return ret_ctx;
}

#else

static symee_context
gen_context_assign_ref (symee_context ctx, gimple stmt)
{
  tree lhs = gimple_assign_lhs (stmt);
  tree rhs = gimple_assign_rhs_to_tree (stmt);
  symee_expr rhs_expr = eval_expr_tree (ctx, rhs);
  symee_context ret_ctx = symee_context_copy_init (ctx);
  
  gcc_assert (REFERENCE_CLASS_P (lhs));

  struct loop *inner_loop = loop_containing_stmt (stmt);
  struct data_reference *dr = create_data_ref (inner_loop, inner_loop, lhs, stmt, false);
  tree base_address = unshare_expr (DR_BASE_ADDRESS (dr));
  tree base = base_address;

  if (TREE_CODE (base_address) == ADDR_EXPR)
    base = TREE_OPERAND (base_address, 0);

  /* <base, */
  if (base) {
    tree dsum = NULL;
    tree array_decl = base;
    tree array_index = size_binop (PLUS_EXPR, DR_INIT (dr), DR_OFFSET (dr)); // TODO:
    tree base_dsum = symee_get_state_sym (ctx, array_decl);

    if (base_dsum == NULL)
      base_dsum = array_decl;

    dsum = build4 (SYMEE_DIRECT_SUM, void_type_node,
                   base_dsum, rhs_expr, array_index, array_decl);
    symee_set_state_sym (ret_ctx, array_decl, dsum);

    if (dump_details)
      dump_symbolic_expr (dump_file, array_decl, dsum);
  }

  symee_set_state_sym (ret_ctx, lhs, rhs_expr);
  if (dump_details)
    dump_symbolic_expr (dump_file, lhs, rhs_expr);

  return ret_ctx;
}

#endif


static symee_context
gen_context_assign_1 (symee_context ctx, gimple stmt)
{
  tree lhs = gimple_assign_lhs (stmt);
  tree rhs = gimple_assign_rhs_to_tree (stmt);
  symee_expr expr = eval_expr_tree (ctx, rhs);

  if (dump_details)
    dump_symbolic_expr (dump_file, lhs, expr);

  if (eval_reference &&
      REFERENCE_CLASS_P (lhs)) {
    if (symee_use_ref_list)
      ctx = gen_context_ref (ctx, stmt, lhs, expr);
    else
      ctx = gen_context_assign_ref (ctx, stmt);
  }

  else
    symee_set_state_sym (ctx, lhs, expr);

  if (symee_run_vrp) {
    extern void symee_dump_symbolic_range (symee_context, symee_expr);
    symee_dump_symbolic_range (ctx, lhs);
  }
  
  return ctx;
}


static symee_context
gen_context_assign (symee_context ctx, gimple stmt)
{
  tree lhs = gimple_assign_lhs (stmt);
  location_t loc = gimple_location (stmt);
  enum tree_code subcode = gimple_assign_rhs_code (stmt);

  switch (get_gimple_rhs_class (subcode)) {
  case GIMPLE_SINGLE_RHS:
    {
      tree rhs = gimple_assign_rhs1 (stmt);
      symee_expr expr = symee_get_state_sym (ctx, rhs);
      if (dump_details)
        dump_symbolic_expr (dump_file, lhs, expr);
      symee_set_state_sym (ctx, lhs, expr);
    }
    break;

  case GIMPLE_UNARY_RHS:
    {
      tree rhs = gimple_assign_rhs1 (stmt);
      symee_expr expr = symee_get_state_sym (ctx, rhs);
      if (dump_file)
        dump_symbolic_expr (dump_file, lhs, expr);
      symee_set_state_sym (ctx, lhs, expr);
    }
    break;

  case GIMPLE_BINARY_RHS:
    {
      if (gimple_assign_rhs_code (stmt) == PLUS_EXPR
          || gimple_assign_rhs_code (stmt) == MINUS_EXPR
          || gimple_assign_rhs_code (stmt) == MULT_EXPR) {
        tree rhs1 = gimple_assign_rhs1 (stmt);
        symee_expr expr1 = symee_get_state_sym (ctx, rhs1);
        tree rhs2 = gimple_assign_rhs2 (stmt);
        symee_expr expr2 = symee_get_state_sym (ctx, rhs2);

        if (expr1 != rhs1 || expr2 != rhs2) {
          // TODO: duplicate ???
          symee_expr result =
            symee_fold_binary_loc (loc, subcode,
                                   TREE_TYPE (gimple_assign_lhs (stmt)),
                                   expr1, expr2);
          if (dump_file)
            dump_symbolic_expr (dump_file, lhs, result);
          symee_set_state_sym (ctx, lhs, result);
        }
        else {
          symee_expr expr = gimple_assign_rhs_to_tree (stmt);
          if (dump_details)
            dump_symbolic_expr (dump_file, lhs, expr);
          symee_set_state_sym (ctx, lhs, expr);
        }
      }
    }
    break;

  default:
    gen_context_assign_1 (ctx, stmt);
    break;
  }

  return ctx;
}


static symee_context
gen_context_asm (symee_context ctx, gimple stmt)
{
  gcc_unreachable ();
  
  return ctx;
}


struct merge_callee_context_data {
  gimple call_stmt;
  symee_context ctx;
  symee_context in_ctx;
  symee_context out_ctx;
  symee_context ret_ctx;
  symee_context actual_out_ctx;
};

static bool
merge_callee_context_1 (const void *key, void **slot, void *data)
{
  struct merge_callee_context_data* mdata =
    (struct merge_callee_context_data*) data;
  symee_const_expr sym = (symee_const_expr) key;
  symee_expr expr = *(symee_expr*)slot;
  symee_expr basesym = get_base_sym (sym);
  symee_expr retsym = NULL;

  retsym = eval_expr_tree (mdata->actual_out_ctx, sym);
  if (!retsym)
    retsym = sym;
  
  symee_set_state_sym (mdata->ret_ctx, retsym, expr);

  return true;
}


static bool
merge_callee_context_1_1 (const void *key, void **slot, void *data)
{
  struct merge_callee_context_data* mdata =
    (struct merge_callee_context_data*) data;
  symee_const_expr sym = (symee_const_expr) key;
  symee_expr expr = *(symee_expr*)slot;
  symee_expr basesym = get_base_sym (sym);
  symee_expr retsym = NULL;

  /* global out */
  if (DECL_P (basesym) && is_global_var (basesym)) {
    retsym = sym;
  }

  /* formal out */
  if (REFERENCE_CLASS_P (sym) &&
      TREE_CODE (basesym) == SSA_NAME) {
    symee_expr actual_sym = NULL;
    if (eval_formal_outsym) {
      symee_context formal_out_ctx = symee_context_create ();
      formal_out_ctx->state = symbolic_pointer_map_copy (current_node_info->formal_out);
      eval_context (mdata->actual_out_ctx, formal_out_ctx);
      actual_sym = eval_expr_tree (formal_out_ctx, sym);
    }
    else
      actual_sym = eval_expr_tree (mdata->out_ctx, sym);
    gcc_assert (actual_sym);
    symee_set_state_sym (mdata->ret_ctx, actual_sym, expr);
  }

  if (retsym)
    symee_set_state_sym (mdata->ret_ctx, sym, expr);

  return true;
}


/* ctx : context before call_stmt
   in_ctx : input ctx of callee
   out_ctx : output ctx of callee

   return context of after call_stmt */

static symee_context
merge_callee_context (symee_context ctx, gimple call_stmt,
                      symee_context in_ctx, symee_context out_ctx,
                      symee_context actual_out_ctx)
{
  symee_context ret_ctx = symee_context_copy (ctx);
  struct merge_callee_context_data mdata =
    {call_stmt, ctx, in_ctx, out_ctx, ret_ctx, actual_out_ctx};

  symbolic_pointer_map_traverse (out_ctx->state, merge_callee_context_1, &mdata);

  return ret_ctx;
}


/* 'call_stmt'' has no callee cgraph_node */

static symee_context
gen_context_call_indirect (symee_context ctx, gimple call_stmt)
{
  struct cgraph_node* node = current_cgraph_node;
  struct cgraph_edge* edge = cgraph_edge (node, call_stmt);
  tree lhs = gimple_call_lhs (call_stmt);
  tree ret_type = gimple_call_return_type (call_stmt);
  tree tmp_var = NULL_TREE;
  if (lhs) {
    char var_name [256] = "icall";
    tree fn = gimple_call_fn (call_stmt);
    if (TREE_CODE (fn) == SSA_NAME)
      fn = SSA_NAME_VAR (fn);
    if (DECL_P (fn)) {
      strcat (var_name, ".");
      if (DECL_NAME (fn))
        strcat (var_name, IDENTIFIER_POINTER (DECL_NAME (fn)));
      else {
        strcat (var_name, "D.");
        sprintf (var_name + strlen (var_name), "%d", DECL_UID (fn));
      }
    }
    strcat (var_name, ".suffix");
    tmp_var = add_new_static_var (ret_type);
    DECL_NAME (tmp_var) = create_tmp_var_name (var_name);

    symee_set_state_sym (ctx, lhs, tmp_var);
    if (dump_details)
      dump_symbolic_expr (dump_file, lhs, tmp_var);
  }

  return ctx;
}


/* callee cgraph_node of 'call_stmt' has no symee_context */

static symee_context
gen_context_call_no_body (symee_context ctx, gimple call_stmt)
{
  struct cgraph_node* node = current_cgraph_node;
  struct cgraph_edge* edge = cgraph_edge (node, call_stmt);
  struct cgraph_node* callee_node = edge->callee;
  tree lhs = gimple_call_lhs (call_stmt);
  tree ret_type = gimple_call_return_type (call_stmt);
  tree fndecl = gimple_call_fndecl (call_stmt);
  const char* fname = IDENTIFIER_POINTER (DECL_NAME (fndecl));
  tree tmp_var = NULL_TREE;

  if (is_builtin_fn (fndecl)
      && DECL_BUILT_IN_CLASS (fndecl) == BUILT_IN_NORMAL)
    switch (DECL_FUNCTION_CODE (fndecl))
      {
      case BUILT_IN_MALLOC:
      case BUILT_IN_CALLOC:
      case BUILT_IN_ALLOCA:
        {
          /* base_pointer + 1 */
          tree rhs_ptr = build2 (POINTER_PLUS_EXPR, ptr_type_node,
                                 current_node_info->heap_pointer,
                                 size_int (1));
          symee_set_state_sym (ctx, lhs, rhs_ptr);
          return ctx;
        }
        break;
      case BUILT_IN_REALLOC:
        {
          tree old_ptr = gimple_call_arg (call_stmt, 0);
          /* TODO: */
          symee_set_state_sym (ctx, lhs, eval_expr_tree (ctx, old_ptr));
          return ctx;
        }
        break;
      case BUILT_IN_FREE:
        {
          tree old_ptr = gimple_call_arg (call_stmt, 0);
          if (is_gimple_variable (old_ptr))
            symee_delete_state_sym (ctx, old_ptr);
          return ctx;
        }
        break;
      default:
        break;
      }

  
  if (strcmp (fname, "assert") == 0) {
    tree expr = gimple_call_arg (call_stmt, 0);
    gcc_assert (expr);
    if (ctx->path_cond) {
      symee_expr and_expr = build2 (TRUTH_AND_EXPR, boolean_type_node,
                                    ctx->path_cond, expr);
      ctx->path_cond = eval_expr_tree (ctx, and_expr);
    }
    else
      ctx->path_cond = eval_expr_tree (ctx, expr);

    if (dump_details) {
      fprintf (dump_file, "assert change path_cond:\n");
      dump_symbolic_expr (dump_file, expr, ctx->path_cond);
    }
    return ctx;
  }

  if (lhs && strcmp (fname, "malloc") == 0)
    {
      /* base_pointer + 1 */
      tree rhs_ptr = build2 (POINTER_PLUS_EXPR, ptr_type_node,
                             current_node_info->heap_pointer,
                             size_int (1));
      symee_set_state_sym (ctx, lhs, rhs_ptr);

      symee_set_range_base (ctx, lhs, tmp_var);
      tree size = gimple_call_arg (call_stmt, 0);
      tree min = tmp_var;
      tree max = build2 (PLUS_EXPR, TREE_TYPE (lhs), min, size);
      symee_range range = symee_range_create (0, min, max);
      symee_set_sym_index_range (ctx, lhs, range);

      return ctx;
    }

  if (lhs) {
    char var_name [256] = "call.";
    strcat (var_name, fname);
    strcat (var_name, ".suffix");
    tmp_var = add_new_static_var (ret_type);
    DECL_NAME (tmp_var) = create_tmp_var_name (var_name);

    symee_set_state_sym (ctx, lhs, tmp_var);
    if (dump_details)
      dump_symbolic_expr (dump_file, lhs, tmp_var);

    return ctx;
  }
  
  return ctx;
}


static symee_context
gen_context_call_2 (symee_context ctx, gimple call_stmt)
{
  struct cgraph_node* caller_node = current_cgraph_node;
  struct cgraph_edge* edge = cgraph_edge (caller_node, call_stmt);
  struct cgraph_node* callee_node = edge->callee;
  symee_context in_ctx = symee_context_copy (ctx);
  symee_context callee_ctx = cgraph_node_get_context (callee_node);
  symee_context out_ctx = NULL;
  symee_context actual_out_ctx = symee_context_create ();
  symee_context ret_ctx = symee_context_copy (ctx);
  tree fndecl = gimple_call_fndecl (call_stmt);
  tree parm;
  size_t i = 0;
  symee_expr lhs = gimple_call_lhs (call_stmt);
  
  /* parameter in */
  for (parm = DECL_ARGUMENTS (fndecl); parm; parm = DECL_CHAIN (parm)) {
    if (i < gimple_call_num_args (call_stmt)) {
      tree argi = gimple_call_arg (call_stmt, i);
      symee_expr actual_expr = eval_expr_tree (ctx, argi);
      if (eval_formal_insym) {
        symee_expr insym = cgraph_node_get_insym (callee_node, parm);
        symee_set_state_sym (in_ctx, insym, actual_expr);
      }
      else
        symee_set_state_sym (in_ctx, parm, actual_expr);

      if (eval_formal_outsym) {
        symee_expr outsym = cgraph_node_get_outsym (callee_node, parm);
        symee_set_state_sym (actual_out_ctx, outsym, actual_expr);
      }
    }
    else {
      symee_assert (0, (stderr, "SYMEE: can not process variable call arguments\n"));
    }
    
    i++;
  }

  /* re-eval callee_ctx under in_ctx */
  out_ctx = eval_context (in_ctx, callee_ctx);

  /* return value */
  symee_expr retsym = cgraph_node_get_retsym (callee_node);
  symee_expr retval = symee_delete_state_sym (out_ctx, retsym);
  if (lhs) {
    if (retval == NULL || retval == retsym) {
      symee_warn (0, (stderr,
                      "SYMEE_WARNING: null return result of call %s\n",
                      cgraph_node_name (callee_node)));
      symee_set_state_sym (ret_ctx, lhs, add_new_sym (lhs));
    }
    else
      symee_set_state_sym (ret_ctx, lhs, retval);
    if (dump_details)
      dump_symbolic_expr (dump_file, lhs, retval);
  }


  /* parameter out */
  ret_ctx = merge_callee_context (ret_ctx, call_stmt, in_ctx, out_ctx, actual_out_ctx);

#if 0
  /* parameter out */
  for (parm = DECL_ARGUMENTS (fndecl); parm; parm = DECL_CHAIN (parm)) {
    if (i < gimple_call_num_args (call_stmt)) {
      tree argi = gimple_call_arg (call_stmt, i);
      symee_expr actual_expr = eval_expr_tree (ctx, argi);
      if (eval_formal_insym) {
        symee_expr insym = cgraph_node_get_insym (callee_node, parm);
        symee_set_state_sym (in_ctx, insym, actual_expr);
      }
      else
        symee_set_state_sym (in_ctx, parm, actual_expr);
    }
    else {
      symee_assert (0, (stderr,
                        "SYMEE: can not process variable call arguments\n"));
    }
    
    i++;
  }
#endif
  
  if (dump_details) {
    fprintf (dump_file, "result context of merge_callee_context\n");
    dump_context_brief (dump_file, ret_ctx);
  }

  return ret_ctx;
}


/* instantiate formal parameters with actual arguments

   for example

   function decl:
   void foo (int a);

   function call:
   foo (b);

   We generate new context with new state
   a = b;

   if we had introduced new symbol for formal, such as
   a = a_in;
   now we have
   a_in = b;

   as the parameter, if we had introduced new symbol for actual, such as
   b_in = b;
   we have
   a_in = b_in;

   now we can re-evaluate local context in foo.

   keep this interface clear, so it can be used by different environment */

static symee_context
gen_context_parameter_in (symee_context ctx, cgraph_edge_p edge)
{
  gimple call_stmt = edge->call_stmt;
  struct cgraph_node* caller_node = edge->caller;
  struct cgraph_node* callee_node = edge->callee;

  symee_context in_ctx = symee_context_copy (ctx);
  tree fndecl = gimple_call_fndecl (call_stmt);
  tree parm;
  size_t i = 0;

  /* use caller's node_info */
  symee_cgraph_node_info old_info = current_node_info;
  current_node_info = get_cgraph_node_info (caller_node);
  
  for (parm = DECL_ARGUMENTS (fndecl); parm; parm = DECL_CHAIN (parm)) {
    if (i < gimple_call_num_args (call_stmt)) {
      tree argi = gimple_call_arg (call_stmt, i);
      symee_expr actual_expr = eval_expr_tree (ctx, argi);
      if (eval_formal_insym) {
        symee_expr insym = cgraph_node_get_insym (callee_node, parm);
        symee_set_state_sym (in_ctx, insym, actual_expr);
      }
      else
        symee_set_state_sym (in_ctx, parm, actual_expr);
    }
    else {
      symee_assert (0, (stderr, "SYMEE: can not process variable call arguments\n"));
    }
    
    i++;
  }
#if 0
  for (parm = DECL_ARGUMENTS (fndecl); parm; parm = DECL_CHAIN (parm)) {
    if (i < gimple_call_num_args (call_stmt)) {
      tree actual_sym = gimple_call_arg (call_stmt, i);
      symee_expr actual_expr = symee_get_state_sym (ctx, actual_sym);
      symee_set_state_sym (in_ctx, parm, actual_expr);
    }
    else {
      fprintf (stderr, "SYMEE: can not process variable call arguments\n");
      gcc_unreachable ();
    }
    
    i++;
  }
#endif

  current_node_info = old_info;

  return in_ctx;
}


static symee_context
gen_context_call_1 (symee_context ctx, gimple call_stmt)
{
  struct cgraph_node* caller_node = current_cgraph_node;
  struct cgraph_edge* edge = cgraph_edge (caller_node, call_stmt);
  struct cgraph_node* callee_node = edge->callee;
  symee_context callee_ctx = cgraph_node_get_context (callee_node);
  symee_context ret_ctx = symee_context_copy (callee_ctx);
  tree fndecl = gimple_call_fndecl (call_stmt);
  tree parm;
  size_t i = 0;
  symee_expr lhs = gimple_call_lhs (call_stmt);
  symee_context in_ctx = symee_context_copy (ctx);

  if (dump_file && (dump_flags & TDF_DETAILS)) {
    fprintf (dump_file, "in_ctx:\n");
    dump_context (dump_file, in_ctx);
  }
  
  for (parm = DECL_ARGUMENTS (fndecl); parm; parm = DECL_CHAIN (parm)) {
    if (i < gimple_call_num_args (call_stmt)) {
      tree actual_sym = gimple_call_arg (call_stmt, i);
      symee_expr actual_expr = symee_get_state_sym (ctx, actual_sym);
      symee_set_state_sym (in_ctx, parm, actual_expr);
    }
    else {
      fprintf (stderr, "SYMEE: can not process variable call arguments\n");
      gcc_unreachable ();
    }
    
    i++;
  }

  if (dump_file && (dump_flags & TDF_DETAILS)) {
    fprintf (dump_file, "in_ctx: after processing parameter in\n");
    dump_context (dump_file, in_ctx);
  }

  {
    ret_ctx = call_eval_context (in_ctx, ret_ctx);
    if (lhs) {
      symee_expr rhs = find_result_decl (ret_ctx);
      symee_set_state_sym (ret_ctx, lhs, rhs);
    }
  }
  
  if (dump_file && (dump_flags & TDF_DETAILS)) {
    fprintf (dump_file, "in_ctx: after eval_context\n");
    dump_context (dump_file, in_ctx);
    fprintf (dump_file, "call_ctx: after eval_context\n");
    dump_context_brief (dump_file, ret_ctx);
  }
  
  // merge ret_ctx into ctx ?
  
  return ret_ctx;
}


static symee_context
gen_context_call (symee_context ctx, gimple call_stmt)
{
  struct cgraph_node* node = current_cgraph_node;
  struct cgraph_edge* edge = cgraph_edge (node, call_stmt);
  struct cgraph_node* callee_node = edge->callee;
  symee_context callee_ctx = NULL;

  if (!callee_node)
    return gen_context_call_indirect (ctx, call_stmt);

  callee_ctx = cgraph_node_get_context (callee_node);

  /* no body or no ctx for callee */
  if (!callee_ctx)
    return gen_context_call_no_body (ctx, call_stmt);

  return gen_context_call_2 (ctx, call_stmt);
}


static symee_context
push_cfun_additional (symee_context ctx, gimple call_stmt)
{

}


static symee_context
pop_cfun_additional (symee_context ctx, gimple call_stmt)
{
  scev_finalize ();
  scev_initialize ();
}


static symee_context
gen_context_call_r_1 (symee_context ctx, gimple call_stmt)
{
  struct cgraph_node* caller_node = current_cgraph_node;
  struct cgraph_edge* edge = cgraph_edge (caller_node, call_stmt);
  struct cgraph_node* callee_node = edge->callee;
  symee_context in_ctx = symee_context_copy (ctx);
  symee_context callee_ctx = cgraph_node_get_context (callee_node);
  symee_context out_ctx = NULL;
  symee_context actual_out_ctx = symee_context_create ();
  symee_context ret_ctx = symee_context_copy (ctx);
  tree fndecl = gimple_call_fndecl (call_stmt);
  tree parm;
  size_t i = 0;
  symee_expr lhs = gimple_call_lhs (call_stmt);
  
  /* parameter in */
  for (parm = DECL_ARGUMENTS (fndecl); parm; parm = DECL_CHAIN (parm)) {
    if (i < gimple_call_num_args (call_stmt)) {
      tree argi = gimple_call_arg (call_stmt, i);
      symee_expr actual_expr = eval_expr_tree (ctx, argi);
      if (eval_formal_insym) {
        symee_expr insym = cgraph_node_get_insym (callee_node, parm);
        symee_set_state_sym (in_ctx, insym, actual_expr);
      }
      else
        symee_set_state_sym (in_ctx, parm, actual_expr);

      if (eval_formal_outsym) {
        symee_expr outsym = cgraph_node_get_outsym (callee_node, parm);
        symee_set_state_sym (actual_out_ctx, outsym, actual_expr);
      }
    }
    else {
      symee_assert (0, (stderr, "SYMEE: can not process variable call arguments\n"));
    }
    
    i++;
  }

#if 0
  /* re-eval callee_ctx under in_ctx */
  out_ctx = eval_context (in_ctx, callee_ctx);
#endif

  gen_context_cgraph_node (in_ctx, callee_node);
  
  out_ctx = cgraph_node_get_context (callee_node);
  
  /* return value */
  symee_expr retsym = cgraph_node_get_retsym (callee_node);
  symee_expr retval = symee_delete_state_sym (out_ctx, retsym);
  if (lhs) {
    symee_set_state_sym (ret_ctx, lhs, retval);
    if (dump_details)
      dump_symbolic_expr (dump_file, lhs, retval);
  }


  /* parameter out */
  ret_ctx = merge_callee_context (ret_ctx, call_stmt,
                                  in_ctx, out_ctx, actual_out_ctx);

#if 0
  /* parameter out */
  for (parm = DECL_ARGUMENTS (fndecl); parm; parm = DECL_CHAIN (parm)) {
    if (i < gimple_call_num_args (call_stmt)) {
      tree argi = gimple_call_arg (call_stmt, i);
      symee_expr actual_expr = eval_expr_tree (ctx, argi);
      if (eval_formal_insym) {
        symee_expr insym = cgraph_node_get_insym (callee_node, parm);
        symee_set_state_sym (in_ctx, insym, actual_expr);
      }
      else
        symee_set_state_sym (in_ctx, parm, actual_expr);
    }
    else {
      symee_assert (0, (stderr, "SYMEE: can not process variable call arguments\n"));
    }
    
    i++;
  }
#endif
  
  if (dump_details) {
    fprintf (dump_file, "result context of merge_callee_context\n");
    dump_context_brief (dump_file, ret_ctx);
  }

  return ret_ctx;
}


static symee_context
gen_context_call_r (symee_context ctx, gimple call_stmt)
{
  struct cgraph_node* node = current_cgraph_node;
  struct cgraph_edge* edge = cgraph_edge (node, call_stmt);
  struct cgraph_node* callee_node = edge->callee;
  symee_context callee_ctx = NULL;
  symee_context ret_ctx = NULL;
  
  if (!callee_node)
    return gen_context_call_indirect (ctx, call_stmt);

  callee_ctx = cgraph_node_get_context (callee_node);

  /* no body or no ctx for callee */
  if (!callee_ctx)
    return gen_context_call_no_body (ctx, call_stmt);

  ret_ctx = gen_context_call_r_1 (ctx, call_stmt);

  pop_cfun_additional (ret_ctx, call_stmt);
  
  return ret_ctx;
}


static symee_context
gen_context_goto (symee_context ctx, gimple stmt)
{
  gcc_assert (0);
  return ctx;
}


static symee_context
gen_context_switch (symee_context ctx, gimple stmt)
{
  tree index = gimple_switch_index (stmt);
  symee_expr sym = eval_expr_tree (ctx, index);
  size_t n = gimple_switch_num_labels (stmt);
  tree default_label = gimple_switch_default_label (stmt);
  tree default_cond_expr = NULL_TREE;
  size_t i;

  /* default_label must be label 0 */
  for (i = 1; i < n; i++) {
    tree case_label_expr = gimple_switch_label (stmt, i);
    tree case_label = CASE_LABEL (case_label_expr);
    basic_block case_bb = label_to_block (case_label);
    edge case_edge = find_edge (gimple_bb (stmt), case_bb);
    tree case_low = CASE_LOW (case_label_expr);
    tree case_high = CASE_HIGH (case_label_expr);
    tree case_cond_expr = NULL_TREE;
    tree not_case_cond_expr = NULL_TREE;
    symee_expr path_cond = NULL_TREE;
    symee_context case_ctx = symee_context_copy (ctx);
    
    /* index >= low && index <= high */
    if (case_high) {
      tree ge_expr = build2 (GE_EXPR, boolean_type_node, index, case_low);
      tree le_expr = build2 (LE_EXPR, boolean_type_node, index, case_high);
      case_cond_expr = build2 (TRUTH_AND_EXPR, boolean_type_node,
                               ge_expr, le_expr);
    }
    else
      case_cond_expr = build2 (EQ_EXPR, boolean_type_node, index, case_low);

    not_case_cond_expr = build1 (TRUTH_NOT_EXPR, boolean_type_node,
                                 case_cond_expr);
    if (default_cond_expr) {
      default_cond_expr = build2 (TRUTH_AND_EXPR, boolean_type_node,
                                  default_cond_expr, not_case_cond_expr);
    }
    else
      default_cond_expr = not_case_cond_expr;

    path_cond = eval_expr_tree (ctx, case_cond_expr);
    if (!path_cond)
      path_cond = case_cond_expr;
    
    symee_set_path_cond (case_ctx, path_cond);

    /* update case_ctx by case value if ... */
    if (!case_high) {
      if (1)
        symee_set_state_sym (case_ctx, index, case_low);
      else
        symee_context_update_path_cond (case_ctx);
    }

    set_edge_context (case_edge, case_ctx);

    if (dump_details) {
      fprintf (dump_file, "switch: case %zd path_cond = ", i);
      print_generic_expr (dump_file, case_ctx->path_cond, 0);
      fprintf (dump_file, "\n");
    }
  }

  /* default_label, 0'th label */
  symee_context default_ctx = symee_context_copy (ctx);
  basic_block default_bb = label_to_block (CASE_LABEL (default_label));
  edge default_edge = find_edge (gimple_bb (stmt), default_bb);
  tree default_path_cond = eval_expr_tree (ctx, default_cond_expr);
  if (!default_path_cond)
    default_path_cond = default_cond_expr;

  symee_set_path_cond (default_ctx, default_path_cond);
  /* default value of index seldom be const, no need to eval */
  set_edge_context (default_edge, default_ctx);
  
  if (dump_details) {
    fprintf (dump_file, "switch: default path_cond = ");
    print_generic_expr (dump_file, default_ctx->path_cond, 0);
    fprintf (dump_file, "\n");
  }

  return ctx;
}


static symee_context
gen_context_cond (symee_context ctx, gimple stmt)
{
  enum tree_code cond_expr_code = gimple_cond_code (stmt);
  tree l = gimple_cond_lhs (stmt);
  tree le = NULL; //eval_expr_tree (ctx, l);
  tree cond_expr_lhs = le ? le : l;
  tree r = gimple_cond_rhs (stmt);
  tree re = NULL; //eval_expr_tree (ctx, r);
  tree cond_expr_rhs = re ? re : r;
  tree cond_expr = build2 (cond_expr_code, boolean_type_node,
                           cond_expr_lhs, cond_expr_rhs);
  tree not_cond_expr = build1 (TRUTH_NOT_EXPR, boolean_type_node, cond_expr);
  tree fold_cond_expr = eval_expr_tree (ctx, cond_expr);
  tree true_cond_expr = fold_cond_expr ? fold_cond_expr : cond_expr;
  tree fold_not_cond_expr = eval_expr_tree (ctx, not_cond_expr);
  tree false_cond_expr = fold_not_cond_expr ? fold_not_cond_expr : not_cond_expr;
  tree then_label = gimple_cond_true_label (stmt);
  tree else_label = gimple_cond_false_label (stmt);

  {
    edge true_edge;
    edge false_edge;

    extract_true_false_edges_from_block (gimple_bb (stmt),
                                         &true_edge, &false_edge);
    basic_block target_bb = true_edge->dest;
    gimple target_stmt = first_stmt (target_bb);
    symee_context target_ctx = symee_context_copy (ctx);
    symee_set_path_cond (target_ctx, true_cond_expr);
    set_entrycontext (target_stmt, target_ctx);
    set_edge_context (true_edge, target_ctx);
    target_stmt = first_stmt (false_edge->dest);
    target_ctx = symee_context_copy (ctx);
    symee_set_path_cond (target_ctx, false_cond_expr);
    set_entrycontext (target_stmt, target_ctx);
    set_edge_context (false_edge, target_ctx);
  }

  /* return un-touched ctx */
  return ctx;
}


static symee_context
gen_context_phi (symee_context ctx, gimple phi)
{
  size_t n = gimple_phi_num_args (phi);
  size_t i;
  tree lhs = gimple_phi_result (phi);
  tree rhs = gimple_phi_arg_def (phi, 0);

  gcc_assert (n > 0);

  if (dump_details) {
    fprintf (dump_file, "processing gimple phi (%p):\n", phi);
    print_gimple_stmt (dump_file, phi, 0, TDF_VOPS|TDF_MEMSYMS);
  }

  current_stmt = phi;

  for (i = 0; i < n; i++) {
    tree op = gimple_phi_arg_def (phi, i);
    edge e = gimple_phi_arg_edge (phi, i);
    symee_context edge_ctx = get_edge_context (e);
    /* FIXME: */
    if (!edge_ctx)
      continue;
    gcc_assert (edge_ctx);
    if (i == 0) {
      rhs = symee_get_state_sym (edge_ctx, op);
      continue;
    }
    symee_expr path_cond =
      !edge_ctx ? boolean_true_node : (edge_ctx->path_cond ?
                                       edge_ctx->path_cond : boolean_true_node);
    symee_expr cond_expr = build3 (COND_EXPR, TREE_TYPE (lhs), path_cond,
                                   symee_get_state_sym (edge_ctx, op), rhs);
    rhs = cond_expr;
  }

  gcc_assert (rhs);
  symee_set_state_sym (ctx, lhs, rhs);

  if (dump_details) {
    dump_symbolic_expr (dump_file, lhs, rhs);
  }
  
  return ctx;
}


static symee_context
gen_context_phis (symee_context ctx, basic_block bb)
{
  gimple_stmt_iterator gsi;
  for (gsi = gsi_start_phis (bb); !gsi_end_p (gsi); gsi_next (&gsi)) {
    ctx = gen_context_phi (ctx, gsi_stmt (gsi));
  }
  
  return ctx;
}


/* gimple_label statement can not change program context.
   TODO: we maybe need to merge contexts from precedence edge. */

static inline symee_context
gen_context_label (symee_context ctx, gimple stmt)
{
  return ctx;
}


/* gimple_return statement do not change program context. */

static inline symee_context
gen_context_return (symee_context ctx, gimple stmt)
{
  tree retval = gimple_return_retval (stmt);

  if (!retval)
    return ctx;
  
  location_t loc = EXPR_LOCATION (retval);
  tree result_decl = current_node_info->result_decl;
  // gcc_assert (!result_decl);
  if (!result_decl)
    result_decl = build_decl (loc, RESULT_DECL, retval, TREE_TYPE (retval));
  symee_expr expr = eval_expr_tree (ctx, retval);
  
  symee_set_state_sym (ctx, result_decl, expr);

  current_node_info->result_decl = result_decl;
  
  return ctx;
}


static symee_context
gen_context_debug (symee_context ctx, gimple stmt)
{
  gcc_assert (stmt && gimple_code (stmt) == GIMPLE_DEBUG);
  symee_expr var = gimple_debug_bind_get_var (stmt);
  symee_expr value = gimple_debug_bind_get_value (stmt);
  symee_expr expr = symee_get_state_sym (ctx, var);

  /* TODO: compare value and expr.
     can we change state var = value here ? */

  return ctx;
}


static symee_context
gen_context_stmt_ignore (symee_context ctx, gimple stmt)
{
  /* gimple code for eh and omp will be not processed. */
  if (dump_file)
    fprintf (dump_file, "NYI: %s statement will be not processed!\n",
             gimple_code_name [(int) gimple_code (stmt)]);

  fprintf (stderr, "NYI: %s statement will not be processed!\n",
           gimple_code_name [(int) gimple_code (stmt)]);

  return ctx;
}


static symee_context
gen_context_stmt_1 (symee_context ctx, gimple stmt)
{
  if (dump_file)
    fprintf (dump_file, "NYI: %s statement can not be processed now!\n",
             gimple_code_name [(int) gimple_code (stmt)]);

  fprintf (stderr, "NYI: %s statement can not be processed now!\n",
           gimple_code_name [(int) gimple_code (stmt)]);

  gcc_unreachable ();

  return ctx;
}


static symee_context
gen_context_stmt_vops (symee_context ctx, gimple stmt)
{
  tree lhs = gimple_vdef (stmt);
  tree rhs = gimple_vuse (stmt);

  if (!rhs)
    return ctx;
#if 0
  if (!lhs)
    lhs = gimple_get_lhs (stmt);
#endif
  if (!lhs)
    return ctx;
  
  if (dump_details) {
    dump_symbolic_expr (dump_file, lhs, rhs);
  }
  
  if (eval_reference &&
      REFERENCE_CLASS_P (lhs))
    ctx = gen_context_ref (ctx, stmt, lhs, rhs);
  else
    symee_set_state_sym (ctx, lhs, rhs);

  return ctx;
}


static symee_context
gen_context_stmt (symee_context ctx, gimple stmt)
{
  symee_context ret_ctx = NULL;
  
  if (dump_file) {
    fprintf (dump_file, "\nprocessing gimple stmt (%p):\n", stmt);
    print_gimple_stmt (dump_file, stmt, 0, dump_flags);
  }

  /* TODO: vdef/vuse */
  if (1) {
    ssa_op_iter iter;
    use_operand_p use_p;
    tree op;

    FOR_EACH_SSA_USE_OPERAND (use_p, stmt, iter, SSA_OP_USE|SSA_OP_VUSE)
      {
        op = USE_FROM_PTR (use_p);
        gcc_assert (!REFERENCE_CLASS_P (op));
      }
  }
  
  current_stmt = stmt;

  set_entrycontext (stmt, ctx);
  
  switch (gimple_code (stmt)) {
  case GIMPLE_ASSIGN:
    ret_ctx = gen_context_assign_1 (ctx, stmt);
    break;
    
  case GIMPLE_CALL:
    if (symee_forward_recursive)
      ret_ctx = gen_context_call_r (ctx, stmt);
    else
      ret_ctx = gen_context_call (ctx, stmt);
    break;
    
  case GIMPLE_COND:
    ret_ctx = gen_context_cond (ctx, stmt);
    break;
    
  case GIMPLE_SWITCH:
    ret_ctx = gen_context_switch (ctx, stmt);
    break;
    
  case GIMPLE_GOTO:
    ret_ctx = gen_context_goto (ctx, stmt);
    break;
    
  case GIMPLE_LABEL:
    ret_ctx = gen_context_label (ctx, stmt);
    break;
    
  case GIMPLE_RETURN:
    ret_ctx = gen_context_return (ctx, stmt);
    break;
    
  case GIMPLE_DEBUG:
    ret_ctx = gen_context_debug (ctx, stmt);
    break;
    
  case GIMPLE_PHI:
    ret_ctx = gen_context_phi (ctx, stmt);
    break;
    
  case GIMPLE_ASM:
  case GIMPLE_BIND:
  case GIMPLE_ERROR_MARK:
    ret_ctx = gen_context_stmt_1 (ctx, stmt);
    break;
    
  case GIMPLE_NOP:
  case GIMPLE_PREDICT:
  default:
    ret_ctx = gen_context_stmt_ignore (ctx, stmt);
    break;
  }

  if (eval_memsyms)
    ret_ctx = gen_context_stmt_vops (ret_ctx, stmt);
  
  return ret_ctx;
}


/*
  'ctx' is the context after preheader.
  If variable is not defined in loop 'l', we can get it's value from 'ctx'.
 */

static symee_context
gen_context_loop_body (symee_context ctx, struct loop* l)
{
  basic_block *bbs = get_loop_body_in_dom_order (l);
  basic_block bb = l->header;
  symee_context in_ctx = symee_context_create ();
  symee_context ret_ctx = NULL;
  
  gcc_assert (bbs[0] == bb);

  ret_ctx = gen_context_bb (in_ctx, bb);

  if (dump_details)
    {
      fprintf (dump_file, "gen_context_loop:\n\n");
      dump_bb (bb, dump_file, 0);
      fprintf (dump_file, "\n\n   --loop-->   \n\n");
      dump_context_brief (dump_file, ret_ctx);
    }
  
  return ctx;
}


symee_context
gen_context_loop (symee_context ctx, struct loop* l)
{
  return gen_context_loop_body (ctx, l);
  
  return ctx;
}


symee_context
gen_context_loops (symee_context ctx, struct loops* loops)
{
  loop_iterator li;
  struct loop* l;

  gcc_assert (current_loops);
  
  FOR_EACH_LOOP (li, l, LI_FROM_INNERMOST)
    {
      gen_context_loop (NULL, l);
    }

  return ctx;
}


static void
gen_context_entry (basic_block bb)
{
  if (is_cfg_entry_bb (bb))
    init_start_node (bb);
  else
    merge_context (bb);

  if (is_loop_preheader (bb))
    init_preheader (bb);
  else if (is_loop_postexit (bb))
    update_postexit (bb);
}


static void
finalize_cfg_exit_bb (basic_block bb)
{
  return;
}


#if 0
/* 'bb' is cfg entry */

static void
gen_context_bb_cfg_enrty (symee_context in_ctx, basic_block bb)
{
  symee_context ctx = symee_context_copy_init (in_ctx);
  set_bb_entry_context (bb, ctx);
  gcc_assert (single_succ_p (bb));
  set_edge_context (single_succ_edge (bb), ctx);

  int formal_id = 0;
  tree parm = DECL_ARGUMENTS (current_function_decl);
  for (; parm; parm = TREE_CHAIN (parm)) {
    VEC_safe_push (tree, heap, current_node_info->parm_decls, parm);

    if (eval_formal_insym) {
      tree parm_in = add_new_parm_sym (parm, current_function_decl, true);
      symee_set_state_sym (ctx, parm, parm_in);

      void** slot = symbolic_pointer_map_insert (current_node_info->formal_in, parm);
      *slot = parm_in;
    }

    if (eval_formal_outsym) {
      tree parm_out = add_new_parm_sym (parm, current_function_decl, false);
      symee_set_state_sym (ctx, parm_out, parm);

      void** slot = symbolic_pointer_map_insert (current_node_info->formal_out, parm);
      *slot = parm_out;
    }
    
    formal_id++;
  }

  return;
}
#endif


/* Merge ctx1 and ctx2 only with different sym. (one or the other but not both)
   If sym exists in one and only one ctx, sym will be in ret_ctx. Or ret_ctx
   will be not in ret_ctx. If sym exists in both ctx1 and ctx2, it will be
   processed by phi processing */

static bool
xor_merge_context_1 (const void* key, void** slot, void* data)
{
  symee_const_expr sym = (symee_const_expr) key;
  symee_expr expr = *(symee_expr*)slot;
  symee_context ctx = (symee_context) data;

  void **ctx_slot = symbolic_pointer_map_contains (ctx->state, sym);
  /* xor merge */
  if (!ctx_slot) {
    ctx_slot = symbolic_pointer_map_insert (ctx->state, sym);
    gcc_assert (ctx_slot);
    *ctx_slot = expr;
  }

  return true;
}


/* xor merge src_ctx into dest_ctx */

static symee_context
xor_merge_context (symee_context dest_ctx, symee_context src_ctx)
{
  tree path_cond_or = NULL_TREE;
  symee_context ret_ctx = NULL;
  symee_expr expr = NULL;

  gcc_assert (dest_ctx);
  if (!src_ctx)
    return dest_ctx;
  
  symbolic_pointer_map_traverse (src_ctx->state, xor_merge_context_1, dest_ctx);
  
  if (dest_ctx->path_cond)
    path_cond_or =
      build2 (TRUTH_OR_EXPR, TREE_TYPE (dest_ctx->path_cond),
              dest_ctx->path_cond, src_ctx->path_cond);
  else
    path_cond_or = src_ctx->path_cond;

  expr = eval_expr_tree (dest_ctx, path_cond_or);
  if (!expr)
    expr = path_cond_or;

  if (symee_is_const_true (expr))
    dest_ctx->path_cond = boolean_true_node;
  else if (symee_is_const_false (expr))
    dest_ctx->path_cond = boolean_false_node;
  else
    dest_ctx->path_cond = expr;

  return dest_ctx;
}


/* merge all incoming context of 'bb' */

static symee_context
gen_context_bb_merge_edges (basic_block bb, VEC(edge,gc) *incoming_edges)
{
  symee_context ret_ctx = symee_context_create ();
  edge e;
  edge_iterator ei;
  int i = 0;
  gimple_stmt_iterator gsi;

  for (gsi = gsi_start_phis (bb); !gsi_end_p (gsi); gsi_next (&gsi)) {
    gimple phi = gsi_stmt (gsi);
    bool ignore = false;
    int arg_index = 0;
    for (arg_index = 0; arg_index < gimple_phi_num_args (phi); arg_index++) {
      edge arg_edge = gimple_phi_arg_edge (phi, arg_index);
      bool find = false;
      FOR_EACH_EDGE (e, ei, incoming_edges) {
        if (arg_edge == e)
          find = true;
      }
      if (!find)
        ignore = true;
    }
    if (ignore)
      continue;

    ret_ctx = gen_context_phi (ret_ctx, gsi_stmt (gsi));
  }

  FOR_EACH_EDGE (e, ei, incoming_edges) {
    ret_ctx = xor_merge_context (ret_ctx, get_edge_context (e));
  }
  
  gcc_assert (ret_ctx);
  
  if (dump_file && dump_details) {
    fprintf (dump_file, "SYMEE: %s merge incoming context into %p\n",
             __func__, ret_ctx);
    dump_context_brief (dump_file, ret_ctx);
  }

  return ret_ctx;
}


/* merge all incoming context of 'bb' */

static symee_context
gen_context_bb_merge (basic_block bb)
{
  symee_context ret_ctx = symee_context_create ();
  edge e;
  edge_iterator ei;
  int i = 0;
  gimple_stmt_iterator gsi;

  for (gsi = gsi_start_phis (bb); !gsi_end_p (gsi); gsi_next (&gsi)) {
    ret_ctx = gen_context_phi (ret_ctx, gsi_stmt (gsi));
  }

  FOR_EACH_EDGE (e, ei, bb->preds) {
    ret_ctx = xor_merge_context (ret_ctx, get_edge_context (e));
  }
  
  gcc_assert (ret_ctx);
  
  if (dump_file && dump_details) {
    fprintf (dump_file, "SYMEE: %s merge incoming context into %p\n",
             __func__, ret_ctx);
    dump_context_brief (dump_file, ret_ctx);
  }

  return ret_ctx;
}


/* merge incoming ctx, and generate ctx for all phis */

static symee_context
gen_context_bb_entry (symee_context ctx, basic_block bb)
{
  symee_context ret_ctx = ctx;
  if (ret_ctx)
    return ret_ctx;

  /* avoid re-computing */
  ret_ctx = get_bb_entry_context (bb);
  if (ret_ctx)
    return ret_ctx;

  /* fallthrough */
  if (single_pred_p (bb)) {
    ret_ctx = get_edge_context (single_pred_edge (bb));
    if (ret_ctx)
      return ret_ctx;
  }

  /* TODO: back edge */
  if (is_loop_header (bb)) {
    edge ph_edge = loop_preheader_edge (bb->loop_father);
    ret_ctx = get_edge_context (ph_edge);
    if (ret_ctx)
      return ret_ctx;
  }

  /* multiple incoming edge */
  ret_ctx = gen_context_bb_merge (bb);

  set_bb_entry_context (bb, ret_ctx);
  
  gcc_assert (ret_ctx);
  return ret_ctx;
}


static symee_context
gen_context_bb_postexit_1 (symee_context ctx, basic_block bb)
{
  gcc_assert (is_loop_postexit (bb));
  struct loop* l = get_loop_of_postexit (bb);
  gcc_assert (l);

  gimple_stmt_iterator gsi;
  for (gsi = gsi_start_phis (bb); !gsi_end_p (gsi); gsi_next (&gsi)) {
    gimple phi = gsi_stmt (gsi);
    tree lhs = gimple_phi_result (phi);
    tree rhs = gimple_phi_arg_def (phi, 0);
    size_t n = gimple_phi_num_args (phi);
    size_t i = 0;

    for (i = 0; i < n; i++) {
      edge e = gimple_phi_arg_edge (phi, i);
      symee_context edge_ctx = get_edge_context (e);
      tree rec_var = gimple_phi_arg_def (phi, 0);
      tree chrec = analyze_scalar_evolution (l, rec_var);
      tree inst_chrec = instantiate_parameters (l, chrec);
      tree value_chrec = chrec_apply (l->num, inst_chrec,
                                      number_of_latch_executions (l));
      tree rhs = value_chrec;
      if (value_chrec != chrec_dont_know)
        rhs = eval_expr_tree (ctx, value_chrec);
      if (!rhs)
        rhs = value_chrec;

      if (dump_file) {
        fprintf (dump_file, "processing gimple phi (%p):\n", phi);
        print_gimple_stmt (dump_file, phi, 0, TDF_VOPS|TDF_MEMSYMS);
      }
      if (dump_details) {
        print_generic_expr (dump_file, rec_var, 0);
        fprintf (dump_file, " = ");
        print_generic_expr (dump_file, chrec, 0);
        fprintf (dump_file, " = ");
        print_generic_expr (dump_file, inst_chrec, 0);
        fprintf (dump_file, " => ");
        print_generic_expr (dump_file, value_chrec, 0);
        fprintf (dump_file, "\n");
      }
      if (dump_file) {
        dump_symbolic_expr (dump_file, lhs, rhs);
      }
      symee_set_state_sym (ctx, lhs, rhs);
    }
  }
  
  return ctx;
}


#define RECURRENCE_STATISTICS

#ifdef RECURRENCE_STATISTICS

/* recurrence statistics */

struct GTY(()) recurrence_stats {
  int num_phis;
  int num_chrecs;
  int num_instantiate_parameters;
  int num_apply_nb_iterations;

  int num_no_chrecs;
  int num_no_iterations;

  int num_no_chrecs_ssa;
};

static GTY(()) struct recurrence_stats recur_stats;

static GTY(()) struct recurrence_stats global_recur_stats;

int flag_recurrence_statistics = 1;

static inline void
recurrence_stats_start (void)
{
  memset (&recur_stats, 0, sizeof (struct recurrence_stats));
}

static inline void
recurrence_stats_push (void)
{
  global_recur_stats.num_phis += recur_stats.num_phis;
  global_recur_stats.num_chrecs += recur_stats.num_chrecs;
  global_recur_stats.num_instantiate_parameters += recur_stats.num_instantiate_parameters;
  global_recur_stats.num_apply_nb_iterations += recur_stats.num_apply_nb_iterations;
  global_recur_stats.num_no_chrecs += recur_stats.num_no_chrecs;
}

static inline void
recurrence_stats_print_global (FILE* fp)
{
  double d = 0.0;
  
  fprintf (fp,
           "global recurrence_stats: %d phis "
           "(%d chrecs (%d param (%d iteration)), %2.2lf%%)\n",
           global_recur_stats.num_phis,
           global_recur_stats.num_chrecs,
           global_recur_stats.num_instantiate_parameters,
           global_recur_stats.num_apply_nb_iterations,
           global_recur_stats.num_phis ?
           (global_recur_stats.num_apply_nb_iterations * 100.0
            / global_recur_stats.num_phis)
           : d);
}

static inline void
recurrence_stats_print (FILE* fp)
{
  recurrence_stats_push ();
  
  fprintf (fp,
           "recurrence_stats (%s): %d phis "
           "(%d chrecs (%d param (%d iteration)), %d no_chrecs)\n",
           current_function_name (),
           recur_stats.num_phis,
           recur_stats.num_chrecs,
           recur_stats.num_instantiate_parameters,
           recur_stats.num_apply_nb_iterations,
           recur_stats.num_no_chrecs);
}

#define RECURRENCE_STATS_INC(field)                 \
  if (flag_recurrence_statistics)               \
    recur_stats.field++;

#else

static inline void recurrence_stats_start (void) { return; }
static inline void recurrence_stats_print (FILE* fp) { return; }
static inline void recurrence_stats_print_global (FILE* fp) { return; }
#define RECURRENCE_STATS_INC(field)

#endif


/* Given a loop phi
   s_1 = PHI <s_2, s_...>
   If s_2 is defined in loop, so s_1 is an recurrence.
   Try to get closed form by chrec, firstly, if not, we must
   give assumed SYMEE_MU expr for s_2;

   s_1 = SYMEE_MU (s_2, expr (s_1));
 */

static symee_expr
gen_conext_generic_recurrence (symee_context ctx,
                               struct loop *l, symee_expr sym)
{
  tree mu = NULL_TREE;

  mu = build3 (SYMEE_MU, TREE_TYPE (sym), sym, NULL, NULL);

  symee_set_state_sym (ctx, sym, mu);
  
  return mu;
}


static tree
get_ref_expr_and_stmts (gimple stmt, tree ref, VEC(gimple,heap) **stmts)
{
  tree vuse = gimple_vuse (stmt);

  while (vuse)
    {
      gimple def_stmt = SSA_NAME_DEF_STMT (vuse);
      VEC_safe_push (gimple, heap, *stmts, def_stmt);
      vuse = gimple_vuse (def_stmt);
    }

  return NULL_TREE;
}


static symee_expr
postexit_sym (symee_context ctx, struct loop* l, symee_expr sym)
{
  tree ret_expr = NULL_TREE;
  tree chrec_expr = NULL_TREE;
  tree chrec = NULL_TREE;
  tree inst_chrec = NULL_TREE;
  tree value_chrec = NULL_TREE;
  tree nb_iterations = number_of_latch_executions (l);

  RECURRENCE_STATS_INC(num_phis);
  
  if (nb_iterations == chrec_dont_know) {
    nb_iterations = add_new_sym (sym);
    RECURRENCE_STATS_INC(num_no_iterations);
  }

  if (TREE_CODE (sym) == ARRAY_REF) {
    tree index = TREE_OPERAND (sym, 1);
    chrec = analyze_array_evolution (l, sym, &index);
  }
  else
    chrec = analyze_scalar_evolution (l, sym);
  
  if (TREE_CODE (chrec) == POLYNOMIAL_CHREC) {
    RECURRENCE_STATS_INC(num_chrecs);
    inst_chrec = instantiate_parameters (l, chrec);
    if (TREE_CODE (inst_chrec) == POLYNOMIAL_CHREC) {
      RECURRENCE_STATS_INC(num_instantiate_parameters);
      value_chrec = chrec_apply (l->num, inst_chrec, nb_iterations);
      if (TREE_CODE (value_chrec) != POLYNOMIAL_CHREC
          && value_chrec != chrec_dont_know) {
        RECURRENCE_STATS_INC(num_apply_nb_iterations);
        chrec_expr = value_chrec;
      }
    }
  }
  else {
    chrec_expr = add_new_sym (sym);
    RECURRENCE_STATS_INC(num_no_chrecs);

    /* TODO: */
    if (dump_details)
      {
        
      }

  }
  
  if (dump_details) {
    print_generic_expr (dump_file, sym, 0);
    fprintf (dump_file, " = ");
    print_generic_expr (dump_file, chrec, 0);
    fprintf (dump_file, " => <");
    print_generic_expr (dump_file, inst_chrec, 0);
    fprintf (dump_file, ", (nb_iter = ");
    print_generic_expr (dump_file, nb_iterations, 0);
    fprintf (dump_file, ")> ");
    fprintf (dump_file, " => ");
    print_generic_expr (dump_file, value_chrec, 0);
    fprintf (dump_file, "\n");
  }

  if (!chrec_expr)
    chrec_expr = add_new_sym (sym);

  ret_expr = eval_expr_tree (ctx, chrec_expr);
  if (!ret_expr)
    ret_expr = chrec_expr;

  if (dump_file) {
    dump_symbolic_expr (dump_file, sym, ret_expr);
  }

  return ret_expr;
}


static symee_context
gen_context_phi_postexit (symee_context ctx, gimple phi)
{
  size_t n = gimple_phi_num_args (phi);
  size_t i;
  tree lhs = gimple_phi_result (phi);
  tree rhs = gimple_phi_arg_def (phi, 0);

  /* TODO: non-scalar ssa_name */
  // gcc_assert (is_gimple_reg (gimple_phi_result (phi)));
  gcc_assert (n > 0);

  if (dump_file) {
    fprintf (dump_file, "processing gimple phi (%p):\n", phi);
    print_gimple_stmt (dump_file, phi, 0, TDF_VOPS|TDF_MEMSYMS);
  }

  current_stmt = phi;
  
  for (i = 0; i < n; i++) {
    tree op = gimple_phi_arg_def (phi, i);
    edge e = gimple_phi_arg_edge (phi, i);
    struct loop* l = e->src->loop_father;
    symee_context edge_ctx = get_edge_context (e);
    /* FIXME: */
    if (!edge_ctx)
      continue;
    gcc_assert (edge_ctx);
    symee_expr op_expr = symee_get_state_sym (edge_ctx, op);
    if (loop_exit_edge_p (l, e))
      op_expr = postexit_sym (edge_ctx, l, op);

    gcc_assert (edge_ctx);

    if (i == 0) {
      rhs = op_expr;
      continue;
    }
    symee_expr path_cond =
      !edge_ctx ? boolean_true_node : (edge_ctx->path_cond ?
                                       edge_ctx->path_cond : boolean_true_node);
    symee_expr cond_expr = build3 (COND_EXPR, TREE_TYPE (lhs), path_cond,
                                   op_expr, rhs);
    rhs = cond_expr;
  }

  gcc_assert (rhs);
  symee_set_state_sym (ctx, lhs, rhs);

  if (dump_file) {
    dump_symbolic_expr (dump_file, lhs, rhs);
  }
  
  return ctx;
}


static symee_context
gen_context_bb_postexit (symee_context ctx, basic_block bb)
{
  gimple_stmt_iterator gsi;

  gcc_assert (is_loop_postexit (bb));
  
  for (gsi = gsi_start_phis (bb); !gsi_end_p (gsi); gsi_next (&gsi)) {
    ctx = gen_context_phi_postexit (ctx, gsi_stmt (gsi));
  }
  
  return ctx;
}


static symee_context
gen_context_loop_header (symee_context ctx, basic_block bb)
{
  if (!symee_sym_exec_loop)
    return ctx;
  
  VEC(symee_context,heap) *rctx;
  struct loop *loop = bb->loop_father;

  gcc_assert (loop_preheader_edge (loop)->src);
  gcc_assert (first_stmt (loop->header));
  
  symee_context_set_prev (ctx, loop_preheader_edge (loop)->src);
  symee_context_set_next (ctx, first_stmt (loop->header));

  if (dump_details) {
    fprintf (dump_file, "|||||||loop enter context begin|||||||\n");
    dump_context_brief (dump_file, ctx);
    fprintf (dump_file, "|||||||loop enter context end|||||||\n");
  }

  rctx = execute_loop (ctx, bb->loop_father);

  if (dump_details) {
    fprintf (dump_file, "|||||||loop exit context begin|||||||\n");
    dump_vec_context (dump_file, rctx);
    fprintf (dump_file, "|||||||loop exit context end|||||||\n");
  }
  
  return ctx;
}


static symee_context
gen_context_bb_stmts_without_branch (symee_context ctx, basic_block bb)
{
  gimple_stmt_iterator gsi;
  symee_context ret_ctx = symee_context_copy_init (ctx);
  gimple last_stmt = gsi_stmt (gsi_last_bb (bb));

  for (gsi = gsi_start_bb (bb);
       !gsi_end_p (gsi) && !gsi_one_before_end_p (gsi);
       gsi_next (&gsi))
    {
      gimple stmt = gsi_stmt (gsi);
      ret_ctx = gen_context_stmt (ret_ctx, stmt);
    }

  if (gimple_code (last_stmt) != GIMPLE_COND
      && gimple_code (last_stmt) != GIMPLE_SWITCH)
    ret_ctx = gen_context_stmt (ret_ctx, last_stmt);

  return ret_ctx;
}


static symee_context
gen_context_bb_stmts (symee_context ctx, basic_block bb)
{
  gimple_stmt_iterator gsi;
  symee_context ret_ctx = symee_context_copy_init (ctx);

  for (gsi = gsi_start_bb (bb); !gsi_end_p (gsi); gsi_next (&gsi))
    {
      gimple stmt = gsi_stmt (gsi);
      ret_ctx = gen_context_stmt (ret_ctx, stmt);
    }

  return ret_ctx;
}


static symee_context
gen_context_bb_phis (symee_context ctx, basic_block bb)
{
  gimple_stmt_iterator gsi;
  symee_context ret_ctx = symee_context_copy_init (ctx);

  for (gsi = gsi_start_phis (bb); !gsi_end_p (gsi); gsi_next (&gsi))
    {
      ret_ctx = gen_context_phi (ret_ctx, gsi_stmt (gsi));
    }

  return ret_ctx;
}


static symee_context
gen_context_bb_2 (symee_context ctx, basic_block bb)
{
  symee_context ret_ctx = NULL;
  gimple_stmt_iterator gsi;

  if (bb_is_visited (bb))
    return ctx;

  if (dump_file && dump_details)
    fprintf (dump_file, "\nSYMEE: starting to process <bb %d>\n\n", bb->index);
  
  /* merge context from incoming edge */
  if (!ctx)
    ctx = gen_context_bb_entry (NULL, bb);

  current_context = ctx;

  if (is_loop_header (bb))
    ;//    ret_ctx = gen_context_loop_header (ctx, bb);
  else if (is_loop_postexit (bb))
    ret_ctx = gen_context_bb_postexit (ctx, bb);
  else
    ret_ctx = gen_context_phis (ctx, bb);

  
  /* generate context for all gimple statements */
  for (gsi = gsi_start_bb (bb); !gsi_end_p (gsi); gsi_next (&gsi)) {
    gimple stmt = gsi_stmt (gsi);
    ctx = gen_context_stmt (ctx, stmt);
    set_exitcontext (stmt, ctx);
  }

  /* generate post bb context */
  if (single_succ_p (bb)) {
    edge e = single_succ_edge (bb);
    symee_context edge_ctx = get_edge_context (e);
    if (last_stmt (bb))
      gcc_assert (ctx == get_exitcontext (last_stmt (bb)));
    gcc_assert (ctx);
    if (!edge_ctx)
      set_edge_context (e, ctx);
  }

  /* we have processed out edge, when we process last_stmt of bb ? */
  else {
  }
  
  if (is_loop_header (bb)) {
    basic_block *bbs = get_loop_body_in_dom_order (bb->loop_father);
    int i;
#if 0
    gcc_assert (bbs[0] == bb);
    for (i = 1; i < bb->loop_father->num_nodes; i++) {
      if (bb_is_visited (bbs[i]))
        continue;
      gen_context_bb (ctx, bbs[i]);
      bb_set_visited (bbs[i]);
    }
#endif
#if 0
    for (i = 0; i < current_node_info->rpo_cnt; i++) {
      basic_block loopbb = BASIC_BLOCK (current_node_info->rpo[i]);
      if (loopbb != bb
          && flow_bb_inside_loop_p (bb->loop_father, loopbb)
          && !bb_is_visited (loopbb)) {
        gen_context_bb (NULL, loopbb);
        bb_set_visited (loopbb);
      }
    }
#endif
  }
  
  if (is_loop_postbody (bb))
    update_preheader (bb);

  bb_set_visited (bb);

  return ctx;
}


/* if ctx is NULL, get ctx from predecessors */

static symee_context
gen_context_bb (symee_context ctx, basic_block bb)
{
  symee_context ret_ctx = NULL;
  gimple_stmt_iterator gsi;

  if (bb_is_visited (bb))
    return ctx;

  if (dump_file && dump_details)
    fprintf (dump_file, "\nSYMEE: starting to process <bb %d>\n\n", bb->index);
  
  /* merge context from incoming edge */
  if (!ctx)
    ctx = gen_context_bb_entry (NULL, bb);

  current_context = ctx;

  if (is_loop_header (bb))
    ret_ctx = gen_context_loop_header (ctx, bb);
  else if (is_loop_postexit (bb))
    ret_ctx = gen_context_bb_postexit (ctx, bb);
  else
    ret_ctx = gen_context_phis (ctx, bb);

  
  /* generate context for all gimple statements */
  for (gsi = gsi_start_bb (bb); !gsi_end_p (gsi); gsi_next (&gsi)) {
    gimple stmt = gsi_stmt (gsi);
    ctx = gen_context_stmt (ctx, stmt);
    set_exitcontext (stmt, ctx);
  }

  /* generate post bb context */
  if (single_succ_p (bb)) {
    edge e = single_succ_edge (bb);
    symee_context edge_ctx = get_edge_context (e);
    if (last_stmt (bb))
      gcc_assert (ctx == get_exitcontext (last_stmt (bb)));
    gcc_assert (ctx);
    if (!edge_ctx)
      set_edge_context (e, ctx);
  }

  /* we have processed out edge, when we process last_stmt of bb ? */
  else {
  }
  
  if (is_loop_header (bb)) {
    basic_block *bbs = get_loop_body_in_dom_order (bb->loop_father);
    int i;
#if 0
    gcc_assert (bbs[0] == bb);
    for (i = 1; i < bb->loop_father->num_nodes; i++) {
      if (bb_is_visited (bbs[i]))
        continue;
      gen_context_bb (ctx, bbs[i]);
      bb_set_visited (bbs[i]);
    }
#endif
    for (i = 0; i < current_node_info->rpo_cnt; i++) {
      basic_block loopbb = BASIC_BLOCK (current_node_info->rpo[i]);
      if (loopbb != bb
          && flow_bb_inside_loop_p (bb->loop_father, loopbb)
          && !bb_is_visited (loopbb)) {
        gen_context_bb (NULL, loopbb);
        bb_set_visited (loopbb);
      }
    }
  }
  
  if (is_loop_postbody (bb))
    update_preheader (bb);

  bb_set_visited (bb);

  return ctx;
}


static void
gen_context_exit (basic_block bb)
{
  gimple_seq seq = bb_seq (bb);
  gimple_stmt_iterator gsi = gsi_start_bb (bb);

  // eval_exit_context (bb)
  for (; !gsi_end_p (gsi); gsi_next (&gsi)) {
    gimple stmt = gsi_stmt (gsi);

    gen_context_stmt (current_context, stmt);
  }

  if (is_loop_postbody (bb))
    update_preheader (bb);
}


static void
check_loop (loop_p l)
{
  basic_block *bbs = get_loop_body_in_dom_order (l);
  int i;

  gcc_assert (l->header == bbs[0]);
  
  for (i = 0; i < l->num_nodes; i++) {
    
  }
}


static void
check_loops (void)
{
  loop_iterator li;
  loop_p l;

  FOR_EACH_LOOP (li, l, 0) {
    check_loop (l);
  }
}


static void
dump_integer_set (FILE* fp, int* order, int count)
{
  int i;
  
  fprintf (dump_file, "{");
  for (i = 0; i < count - 1; i++) {
    fprintf (dump_file, "%d, ", order[i]);
  }
  fprintf (dump_file, "%d}", order[count-1]);

  return;
}


static int
bb_comp (const void* arg1, const void* arg2)
{
  basic_block bb1 = *(basic_block*)arg1;
  basic_block bb2 = *(basic_block*)arg2;
  edge e1 = find_edge (bb1, bb2);
  edge e2 = find_edge (bb2, bb1);

  symee_assert (!(e1 && e2), (stderr, "bb_comp cycle\n"));
  
  if (e1 != NULL)
    return 1;
  if (e2 != NULL)
    return -1;

  return 0;
}


static bool
sort_sons (basic_block parent, basic_block *sons, int count)
{
  basic_block* tmp = XCNEWVEC (basic_block, count);
  int tmp_index = 0;
  if (is_loop_header (parent)) {
    int i;
    for (i = 0; i < count; i++) {
      if (parent->loop_father == sons[i]->loop_father) {
        tmp[tmp_index] = sons[i];
        tmp_index++;
        sons[i] = NULL;
      }
    }
  }

  int i;
  for (i = 0; i < count; i++) {
    if (sons[i] == NULL)
      continue;
    tmp[tmp_index] = sons[i];
    tmp_index++;
  }

  for (i = 0; i < count; i++)
    sons[i] = tmp[i];

  XDELETEVEC (tmp);
}


static int
compute_dom_order (enum cdi_direction dir, basic_block root,
                   int* order, int* start_index)
{
  int do_cnt = 0;
  basic_block son;

  order[*start_index] = root->index;
  (*start_index)++;
  do_cnt++;

  basic_block* sons =
    XCNEWVEC (basic_block, last_basic_block + NUM_FIXED_BLOCKS);
  int num_sons = 0;
  for (son = first_dom_son (dir, root); son; son = next_dom_son (dir, son)) {
    sons[num_sons] = son;
    num_sons++;
  }
  sort_sons (root, sons, num_sons);
  int i = 0;
  for (i = 0; i < num_sons; i++) {
    do_cnt += compute_dom_order (dir, sons[i], order, start_index);
  }

  XDELETEVEC (sons);

  return do_cnt;
}


static int
compute_order (int* order)
{
  int* post_order = XCNEWVEC (int, last_basic_block + NUM_FIXED_BLOCKS);
  int po_cnt = post_order_compute (post_order, false, false);
  int* dfs = XCNEWVEC (int, last_basic_block + NUM_FIXED_BLOCKS);
  int* rpo = XCNEWVEC (int, last_basic_block + NUM_FIXED_BLOCKS);
  int rpo_cnt = pre_and_rev_post_order_compute (dfs, rpo, false);
  int* ipo = XCNEWVEC (int, last_basic_block + NUM_FIXED_BLOCKS);
  int ipo_cnt = inverted_post_order_compute (ipo);
  int i = 0;
  basic_block bb = NULL;
  int j = 0;

  current_node_info->rpo = rpo;
  current_node_info->rpo_cnt = rpo_cnt;

  int* dom_order = XCNEWVEC (int, last_basic_block + NUM_FIXED_BLOCKS);
  int start_index = 0;
  int do_cnt = compute_dom_order (CDI_DOMINATORS, ENTRY_BLOCK_PTR,
                                  dom_order, &start_index);
  gcc_assert (do_cnt == start_index);

  for (i = 0; i < rpo_cnt; i++)
    order[i] = dom_order[i];

  if (dump_details) {
    debug_dominance_tree (CDI_DOMINATORS, ENTRY_BLOCK_PTR);
    fprintf (dump_file, "\ndom = ");
    dump_integer_set (dump_file, dom_order, do_cnt);

    fprintf (dump_file, "\ndfs = ");
    dump_integer_set (dump_file, dfs, rpo_cnt);
    fprintf (dump_file, "\nrpo = ");
    dump_integer_set (dump_file, rpo, rpo_cnt);
    fprintf (dump_file, "\npo  = ");
    dump_integer_set (dump_file, post_order, po_cnt);
    fprintf (dump_file, "\nipo = ");
    dump_integer_set (dump_file, ipo, ipo_cnt);
  }

  return rpo_cnt;
}


static VEC(edge,gc)*
loop_header_succs (basic_block bb)
{
  if (is_loop_header (bb)) {
    int len = VEC_length(edge,bb->succs);
    VEC(edge,gc) *in_loop_succs = VEC_alloc (edge,gc,len);
    VEC(edge,gc) *others = VEC_alloc (edge,gc,len);
    edge e;
    edge_iterator ei;
    FOR_EACH_EDGE (e, ei, bb->succs) {
      if (e->dest->loop_father == bb->loop_father)
        VEC_safe_push(edge,gc,in_loop_succs,e);
      else
        VEC_safe_push(edge,gc,others,e);
    }

    FOR_EACH_EDGE (e, ei, others) {
      VEC_safe_push(edge,gc,in_loop_succs,e);
    }
 
    return in_loop_succs;
  }
  else
    return bb->succs;
}


static int
rpo_compute (int *pre_order, int *rev_post_order,
             bool include_entry_exit)
{
  edge_iterator *stack;
  int sp;
  int pre_order_num = 0;
  int rev_post_order_num = n_basic_blocks - 1;
  sbitmap visited;

  /* Allocate stack for back-tracking up CFG.  */
  stack = XNEWVEC (edge_iterator, n_basic_blocks + 1);
  sp = 0;

  if (include_entry_exit)
    {
      if (pre_order)
	pre_order[pre_order_num] = ENTRY_BLOCK;
      pre_order_num++;
      if (rev_post_order)
	rev_post_order[rev_post_order_num--] = ENTRY_BLOCK;
    }
  else
    rev_post_order_num -= NUM_FIXED_BLOCKS;

  /* Allocate bitmap to track nodes that have been visited.  */
  visited = sbitmap_alloc (last_basic_block);

  /* None of the nodes in the CFG have been visited yet.  */
  sbitmap_zero (visited);

  /* Push the first edge on to the stack.  */
  stack[sp++] = ei_start (ENTRY_BLOCK_PTR->succs);

  while (sp)
    {
      edge_iterator ei;
      basic_block src;
      basic_block dest;

      /* Look at the edge on the top of the stack.  */
      ei = stack[sp - 1];
      src = ei_edge (ei)->src;
      dest = ei_edge (ei)->dest;

      /* Check if the edge destination has been visited yet.  */
      if (dest != EXIT_BLOCK_PTR && ! TEST_BIT (visited, dest->index))
	{
	  /* Mark that we have visited the destination.  */
	  SET_BIT (visited, dest->index);

	  if (pre_order)
	    pre_order[pre_order_num] = dest->index;

	  pre_order_num++;

	  if (EDGE_COUNT (dest->succs) > 0) {
	    /* Since the DEST node has been visited for the first
	       time, check its successors.  */
            VEC(edge,gc) **succs = XCNEW (VEC(edge,gc)*);
            *succs = loop_header_succs (dest);
	    stack[sp++] = ei_start (*succs);
          }
          else if (rev_post_order)
	    /* There are no successors for the DEST node so assign
	       its reverse completion number.  */
	    rev_post_order[rev_post_order_num--] = dest->index;
	}
      else
	{
	  if (ei_one_before_end_p (ei) && src != ENTRY_BLOCK_PTR
	      && rev_post_order)
	    /* There are no more successors for the SRC node
	       so assign its reverse completion number.  */
	    rev_post_order[rev_post_order_num--] = src->index;

	  if (!ei_one_before_end_p (ei))
	    ei_next (&stack[sp - 1]);
	  else
	    sp--;
	}
    }

  free (stack);
  sbitmap_free (visited);

  if (include_entry_exit)
    {
      if (pre_order)
	pre_order[pre_order_num] = EXIT_BLOCK;
      pre_order_num++;
      if (rev_post_order)
	rev_post_order[rev_post_order_num--] = EXIT_BLOCK;
      /* The number of nodes visited should be the number of blocks.  */
      gcc_assert (pre_order_num == n_basic_blocks);
    }
  else
    /* The number of nodes visited should be the number of blocks minus
       the entry and exit blocks which are not visited here.  */
    gcc_assert (pre_order_num == n_basic_blocks - NUM_FIXED_BLOCKS);

  return pre_order_num;
}


static symee_context
gen_context_cgraph_node (symee_context ctx, cgraph_node_ptr node)
{
  tree decl = node->decl;
  tree old_decl = current_function_decl;
  symee_cgraph_node_info old_node_info = NULL;
  
  push_cfun (DECL_STRUCT_FUNCTION (decl));
  current_function_decl = decl;

  old_node_info = current_node_info;
  current_node_info = init_cgraph_node_info (node);
  
  if (dump_file) {
    fprintf (dump_file, "\nSYMEE: Generate context for function %s (%p)\n",
             IDENTIFIER_POINTER (DECL_NAME (decl)), decl);
    fprintf_loc (dump_file, DECL_SOURCE_LOCATION (decl),
                 "generating context for cgraph node %s\n\n",
                 cgraph_node_name (node));

    //fprintf_loc (stderr, DECL_SOURCE_LOCATION (decl), NULL);
    fprintf (stderr, "SYMEE: Generate context for function %s (%p)\n",
             IDENTIFIER_POINTER (DECL_NAME (decl)), decl);
  }

  update_ssa (TODO_update_ssa);
  loop_optimizer_init (LOOPS_NORMAL | LOOPS_HAVE_RECORDED_EXITS);
  rewrite_into_loop_closed_ssa (NULL, TODO_update_ssa_any);
  scev_initialize ();
  
  if (symee_run_vrp) {
    push_dump_file ();
    symee_vrp ();
    pop_dump_file ();
  }

  check_loops ();

#ifdef LOOP_BY_LOOP  
  gen_context_loops (NULL, current_loops);
#endif

  if (dump_file)
    gimple_dump_cfg (dump_file, dump_flags);

  int* rpo = XCNEWVEC (int, last_basic_block + NUM_FIXED_BLOCKS);
#if 0
  int rpo_cnt = pre_and_rev_post_order_compute (NULL, rpo, false);
#else
  int rpo_cnt = rpo_compute (NULL, rpo, false);
#endif
  int i;

  if (dump_details) {
    fprintf (dump_file, "\nrpo = ");
    dump_integer_set (dump_file, rpo, rpo_cnt);
    fprintf (dump_file, "\n\n");

    int* irr = XCNEWVEC (int, rpo_cnt);
    int j = 0;
    for (i = 0; i < rpo_cnt; i++) {
      basic_block bb = BASIC_BLOCK (rpo[i]);
      if (bb->flags & BB_IRREDUCIBLE_LOOP) {
        irr[j] = i;
        j++;
      }
    }
    fprintf (dump_file, "\nirr = ");
    dump_integer_set (dump_file, irr, j);
    fprintf (dump_file, "\n\n");
  }
  
  current_node_info->rpo = rpo;
  current_node_info->rpo_cnt = rpo_cnt;

  recurrence_stats_start ();
  
  // TODO: cfg entry
  gen_context_bb_cfg_entry (ctx, ENTRY_BLOCK_PTR);
  //init_start_node (ENTRY_BLOCK_PTR);

  for (i = 0; i < rpo_cnt; i++) {
    basic_block bb = BASIC_BLOCK (rpo[i]);
    if (bb_is_visited (bb))
      continue;
    current_context = gen_context_bb (NULL, bb);
    bb_set_visited (bb);
  }

  // TODO: cfg exit
  gen_context_bb_cfg_exit (current_context, EXIT_BLOCK_PTR);
  //gen_context_bb (NULL, EXIT_BLOCK_PTR);

  //cgraph_node_set_context (node, current_context);

  if (dump_file && dump_details)
    dump_context (dump_file, current_context);

  if (dump_file) {
    recurrence_stats_print (dump_file);

    fprintf (dump_file,
             "\n\nSYMEE: finish to generate context for function %s\n",
             IDENTIFIER_POINTER (DECL_ASSEMBLER_NAME (current_function_decl)));
    dump_context_brief (dump_file, current_context);
    fprintf (dump_file, "\n\n");
    fprintf (dump_file, "exit_context for function %s\n",
             IDENTIFIER_POINTER (DECL_ASSEMBLER_NAME (current_function_decl)));
    dump_context_brief (dump_file, current_node_info->exit_context);
    fprintf (dump_file, "\n\n");
  }

  if (symee_run_vrp) {
    push_dump_file ();
    symee_vrp_finalize ();
    pop_dump_file ();
  }

  scev_finalize ();
  loop_optimizer_finalize ();

  current_node_info = old_node_info;
  
  pop_cfun ();
  current_function_decl = old_decl;

  return;
}


/* To eval an expr through a stmt backward
   1) generate local context with fake input context.
   2) eval expr under result context
*/

static symee_expr
propagate_backward_stmt (gimple stmt, symee_expr expr)
{
  symee_expr ret_expr = NULL_TREE;
  symee_context fake_ctx = symee_context_create ();
  symee_context ctx = gen_context_stmt (fake_ctx, stmt);

  if (ctx)
    ret_expr = substitute_backward (ctx, expr);
  else
    ret_expr = expr;

  return ret_expr;
}


struct symee_expr_operands {
  VEC(tree, heap) *operands;
  VEC(tree, heap) *parents;
  VEC(gimple, heap) *def_stmts;
  gimple common_cfg_pred;
  gimple common_dom_pred;
};


static struct symee_expr_operands *
extract_all_operands (symee_expr expr)
{
  return NULL;
}


static symee_expr
propagate_backward_x (gimple before_stmt, symee_expr expr)
{
  symee_expr ret_expr = NULL_TREE;
  struct symee_expr_operands *ops = extract_all_operands (expr);

  if (ops->common_cfg_pred)
    ;

  return ret_expr;
}


static symee_expr
propagate_backward (cgraph_node_ptr node)
{
  tree decl = node->decl;
  tree old_decl = current_function_decl;
  symee_cgraph_node_info old_node_info = NULL;
  
  push_cfun (DECL_STRUCT_FUNCTION (decl));
  current_function_decl = decl;

  old_node_info = current_node_info;
  current_node_info = init_cgraph_node_info (node);

  update_ssa (TODO_update_ssa);
  loop_optimizer_init (LOOPS_NORMAL | LOOPS_HAVE_RECORDED_EXITS);
  rewrite_into_loop_closed_ssa (NULL, TODO_update_ssa_any);
  scev_initialize ();

  int* rpo = XCNEWVEC (int, last_basic_block + NUM_FIXED_BLOCKS);
#if 0
  int rpo_cnt = pre_and_rev_post_order_compute (NULL, rpo, false);
#else
  int rpo_cnt = rpo_compute (NULL, rpo, false);
#endif
  int i;

  symee_expr target_expr = NULL;
  
  for (i = rpo_cnt - 1; i >= 0; i--) {
    basic_block bb = BASIC_BLOCK (rpo[i]);
    if (bb_is_visited (bb))
      continue;
    gimple_stmt_iterator gsi;
    for (gsi = gsi_last_bb (bb); !gsi_end_p (gsi); gsi_prev (&gsi))
      {
        /* just for testing */
        if (gimple_code (gsi_stmt (gsi)) == GIMPLE_RETURN)
          {
            target_expr = gimple_return_retval (gsi_stmt (gsi));
            continue;
          }

        target_expr = propagate_backward_stmt (gsi_stmt (gsi), target_expr);
      }
    bb_set_visited (bb);
  }

  if (dump_file) {
    fprintf (dump_file, "backward substitute for retval = \t");
    print_generic_expr (dump_file, target_expr, dump_flags);
    fprintf (dump_file, "\n\n\n");
  }
  
  scev_finalize ();
  loop_optimizer_finalize ();

  current_node_info = old_node_info;
  
  pop_cfun ();
  current_function_decl = old_decl;
}


static void
gen_context_cgraph_node_malloc (struct cgraph_node* node)
{
}


static void
gen_context_cgraph_node_free (struct cgraph_node* node)
{
}


static void
gen_context_cgraph_node_clone (struct cgraph_node* node)
{
  fprintf (stderr, "NYI: function %s is clone of %s\n",
           cgraph_node_name (node),
           cgraph_node_name (node->clone_of));
}


static void
gen_context_cgraph_node_no_body (struct cgraph_node* node)
{
  const char* node_name =
    IDENTIFIER_POINTER (DECL_ASSEMBLER_NAME (node->decl));

  /* some special builtin function */
  if (strcmp (node_name, "malloc") == 0)
    gen_context_cgraph_node_malloc (node);
  if (strcmp (node_name, "free") == 0)
    gen_context_cgraph_node_free (node);
  
  if (dump_file && dump_details) {
    fprintf (dump_file, "SYMEE: %s is not yet analyzed!\n",
             IDENTIFIER_POINTER (DECL_ASSEMBLER_NAME (node->decl)));
  }
}


static void
dump_cgraph_gv (FILE* fp)
{
  cgraph_node_ptr node;
  cgraph_edge_p edge;
  
  fprintf (fp, "digraph cgraph {\n");
  fprintf (fp, "  node [shape=box];\n\n");
  for (node = cgraph_nodes; node; node = node->next) {
    if (!node->analyzed)
      continue;
    fprintf (fp, "  %d [label=\"%s\"];\n", node->uid,
             cgraph_node_name (node));
  }

  for (node = cgraph_nodes; node; node = node->next) {
    for (edge = node->callees; edge; edge = edge->next_callee) {
      if (!edge->callee->analyzed)
        continue;
      fprintf (fp, "  %d -> %d;\n", node->uid, edge->callee->uid);
    }
  }
  fprintf (fp, "}\n\n");

  return;
}


static void
dump_cgraph_postorder (FILE* fp, struct cgraph_node** order, int nnodes)
{
  int i;
  
  fprintf (fp, "\n%s : cgraph_postorder is\n", __func__);
  fprintf (fp, "{");
  for (i = nnodes - 1; i > 0; i--) {
    fprintf (fp, "%s, ",
             IDENTIFIER_POINTER (DECL_ASSEMBLER_NAME (order[i]->decl)));
  }
  fprintf (fp, "%s}\n\n",
           IDENTIFIER_POINTER (DECL_ASSEMBLER_NAME (order[0]->decl)));

  /* dump cgraph */
  dump_cgraph_gv (fp);
  
  return;
}


/* browse call cgraph, generate context for each cgraph node.
   gen_context_cgraph is internal function, please use
   symee_generate_conext as external interface. */

static void
gen_context_cgraph (void)
{
  int i;
  struct cgraph_node* node;
  int nnodes;
  struct cgraph_node** order =
    XCNEWVEC (struct cgraph_node*, cgraph_n_nodes);

  /* TODO: */
  symee_assert (!flag_wpa, (stderr, "SYMEE: can not run in wpa mode\n"));
  
  /* traverse call graph bottom-up, and use generated callee context
     for call stmts */
  nnodes = cgraph_postorder (order);

  if (dump_details)
    dump_cgraph_postorder (dump_file, order, nnodes);
  
  for (i = nnodes - 1; i >= 0; i--) {
    node = order[i];

    /* TODO: process recursive call */
    
    if (node->analyzed) {
      if (!node->clone_of)
        gen_context_cgraph_node (NULL, node);
      else
        gen_context_cgraph_node_clone (node);
    }

    /* TODO: */
    else
      gen_context_cgraph_node_no_body (node);
  }
#if 1
  recurrence_stats_print_global (stderr);
  void symee_yices_stats_print (FILE*);
  symee_yices_stats_print (stderr);
  
  if (dump_file) {
    recurrence_stats_print_global (dump_file);
    symee_yices_stats_print (dump_file);
  }
#endif
  return;
}

extern void
symee_generate_context (void)
  __attribute__ ((weak, alias ("gen_context_cgraph")));

#if 0
static void
symee_inline_calls_1 (cgraph_node_ptr node)
{
  struct cgraph_edge *e;

  for (e = node->callees; e; e = e->next_callee)
    {
      if (cgraph_edge_recursive_p (e))
	{
	  if (dump_file)
	    fprintf (dump_file, "  Not inlining recursive call to %s.\n",
		     cgraph_node_name (e->callee));
	  e->inline_failed = CIF_RECURSIVE_INLINING;
	  continue;
	}

      if (dump_file)
	fprintf (dump_file, "  Inlining %s into %s (always_inline).\n",
		 cgraph_node_name (e->callee),
		 cgraph_node_name (e->caller));
      extern bool cgraph_mark_inline_edge (struct cgraph_edge *e, bool update_original,
                                           VEC (cgraph_edge_p, heap) **new_edges);
      cgraph_mark_inline_edge (e, true, NULL);
    }

  return;
}
#endif


static void
symee_real_flatten_1 (struct cgraph_node *root_node)
{
  int i;
  struct cgraph_node* node;
  int nnodes;
  struct cgraph_node** order =
    XCNEWVEC (struct cgraph_node*, cgraph_n_nodes);

  nnodes = cgraph_postorder (order);

  symee_flatten (root_node);

  for (i = nnodes - 1; i >= 0; i--) {
    node = order[i];

    /* node->aux has been set by symee_flatten with 'node' itself */
    if (node->aux == node) {
      tree old_decl = current_function_decl;
      push_cfun (DECL_STRUCT_FUNCTION (node->decl));
      current_function_decl = node->decl;

      /* call symee_inline_transform for each inlinable callee */
      symee_inline_transform (node);

      if (node == root_node) {
        if (dump_details) {
          fprintf (dump_file, "After inlining %s\n", cgraph_node_name (node));

          int num_calls = 0;
          basic_block bb = NULL;
          FOR_EACH_BB (bb) {
            gimple_stmt_iterator gsi;
            for (gsi = gsi_start_bb (bb); !gsi_end_p (gsi); gsi_next (&gsi)) {
              if (gimple_code (gsi_stmt (gsi)) == GIMPLE_CALL) {
                struct cgraph_edge *edge = cgraph_edge (node, gsi_stmt (gsi));
                if (!edge->callee)
                  fprintf (dump_file, "\tindirect call\n");
                else if (cgraph_function_body_availability (edge->callee)
                         >= AVAIL_OVERWRITABLE) {
                  fprintf (dump_file, "\tcall %s\n",
                           cgraph_node_name (edge->callee));
                }
                else
                  fprintf (dump_file, "\topaque call %s\n",
                           cgraph_node_name (edge->callee));
                /**/
                num_calls++;
              }
            }
          }

          fprintf (dump_file, "%s still has #calls %d\n",
                   cgraph_node_name (node), num_calls);
        }
      }

      pop_cfun ();
      current_function_decl = old_decl;

    }
  }

  return;
}


/* Inline all calls in function 'promising_node_name' recursively as possible.
   This function wraps inline decision 'symee_flatten' and real inline
   operation 'symee_inline_transform' into single function.
   TODO: change parameter cgraph_node_ptr */

static void
symee_real_flatten (const char *promising_node_name)
{
  int i;
  struct cgraph_node* node;
  int nnodes;
  struct cgraph_node** order =
    XCNEWVEC (struct cgraph_node*, cgraph_n_nodes);

  nnodes = cgraph_postorder (order);

  for (i = nnodes - 1; i >= 0; i--) {
    node = order[i];

    if (node->analyzed
        && strcmp (cgraph_node_name (node), promising_node_name) == 0) {
      symee_flatten (node);
    }
  }

  for (i = nnodes - 1; i >= 0; i--) {
    node = order[i];

    /* node->aux has been set by symee_flatten with 'node' itself */
    if (node->aux == node) {
      tree old_decl = current_function_decl;
      push_cfun (DECL_STRUCT_FUNCTION (node->decl));
      current_function_decl = node->decl;

      /* call symee_inline_transform for each inlinable callee */
      symee_inline_transform (node);
      update_ssa (TODO_update_ssa);
      
      if (strcmp (cgraph_node_name (node), promising_node_name) == 0) {
        if (dump_details) {
          fprintf (dump_file, "After inlining %s\n", cgraph_node_name (node));

          basic_block bb = NULL;
          FOR_EACH_BB (bb) {
            gimple_stmt_iterator gsi;
            for (gsi = gsi_start_bb (bb); !gsi_end_p (gsi); gsi_next (&gsi)) {
              if (gimple_code (gsi_stmt (gsi)) == GIMPLE_CALL) {
                struct cgraph_edge *edge = cgraph_edge (node, gsi_stmt (gsi));
                if (!edge->callee)
                  fprintf (dump_file, "\tindirect call\n");
                else if (cgraph_function_body_availability (edge->callee)
                         >= AVAIL_OVERWRITABLE) {
                  fprintf (dump_file, "\tcall %s\n",
                           cgraph_node_name (edge->callee));
                }
                else
                  fprintf (dump_file, "\topaque call %s\n",
                           cgraph_node_name (edge->callee));
              }
            }
          }

          // gimple_dump_cfg (dump_file, dump_flags);
        }
      }

      pop_cfun ();
      current_function_decl = old_decl;

    }
  }

  return;
}


/* Inline some functions to ease symee
   according to smart/user-defined inlining decisions

   TODO: design smart inline decisions.
 */

static void
symee_inline_calls (void)
{
  int i;
  struct cgraph_node* node;
  int nnodes;
  struct cgraph_node** order =
    XCNEWVEC (struct cgraph_node*, cgraph_n_nodes);
  const char *promising_node_name = NULL;

  /* TODO: */
  symee_assert (!flag_wpa, (stderr, "SYMEE: can not run in wpa mode\n"));

  /* flatten the funtion designated by command option -fipa-symee-inline= */
  if (flag_ipa_symee_inline)
    symee_real_flatten (flag_ipa_symee_inline);

  /* TODO: general inlining decisions */
  else
    return;
}


static unsigned int
ipa_symee (void)
{
  symee_inline_calls ();

  symee_initialize (flag_ipa_symee);

  symee_generate_context ();

  symee_finalize ();

  return 0;
}


static bool
gate_ipa_symee (void)
{
  /* If there is a symee client, symee will be executed in client driver */
  if (flag_ipa_symee_client)
    return false;

  return flag_ipa_symee > 0;
}


/* just for testing, move into symbolic-analysis.c */

struct simple_ipa_opt_pass pass_ipa_symee =
{
 {
  SIMPLE_IPA_PASS,
  "symee",              		/* name */
  gate_ipa_symee,                       /* gate */
  ipa_symee,                            /* execute */
  NULL,					/* sub */
  NULL,					/* next */
  0,					/* static_pass_number */
  0,                                    /* tv_id */
  PROP_ssa,				/* properties_required */
  0,					/* properties_provided */
  0,					/* properties_destroyed */
  0,					/* todo_flags_start */
  TODO_dump_func			/* todo_flags_finish */
 }
};


/* helper function */

static void
propagate_context_forward_1 (cgraph_node_ptr node, symee_context ctx)
{
  symee_cgraph_node_info old_info = current_node_info;
  current_node_info = get_cgraph_node_info (node);

  gen_context_cgraph_node (ctx, node);
  
  current_node_info = old_info;
}


/* propagate context starting from entry of 'node' */

extern void
propagate_context_forward (cgraph_node_ptr node, symee_context ctx)
{
  symee_cgraph_node_info old_info = current_node_info;
  current_node_info = get_cgraph_node_info (node);

  if (node && node->analyzed && !node->clone_of)
    propagate_context_forward_1 (node, ctx);

  cgraph_edge_p cs = NULL;
  for (cs = node->callees; cs; cs = cs->next_callee) {
    cgraph_node_ptr callee_node = cs->callee;

    if (!callee_node || !callee_node->analyzed || callee_node->clone_of)
      continue;

    gimple call_stmt = cs->call_stmt;
    symee_context before_call_ctx = symee_get_entrycontext (node, call_stmt);
    symee_context inctx = gen_context_parameter_in (before_call_ctx, cs);
    propagate_context_forward (callee_node, inctx);

    /* TODO: */
  }

  current_node_info = old_info;
  
  return;
}


/* Traverse into caller and visit the context recursively.
   If return false, stop propagating.

   Please use this function start from function entry */

extern bool
propagate_context_backward (cgraph_node_ptr node, symee_context ctx,
                            symee_context_visit_fn func, void* data)
{
  int res = true;
  symee_cgraph_node_info old_info = current_node_info;
  current_node_info = get_cgraph_node_info (node);

  res = func (ctx, data);

  /* stop if return flase */
  if (res == false)
    goto stop;
  
  /* demand driven into caller.
     TODO: how to process recursive call ? */
  cgraph_edge_p cs = NULL;
  for (cs = node->callers; cs; cs = cs->next_caller) {
    cgraph_node_ptr caller_node = cs->caller;
    /* TODO: how about clone ? */
    if (caller_node->clone_of)
      goto stop;
    gimple call_stmt = cs->call_stmt;
    symee_context before_call_ctx = symee_get_entrycontext (caller_node, call_stmt);
    symee_context inctx = gen_context_parameter_in (before_call_ctx, cs);
    res = propagate_context_backward (caller_node, inctx, func, data);
    /* TODO: how to process results from different callers ? */
    symee_context_destory (inctx);
    if (res == false)
      goto stop;
  }

 stop:
  current_node_info = old_info;
  return res;
}


/* TODO: */

extern symee_context
gen_context_cgraph_forward (symee_context ctx, cgraph_node_ptr node)
{
  symee_forward_recursive = true;
  symee_run_vrp = false;

  gen_context_cgraph_node (ctx, node);
}

#include "gt-symbolic-expr-eval.h"
