/* symbolic-analysis.c */

#include "config.h"
#include "system.h"
#include "coretypes.h"
#include "tree.h"
#include "target.h"
#include "gimple.h"
#include "cgraph.h"
#include "flags.h"
#include "timevar.h"
#include "diagnostic.h"
#include "params.h"
#include "cfgloop.h"
#include "pointer-set.h"
#include "bitmap.h"
#include "tree-flow.h"
#include "tree-pass.h"
#include "tree-chrec.h"
#include "tree-scalar-evolution.h"
#include "tree-data-ref.h"
#include "tree-pretty-print.h"
#include "tree-dump.h"
#include <string.h>

#include "symbolic-expr-eval.h"


/*
 * TOOL 1. program slicing
 */


struct bs_d {
  gimple stmt;
  htab_t visited_stmts;
  VEC(gimple, heap) *worklist;

  VEC(tree, heap) *visited_params;
  VEC(gimple, heap) *visited_calls;
  VEC(gimple, heap) *visited_globals;
};


static inline struct bs_d*
bs_alloc (void)
{
  struct bs_d* bs = XCNEW (struct bs_d);

  return bs;
}


static inline void
bs_free (struct bs_d* bs)
{
  XDELETE (bs);
}


static struct bs_d*
bs_create (gimple stmt)
{
  struct bs_d* bs = bs_alloc ();
  bs->stmt = stmt;
  bs->visited_stmts =
    htab_create_ggc (100,
                     htab_hash_pointer,
                     htab_eq_pointer,
                     NULL);

  return bs;
}


static void
bs_delete (struct bs_d* bs)
{
  if (bs) {
    if (bs->visited_stmts)
      htab_delete (bs->visited_stmts);
    if (bs->worklist)
      VEC_free (gimple, heap, bs->worklist);
    bs_free (bs);
  }
}


static int
bs_dump_stmt (void **slot, void* fp)
{
  print_gimple_stmt ((FILE*)fp, *(gimple*)slot, 0, dump_flags);
}


void
bs_dump (FILE* fp, struct bs_d* bs)
{
  fprintf (fp, "backward slicing:\n");
  print_gimple_stmt (fp, bs->stmt, 0, dump_flags);
  fprintf (fp, "-->\n");
  htab_traverse (bs->visited_stmts, bs_dump_stmt, dump_file);

  int i;
  tree param;
  
  fprintf (fp, "visited params: ");
  FOR_EACH_VEC_ELT (tree, bs->visited_params, i, param)
    {
      print_generic_expr (fp, param, dump_flags);
      fprintf (fp, "\t\t");
    }
  fprintf (fp, "\n\n");
}


static int gimple_count = 0;
static int bs_count = 0;
static double bs_size = 0.0;

static void
bs_stats_start (tree fn)
{
  gimple_stmt_iterator gsi;
  basic_block bb = NULL;

  gimple_count = 0;
  bs_count = 0;
  bs_size = 0.0;
  
  FOR_EACH_BB (bb) {
    for (gsi = gsi_start_bb (bb); !gsi_end_p (gsi); gsi_next (&gsi)) {
      gimple_count++;
    }
  }

  return;
}


static void
bs_stats_finish (void)
{
  if (dump_file)
    {
      fprintf (dump_file,
               "bs_stats: function %s has %d gimple stmts and %d slices"
               " with average size %lf\n",
               IDENTIFIER_POINTER (DECL_NAME (current_function_decl)),
               gimple_count, bs_count,
               (bs_size/bs_count));
    }
}


static void
bs_stats_push (struct bs_d* bs)
{
  if (bs && bs->visited_stmts)
    {
      int slice_size = htab_elements (bs->visited_stmts);
      bs_count++;
      bs_size += slice_size;
    }
}


static void
bs_register_for_ipa (struct bs_d *p, gimple stmt)
{
  switch (gimple_code (stmt))
    {
    case GIMPLE_NOP:
      break;
    case GIMPLE_CALL:
      VEC_safe_push (gimple, heap, p->visited_calls, stmt);
      break;
    default:
      break;
    }

  return;
}


static tree
bs_walk_stmt_op (tree *tp, int *walk_subtrees, void *data)
{
  struct walk_stmt_info *wi = (struct walk_stmt_info *) data;
  struct bs_d *p = (struct bs_d *) wi->info;
  tree op = *tp;

  gcc_assert (op);
  
  *walk_subtrees = true;

  if (TREE_CODE (op) == SSA_NAME)
    {
      gimple def_stmt = SSA_NAME_DEF_STMT (op);
      if (!htab_find (p->visited_stmts, def_stmt))
        VEC_safe_push (gimple, heap, p->worklist, def_stmt);

      if (gimple_code (def_stmt) == GIMPLE_NOP)
        {
          /* TODO: formals and globals for ipa slicer ??? */
          VEC_safe_push (tree, heap, p->visited_params, SSA_NAME_VAR (op));
        }
    }
  
  return NULL_TREE;
}


static tree
bs_walk_stmt (gimple_stmt_iterator *gsi_p, bool *handled_ops_p,
              struct walk_stmt_info *wi)
{
  struct bs_d *p = (struct bs_d *) wi->info;
  gimple cur_stmt = gsi_stmt (*gsi_p);

  if (gimple_vuse (cur_stmt))
    {
      gimple def_stmt = SSA_NAME_DEF_STMT (gimple_vuse (cur_stmt));
      if (!htab_find (p->visited_stmts, def_stmt))
        VEC_safe_push (gimple, heap, p->worklist, def_stmt);

      bs_register_for_ipa (p, cur_stmt);
    }

  *handled_ops_p = false;

  return NULL_TREE;
}


struct bs_d*
bs_slice (gimple_stmt_iterator gsi)
{
  gimple stmt = gsi_stmt (gsi);
  struct bs_d *bs = bs_create (stmt);
  
  VEC_safe_push (gimple, heap, bs->worklist, stmt);

  while (VEC_length (gimple, bs->worklist) != 0)
    {
      /* pick up one stmt to visit */
      gimple cur_stmt = VEC_pop (gimple, bs->worklist);
      void **slot = htab_find_slot (bs->visited_stmts, cur_stmt, INSERT);
      if (*slot == NULL) {
        *slot = cur_stmt;

        bs_register_for_ipa (bs, cur_stmt);

        if (gimple_code (cur_stmt) != GIMPLE_NOP) {
          struct walk_stmt_info wi;
          memset (&wi, 0, sizeof (wi));
          wi.info = bs;

          gimple_stmt_iterator cur_gsi = gsi_for_stmt (cur_stmt);
          walk_gimple_stmt (&cur_gsi, bs_walk_stmt, bs_walk_stmt_op, &wi);
        }
      }
    }

  VEC_free (gimple, heap, bs->worklist);

  return bs;
}


static int
dump_slice_stmt_1 (void **slot, void* fp)
{
  print_gimple_stmt ((FILE*)fp, *(gimple*)slot, 0, dump_flags);
}


static void
backward_slicing (gimple stmt)
{
  htab_t visited_stmts = htab_create_ggc (100, htab_hash_pointer, htab_eq_pointer, NULL);
  VEC(gimple, heap)* worklist = NULL;

  VEC_safe_push (gimple, heap, worklist, stmt);

  while (VEC_length (gimple, worklist) != 0)
    {
      /* pick up one stmt to visit */
      gimple cur_stmt = VEC_pop (gimple, worklist);
      void **slot = htab_find_slot (visited_stmts, cur_stmt, INSERT);
      if (*slot == NULL) {
        *slot = cur_stmt;

        if (gimple_has_ops (cur_stmt)) {
          unsigned i = 0;
          for (i = 0; i < gimple_num_ops (cur_stmt); i++)
            {
              tree op = gimple_op (cur_stmt, i);
              /* walk op tree to find all def stmt */
              if (op && TREE_CODE (op) == SSA_NAME)
                {
                  gimple def_stmt = SSA_NAME_DEF_STMT (op);
                  if (!htab_find (visited_stmts, def_stmt))
                    VEC_safe_push (gimple, heap, worklist, def_stmt);
                }
            }
        }
        if (gimple_vuse (cur_stmt))
          {
            gimple def_stmt = SSA_NAME_DEF_STMT (gimple_vuse (cur_stmt));
            if (!htab_find (visited_stmts, def_stmt))
              VEC_safe_push (gimple, heap, worklist, def_stmt);
          }
      }
    }

  if (dump_file)
    {
      int slice_size = htab_elements (visited_stmts);
      fprintf (dump_file, "backward slicing: %d\n", slice_size);
      print_gimple_stmt (dump_file, stmt, 0, dump_flags);
      fprintf (dump_file, "-->\n");
      htab_traverse (visited_stmts, dump_slice_stmt_1, dump_file);
      fprintf (dump_file, "\n\n");
    }

  return;
}


static void
bs_node (cgraph_node_ptr node)
{
  tree decl = node->decl;
  tree old_decl = current_function_decl;

  push_cfun (DECL_STRUCT_FUNCTION (decl));
  current_function_decl = decl;

  bs_stats_start (decl);
  
  basic_block bb = NULL;
  FOR_EACH_BB (bb) {
    gimple_stmt_iterator gsi;
    for (gsi = gsi_start_bb (bb); !gsi_end_p (gsi); gsi_next (&gsi)) {
      if (1 || gimple_code (gsi_stmt (gsi)) == GIMPLE_RETURN)
        {
          struct bs_d* bs = bs_slice (gsi);

          bs_stats_push (bs);
          
          if (dump_file)
            bs_dump (dump_file, bs);

          bs_delete (bs);
        }
    }
  }
  
  bs_stats_finish ();
  
  pop_cfun ();
  current_function_decl = old_decl;
}


/* backward slicing engine, just for testing */

static unsigned int
ipa_bs (void)
{
  int i;
  struct cgraph_node* node;
  int nnodes;
  struct cgraph_node** order =
    XCNEWVEC (struct cgraph_node*, cgraph_n_nodes);

  nnodes = cgraph_postorder (order);

  if (dump_file)
    fprintf (dump_file, "sa: backward slicing\n");
  
  for (i = 0; i < nnodes; i++) {
    node = order[i];
    if (node->analyzed && !node->clone_of) {
      bs_node (node);
    }
  }
}


/*
 * TOOL 2. dump ssa graph
 */


void
dump_du (FILE *file, tree var)
{
  imm_use_iterator iter;
  use_operand_p use_p;
  gimple stmt;
  
  gcc_assert (var && TREE_CODE (var) == SSA_NAME);

  print_generic_expr (file, var, TDF_SLIM);
  fprintf (file, " : -->");
  if (has_zero_uses (var))
    fprintf (file, " no uses.\n");
  else
    if (has_single_use (var))
      fprintf (file, " single use.\n");
    else
      fprintf (file, "%d uses.\n", num_imm_uses (var));

  FOR_EACH_IMM_USE_STMT (stmt, iter, var)
    {
      print_gimple_stmt (dump_file, stmt, 0, dump_flags);
      fprintf (file, "%p: ", stmt);
      FOR_EACH_IMM_USE_ON_STMT (use_p, iter)
        {
          fprintf (file, "%p: ", USE_STMT (use_p));
        }
      fprintf (file, "\n");
    }
  fprintf(file, "\n");
}


static void
traverse_ssa_graph_node (cgraph_node_ptr node)
{
  tree decl = node->decl;
  tree old_decl = current_function_decl;

  push_cfun (DECL_STRUCT_FUNCTION (decl));
  current_function_decl = decl;
  
  basic_block bb = NULL;
  FOR_EACH_BB (bb) {
    gimple_stmt_iterator gsi;
    for (gsi = gsi_start_bb (bb); !gsi_end_p (gsi); gsi_next (&gsi)) {
      if (gimple_code (gsi_stmt (gsi)) == GIMPLE_ASSIGN)
        {
          tree lhs = gimple_assign_lhs (gsi_stmt (gsi));
          if (TREE_CODE (lhs) == SSA_NAME)
            dump_du (dump_file, lhs);
        }
    }
  }
  
  pop_cfun ();
  current_function_decl = old_decl;
}


static unsigned int
ipa_traverse_ssa_graph (void)
{
  int i;
  struct cgraph_node* node;
  int nnodes;
  struct cgraph_node** order =
    XCNEWVEC (struct cgraph_node*, cgraph_n_nodes);

  nnodes = cgraph_postorder (order);

  if (dump_file)
    fprintf (dump_file, "SYMEE Client: dump ssa graph\n");
  
  for (i = 0; i < nnodes; i++) {
    node = order[i];
    if (node->analyzed && !node->clone_of) {
      traverse_ssa_graph_node (node);
    }
  }
}


/*
 * TOOL 3. dump ssa graph into summary matrix
 */

/* summary matrix

          s1 s2 s3 ... sn
     v1
     v2
     v3   d  u
     .
     .          k      u
     .
     vn
 */

#define INDENT(fp,n)                            \
  do { int i; for (i = 0; i < n; i++) fprintf (fp, " "); } while (0)


static void
dump_ssa_graph (cgraph_node_ptr node)
{
  tree decl = node->decl;
  tree old_decl = current_function_decl;

  push_cfun (DECL_STRUCT_FUNCTION (decl));
  current_function_decl = decl;

  fprintf (dump_file, "ssa_graph: %7d ssa_name's and ? gimple stmt's (%s)\n",
           num_ssa_names, cgraph_node_name (node));
  size_t i;
  for (i = 1; i < num_ssa_names; i++)
    {
      tree t = ssa_name (i);
      if (t) {
        print_generic_expr (dump_file, ssa_name (i), 0);
        gimple def_stmt = SSA_NAME_DEF_STMT (t);
        INDENT (dump_file, 50);
        print_gimple_stmt (dump_file, def_stmt, 50, dump_flags);
      }
    }
  
  pop_cfun ();
  current_function_decl = old_decl;
}


static unsigned int
ipa_dump_ssa_graph (void)
{
  int i;
  struct cgraph_node* node;
  int nnodes;
  struct cgraph_node** order =
    XCNEWVEC (struct cgraph_node*, cgraph_n_nodes);

  nnodes = cgraph_postorder (order);

  if (dump_file)
    fprintf (dump_file, "SYMEE Client: dump ssa graph\n");
  
  for (i = 0; i < nnodes; i++) {
    node = order[i];
    if (node->analyzed && !node->clone_of) {
      dump_ssa_graph (node);
    }
  }

  return 0;
}


/*
 * TOOL 4. compare a pair of expressions or address expressions
 */


struct cmp_expr_data {
  symee_context ctx;
  int prop_result;
  
  tree exp1;
  tree exp2;

  tree result;
};


static bool
cmp_expr_1 (symee_context ctx, void* data)
{
  struct cmp_expr_data* cmp_data = (struct cmp_expr_data*) data;

  tree exp1 = cmp_data->exp1;
  tree exp2 = cmp_data->exp2;
  
  tree eval_exp1 = symee_eval (ctx, exp1);
  tree eval_exp2 = symee_eval (ctx, exp2);
  
  tree cmp = build2 (EQ_EXPR, boolean_type_node, eval_exp1, eval_exp2);
  tree eval_cmp = symee_eval (ctx, cmp);

  if (!eval_cmp)
    eval_cmp = cmp;

  if (eval_cmp == boolean_true_node
      || eval_cmp == boolean_false_node
      || symee_is_const_true (eval_cmp)
      /* || is_const_false (eval_cmp) */) {
    cmp_data->prop_result = false;
    cmp_data->result = eval_cmp;

    /* stop */
    return false;
  }

  cmp_data->exp1 = eval_exp1;
  cmp_data->exp2 = eval_exp2;
  cmp_data->prop_result = true;
  
  /* continue */
  return true;
}


static void
compare_addr_expr_node_1 (cgraph_node_ptr node)
{
  loop_iterator li;
  struct loop *l;
  tree decl = node->decl;
  tree old_decl = current_function_decl;

  push_cfun (DECL_STRUCT_FUNCTION (decl));
  current_function_decl = decl;

  if (1)
    fprintf (stderr, "D-D compare_expr: %s\n", cgraph_node_name (node));
  
  update_ssa (TODO_update_ssa);
  if (!current_loops)
    loop_optimizer_init (LOOPS_NORMAL | LOOPS_HAVE_RECORDED_EXITS);
  rewrite_into_loop_closed_ssa (NULL, TODO_update_ssa);
  scev_initialize ();

  FOR_EACH_LOOP (li, l, 0)
    {
      VEC (data_reference_p, heap) *datarefs = NULL;
      VEC (ddr_p, heap) *dependence_relations = NULL;
      VEC (loop_p, heap) *loop_nest = NULL;

      if (!l
          || !find_loop_nest (l, &loop_nest)
          || find_data_references_in_loop (l, &datarefs) == chrec_dont_know)
        {
          continue;
        }

      struct data_dependence_relation *ddr;
      struct data_reference *a, *b;
      unsigned int i, j;

      FOR_EACH_VEC_ELT (data_reference_p, datarefs, i, a)
        for (j = i+1; VEC_iterate (data_reference_p, datarefs, j, b); j++)
          if (1 || DR_IS_WRITE (a) || DR_IS_WRITE (b))
            {
              symee_context node_ctx = symee_get_context (node);
              struct cmp_expr_data data = {node_ctx, 1, a->ref, b->ref, 0};

              /* a and b can be under different context.
                 before find common context, we can compare them
                 under different context.
                 This is safe for contexts which can be represent
                 by function formals and globals
              */
              tree aexp = symee_eval (symee_get_entrycontext (node, a->stmt), a->ref);
              tree bexp = symee_eval (symee_get_entrycontext (node, b->stmt), b->ref);
              tree cmp = build2 (EQ_EXPR, boolean_type_node, aexp, bexp);
              tree eval_cmp = symee_eval (node_ctx, cmp);
              if (eval_cmp)
                eval_cmp = cmp;
              if (eval_cmp == boolean_true_node
                  || eval_cmp == boolean_false_node
                  || symee_is_const_true (eval_cmp))
                {
                  data.prop_result = false;
                  data.result = eval_cmp;
                }

              /* From current node entry, they have common context */
              else
                {
                  data.exp1 = aexp;
                  data.exp2 = bexp;
                  propagate_context_backward (node, node_ctx, cmp_expr_1, &data);
                }
              
              bool anderson_alias = refs_may_alias_p (a->ref, b->ref);
              bool alias = symee_dr_may_alias_p (a, b);

              if (dump_file) {
                fprintf (dump_file, "sa: da-by-cmp-addr-expr <");
                print_generic_expr (dump_file, a->ref, dump_flags);
                fprintf (dump_file, ", ");
                print_generic_expr (dump_file, b->ref, dump_flags);
                fprintf (dump_file, ">  --%1d,%1d-->  ", anderson_alias, alias);
                fprintf (dump_file, "<%d,%d> ",
                         gimple_lineno (a->stmt),
                         gimple_lineno (b->stmt));
                if (data.prop_result == true)
                  fprintf (dump_file, " (NO_RESULT)\n");
                else {
                  if (data.result == boolean_true_node)
                    fprintf (dump_file, " (EQUAL)\n");
                  else if (data.result == boolean_false_node)
                    fprintf (dump_file, " (NOT EQUAL)\n");
                  else
                    fprintf (dump_file, " (UNKOWN)\n");
                }
              }
            }
    }
  
  pop_cfun ();
  current_function_decl = old_decl;

}


static void
compare_addr_expr_node (cgraph_node_ptr node)
{
  loop_iterator li;
  struct loop *l;
  tree decl = node->decl;
  tree old_decl = current_function_decl;

  push_cfun (DECL_STRUCT_FUNCTION (decl));
  current_function_decl = decl;

  if (1)
    fprintf (stderr, "D-D compare_expr: %s\n", cgraph_node_name (node));
  
  update_ssa (TODO_update_ssa);
  if (!current_loops)
    loop_optimizer_init (LOOPS_NORMAL | LOOPS_HAVE_RECORDED_EXITS);
  rewrite_into_loop_closed_ssa (NULL, TODO_update_ssa);
  scev_initialize ();

  FOR_EACH_LOOP (li, l, 0)
    {
      VEC (data_reference_p, heap) *datarefs = NULL;
      VEC (ddr_p, heap) *dependence_relations = NULL;
      VEC (loop_p, heap) *loop_nest = NULL;

      if (!l
          || !find_loop_nest (l, &loop_nest)
          || find_data_references_in_loop (l, &datarefs) == chrec_dont_know)
        {
          continue;
        }

      struct data_dependence_relation *ddr;
      struct data_reference *a, *b;
      unsigned int i, j;

      FOR_EACH_VEC_ELT (data_reference_p, datarefs, i, a)
        for (j = i+1; VEC_iterate (data_reference_p, datarefs, j, b); j++)
          if (1 || DR_IS_WRITE (a) || DR_IS_WRITE (b))
            {
              symee_context node_ctx = symee_get_context (node);
              tree a_addr = build_fold_addr_expr (a->ref);
              tree b_addr = build_fold_addr_expr (b->ref);
              struct cmp_expr_data data = {node_ctx, 1, a_addr, b_addr, 0};

              /* a and b can be under different context.
                 before find common context, we can compare them
                 under different context.
                 This is safe for contexts which can be represent
                 by function formals and globals
              */
              tree aexp = symee_eval (symee_get_entrycontext (node, a->stmt), a_addr);
              tree bexp = symee_eval (symee_get_entrycontext (node, b->stmt), b_addr);
              tree cmp = build2 (EQ_EXPR, boolean_type_node, aexp, bexp);
              tree eval_cmp = symee_eval (node_ctx, cmp);
              if (eval_cmp)
                eval_cmp = cmp;
              if (eval_cmp == boolean_true_node
                  || eval_cmp == boolean_false_node
                  || symee_is_const_true (eval_cmp))
                {
                  data.prop_result = false;
                  data.result = eval_cmp;
                }

              /* From current node entry, they have common context */
              else
                {
                  data.exp1 = aexp;
                  data.exp2 = bexp;
                  propagate_context_backward (node, node_ctx, cmp_expr_1, &data);
                }
              
              bool anderson_alias = refs_may_alias_p (a->ref, b->ref);
              bool alias = symee_dr_may_alias_p (a, b);

              if (dump_file) {
                fprintf (dump_file, "sa: da-by-cmp-addr-expr <");
                print_generic_expr (dump_file, a_addr, dump_flags);
                fprintf (dump_file, ", ");
                print_generic_expr (dump_file, b_addr, dump_flags);
                fprintf (dump_file, ">  --%1d,%1d-->  ", anderson_alias, alias);
                fprintf (dump_file, "<%d,%d> ",
                         gimple_lineno (a->stmt),
                         gimple_lineno (b->stmt));
                if (data.prop_result == true)
                  fprintf (dump_file, " (NO_RESULT)\n");
                else {
                  if (data.result == boolean_true_node)
                    fprintf (dump_file, " (EQUAL)\n");
                  else if (data.result == boolean_false_node)
                    fprintf (dump_file, " (NOT EQUAL)\n");
                  else
                    fprintf (dump_file, " (UNKOWN)\n");
                }
              }
            }
    }
  
  pop_cfun ();
  current_function_decl = old_decl;

}


static unsigned int
compare_addr_expr (void)
{
  int i;
  struct cgraph_node* node;
  int nnodes;
  struct cgraph_node** order =
    XCNEWVEC (struct cgraph_node*, cgraph_n_nodes);

  nnodes = cgraph_postorder (order);

  if (1)
    fprintf (stderr, "SYMEE Client: demand driven compare_expr\n");
  
  if (dump_file)
    fprintf (dump_file, "SYMEE Client: demand driven compare_expr\n");
  
  symee_init_alias ();

  for (i = 0; i < nnodes; i++) {
    node = order[i];
    if (node->analyzed && !node->clone_of) {
      compare_addr_expr_node (node);
    }
  }

  symee_fini_alias ();
}


static void
compare_expr_node (cgraph_node_ptr node)
{
  tree decl = node->decl;
  tree old_decl = current_function_decl;
  symee_cgraph_node_info old_node_info = current_node_info;
  push_cfun (DECL_STRUCT_FUNCTION (decl));
  current_function_decl = decl;

  current_node_info = symee_get_cgraph_node_info (node);

  basic_block bb = NULL;
  FOR_EACH_BB (bb) {
    gimple_stmt_iterator gsi;
    for (gsi = gsi_start_bb (bb); !gsi_end_p (gsi); gsi_next (&gsi)) {
      gimple stmt = gsi_stmt (gsi);
      if (gimple_code (stmt) == GIMPLE_COND) {
        tree lhs = gimple_cond_lhs (stmt);
        tree rhs = gimple_cond_rhs (stmt);
        if (lhs && rhs) {
          symee_context cur_ctx = symee_get_entrycontext (node, stmt);
          struct cmp_expr_data data = {cur_ctx, 1, lhs, rhs, 0};

          tree cond_expr = build2 (gimple_cond_code (stmt),
                                   boolean_type_node, lhs, rhs);
          cond_expr = symee_eval (cur_ctx, cond_expr);
          if (cond_expr == boolean_true_node
              || cond_expr == boolean_false_node
              || symee_is_const_true (cond_expr)
              || symee_is_const_false (cond_expr))
            {
              fprintf (dump_file, "sa: compare <> is const");
            }
          

          propagate_context_backward (node, cur_ctx, cmp_expr_1, &data);

          if (dump_file) {
            fprintf (dump_file, "sa: compare <");
            print_generic_expr (dump_file, lhs, dump_flags);
            fprintf (dump_file, ", ");
            print_generic_expr (dump_file, rhs, dump_flags);
            fprintf (dump_file, ">  -->  ");

            if (data.prop_result == true)
              fprintf (dump_file, " (NO_RESULT)\n");
            else {
              if (data.result == boolean_true_node)
                fprintf (dump_file, " (EQUAL)\n");
              else if (data.result == boolean_false_node)
                fprintf (dump_file, " (NOT EQUAL)\n");
              else
                fprintf (dump_file, " (UNKOWN)\n");
            }
              
          }
        }
      }
    }
  }
  
  pop_cfun ();
  current_function_decl = old_decl;
}


static unsigned int
compare_expr (void)
{
  int i;
  struct cgraph_node* node;
  int nnodes;
  struct cgraph_node** order =
    XCNEWVEC (struct cgraph_node*, cgraph_n_nodes);

  nnodes = cgraph_postorder (order);

  if (1)
    fprintf (stderr, "SYMEE Client: demand driven compare_expr\n");
  
  if (dump_file)
    fprintf (dump_file, "SYMEE Client: demand driven compare_expr\n");
  
  //symee_init_alias ();

  for (i = 0; i < nnodes; i++) {
    node = order[i];
    if (node->analyzed && !node->clone_of) {
      compare_expr_node (node);
    }
  }

  //symee_fini_alias ();
}


/*
 * TOOL 5. propagate context forward
 */

static void
test_forward_propagation (void)
{
  int i;
  struct cgraph_node* node;
  int nnodes;
  struct cgraph_node** order =
    XCNEWVEC (struct cgraph_node*, cgraph_n_nodes);

  nnodes = cgraph_postorder (order);

  if (dump_file)
    fprintf (dump_file, "SYMEE Client: test forward propagation\n");
  
  for (i = 0; i < nnodes; i++) {
    node = order[i];
    if (node && node->analyzed && !node->clone_of) {
      fprintf (stderr, "forward propagation: start from %s\n",
               cgraph_node_name (node));
      propagate_context_forward (node, NULL);
    }
  }
}


/*
 * TOOL 6. generate context forward recursively
 */


static void
test_gen_context_cgraph_r (void)
{
  int i;
  struct cgraph_node* node;
  int nnodes;
  struct cgraph_node** order =
    XCNEWVEC (struct cgraph_node*, cgraph_n_nodes);

  nnodes = cgraph_postorder (order);

  if (dump_file)
    fprintf (dump_file, "SYMEE Client: test forward propagation\n");
  
  for (i = 0; i < nnodes; i++) {
    node = order[i];
    if (node && node->analyzed && !node->clone_of) {
      fprintf (stderr, "gen_context_cgraph_r: start from %s\n",
               cgraph_node_name (node));
      gen_context_cgraph_forward (NULL, node);
    }
  }
}


/*
 * TOOL 7. demand-driven privatization analysis
 */

/* Find killing sites for given reference (gcc has done it for ssa_name)

   For a given reference for which gcc can generate corresponding
   data_reference, we can get all may-def sites by following ssa_chain of .MEM.
   For each site we need to dicide whether it is a killing site. We may
   find a set of killing sites which consist of a kill closure.

   This pass is not a symee client */

typedef struct {
  gimple stmt;

  /* result */
  htab_t killing_stmts;

  /* temporary */
  htab_t visited_stmts;
  VEC (gimple, heap) *worklist;
} ddpa_t;


static ddpa_t*
ddpa_init (gimple stmt)
{
  ddpa_t *p = XCNEW (ddpa_t);
  p->stmt = stmt;
  p->killing_stmts = htab_create_ggc (100, htab_hash_pointer, htab_eq_pointer, NULL);
  p->visited_stmts = htab_create_ggc (100, htab_hash_pointer, htab_eq_pointer, NULL);

  return p;
}


static void
ddpa_fini (ddpa_t *p)
{
  if (!p)
    return;
  if (p->worklist)
    VEC_free (gimple, heap, p->worklist);
  if (p->visited_stmts)
    htab_delete (p->visited_stmts);
  if (p->killing_stmts)
    htab_delete (p->killing_stmts);
  XDELETE (p);
}


static void
ddpa_kill_ref (gimple def_stmt, tree ref, ddpa_t *ddpa_data)
{
  if (dump_details) {
    fprintf (dump_file, "ref: ");
    print_generic_expr (dump_file, ref, dump_flags);
  }

  if (stmt_kills_ref_p (def_stmt, ref)) {
    if (dump_details)
      fprintf (dump_file, "\t\tkilled by stmt: ");
  }
  else if (!stmt_may_clobber_ref_p (def_stmt, ref)) {
    /* TODO: just ignore ? */
    if (dump_details)
      fprintf (dump_file, "\t\tnot clobbered by stmt: ");
  }
  else {
    if (dump_details)
      fprintf (dump_file, "\t\tclobbered but not killed by stmt: ");
    
    VEC_safe_push (gimple, heap, ddpa_data->worklist, def_stmt);
  }

  if (dump_details)
    print_gimple_stmt (dump_file, def_stmt, 0, dump_flags);
}


static void
ddpa_kill (gimple def_stmt, gimple use_stmt, ddpa_t *ddpa_data)
{
  unsigned int i = 0;
  switch (gimple_code (use_stmt)) {
  case GIMPLE_CALL:
    /* FIXME: We do not know which reference should be killed */
    for (i = 0; i < gimple_call_num_args (use_stmt); i++) {
      tree ref = gimple_call_arg (use_stmt, 0);
      if (REFERENCE_CLASS_P (ref))
        ddpa_kill_ref (def_stmt, ref, ddpa_data);
    }
    break;
  default:
    for (i = 0; i < gimple_num_ops (use_stmt); i++) {
      tree ref = gimple_op (use_stmt, i);
      if (REFERENCE_CLASS_P (ref))
        ddpa_kill_ref (def_stmt, ref, ddpa_data);
    }
    break;
  }
  
  return;
}


void
ddpa (gimple stmt)
{
  basic_block bb = gimple_bb (stmt);
  struct loop *l = loop_containing_stmt (stmt);
  tree ref = NULL;
  tree *op0, *op1;
  
  if (gimple_code (stmt) == GIMPLE_ASSIGN)
    {
      tree base;
      op0 = gimple_assign_lhs_ptr (stmt);
      op1 = gimple_assign_rhs1_ptr (stmt);

      if (DECL_P (*op1)
	  || (REFERENCE_CLASS_P (*op1)
	      && (base = get_base_address (*op1))
	      && TREE_CODE (base) != SSA_NAME))
	{
	  ref = *op1;
	}
    }

  if (dump_details) {
    if (ref) {
      fprintf (dump_file, "find kill sites for ref (");
      print_generic_expr (dump_file, ref, dump_flags);
      fprintf (dump_file, ") in statement:\n");
    }
    else
      fprintf (dump_file, "no reference to kill in statement:\n");
    print_gimple_stmt (dump_file, stmt, 0, dump_flags);
    fprintf (dump_file, "\n");
  }
  
  if (ref) {
    data_reference_p dr = create_data_ref (l, l, ref, stmt, true);
  }
  else
    return;
  
  ddpa_t *ddpa_data = ddpa_init (stmt);

  VEC_safe_push (gimple, heap, ddpa_data->worklist, stmt);

  while (VEC_length (gimple, ddpa_data->worklist))
    {
      gimple use_stmt = VEC_pop (gimple, ddpa_data->worklist);
      gimple def_stmt = SSA_NAME_DEF_STMT (gimple_vuse (use_stmt));

      if (!htab_find (ddpa_data->visited_stmts, def_stmt))
        ddpa_kill (def_stmt, use_stmt, ddpa_data);
    }

  ddpa_fini (ddpa_data);
}


void
ddpa_node (cgraph_node_ptr node)
{
  tree decl = node->decl;
  tree old_decl = current_function_decl;

  push_cfun (DECL_STRUCT_FUNCTION (decl));
  current_function_decl = decl;

  bs_stats_start (decl);
  
  basic_block bb = NULL;
  FOR_EACH_BB (bb) {
    gimple_stmt_iterator gsi;
    for (gsi = gsi_start_bb (bb); !gsi_end_p (gsi); gsi_next (&gsi)) {
      if (gimple_vuse (gsi_stmt (gsi)))
        {
          ddpa (gsi_stmt (gsi));
        }
    }
  }
  
  bs_stats_finish ();
  
  pop_cfun ();
  current_function_decl = old_decl;

}


static unsigned int
ipa_ddpa (void)
{
  int i;
  struct cgraph_node* node;
  int nnodes;
  struct cgraph_node** order =
    XCNEWVEC (struct cgraph_node*, cgraph_n_nodes);
  const char *promising_func_name = "P7Viterbi";

  symee_init_alias ();
  
  nnodes = cgraph_postorder (order);

  if (dump_file)
    fprintf (dump_file, "ipa_ddpa: demand driven privatization analysis\n\n");

  for (i = 0; i < nnodes; i++) {
    node = order[i];
    if (node->analyzed && !node->clone_of) {
      if (strcmp (cgraph_node_name (node), promising_func_name) == 0)
        ddpa_node (node);
    }
  }

  symee_fini_alias ();

  return 0;
}


/*
 * TOOL 8. ipa_symee_vrp
 */

static void
ipa_symee_vrp_node (cgraph_node_ptr node)
{
  tree decl = node->decl;
  tree old_decl = current_function_decl;
  symee_cgraph_node_info old_node_info = current_node_info;
  push_cfun (DECL_STRUCT_FUNCTION (decl));
  current_function_decl = decl;

  current_node_info = symee_get_cgraph_node_info (node);


  size_t i;
  for (i = 1; i < num_ssa_names; i++)
    {
      tree t = ssa_name (i);
      if (t) {
        gimple def_stmt = SSA_NAME_DEF_STMT (t);
        symee_context ctx = symee_get_entrycontext (node, def_stmt);
        symee_dump_symbolic_range (ctx, t);
      }
    }
  
  basic_block bb = NULL;
  FOR_EACH_BB (bb) {
    gimple_stmt_iterator gsi;
    for (gsi = gsi_start_bb (bb); !gsi_end_p (gsi); gsi_next (&gsi)) {
      gimple stmt = gsi_stmt (gsi);
    }
  }
  
  pop_cfun ();
  current_function_decl = old_decl;
}


static unsigned int
ipa_symee_vrp (void)
{
  int i;
  struct cgraph_node* node;
  int nnodes;
  struct cgraph_node** order =
    XCNEWVEC (struct cgraph_node*, cgraph_n_nodes);

  nnodes = cgraph_postorder (order);

  if (dump_file)
    fprintf (dump_file, "symee client: symbolic vrp\n");
  
  symee_init_alias ();

  for (i = 0; i < nnodes; i++) {
    node = order[i];
    if (node->analyzed && !node->clone_of) {
      ipa_symee_vrp_node (node);
    }
  }

  symee_fini_alias ();
}


/*
 * TOOL . symee client driver
 */

/* predefined client entries */

typedef unsigned int (*symee_client_entry_func_t) (void);

struct symee_client_entry {
  const char *name;
  symee_client_entry_func_t entry;
};
  
struct symee_client_entry client_entries [] = {
  {"cmp-expr", compare_expr},
  {"cmp-addr-expr", compare_addr_expr},
  {"symee-vrp", ipa_symee_vrp},

  /* TODO: */
  {"check-dep", NULL},
  {"private", NULL},
  
  /* keep the last entry empty */
  {NULL, NULL}
};

#define NUM_CLIENT_ENTRIES                                              \
  (sizeof (client_entries) / sizeof (struct symee_client_entry) - 1)

static inline symee_client_entry_func_t
get_client_entry (const char *client_name)
{
  size_t i = 0;
  for (i = 0; i < NUM_CLIENT_ENTRIES; i++)
    {
      if (strcmp (client_entries[i].name, client_name) == 0)
        return client_entries[i].entry;
    }
  return NULL;
}


/* general symee client entry */

static unsigned int
ipa_symee_client (void)
{
  symee_client_entry_func_t client_entry =
    get_client_entry (flag_ipa_symee_client);

  if (!client_entry) {
    fprintf (stderr, "symee client: no such client %s\n",
             flag_ipa_symee_client);
    return 0;
  }

  if (dump_file)
    fprintf (dump_file, "/* symee client: %s */\n\n",
             flag_ipa_symee_client);
  
  symee_initialize (0x7fffffff);

  symee_generate_context ();

  if (flag_ipa_symee_ggc)
    {
      dump_memory_report (false);
    }
  
  symee_yices_stats_start ();
  
  /* perform client analysis */
  client_entry ();

  symee_yices_stats_print (stderr);
  if (dump_file)
    symee_yices_stats_print (dump_file);
  
  symee_finalize ();

  return 0;
}


static bool
gate_ipa_symee_client(void)
{
  return flag_ipa_symee_client != NULL;
}


struct simple_ipa_opt_pass pass_ipa_symee_client =
{
 {
  SIMPLE_IPA_PASS,
  "symee_client",                      	/* name */
  gate_ipa_symee_client,                /* gate */
  ipa_symee_client,                     /* execute */
  NULL,					/* sub */
  NULL,					/* next */
  0,					/* static_pass_number */
  0,                                    /* tv_id */
  PROP_ssa,				/* properties_required */
  0,					/* properties_provided */
  0,					/* properties_destroyed */
  0,					/* todo_flags_start */
  TODO_dump_func			/* todo_flags_finish */
 }
};



/******************************************************************************
 *
 * test driver for symbolic analysis which does not use symee
 *
 ******************************************************************************/

static unsigned int
ipa_sa (void)
{
  //ipa_bs ();
  ipa_dump_ssa_graph ();
  
  return 0;
}


static bool
gate_ipa_sa(void)
{
  return flag_ipa_sa > 0;
}


struct simple_ipa_opt_pass pass_ipa_sa =
{
 {
  SIMPLE_IPA_PASS,
  "sa",                         	/* name */
  gate_ipa_sa,                          /* gate */
  ipa_sa,                               /* execute */
  NULL,					/* sub */
  NULL,					/* next */
  0,					/* static_pass_number */
  0,                                    /* tv_id */
  PROP_ssa,				/* properties_required */
  0,					/* properties_provided */
  0,					/* properties_destroyed */
  0,					/* todo_flags_start */
  TODO_dump_func			/* todo_flags_finish */
 }
};


