/* gcalc language specific functions for GENERIC generation and type
   checking
   Copyright (C) 2010
   Free Software Foundation, Inc.

   This file is part of GCC.

   GCC is free software; you can redistribute it and/or modify it under
   the terms of the GNU General Public License as published by the Free
   Software Foundation; either version 3, or (at your option) any later
   version.

   GCC is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
   for more details.

   You should have received a copy of the GNU General Public License
   along with GCC; see the file COPYING3.  If not see
   <http://www.gnu.org/licenses/>.  
*/

#include "config.h"
#include "system.h"
#include "ansidecl.h"
#include "coretypes.h"
#include "tm.h"
#include "opts.h"
#include "tree.h"
#include "tree-iterator.h"
#include "tree-pass.h"
#include "gimple.h"
#include "toplev.h"
#include "debug.h"
#include "options.h"
#include "flags.h"
#include "convert.h"
#include "diagnostic-core.h"
#include "langhooks.h"
#include "langhooks-def.h"
#include "target.h"
#include "cgraph.h"

#include <gmp.h>
#include <mpfr.h>

#include "vec.h"
#include "hashtab.h"

#include "calc1.h"
#include "calc_lang.h"

#define BUILD_LOCUS_INFO(loc) linemap_line_start(line_table, loc->line, 0)

VEC(tree,gc) * main_block_decls;

/* The symbol table of gcalc is implemented using a very simple hash table
 * which contains VAR_DECL tree nodes. For real front-ends, the symbol
 * table entry structure might be more sophisticated, but a VAR_DECL tree
 * node really accomplishes the needs
 */
htab_t calc_sym_table = NULL;

/* The list of gcalc instructions. The get filled by the bison parser and
 * its parser actions */
tree main_stmts = NULL_TREE;
tree file_name_string = NULL_TREE;

/* Symbol table (htab) call back function for hash value generation */
static hashval_t 
calc_htab_hash (const void *item)
{
  const tree name = ((const symtab_entry_t*)item)->name;
  const char *decl_name = IDENTIFIER_POINTER(name);

  return (decl_name[0]) & 31;
}

/* Symbol table (htab) call back function for entry comparison. This is 
 * just a simple string comparison on the declaration name
 */
static int 
calc_htab_eq (const void *lhs, const void *rhs)
{
  const tree lhs_name = ((const symtab_entry_t*)lhs)->name;
  const tree rhs_name = ((const symtab_entry_t*)rhs)->name;

  return !strcmp(IDENTIFIER_POINTER(lhs_name),
                 IDENTIFIER_POINTER(rhs_name));
}

/* Symbol table (htab) call back function for entry cleanup */
static void 
calc_htab_del (void *item ATTRIBUTE_UNUSED)
{
  return;
}

/* Symbol table (htab) call back function for traversal purpose.
 * entry is a pointer to the slot containing the real element, though
 * the expression (tree)*entry is the VAR_DECL tree node. arg is the
 * pointer to the global_decl from calc_getdecls
 */
int 
calc_htab_trav (void **entry, void *arg)
{
  tree *first = (tree*)arg;

  symtab_entry_t *sym = *(symtab_entry_t**)entry;
  TREE_CHAIN(sym->decl) = *first;
  *first = sym->decl;

  return true;
}

/* Symbol table initializaton */
int 
init_sym_tbl (void)
{
  calc_sym_table = htab_create(32, calc_htab_hash, calc_htab_eq, calc_htab_del);
  return 0;
}

/* Symbol table finalization */
int 
fini_sym_tbl (void)
{
  htab_delete (calc_sym_table);

  return 0;
}

/* Type checking function for the LHS of an assign expression, e.g.
 * LHS = RHS. Only an IDENTIFIER_NODE and a MODIFY_EXPR are valid
 * LHS expressions. The MODIFY_EXPR might come from expressions like
 * a = (b = 12);
 * 
 * If the input is an IDENTIFIER_NODE, the IDENTIFIER_NODE is converted
 * into a VAR_DECL tree node, otherwise the original tree node is returned
 */
tree
verify_and_convert_lhs (tree expr, calc_location_t *expr_loc)
{
  if (TREE_CODE(expr) == IDENTIFIER_NODE)
    {
      symtab_entry_t *sym = XNEW(struct symtab_entry);

      sym->name = expr;
      if (htab_find(calc_sym_table, sym) == HTAB_EMPTY_ENTRY)
        {
          tree decl = build_decl(BUILD_LOCUS_INFO(expr_loc), VAR_DECL, expr, integer_type_node);
          sym->decl = decl;

          /* Add the VAR_DECL into the symbol table */
          PTR *slot = htab_find_slot(calc_sym_table, sym, INSERT);
          *slot = sym;
          
          VEC_safe_push(tree,gc,main_block_decls,decl);

          return decl;
        }
      else
        {
          return sym->decl;
        }
    }
  else if (TREE_CODE(expr) == MODIFY_EXPR)
    {
      /* If we got a MODIFY_EXPR, we need to add the MODIFY_EXPR to the statement list and
       * convert the MODIFY_EXPR into the VAR_DECL of the LHS expression
       */

      /* Maybe, needs to be fixed. Will check if it's possible to put a MODIFY_EXPR into a
       * MODIFY_EXPR
       */
      return expr;
    }

  return error_mark_node;
}

/* Type checking function for the RHS of an assign statement. An IDENTIFIER_NODE is
 * checked against the symbol table and the corresponding VAR_DECL is returnd, if available.
 * Otherwise an error is thrown. In all other cases, the input tree node is 
 * just returned
 */
tree
verify_and_convert_rhs (tree expr, calc_location_t *expr_loc)
{
  if (TREE_CODE(expr) == IDENTIFIER_NODE)
    {
      symtab_entry_t sym, *res;

      sym.name = expr;
      if ((res = (symtab_entry_t*)htab_find(calc_sym_table, &sym)) == HTAB_EMPTY_ENTRY)
        {
          error_at(BUILD_LOCUS_INFO(expr_loc), "variable %q+E isn't yet defined", expr);
          exit(FATAL_EXIT_CODE);
        }
      else
        return res->decl;
    }

  return expr;
}

/* Wrapper for the MODIFY_EXPR, e.g. LHS = RHS */
tree
build_assign_expr (tree lhs, calc_location_t *lhs_loc, tree rhs, calc_location_t *rhs_loc ATTRIBUTE_UNUSED)
{
  lhs = verify_and_convert_lhs(lhs, lhs_loc);
  rhs = verify_and_convert_rhs(rhs, lhs_loc);

  if (lhs != error_mark_node && rhs != error_mark_node)
    {
      tree expr = build2(MODIFY_EXPR, integer_type_node, lhs, rhs);
      SET_EXPR_LOCATION(expr, BUILD_LOCUS_INFO(lhs_loc));

      return expr;
    }

  return error_mark_node;
}

/* Wrapper for the MULT_EXPR, e.g. FAC1 * FAC2 */
tree 
build_mult_expr (tree factor1, calc_location_t *fac1_loc, tree factor2, calc_location_t *fac2_loc)
{
  factor1 = verify_and_convert_rhs(factor1, fac1_loc);
  factor2 = verify_and_convert_rhs(factor2, fac2_loc);

  if (factor1 != error_mark_node && factor2 != error_mark_node)
    {
      tree expr = build2(MULT_EXPR, integer_type_node, factor1, factor2);
      SET_EXPR_LOCATION(expr, BUILD_LOCUS_INFO(fac1_loc));

      return expr;
    }

  return error_mark_node;
}

/* Wrapper for the DIV_EXPR, e.g. DIV1 / DIV2. This is integer
 * division only. We'll also implement a runtime check for the
 * divisor (zero-check)
 *
 * Note: results will loose precision if we do 4/2 = 2; but if we
 * have 3/2 we will get 1.5 which will be rounded to the nearest integer
 */
tree 
build_div_expr (tree dividend, calc_location_t *dividend_loc, tree divisor, calc_location_t *divisor_loc)
{
  dividend = verify_and_convert_rhs(dividend, dividend_loc);
  divisor = verify_and_convert_rhs(divisor, divisor_loc);

  if (dividend != error_mark_node && divisor != error_mark_node)
    {
      tree div_expr = build2(TRUNC_DIV_EXPR, integer_type_node, dividend, divisor);

      if (flag_dynamic_div_check)
        {
          /* Whenever the dynamic check is enabled, the compiler will add code to check the value
          * of the divisor before the divison is executed. If the value is zero, the compiler
          * will add a call to an abort function in the runtime environment of gcalc
          */
          tree tmp_var = build_decl(BUILTINS_LOCATION, VAR_DECL, NULL_TREE, integer_type_node);
          DECL_IGNORED_P(tmp_var) = true;
          DECL_ARTIFICIAL(tmp_var) = true;

          /* the comparison expression */
          tree comp_expr = build2(NE_EXPR, integer_type_node, divisor, build_int_cst(integer_type_node, 0));

          /* the COND_EXPR basically allows to have a (return) type, but due to the fact that we generate a
           * function call to a runtime library function in the else part, we'll create a temporary variable 
           * and assign to a value in both cases.
           */

          /* the if-part */
          tree if_stmts = alloc_stmt_list();
          tree if_mod_expr = build2(MODIFY_EXPR, integer_type_node, tmp_var, div_expr);
          append_to_statement_list(if_mod_expr, &if_stmts);

          /* the else-part */
          tree else_stmts = alloc_stmt_list();
          /* we generate an expression assigning '0', but this will never be executed because the runtime call
           * will gracefully exit
           */
          tree else_mod_expr = build2(MODIFY_EXPR, integer_type_node, tmp_var, build_int_cst(integer_type_node, 0));

          if (file_name_string == NULL_TREE)
            {
              char *str = xstrdup(in_fnames[0]);
              file_name_string = build_string(strlen(str), str);
              TREE_TYPE(file_name_string) = build_array_type(char_type_node, build_index_type(build_int_cst(NULL_TREE, strlen(str) + 1)));
            }

          tree params = NULL_TREE;
          chainon( params, tree_cons (NULL_TREE, build_pointer_type(char_type_node), NULL_TREE) );                           
          chainon( params, tree_cons (NULL_TREE, integer_type_node, NULL_TREE) );                           

          tree fntype = build_function_type( void_type_node, params );
          tree exc_call = build_decl( UNKNOWN_LOCATION, FUNCTION_DECL,                                    
                            get_identifier("gcalc_runtime_exception"), fntype);
                                                                        
          DECL_EXTERNAL( exc_call ) = true;                                                               
          TREE_PUBLIC( exc_call ) = true;                                     

          //tree * args_vec = XNEWVEC( tree, 2);
          //args_vec[0] = name_string;
          //args_vec[1] = build_int_cst(integer_type_node, dividend_loc->line);
          //tree call_expr = build_call_expr_loc_array( BUILTINS_LOCATION, exc_call, 2, args_vec );

          //tree call_expr = build_call_expr(exc_call, 2, name_string, build_int_cst(integer_type_node, dividend_loc->line));
          tree addr_expr = build1(ADDR_EXPR, build_pointer_type(char_type_node), file_name_string);
          tree call_expr = build_call_expr(exc_call, 2, addr_expr, build_int_cst(integer_type_node, dividend_loc->line));

          append_to_statement_list(call_expr, &else_stmts);
          append_to_statement_list(else_mod_expr, &else_stmts);

          /* The else-part is currently unset, but needs to be fixed once the interface to the runtime
           * environment is available */
          tree cond_expr = build3(COND_EXPR, void_type_node, comp_expr, if_stmts, else_stmts);
          append_to_statement_list(cond_expr, &main_stmts);

          return tmp_var;
        }

      return div_expr;
    }

    return error_mark_node;
}

/* Wrapper for the SUB_EXPR, e.g. SUB1 - SUB1 */
tree 
build_sub_expr (tree minuend, calc_location_t *min_loc, tree subtrahend, calc_location_t *sub_loc)
{
  minuend = verify_and_convert_rhs(minuend, min_loc);
  subtrahend = verify_and_convert_rhs(subtrahend, sub_loc);

  if (minuend != error_mark_node && subtrahend != error_mark_node)
    {
      tree expr = build2(MINUS_EXPR, integer_type_node, minuend, subtrahend);
      SET_EXPR_LOCATION(expr, BUILD_LOCUS_INFO(min_loc));

      return expr;
    }
                    
  return error_mark_node;
}

/* Wrapper for the ADD_EXPR, e.g. ADD1 + ADD2 */
tree 
build_add_expr (tree summand1, calc_location_t *sum1_loc, tree summand2, calc_location_t *sum2_loc)
{
  summand1 = verify_and_convert_rhs(summand1, sum1_loc);
  summand2 = verify_and_convert_rhs(summand2, sum2_loc);

  if (summand1 != error_mark_node && summand2 != error_mark_node)
    {
      tree expr = build2(PLUS_EXPR, integer_type_node, summand1, summand2);
      SET_EXPR_LOCATION(expr, BUILD_LOCUS_INFO(sum1_loc));

      return expr;
    }

  return error_mark_node;
}

/**
 * Build a runtime call to 'extern void gcalc_runtime_print_call (int, ...)'
 *
 * @param loc - location_t of the call to print
 * @param n - number of arguments
 * @param args - vector of tree's of arguments
 **/
tree
build_print_expr (location_t loc, int n, VEC(tree,gc) * args )
{
  tree retval = NULL_TREE;
  if( args )
    {
      tree params = NULL_TREE;
      /* Parameters always enclosed with the void_type_node, to signify end */
      chainon( params, tree_cons (NULL_TREE, integer_type_node, NULL_TREE) );
      chainon( params, tree_cons (NULL_TREE, va_list_type_node, NULL_TREE) );
      chainon( params, tree_cons (NULL_TREE, void_type_node, NULL_TREE) );

      tree fntype = build_function_type( void_type_node, params );
      tree print_call = build_decl( UNKNOWN_LOCATION, FUNCTION_DECL,
				    get_identifier("gcalc_runtime_print_call"),
				    fntype );

      tree restype = TREE_TYPE( print_call );
      tree resdecl = build_decl( UNKNOWN_LOCATION, RESULT_DECL, NULL_TREE,
				 restype );

      DECL_CONTEXT( resdecl ) = print_call;
      DECL_RESULT( print_call ) = resdecl;
      DECL_EXTERNAL( print_call ) = true;
      TREE_PUBLIC( print_call ) = true;

      tree * args_vec = XNEWVEC( tree, n+1 );
      int idx = 0, idy = 1; tree itx = NULL_TREE; 

      args_vec[0] = build_int_cst( integer_type_node, n );

      for( ; VEC_iterate(tree,args,idx,itx); ++idx )
	      {
	        gcc_assert( itx );
	        if( TREE_CODE(itx) == IDENTIFIER_NODE )
	          {
              symtab_entry_t sym, *res;

              sym.name = itx;
	            if( (res = (symtab_entry_t*)htab_find(calc_sym_table, &sym)) == HTAB_EMPTY_ENTRY )
		          {
		            error("variable %q+E isn't yet defined", itx);
		            exit(FATAL_EXIT_CODE);
		          }
	            args_vec[idy] = res->decl;
	           }
          else
	          args_vec[idy] = itx;

	        idy++;
        } 

      retval = build_call_expr_loc_array( loc, print_call, n+1, args_vec );
      SET_EXPR_LOCATION(retval, loc);

    }
  return retval;
}
