/* libprofile.h  -*- C++ -*- */

#ifndef _LIB_PROFILE_H_
#define _LIB_PROFILE_H_

#define READ 0			/*if address read */
#define WRITE 1			/*if address written */
#define NADDR 100000		/*number of unique address to be stored */
#define OK 1
#define SYSERR -1

#include "uthash.h"
#include <string.h>
#include "utlist.h"
#include <utility>
#include <hash_map>
#include <list>
#include <map>
#include <tr1/unordered_map>
#include <set>
#include <unistd.h>
#include <sys/times.h>
#include <assert.h>


//#define DEBUG
#define TIMING
//#define SAMPLING


#define BZERO(b,len) (memset((b), '\0', (len)), (void) 0)

using namespace std;
using namespace __gnu_cxx;

#define TIME_VEC_NUM 10
extern long long profile_count;

extern clock_t Time_Profile[TIME_VEC_NUM];
extern struct tms tmsstart[TIME_VEC_NUM];
extern struct tms tmsend[TIME_VEC_NUM];
extern clock_t tstart[TIME_VEC_NUM];
extern clock_t tend[TIME_VEC_NUM];
extern clock_t total_start;
extern clock_t total_end;


inline void
count_time_start (int i)
{
  tstart[i] = times (&tmsstart[i]);
	Time_Profile[i] = 0;
}

inline void
count_time_end (int i)
{
  tend[i] = times (&tmsend[i]);
  Time_Profile[i] = (tend[i] - tstart[i]);
}


#include "thread_pool.h"


/* Data dependence type.  */

enum dep_type
{
  /* Read After Write (RAW).  */
  dep_RAW = 0x2,

  /* Write After Read (WAR).  */
  dep_WAR = 0x4,

  /* Write After Write (WAW).  */
  dep_WAW = 0x8,

  /* Read After Read (RAR).  */
  dep_RAR = 0x10,

  /* No Dependence */
  dep_NO = 0x20,
};


/* Data dependence kind.  */

enum dep_kind
{
  /* Loop-independent.  */
  LOOP_INDEPENDENT = 0x1,

  /* Loop-carried.  */
  LOOP_CARRIED = 0x2
};


/* A ipa_data_dependence_relation represents a relation from 
   ipa_data_reference A to B.  
   For a loop-independent relation, B is ahead of A in lexicographic order.
   For a loop-carried relation, A is in the iteration after which B is in.    */

class DEPENDENCY
{
  public:

	int _a;		/* The id of the statement depending on B */
  int _b;			/* The id of the statement being depended on */
  int _loop_id;
  dep_type _type;
  dep_kind _kind;
	int _distance;


    DEPENDENCY (int a, int b, dep_kind k, dep_type t, int loop_id, int dis=0):_a (a),
    _b (b), _kind (k), _type (t), _loop_id (loop_id), _distance(dis)
  {
  }

  int a () const
  {
    return _a;
  }
  int b () const
  {
    return _b;
  }
  dep_kind kind () const
  {
    return _kind;
  }
  dep_type type () const
  {
    return _type;
  }
  int loop_id () const
  {
    return _loop_id;
  }

  void set_a (int a)
  {
    _a = a;
  }
  void set_b (int b)
  {
    _b = b;
  }
  void set_kind (dep_kind k)
  {
    _kind = k;
  }
  void set_type (dep_type t)
  {
    _type = t;
  }
  void set_loop_id (int i)
  {
    _loop_id = i;
  }

  const DEPENDENCY &operator= (const DEPENDENCY & d) 
  {
    _a = d._a;
    _b = d._b;
    _kind = d._kind;
		_type = d._type;
		_loop_id = d._loop_id;
		_distance = d._distance;
    return *this;
  }

  bool operator== (const DEPENDENCY & d) const
  {
    if (_a != d._a)
      return false;
    if (_b != d._b)
      return false;
    if (_kind != d._kind)
      return false;
    if (_type != d._type)
      return false;
    if (_loop_id != d._loop_id)
      return false;
    return true;
  }

  bool operator< (const DEPENDENCY & d) const
  {
    if (_a < d._a)
      return true;
    if (_b < d._b)
      return true;
    if (_kind < d._kind)
      return true;
    if (_type < d._type)
      return true;
    if (_loop_id < d._loop_id)
      return true;
    return false;
  }

  void print (FILE *fp, int indent = 0) const
  {
    for (int i = 0; i < indent; i++)
      fprintf (fp, " ");

    fprintf (fp, "LOOP %d  ", _loop_id);

    switch (_type)
    {
      /* Read After Write (RAW).      */
      case dep_RAW:
        fprintf (fp, "RAW  ");
        break;

      /* Write After Read (WAR).      */
      case dep_WAR:
        fprintf (fp, "WAR  ");
        break;

      /* Write After Write (WAW).  */
      case dep_RAR:
        fprintf (fp, "RAR  ");
        break;

      case dep_WAW:
        fprintf (fp, "WAW  ");
        break;

      default:
        fprintf (fp, "ERROR : ");
      break;
    }

    fprintf (fp, "%d --> %d	  ", _a, _b);


    if (_kind == LOOP_INDEPENDENT)
      fprintf (fp, "LOOP INDEPENDENT\n");
    else
      fprintf (fp, "LOOP CARRIED \n");

  }


  void print (const char *fname, int indent = 0) const
  {

		FILE *fp = fopen (fname, "w+");
		print (fp);
		fclose (fp);
  }

};




class DEPENDENCY_VEC : public std::list < DEPENDENCY >
{
public:

  void
  Insert (const DEPENDENCY & d)
  {
    push_back (d);
  }

  bool
  Find (const DEPENDENCY & d)
  {
    for (const_iterator iter = begin (); iter != end (); ++iter)
      if (*iter == d)
				return true;
    return false;
  }

  void
  Unique ()
  {
    std::set < DEPENDENCY > tmp_set;
    iterator   iter = begin ();
    while (iter != end ())
    {
			if (tmp_set.insert (*iter).second)
			  ++iter;
			else
			  iter = erase (iter);
    }
  }

  void
  print (FILE * fp, int indent) const
  {
    for (const_iterator iter = begin (); iter != end (); ++iter)
      iter->print (fp, indent);
  }

};



/* structure for hash table <key,value> */
struct dep_hash_entry 
{
	int memop_id;		/* The statement id */
	DEPENDENCY_VEC dependencies; 
	UT_hash_handle hh;

	
	dep_hash_entry () :
		 memop_id(0),
		 dependencies()
	 { }
};





/* Structure of a memory operation */
class MEMORY_OPERATION
{
private:

  int _id;			/* The mem op id */
  int _iterno:    30;				/* The current iteration  */
  unsigned _rwflag: 1;				/*if address read (0) or written (1) */
  unsigned _killed: 1;				/*if address read (0) or written (1) */

public:
  MEMORY_OPERATION ()
  {
  	_killed = 0;
  }

  MEMORY_OPERATION (int id, int iterno, int rwflag):
  _id (id),
  _iterno (iterno),
  _rwflag (rwflag),
  _killed (0)
  {
  }

  int   id () const  { return _id;  }
  int   iterno () const   {     return       _iterno;  }
  int 	rwflag () const  {    return      _rwflag;  }
  int 	killed () const  {    return      _killed;  }

  void  set_iterno (int i)  {    _iterno = i;  }
  void  set_killed ()  {    _killed = 1;  }

  bool
  operator== (const MEMORY_OPERATION & m) const
  {
    return
      _id == m._id;
  }

  bool
  operator< (const MEMORY_OPERATION & m) const
  {
    return
      _id <
      m.
      _id;
  }

  void
  print (FILE * fp) const
  {
    fprintf (fp, "id = %d, iter = %d, write =%d\n", _id, _iterno, _rwflag);
  }

};


class MEMORY_OPERATION_LIST : public std::list <MEMORY_OPERATION>
{
public:
	int Size() const { return size(); }

	void print(FILE *fp) const 
	{
		fprintf(fp, "size=%d\n", size());
		for (const_iterator iter=begin(); iter != end(); ++iter)
			iter->print(fp);
	}

	bool Find(int id) const
		{
			for (const_iterator iter = begin(); iter != end(); ++iter )
				if (iter->id() == id )
					return true;
			return false;
		}
	
};



#define PTR char*

/* structure for hash table <key,value> */
struct hash_entry
{
  PTR     addr;			/* memory address */
  MEMORY_OPERATION     memop;			/* The latest memory operation */
  DEPENDENCY_VEC    dep_list;
  UT_hash_handle    hh;

  hash_entry ():
  addr (0),
  dep_list ()
  {
  }

  void
  operator= (const hash_entry & h)
  {
    addr = h.addr;
    memop = h.memop;
    dep_list = h.dep_list;
  }

};



struct not_check_dep;

class ID_SET:  public std::set < int >
{

public:
  bool
  Find (int id) const
  {
    return
    find (id) !=
    end ();
  }

  void
  Insert (int id)
  {
    insert (id);
  }

  void
  Union (const ID_SET & s)
  {
    insert (s.begin (), s.end ());
  }

  void
  Union (const not_check_dep * s);


};





struct not_check_dep
{
  int
    len;			/* the length of context loop vector */
  int
    loop_id[1];			/* The loop id vector */
};



/* Map loop to source file info */
struct loop_info
{
  int
    itercount;			/* total iteration number */
};



void
ID_SET::Union (const not_check_dep * s)
{
  if (s->len == 0)
    return;
  insert (&(s->loop_id[0]), &(s->loop_id[s->len]));
}



FILE *fp;


/* structure for hash table <key,value> */
struct pair_hash_entry
{
 PTR  addr;		/* memory address */
 MEMORY_OPERATION_LIST 	memops; 	/* The latest statement */
 UT_hash_handle hh;

 pair_hash_entry () :
 		addr(0),
		memops()
 	{ }		
};

 
enum SD3_STATUS {
  SD3_START = 0,
  SD3_FIRST,
  SD3_LEARNED,
  SD3_WEAK,
  SD3_STRONG,
  SD3_SET,
  SD3_LAST
};





class SD3_STRIDE 
{
public:
  PTR low;
  PTR high;
  ptrdiff_t distance;		// must >= 0
  ptrdiff_t num_strides;
  ptrdiff_t jump_distance;		// must >= 0
  int size; // bit size of memory reference

  int stmt;
  SD3_STATUS status;
  unsigned int read : 1;
  unsigned int killed: 1;

  void init (PTR low = 0, PTR high = 0,
             ptrdiff_t distance = 0, int size = 0,
             SD3_STATUS status = SD3_START, int read = 1, int write = 0,
             int loop_id = -1, int stmt = -1, int ref = -1);

  SD3_STRIDE (void) : status (SD3_START) { init (); }
  SD3_STRIDE (PTR low, PTR high, ptrdiff_t distance, int size = 0,
              SD3_STATUS status = SD3_START, int read = 1, int write = 0,
              int loop_id = -1, int stmt = -1, int ref = -1) {
    init (low, high, distance, size, status, read, write, loop_id, stmt, ref);
  }
  SD3_STRIDE (PTR addr, int size, int read, int loop_id, int stmt, int ref) {
    init (addr, addr, 0, size, SD3_FIRST, read, 0, loop_id, stmt, ref);
  }
  
  bool operator == (const SD3_STRIDE& x) const
  {
    return (low == x.low
            && high == x.high
            && distance == x.distance
            && stmt == x.stmt);
  }

  bool operator < (const SD3_STRIDE& x) const
  {
    return (high < x.low);
  }

	
  bool overlaps (const SD3_STRIDE& x) const
  {
    return  !(*this < x ) && !( x < *this);
  }

  bool contains (PTR addr) const
  {
    return  (low <= addr ) && ( high >= addr);
  }

  bool empty () const
  {
    return  (low ==0) && ( high==0);
  }

  PTR get_low (void) const { return low; }
  void set_low (PTR low) { this->low = low; }
  PTR get_high (void) const { return high; }
  void set_high (PTR high) { this->high = high; }
  ptrdiff_t get_distance (void) const { return distance; }
  void set_distance (ptrdiff_t distance) { this->distance = distance; }
  ptrdiff_t get_num_strides (void) const { return num_strides; }
  void set_num_strides (ptrdiff_t num_strides) { this->num_strides = num_strides; }
  int get_size (void) { return size; }
  void set_size (void) { this->size = size; }
  
  int get_stmt (void) const { return stmt; }
  void set_stmt (int stmt) { this->stmt = stmt; }

  SD3_STATUS get_status (void) const { return status; }
  void set_status (SD3_STATUS status) { this->status = status; }
  
  int get_flag_read (void) const { return read; }
  void set_flag_read (int read) { this->read = read; }
  
  bool check_addr (PTR addr, int size = 0, int stmt = -1, int ref = -1) const;
  bool add_addr (PTR addr, int size = 0, int stmt = -1, int write = 0);
  bool merge (const SD3_STRIDE& x);

	void split (SD3_STRIDE& x, std::multiset <SD3_STRIDE> &strides) const;
  bool compute_dependence (SD3_STRIDE& x, dep_kind dkind, int dloop) const;
  bool check_and_reduce (SD3_STRIDE& x, dep_kind dkind, int dloop, std::multiset <SD3_STRIDE> &strides) const;
};


typedef std::multiset <SD3_STRIDE> SD3_STRIDE_SET;



//typedef std::tr1::unordered_map < address, STMT_SET, hash_address, equal_address > ADDRESS_HASH;


enum PROFILE_KIND
{
  pk_not_profile,		// nothing to be done
  pk_profile,			// record address       only
  pk_check_dependence,		// record address and check dependence 
};

typedef std::map <int, SD3_STRIDE_SET> STRIDE_MAP;


/*structured for storing currently open loops and its iteration numbers */
class   lentry
{
public:
  int     loop_id;
  unsigned int     iter_no;
  PROFILE_KIND    kind;

  hash_entry 			*hashaddr;
  pair_hash_entry	*pair_hash;

	pair_hash_entry *history_point_table;
	pair_hash_entry *pending_point_table;

	STRIDE_MAP history_stride_table;
	STRIDE_MAP pending_stride_table;

	
  lentry ()
 	 :loop_id (0), iter_no (0), hashaddr (NULL), kind (pk_check_dependence), pair_hash(NULL),
 	 history_point_table (NULL), pending_point_table (NULL),
 	 history_stride_table (), pending_stride_table ()
	 {
	 }
	 


  void
  print (FILE * fp) const
  {
    fprintf (fp, "LOOP %d  ITER	%d \n", loop_id, iter_no);
  }

  const SD3_STRIDE* check_stride (PTR addr, int size = 0,
                                  int stmt = -1, int write = 0);
  void insert_stride (const SD3_STRIDE& stride);
	bool merge_stride (SD3_STRIDE_SET &history, SD3_STRIDE_SET &pending);
	void merge_stride_table (void);
  void merge_point_table (void);
  void merge_pending_table (void);
	void check_loop_independent_killed (SD3_STRIDE& stride, SD3_STRIDE_SET &strides);
	void check_loop_independent_dependence (SD3_STRIDE& stride);
	void check_loop_independent_dependence (PTR addr, MEMORY_OPERATION &new_stmt, MEMORY_OPERATION_LIST &stmts);
	void check_loop_carried_dependence (SD3_STRIDE& stride);
	void check_loop_carried_dependence (PTR addr, MEMORY_OPERATION &new_stmt, MEMORY_OPERATION_LIST &stmts);
  void compute_dependence (void);
	bool compute_loop_independent_edge 	(SD3_STRIDE& stride, DEPENDENCY *dep)	;
	bool compute_loop_carried_edge(SD3_STRIDE& stride, DEPENDENCY *dep);

};


static const int SIZE = 100;
static lentry **loop_stack;	/* Start from 1 */
static unsigned int loop_stack_capacity = 0;
static unsigned int sp = 0;
static int initialized = 0;
static unsigned long long pc = 0;	// the current program counter



/* start from 1 */
static inline int
Empty (void)
{
  return sp == 0;

}

static inline int
Length (void)
{
  return sp;
}

static inline void
Push (lentry * loop)
{
  loop_stack[++sp] = loop;
}

static inline void
Pop (void)
{
  delete loop_stack[sp];
  if (sp)
    sp--;
}

static inline lentry *
Top ()
{
  return loop_stack[sp];
}

static inline lentry *
Bottom (void)
{
  return loop_stack[1];
}


static inline lentry *
Elem (int i)
{
  return loop_stack[i];
}

static int
Contains (int loop_id)
{
  int i;
  for (i = sp; i > 0; i--)
    if (Elem (i)->loop_id == loop_id)
      return i;
  return 0;
}


class LOOP_DEPENDENCY
{
private:
  int _loop_id;
  DEPENDENCY_VEC _dependencies;
public:

  LOOP_DEPENDENCY (int loop_id):
  _loop_id (loop_id)
  {
  }

  int loop_id () const   {     return  _loop_id;   }
  DEPENDENCY_VEC &dependencies ()   {     return _dependencies;  }


  void print (FILE * fp, int indent) const
  {
    _dependencies.print (fp, indent);
  }

};


class
  LOOP_DEPENDENCICES:
  public
  std::list <
  LOOP_DEPENDENCY >
{

  public:
    bool
  Insert (const DEPENDENCY & new_dep, bool check_redundancy = false)
  {
    for (iterator iter = begin (); iter != end (); ++iter)
      {
	if (iter->loop_id () == new_dep.loop_id ())
	  {
	    DEPENDENCY_VEC & deps = iter->dependencies ();
	    if (check_redundancy)
	      {
		for (DEPENDENCY_VEC::iterator diter = deps.begin ();
		     diter != deps.end (); ++diter)
		  {
		    if (new_dep == *diter)
		      return false;
		  }
	      }
	    deps.
	    push_back (new_dep);
	    return true;
	  }
      }

    LOOP_DEPENDENCY
    ldep (new_dep.loop_id ());
    ldep.dependencies ().push_back (new_dep);
    push_back (ldep);
    return true;

  }

  void
  print (FILE * fp, int indent) const
  {
    for (const_iterator iter = begin (); iter != end (); ++iter)
      iter->
      print (fp, indent);
  }

};


// < loop-independent, loop-carried >
typedef std::pair <LOOP_DEPENDENCICES,  LOOP_DEPENDENCICES >  LOOP_DEPENDENCICES_PAIR;


class   MEMORY_OPERATION_INFO
{
public:

  int     id;				/* The mem op id, e.g slice id */
	int			uid;
	int			slice_id;
	int     read;
	int 		stride_tolerate;
	bool 		strided;
  LOOP_DEPENDENCICES_PAIR     possible_dependencies;	//       possible dependencies from static analysis
  LOOP_DEPENDENCICES_PAIR     true_dependencies;		//       runtime dependencies

  MEMORY_OPERATION_INFO (int i):
  id (i),
	uid(0),
	strided(true),
	stride_tolerate(0),
	read(0),
	slice_id(0)
  {   }

};

typedef   std::tr1::unordered_map < int, MEMORY_OPERATION_INFO * >  MEMOP_INFO_HASH;

MEMOP_INFO_HASH  mem_info_hash;

class  MEMORY_BLOCK
{
public:

  PTR  low;			/* the lower memory address */
  PTR  high;			/* the upper memory address */
  int  id;				/* the unique id of each block */

  MEMORY_BLOCK ():
  low (0),
  high (0),
  id (0)
  {
  }

  MEMORY_BLOCK (PTR l, PTR h, int i):
  low (l),
  high (h),
  id (i)
  {
  }

  bool
  empty () const
  {
    return
      low == 0 && high == 0;
  }

  bool
  operator< (const MEMORY_BLOCK & m) const
  {
    return
      high <
      m.
      low;
  }

  bool
  operator> (const MEMORY_BLOCK & m) const
  {
    return
      m.
      high <
      low;
  }

  bool
  operator== (const MEMORY_BLOCK & m) const
  {
    return
      low == m.low && high == m.high;
  }

  bool
  contains (const MEMORY_BLOCK & m) const
  {
    return
      low <=
      m.
      low &&
      high >=
      m.
      high;
  }

  bool
  overlaps (const MEMORY_BLOCK & m) const
  {
    return !((*this) < m) && !(m < (*this));
  }

};

// the same key means overlapping
class  MEMOP_BLOCK_MAP:   public   std::set < MEMORY_BLOCK >
{
public:

	int  last_id;			// the last block id

  MEMOP_BLOCK_MAP ()
  {
    last_id = 0;
  }

};

// index by block id
typedef std::vector < const MEMORY_BLOCK *> MEMOP_BLOCK_VEC;

MEMORY_BLOCK   null_block;
MEMOP_BLOCK_MAP mem_block_map;
MEMOP_BLOCK_VEC mem_block_vec;


class
  LOOP_INFO
{
private:

  int
    id;				/* loop id */
  std::set < int >
    profile_mems;		/* The id of mem ops to be profiled */

public:
  LOOP_INFO (int i):
  id (i)
  {
  }

  void
  insert_mem_op (int i)
  {
    profile_mems.insert (i);
  }

  bool
  contain_mem_op (int i) const
  {
    return
      profile_mems.
    find (i) !=
      profile_mems.
    end ();
  }

};

std::set < int >
  not_profile_loops;
std::vector < LOOP_INFO > loops;

LOOP_INFO & get_loop_info (int i)
{
  return loops[i];
}



#endif
