/*   profile.c
 * 
 * The input parameters to be received from original source are:
 * 1. char *addr, 2. loop iteration vector, 4. read/write flag
 * 
 * For each address string, insert into hash table. Key is address string and Value is unique incremented counter.
 * The counter gives the index of the address string in the address array. Each entry in the array is of type 
 * 'struct address', which has a pointer to 'struct loop'. Each struct loop has 2 pointers, 1 for all iterations
 * in the loop and other for next loop which has the same address (inner loop).
 *  
 *    hash table
 *  -------------    struct address
 *  |    |      |    ----------------
 *  |KEY1|INDEX1|    | | | | |
 *  |KEY2|INDEX2|    ----------------
 *  |    |      |      struct address -> struct loop -> struct iteration
 *  -------------    
 * 
 *  function-names <change the fn names later>
 *  1. print_name: name of the variables
 *  2. print_address: address of the variables, 
 *  3. print_edge: loop entry / loop exit
 *  4. print_exit: exit function
 *  
 *  loop and address are recorded separately; either change in profiling or mainitain a linked list/array;
 *  iteration number is counted explicitly on every loop entry function call.
 */



#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
//#include <limits.h>
#include "libprofile.h"
#include "thread_pool.h"
#include "libalias.cxx"
#include "libcount.cxx"




clock_t Time_Profile[TIME_VEC_NUM];
struct tms tmsstart[TIME_VEC_NUM];
struct tms tmsend[TIME_VEC_NUM];
clock_t tstart[TIME_VEC_NUM];
clock_t tend[TIME_VEC_NUM];
clock_t total_start;
clock_t total_end;
static long clktck;
long long profile_count = 0;
static long long stmt_count = 0;
static long long addr_count = 0;
static int max_dep_num = 0;
static int ave_dep_num = 0;


int firstent = 0;		/* global variable to call init() function the first time */
Thread *threadPool;
static pthread_spinlock_t hashdepSpin;


int all_not_profile = 0;	/* set 1 if all loops in current stack need not to be profiled */
const char *npname = "noprofile.info";
const char *loopname = "loop.info";
char *depname = "dependencies.info";
char *errname = "error.info";

/* Generate loop profiling tables recording which memories need to be profiled */
void
generate_loop_table ()
{
  FILE *loopfp = fopen (loopname, "r");

  if (!loopfp)
    return;


  std::set < int >loop_set;

  char *line = NULL;
  size_t len = 0;
  int bytes_read;
  int loop_id;
  char str[100];

  while ((bytes_read = getline (&line, &len, loopfp)) != -1)
    {
      sscanf (line, "%s", str);

      if (strcmp (str, "LOOP") == 0)
	{
	  sscanf (line, "%*s %d", &loop_id);
	  loop_set.insert (loop_id);
	}
    }



  fclose (loopfp);

	LOOP_INFO loop (0);
	loops.push_back (loop);


  int i = 1;
  for (std::set < int >::iterator iter = loop_set.begin ();
       iter != loop_set.end (); ++iter)
    {
      assert (i == *iter);
      LOOP_INFO loop (i);
      loops.push_back (loop);
      ++i;
    }

}

void
loop_insert_mem_op_to_profile (int loop_id, int memop_id)
{
  loops[loop_id].insert_mem_op (memop_id);
}


/* If LOOPFP is NULL, profile for all loops */
void
generate_not_profile_loops ()
{

  FILE *loopfp = fopen (npname, "r");

  if (!loopfp)
    return;


  char *line = NULL;
  size_t len = 0;
  int bytes_read;
  char str[100];
  int loop_id;
  char kind[100];


  while ((bytes_read = getline (&line, &len, loopfp)) != -1)
    {
      sscanf (line, "%s", str);

      if (strcmp (str, "LOOP") == 0)
	{
	  sscanf (line, "%*s %d	 %s", &loop_id, kind);
	  not_profile_loops.insert (loop_id);
	}
    }



  fclose (loopfp);

  for (std::set < int >::iterator iter = not_profile_loops.begin ();
       iter != not_profile_loops.end (); ++iter)
    fprintf (fp, "LOOP %d won't be profiled\n", *iter);

}

static int profiled_loop = 0;

void
generate_memory_op_table (const char *memname)
{



  FILE *memfp = fopen (memname, "r");

  if (!memfp)
    {
      fprintf (stderr, "No file %s found!\n", memname);
      abort ();
    }


  /*
     The expected file format is 

     MEMOP 1      ....
     POSSIBLE INDEPENDENT
     LOOP 1       WAW  1 --> 5    MUST    LOOP INDEPENDENT        
     LOOP 2       RAW  1 --> 6    MAY             LOOP INDEPENDENT
     POSSIBLE CARRIED
     LOOP 3       WAR  1 --> 7    MUST    LOOP CARRIED    
     LOOP 4       RAW  1 --> 8    MAY             LOOP CARRIED

   */

  char *line = NULL;
  size_t len = 0;
  int bytes_read;
  MEMORY_OPERATION_INFO *mem_op = NULL;
  char str_loop[100];
  int loop_id;
  char str_type[100];
  int a, b;
  char str_must[100];

  /* 1 - loop-independent;  2 - loop-carried */
  dep_kind kind;

  while ((bytes_read = getline (&line, &len, memfp)) != -1)
  {
    char str[100];
    sscanf (line, "%s", str);

    if (strcmp (str, "LOOPID") == 0)
    {
      // ignore "LOOPID", go on reading memop_id
      int loop_id;
      sscanf (line, "%*s %d", &loop_id);
      profiled_loop = loop_id;
    }

    else if (strcmp (str, "MEMOP") == 0)
    {
      // ignore "MEMOP", go on reading memop_id
      int mem_id;
      sscanf (line, "%*s %d", &mem_id);
      mem_op = new MEMORY_OPERATION_INFO (mem_id);
      mem_info_hash[mem_id] = mem_op;

    }
    else if (strcmp (str, "UID") == 0)
    {
      // ignore "UID", go on reading memop_id
      int mem_id;
      sscanf (line, "%*s %d", &mem_id);
      mem_op->uid = mem_id;
    }
    else if (strcmp (str, "SLICE") == 0)
    {
      // ignore "SLICE", go on reading memop_id
      int mem_id;
      sscanf (line, "%*s %d", &mem_id);
      mem_op->slice_id = mem_id;
    }
    else if (strcmp (str, "READ") == 0)
    {
      mem_op->read = 1;
    }
    else if (strcmp (str, "WRITE") == 0)
    {
      mem_op->read = 0;
    }

    else if (strcmp (str, "POSSIBLE") == 0)
    {
      sscanf (line, "%*s %s", str);
      if (strcmp (str, "INDEPENDENT") == 0)
      {
        // possible loop-independent dependencies
        kind = LOOP_INDEPENDENT;

      }

      else if (strcmp (str, "CARRIED") == 0)
      {
        // possible loop-carried dependencies
        kind = LOOP_CARRIED;

      }
    }

    else if (strcmp (str, "LOOP") == 0)
    {
      /* Try to recognize the form :          LOOP 86  WAW    576 --> 562     MAY */
      sscanf (line, "%s %d %s %d %*s %d %s", str_loop, &loop_id, str_type,
    	  &a, &b, str_must);
      assert (kind == LOOP_INDEPENDENT || kind == LOOP_CARRIED);

      dep_type type = dep_NO;
      if (strcmp (str_type, "RAW") == 0)
        type = dep_RAW;
      else if (strcmp (str_type, "WAW") == 0)
        type = dep_WAW;
      else if (strcmp (str_type, "WAR") == 0)
        type = dep_WAR;
      else
        {
          fprintf (stderr, "Wrong dep kind\n");
          abort ();
        }


      if (strcmp (str_must, "MAY") == 0)
      {
        DEPENDENCY dep (a, b, kind, type, loop_id);
        if (kind == LOOP_INDEPENDENT)
        	mem_op->possible_dependencies.first.Insert (dep);
        else
      	{
      	  mem_op->possible_dependencies.second.Insert (dep);
      	}
      }

    }

  }



  fclose (memfp);

  if (line)
    free (line);

}



#include "hash_insert.h"

inline void
print_dependencies ()
{

  FILE *depfp = fopen (depname, "w");

  for (MEMOP_INFO_HASH::iterator iter = mem_info_hash.begin ();
       iter != mem_info_hash.end (); ++iter)
    {
      MEMORY_OPERATION_INFO *info = iter->second;
      LOOP_DEPENDENCICES & independent = info->true_dependencies.first;
      LOOP_DEPENDENCICES & carried = info->true_dependencies.second;

      if (independent.empty () && carried.empty ())
	continue;

      /*
         The expected file format is 

         MEMOP 1      ....
         POSSIBLE INDEPENDENT
         LOOP 1       WAW  1 --> 5            LOOP INDEPENDENT        
         LOOP 2       RAW  1 --> 6            LOOP INDEPENDENT
         POSSIBLE CARRIED
         LOOP 3       WAR  1 --> 7            LOOP CARRIED    
         LOOP 4       RAW  1 --> 8            LOOP CARRIED

         OR 

         MEMOP 1      ....
         POSSIBLE DEPENDENCIES UNAVAILABLE    

       */
      fprintf (depfp, "MEMOP %d\n", info->id);
      if (!independent.empty ())
	{
	  fprintf (depfp, "INDEPENDENT\n");
	  independent.print (depfp, 4);
	}
      if (!carried.empty ())
	{
	  fprintf (depfp, "CARRIED\n");
	  carried.print (depfp, 4);
	}



    }

  fclose (depfp);


}


static void
profile_initialize (void)
{

  initialized = 1;
  loop_stack_capacity = SIZE - 1;
  loop_stack = (lentry **) malloc (sizeof (lentry) * loop_stack_capacity);
  loop_stack[0] = new lentry;
  loop_stack[0]->kind = pk_not_profile;


  fp = stdout;
  //      fp = fopen ("profile.res", "w" );
  fprintf (fp, "START PROFILING ... \n");
  //      fclose(fp);

  generate_loop_table ();

  generate_not_profile_loops ();

  generate_memory_op_table ("memop.info");

  clktck = sysconf (_SC_CLK_TCK);


}


/* 	addr 	- address;
		size	- the byte size of memory malloced
		flag	-	0 -- malloc ; 1 -- realloc;	2 -- free
*/

extern "C" void
__print_malloc (PTR old_addr, PTR new_addr, int size, int flag)
{

  switch (flag)
    {
    case 0:			// malloc
      {
	MEMORY_BLOCK block (new_addr, new_addr + size - 1,
			    ++mem_block_map.last_id);
	MEMOP_BLOCK_MAP::iterator iter = mem_block_map.insert (block).first;
	mem_block_vec.push_back (&(*iter));
	break;
      }

    case 1:			// realloc
      {
	const MEMORY_BLOCK & old_block = get_mem_block (old_addr);
	int id = old_block.id;
	mem_block_map.erase (old_block);
	MEMORY_BLOCK new_block (new_addr, new_addr + size - 1, id);
	MEMOP_BLOCK_MAP::iterator iter =
	  mem_block_map.insert (new_block).first;
	mem_block_vec[id] = &(*iter);
	break;
      }

    case 2:			// free
      {
	const MEMORY_BLOCK & old_block = get_mem_block (old_addr);
	int id = old_block.id;
	mem_block_vec[id] = &null_block;
	mem_block_map.erase (old_block);
	break;
      }
    default:
      assert (false);
      break;
    }

}


/* 	addr 	- address;
		size	- the bit size of memory to be accessed
		flag	-	read/write, if address read (0) or written (1) ; 
		id		- memory operation id or slice id;
*/
extern "C" void
__print_address (PTR addr, int size, char flag, int id)
{

  if (!initialized)
    profile_initialize ();

  if (Top ()->kind == pk_not_profile)
    return;

  /*      perform hash operation and check dependence */
  MEMORY_OPERATION memop (id, Top ()->iter_no, flag);
  address_hash_insert (Top (), addr, memop);
  ++stmt_count;

}








/*entry-loop enter/exit; loc - loop global id; s -  */
extern "C" void
__print_edge (int loop_id,	/* The global loop id */
          	      int entry		/* 1- loop entry , 0- loop exit */
                  )
{
  if (!initialized)
    profile_initialize ();

  if (entry)
  {
    if (loop_stack_capacity = Length () + 2 )
    {
      loop_stack_capacity = (loop_stack_capacity << 1) - 1;
      loop_stack = (lentry **) realloc (loop_stack, sizeof (lentry*) * loop_stack_capacity);
    }

    if (Empty () || Top ()->loop_id != loop_id)
    {

      lentry *loop = new lentry;
      loop->loop_id = loop_id;
      loop->iter_no = 1;

      if (Empty ())
      {
        if (not_profile_loops.find (loop_id) !=   	  not_profile_loops.end ())
        	loop->kind = pk_not_profile;

        stmt_count = 0;
        ave_dep_num = 0;
//        fprintf (fp,
 // 	       "\n\n\n ======================== Entering LOOP NEST %d==========================\n",
//  	       loop_id);
      }
      else
      {
        if (Top ()->kind == pk_not_profile)
        	loop->kind = pk_not_profile;
        else if (not_profile_loops.find (loop_id) != not_profile_loops.end ())
      	{
      	  if (Top ()->kind == pk_not_profile)
      	    loop->kind = pk_not_profile;
      	  else
      	    loop->kind = pk_profile;	// profile but not need checking dependence                                             
      	}
      }

      Push (loop);
    }
    else
    {
      Top ()->iter_no++;
    }
  }
  else
  {

    lentry *loop = Top ();

    if (loop->loop_id == loop_id)
  	{
      // loop_infos[loop_id].itercount += loop->iter_no;

  	  /* transfer hash table */
  	  if (Length () > 1)
  	    transfer_loop (Elem (Length () - 1), loop);

  	  clear_loop (loop);
  	  Pop ();


  	  if (0 && Empty ())
	    {

	      fprintf (fp,
		       "\n ========================Exiting of LOOP NEST %d==========================\n",
		       loop_id);
	      fprintf (fp, "max_dep_num = %d\n", max_dep_num);
	      fprintf (fp, "ave_dep_num = %d\n", ave_dep_num);
	      fprintf (fp, "addr count = %lld\n", addr_count);
	      fprintf (fp, "stmt count = %lld\n", stmt_count);


	      fprintf (fp, "profile_count=%lld\n", profile_count);
	      fprintf (fp, "time for print range is : %7.4f sec\n",
		       Time_Profile[0] / (double) clktck);
	      fprintf (fp, "time for insert_and_merge is : %7.4f sec\n",
		       Time_Profile[1] / (double) clktck);
	      fprintf (fp, "time for transfer loop is : %7.4f sec\n",
		       Time_Profile[2] / (double) clktck);
	      fprintf (fp, "time for check dependence is : %7.4f sec\n",
		       Time_Profile[3] / (double) clktck);
	    }
  	}
  }

}



extern "C" void
__print_exit ()
{
  print_dependencies ();
}

extern "C"
{
  #include "libdefs.h"
  #include "libmempool.h"
}


#include "libsd3.cxx"
#include "libpairwise.cxx"
#include "libedge.cxx"
#include "libprivatize.cxx"
#include "libshadow.cxx"

static int *post;

extern "C" void
__thread_initialize(int len)
{
  post = new int[len];
}

extern "C" void
__thread_post(int i, int tid)
{
  post[i] = tid;
}

extern "C" void
__thread_wait(int i, int tid)
{
  while (post[i] != tid - 1 )
    ;
}


