/* -*- C++ -*- */


#include "libprofile.h"



#include <sys/mman.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <vector>
#include <setjmp.h>


#define MEMLEAK 1

MEM_POOL  *local_pool;

int time_limit = 20;  // minutes
bool time_start_p = false;

using namespace std;

enum vf_flags_t 
{
  VF_READ        = 0x1,	/* has been read */
  VF_WRITE       = 0x2  /* has been written */
};

class Virtual_File
{
public:
  vector<char> buffer;
  long  cur;  // current index, not visited yet  
  int   flags;

  Virtual_File() :
    cur(0),
    flags(0)
    {}

    
  bool  has_read () const { return flags & VF_READ; }
  void  set_read( ) { flags |= VF_READ;}
  void  reset_read( ) { flags &= ~VF_READ;}

  bool  has_write () const { return flags & VF_WRITE; }
  void  set_write( ) { flags |= VF_WRITE;}
  void  reset_write( ) { flags &= ~VF_WRITE;}
  
    
};


typedef map<FILE*, Virtual_File* > FILE_POINTER_MAP;
typedef map<string, Virtual_File* > FILE_NAME_MAP;
typedef map<int, Virtual_File* > FILE_DESCRIPTOR_MAP;

FILE_NAME_MAP file_map;
FILE_POINTER_MAP file_pointer_map;
FILE_DESCRIPTOR_MAP file_desc_map;


#define handle_error(msg) \
    do { perror(msg); abort(); } while (0)


   
static void load_file_to_memory(FILE *fp, Virtual_File *varea) 
{ 
  fseek(fp, 0, SEEK_END);
  int size = ftell (fp);
  fseek(fp, 0, SEEK_SET);

  char *buffer = new char[size + 1];
 
  if (size != fread(buffer, sizeof(char), size, fp))
  {
    delete buffer;  
    abort();
  }
  
  varea->buffer.assign(buffer, buffer+size);
  
}

static void load_file_to_memory(int fd, Virtual_File *varea) 
{ 
	int size = lseek(fd, 0, SEEK_END);

	lseek(fd, 0, SEEK_SET);
  
  char *buffer = new char[size + 1];
  
	if (size != read(fd, buffer, size))
  {
    delete buffer;  
    abort();
  }
    
  varea->buffer.assign(buffer, buffer+size);
}


extern "C" 
void __edge_open(int fd, char* fpath, int flags)
{
  Virtual_File *varea = file_map[fpath];
  if (!varea)
  {
    int length;
    struct stat sb;
    int pa_offset = 0;

    varea = new Virtual_File();

    if ( flags == O_RDONLY )
    {
      load_file_to_memory (fd, varea);
    }
    else if ( flags & O_RDWR )
    {
      load_file_to_memory (fd, varea);
    }
    
    file_map[fpath] =  varea;
   
  }
  else
    varea->cur = 0;

  
  file_desc_map[fd] =  varea;  
  
}

extern "C" 
void __edge_close(int fd)
{
  file_desc_map.erase(fd);  
}


extern "C" 
long  __edge_read (int fd, char * ptr, long numbytes)
{

  Virtual_File *varea = file_desc_map[fd];

  assert(varea);

  int len = numbytes;

  
  if ( len == 0)
    return 0;

  if (varea->buffer.size() - varea->cur < len)
    return 0;

  for (int i= 0; i < len; i++ )
    ptr[i] = varea->buffer[i+varea->cur];

  varea->cur += len;
  varea->set_read();
  
  return len;  
}


extern "C" 
ssize_t  __edge_write (int fd, char * ptr, size_t numbytes)
{

  Virtual_File *varea = file_desc_map[fd];

  assert(varea);

  int len = numbytes ;

  if ( varea->cur == varea->buffer.size() ) {
    for (int i= 0; i < len; i++ ) 
      varea->buffer.push_back(ptr[i]);    
  }
  else
  {
    int i = 0;
    while ( i + varea->cur < varea->buffer.size() && i < len )
    {
      varea->buffer[i+varea->cur] = ptr[i];
      i++;
    }
      
    for ( ;i < len; i++ ) 
      varea->buffer.push_back(ptr[i]);    

  }
    

  varea->cur += len;
  varea->set_write();

  return numbytes;
}

extern "C" 
long __edge_lseek (int fd, long position, int startpoint)
{

  Virtual_File *varea = file_desc_map[fd];

  assert(varea);

  /*
  SEEK_SET        Position is number of bytes from beginning of file
  SEEK_END        Position is number of bytes after end of file
  SEEK_CUR        Position is number of bytes after current position

  */
  switch (startpoint)
  {
    case SEEK_SET:
      varea->cur = position;
      break;

    case SEEK_END:
      varea->cur = varea->buffer.size() + position;
      break;
      
    case SEEK_CUR:      
      varea->cur += position;
      break;
      
    default:
      abort();
  }

  if ( varea->cur < 0 )
    varea->cur = 0;
  else if (varea->cur > varea->buffer.size())
    varea->cur = varea->buffer.size();
  
  return varea->cur;
}


extern "C" 
void __edge_fopen(FILE *fp, char* fpath, char* mode)
{
  if (!fp)
    return;

  Virtual_File *varea = file_map[fpath];
  if (!varea)
  {
    int length;
    struct stat sb;
    int pa_offset = 0;

    varea = new Virtual_File();

    if ( strcmp(mode, "r") == 0 || strcmp(mode, "rb") == 0)
    {
      load_file_to_memory (fp, varea);
    }
    else if ( strcmp(mode, "r+") == 0 )
    {
      load_file_to_memory (fp, varea);
    }
    else if ( strcmp(mode, "a+") == 0 )
    {
      load_file_to_memory (fp, varea);
    }
    else if ( strcmp(mode, "w") == 0 || strcmp(mode, "wb") == 0)
    {
    }

    else
    {
      printf("invalid open mode %s\n", mode);
      abort();
    }
    
    file_map[fpath] =  varea;
   
  } 
  else
    varea->cur = 0;
  
  file_pointer_map[fp] = varea;   
}

extern "C" 
void __edge_fclose(FILE *fp)
{
  file_pointer_map.erase(fp);  
}

extern "C" 
size_t __edge_fread ( char * ptr, size_t size, size_t count, FILE * stream )
{

  Virtual_File *varea = file_pointer_map[stream];

  assert(varea);

  if ( count == 0)
    return 0;

  int len = size*count;
  if (varea->buffer.size() - varea->cur < len)
    return 0;

  for (int i= 0; i < len; i++ )
    ptr[i] = varea->buffer[i+varea->cur];

  varea->cur += len;
  varea->set_read();
  
  return count;
}


extern "C" 
size_t __edge_fwrite ( const char * ptr, size_t size, size_t count, FILE * stream )
{

  int len = size*count;

  if ( stream == stdout || stream == stderr)
    return len;


  Virtual_File *varea = file_pointer_map[stream];

  assert(varea);

  
  if ( varea->cur == varea->buffer.size() ) {
    for (int i= 0; i < len; i++ ) 
      varea->buffer.push_back(ptr[i]);    
  }
  else
  {
    int i = 0;
    while ( i + varea->cur < varea->buffer.size() && i < len )
    {
      varea->buffer[i+varea->cur] = ptr[i];
      i++;
    }
      
    for (; i < len; i++ ) 
      varea->buffer.push_back(ptr[i]);    

  }

  varea->cur += len;
  varea->set_write();

  return len;
}

extern "C" 
int __edge_fseek ( FILE * stream, long offset, int origin )
{

  Virtual_File *varea = file_pointer_map[stream];

  assert(varea);

  switch (origin)
  {
    case SEEK_SET:
      varea->cur = offset;
      break;

    case SEEK_END:
      varea->cur = varea->buffer.size() + offset;
      break;
      
    case SEEK_CUR:      
      varea->cur += offset;
      break;
      
    default:
      abort();
  }

  
  if ( varea->cur < 0 )
    varea->cur = 0;
  else if (varea->cur > varea->buffer.size())
    varea->cur = varea->buffer.size();

  return 0; 

}


/*

void rewind ( FILE * stream );

Set position indicator to the beginning
Sets the position indicator associated with stream to the beginning of the file.
A call to rewind is equivalent to:

fseek ( stream , 0L , SEEK_SET );
except that, unlike fseek, rewind clears the error indicator.
On streams open for update (read+write), a call to rewind allows to switch between reading and writing.

Parameters

stream
    Pointer to a FILE object that identifies the stream.


Return Value
none


*/

extern "C"
void __edge_rewind ( FILE * stream )
{

  Virtual_File *varea = file_pointer_map[stream];

  assert(varea);

  varea->cur = 0;
}


extern "C"
long __edge_ftell ( FILE * stream )
{

  Virtual_File *varea = file_pointer_map[stream];

  assert(varea);

  return varea->cur;
}

extern "C"
int __edge_fflush ( FILE * stream )
{
  return 0;
}

extern "C"
int __edge_ferror ( FILE * stream )
{
  return 0;
}

extern "C"
void __edge_cleanerror ( FILE * stream )
{
  return ;
}

extern "C"
int __edge_feof ( FILE * stream )
{
  Virtual_File *varea = file_pointer_map[stream];
  return varea->cur >= varea->buffer.size();
}

/*
  Returns the character currently pointed by the internal file position indicator of the specified 
  stream. The internal file position indicator is then advanced by one character to point to the 
  next character.
  fgetc and getc are equivalent, except that the latter one may be implemented as a macro.

  Parameters
  stream 
      Pointer to a FILE object that identifies the stream on which the operation is to be performed.

  Return Value
  The character read is returned as an int value.
  If the End-of-File is reached or a reading error happens, the function returns EOF and the 
  corresponding error or eof indicator is set. You can use either ferror or feof to determine 
  whether an error happened or the End-Of-File was reached.
*/
extern "C" 
int __edge_fgetc (FILE * stream )
{

  Virtual_File *varea = file_pointer_map[stream];

  assert(varea);

  if (varea->cur >= varea->buffer.size() )
    return EOF;

  varea->set_read();
  return varea->buffer[varea->cur++];
  
}

extern "C" 
int __edge_getc (FILE * stream )
{
  return __edge_fgetc(stream);  
}


extern "C" 
int __edge_fputc ( int character, FILE * stream )
{

  if ( stream == stdout || stream == stderr)
    return character;

  Virtual_File *varea = file_pointer_map[stream];
  assert(varea);
  varea->set_write();

  if ( varea->cur == varea->buffer.size() ) 
      varea->buffer.push_back(character);    
  else
    varea->buffer[varea->cur++] = character;

  varea->cur++;
 
  return character;
}

/*
char * fgets ( char * str, int num, FILE * stream );

Get string from stream
Reads characters from stream and stores them as a C string into str until (num-1) characters have been read or either a newline or a the End-of-File is reached, whichever comes first.
A newline character makes fgets stop reading, but it is considered a valid character and therefore it is included in the string copied to str.
A null character is automatically appended in str after the characters read to signal the end of the C string.

Parameters

str
    Pointer to an array of chars where the string read is stored.
num
    Maximum number of characters to be read (including the final null-character). Usually, the length of the array passed as str is used.
stream
    Pointer to a FILE object that identifies the stream where characters are read from.
    To read from the standard input, stdin can be used for this parameter.


Return Value
On success, the function returns the same str parameter.
If the End-of-File is encountered and no characters have been read, the contents of str remain unchanged and a null pointer is returned.
If an error occurs, a null pointer is returned.
Use either ferror or feof to check whether an error happened or the End-of-File was reached.


*/

extern "C" 
char * __edge_fgets ( char * str, int num, FILE * stream )
{

  Virtual_File *varea = file_pointer_map[stream];

  assert(varea);
  varea->set_read();

  if (varea->cur == varea->buffer.size())
    return NULL;

  int i = 0;
  while (i < num -1)
  {
    if (varea->cur == varea->buffer.size())
      break;
    
    str[i++] = varea->buffer[varea->cur++];

    if (str[i-1] == '\n')
      break;
  }

  str[i] = '\0';
  return str;
  
}

extern "C" 
int __edge_fputs ( char * str, FILE * stream )
{
  return 1;
}

void __skip_delimiter(Virtual_File *varea)
{
  // skip the first deliminators
  while (varea->cur < varea->buffer.size())
  {
    char c = varea->buffer[varea->cur];
    if (c !=' ' && c !='\n' && c !='\t')
      break;
    varea->cur++;
  }
}

/*
  Reads data from the stream and stores them according to the parameter format into the locations 
  pointed by the additional arguments. The additional arguments should point to already allocated 
  objects of the type specified by their corresponding form

  On success, the function returns the number of items successfully read. This count can match the 
  expected number of readings or be less -even zero- in the case of a matching failure.
  In the case of an input failure before any data could be successfully read, EOF is returned.
*/


extern "C" 
int __edge_fscanf ( FILE * stream, const char * format, ... )
{

  Virtual_File *varea = file_pointer_map[stream];
  assert(varea);
  varea->set_read();

  va_list argList;  
  va_start(argList, format);

  int items=0;
  bool percent=false;
  bool longp=false;
  while (*format)
  {
    __skip_delimiter (varea);
    if ( varea->cur >= varea->buffer.size() )
      return EOF; // EOF
    switch(*format) 
    {
      case '%':
        percent = true;
        break;

      case ' ':
      case '\t':
      case '\n':
      {        
        break;
      }
      
      case 'l':            /* int */        
      {
        __skip_delimiter (varea);
           
        if (!percent)
        {
         char c = varea->buffer[varea->cur++];          
         if ( c != *format )
           return items;
         break;
        }
        longp = true;
        break;
      }
        
      case 'd':            /* int */
      {
        __skip_delimiter (varea);
        
        char c;
        if (!percent)
        {
          c = varea->buffer[varea->cur++];          
          if ( c != *format )
            return items;
          break;
        }
        percent = false;

        string str;
        c = varea->buffer[varea->cur];  
        if ( c != '+' && c != '-' && !isdigit(c) )
          return items;
        str += c;
        varea->cur++;
        while (varea->cur < varea->buffer.size())
        {
          c = varea->buffer[varea->cur];
          if ( !isdigit(c))
            break;
          str += c;
          varea->cur++;
        }

        if (!str.empty())
        {
          int *save = va_arg(argList, int*);
          *save = atoi(str.c_str());               
          items++;
        }
        else
          return items;
        break;
      }

      case 'f':            /* float */
      case 'g':            /* float */
      {
        __skip_delimiter (varea);
        
        char c;
        if (!percent)
        {
          c = varea->buffer[varea->cur++];          
          if ( c != *format )
            return items;
          break;
        }
        percent = false;

        string str;
        while (varea->cur < varea->buffer.size())
        {
          c = varea->buffer[varea->cur];
          if ( c != '+' && c != '-'  && c != '.' && c != 'e' && !isdigit(c) )
            break;
          str += c;
          varea->cur++;
        }

        if (!str.empty())
        {
          if (longp)
          {
            double *save = va_arg(argList, double*);
            *save = atof(str.c_str());               
          }
          else
          {
            float *save = va_arg(argList, float*);
            *save = atof(str.c_str());               
          }
            
          items++;
        }
        else
          return items;

        longp = false;
        break;
      }     
        
      case 's':            /* string */
      {
        __skip_delimiter (varea);
        
        if (!percent)
        {
          char c = varea->buffer[varea->cur++];          
          if ( c != *format )
            return items;
          break;
        }
        percent = false;

        string str;
        
        while (varea->cur < varea->buffer.size())
        {
          char c = varea->buffer[varea->cur];
          if (c==' ' || c=='\n' || c=='\t')
            break;
          str += c;
          varea->cur++;
        }

        if (!str.empty())
        {
          char *save = va_arg(argList, char*);
          strcpy(save, str.c_str());
          items++;
        }
        else
          return items;
        break;
         
      }
      
      case 'c':            /* char */
      {
        __skip_delimiter (varea);
        char c = varea->buffer[varea->cur++];          
        
        if (!percent)
        {
          if ( c != *format )
            return items;
        }
        else
        {
          char *save = va_arg(argList, char*);
          *save = c;
          items++;
        }
          
        break;
         
      }

      default:
      {
        if (percent)
          Is_True(false, ("%% %c not support\n", *format));

        __skip_delimiter (varea);
        
        char c = varea->buffer[varea->cur++];          
        if ( c != *format )
          return items;
        break;
      }

            
    }

    format++;
  }


  va_end(argList);        // now buffer contains the formatted message..       
  return items;  
}


extern "C" 
int __edge_vfprintf ( FILE * stream, const char * format, va_list argList )
{
  return 1;
  if ( stream == stdout || stream == stderr)
  {
    int len = vfprintf(stream, format, argList);	
    return len; 
  }
}


extern "C" 
int __edge_fprintf (FILE *stream, const char *format, ...)
{
	va_list argList;	
  va_start(argList, format);
  int len = __edge_vfprintf (stream, format, argList);
  va_end(argList);
  return len;
}



extern "C" 
int __edge_printf (const char *format, ...)
{
  va_list argList;  
  va_start(argList, format);
  int len = __edge_vfprintf (stdout, format, argList);
  va_end(argList);
  return len;
}




extern "C" 
int __edge_puts ( const char * str )
{
  return 1;  
}


extern "C" 
int __edge_putchar ( int character )
{
  return character;  
}


extern "C" 
void * __edge_malloc ( size_t size )
{
#ifdef MEMLEAK
  return MEM_POOL_malloc (local_pool, size);
#else
  return malloc(size);
#endif
}


extern "C" 
void * __edge_calloc ( size_t num, size_t size )
{
#ifdef MEMLEAK
  return MEM_POOL_calloc(local_pool, num, size);
#else
  return calloc(num, size);
#endif
}


extern "C" 
void * __edge_realloc ( void * ptr, size_t size )
{
#ifdef MEMLEAK
  return MEM_POOL_realloc (local_pool, ptr, size);
#else
  return realloc(ptr,size);
#endif
}



extern "C" 
void __edge_free ( void * ptr )
{
  if (!ptr)
    return;
#ifdef MEMLEAK
  MEM_POOL_free(local_pool, ptr);
#else
  free(ptr);
#endif
}


// TODO:    fgetpos  freopen  fsetpos


extern void __edge_set_depfile (const char *deppath)
{
  string tmp = deppath;
  depname = new char[tmp.size()];
  strcpy (depname, deppath);
}


extern void __edge_clean()
{
  delete depname;
}




extern jmp_buf env;

extern "C" void
__edge_profile_initialize (void)
{


  count_time_start (1);

  if ( loop_stack_capacity == 0 )
  {
    loop_stack_capacity = SIZE - 1;
    loop_stack = (lentry **) malloc (sizeof (lentry*) * loop_stack_capacity);
    loop_stack[0] = new lentry;
    loop_stack[0]->kind = pk_not_profile;
  }
  
  sp = 0;
  
  fp = stdout;

  clktck = sysconf (_SC_CLK_TCK);

  for (FILE_NAME_MAP::iterator iter = file_map.begin(); iter != file_map.end(); ++iter)
  {
    Virtual_File *varea = iter->second;
    if ( varea->has_read() && varea->has_write() )
    {
      delete varea;
      iter->second = NULL;
    }      
    else
      varea->cur = 0;
  }

  local_pool = new MEM_POOL;

//  generate_loop_table ();

//  generate_not_profile_loops ();

//  generate_memory_op_table ();

}





extern "C" void
__edge_profile_finalize (void)
{

  while ( !Empty() )
  {
    sd3_clear_loop (Top());
    Pop ();
  }
  
  file_pointer_map.clear();

  for (FILE_NAME_MAP::iterator iter = file_map.begin(); iter != file_map.end(); ++iter)
  {
    Virtual_File *varea = iter->second;
    if ( varea->has_read() && varea->has_write() )
    {
      delete varea;
      iter->second = NULL;
    }      
    else
      varea->cur = 0;
  }

  delete local_pool;

}


extern "C" void
__edge_print_abort () 
{

  __edge_profile_finalize ();
  //longjmp(env, 2) ;
	abort();
}


extern "C" void
__edge_print_exit () 
{

  __edge_profile_finalize ();  
  //longjmp(env, 1) ;
  exit(1);
}



bool
lentry::compute_loop_independent_edge (SD3_STRIDE& a_stride, DEPENDENCY *dep)
{
  assert( loop_id == dep->loop_id() );
  
  /* checking a against b */
  
  SD3_STRIDE_SET &b_strides = pending_stride_table[dep->b()];
  
  std::pair<SD3_STRIDE_SET::iterator, SD3_STRIDE_SET::iterator>
    range_pair = b_strides.equal_range(a_stride);
  
  for ( SD3_STRIDE_SET::iterator biter = range_pair.first; biter != range_pair.second; ++biter )
  {
    const SD3_STRIDE &b_stride = *biter; 
    if ( !b_stride.overlaps (a_stride) )
      break;
  
    if (b_stride.compute_dependence (a_stride, dep->kind(), dep->loop_id() ))
      return true;
  }
  
  return false;

}



bool
lentry::compute_loop_carried_edge (SD3_STRIDE& a_stride, DEPENDENCY *dep)
{

  assert( loop_id == dep->loop_id() );

  /* checking a against b */


  SD3_STRIDE_SET &b_strides = history_stride_table[dep->b()];

  std::pair<SD3_STRIDE_SET::iterator, SD3_STRIDE_SET::iterator>
    range_pair = b_strides.equal_range(a_stride);

  for ( SD3_STRIDE_SET::iterator biter = range_pair.first; biter != range_pair.second; ++biter )
  {
    const SD3_STRIDE &b_stride = *biter; 
    if ( !b_stride.overlaps (a_stride) )
      break;
  
    if (b_stride.compute_dependence (a_stride, dep->kind(), dep->loop_id() ))
      return true;
  }

  return false;
}

// return true if splited
bool
SD3_STRIDE::check_and_reduce (SD3_STRIDE& x, dep_kind dkind, int dloop, SD3_STRIDE_SET &strides) const
{

  if ( !overlaps(x) )
  {
    strides.insert(x);
    return false;
  }

  int dstmt_before = stmt;
  int dstmt_after = x.stmt;
  dep_type dtype = get_dep_type (*this, x);

  if ( dtype == dep_RAR )
  {
    strides.insert(x);
    return false;
  }

  ptrdiff_t gcd_num;
  ptrdiff_t const_num;
  
  if (distance == 0 && x.distance == 0) 
  {
    if (low != x.low)
    {
      strides.insert(x);
      return false;
    }
    
    if (dtype == dep_RAW)      
    {
      DEPENDENCY dep(dstmt_after, dstmt_before, dkind, dtype, dloop);
      sd3_record_dependence (dep);
      return true;
    }
  }
  else if (distance == 0) 
  {
    PTR addr = low;
    if (addr < x.low || addr > x.high)
    {
      strides.insert(x);
      return false;
    }

    PTR tmp = x.low;

    if (abs (addr - x.low) % x.distance)
    {
      strides.insert(x);
      return false;
    }

    /* split x */

    DEPENDENCY dep(dstmt_after, dstmt_before, dkind, dtype, dloop);
    sd3_record_dependence (dep);

    SD3_STRIDE stride_first = x;  
    stride_first.high = addr - x.distance;
    if ( stride_first.low = stride_first.high )
    {
      stride_first.distance = 0;
      stride_first.status = SD3_FIRST;
    }
    
    SD3_STRIDE stride_second = x;  
    stride_second.low = addr + x.distance;
    if ( stride_second.low = stride_second.high )
    {
      stride_second.distance = 0;
      stride_second.status = SD3_FIRST;
    }
    
    strides.insert(stride_first);
    strides.insert(stride_second);     
  }
  
  else if (x.distance == 0) 
  {
    PTR addr = x.low;
    if (addr < low  || addr > high )
    {
      strides.insert(x);
      return false;
    }
  
    if (abs (addr - low) % distance)
    {
      strides.insert(x);
      return false;
    }

    if (dtype == dep_RAW)
    {
      DEPENDENCY dep(dstmt_after, dstmt_before, dkind, dtype, dloop);
      sd3_record_dependence (dep);
      return true;
    }
  }
  
  else
  {
    // distance != 0 && x.distance != 0
    gcd_num = sd3_gcd (distance, x.distance);
    assert (gcd_num);

    if (abs (x.low - low) % gcd_num)
    {
      strides.insert(x);
      return false;
    }

    if (x.distance == distance)
    {
      if ( x.low >= low && x.high <= high )
      {
        DEPENDENCY dep(dstmt_after, dstmt_before, dkind, dtype, dloop);
        sd3_record_dependence (dep);
        return true;
      }

      if ( x.low < low )
      {
        /* split x */
        DEPENDENCY dep(dstmt_after, dstmt_before, dkind, dtype, dloop);
        sd3_record_dependence (dep);
        SD3_STRIDE stride_first = x;  
        stride_first.high = low - x.distance;
        if ( stride_first.low = stride_first.high )
        {
          stride_first.distance = 0;
          stride_first.status = SD3_FIRST;
        }
        strides.insert(stride_first);
      }
   
      if ( x.high > high )
      {
        /* split x */
        DEPENDENCY dep(dstmt_after, dstmt_before, dkind, dtype, dloop);
        sd3_record_dependence (dep);
        SD3_STRIDE stride_second = x;  
        stride_second.low = high + x.distance;
        if ( stride_second.low = stride_second.high )
        {
         stride_second.distance = 0;
         stride_second.status = SD3_FIRST;
        }
        strides.insert(stride_second);
      }

      return true;
    }

    else  
    {
      // different step. Currently under very conservative processing
      strides.insert(x);
      return false;
    }
  }

  strides.insert(x);
  return false;
  
}



void
lentry::check_loop_independent_killed (SD3_STRIDE& stride, SD3_STRIDE_SET &strides)
{

  /* check dependence with stride pending table */
  strides.clear();
  strides.insert(stride);

  // check with each previous statement
  for (STRIDE_MAP::iterator hiter = pending_stride_table.begin(); 
        hiter != pending_stride_table.end(); ++hiter )
  {
    // write down unsolved strides for the current checking
    SD3_STRIDE_SET new_strides;

    // for each stride, check against overlapped strides
    while ( !strides.empty() )
    {
      SD3_STRIDE tmp_stride = *(strides.begin()); 
      strides.erase (strides.begin());
    
      std::pair<SD3_STRIDE_SET::iterator, SD3_STRIDE_SET::iterator>
        range_pair = hiter->second.equal_range(tmp_stride);

      bool splitted = false;
      for ( SD3_STRIDE_SET::iterator iter = range_pair.first; iter != range_pair.second; ++iter )
      {
        const SD3_STRIDE &pending_stride = *iter; 
        if ( !pending_stride.overlaps (tmp_stride) )
          break;
        
        SD3_STRIDE_SET tmp_strides;

        if (pending_stride.check_and_reduce (tmp_stride, LOOP_INDEPENDENT, loop_id, tmp_strides))
        {
          // stride has been splitted 
          splitted = true;
          strides.insert ( tmp_strides.begin(), tmp_strides.end() );
          break;
        }
      }  

      if ( !splitted )
        new_strides.insert (tmp_stride);
    
    }


    strides = new_strides;

  }


}




/* addr - address;
   size - the bit size of memory to be accessed
   flag - read/write, if address read (0) or written (1) ; 
   id - memory operation id or slice id; 
   kind -- 0x1 - loop-independent ,  0x2 - loop-carried
*/

extern "C" void
__edge_print_address (PTR addr, int size, char flag, int id, int kind, DEPENDENCY *dep)
{

  if (Empty ())
    return;

#ifdef TIME_LIMIT
  count_time_end (1);
  if ( Time_Profile[1] / (double) clktck > time_limit * 60 )
  {
    __edge_print_abort();     
  }    
#endif

  SD3_STRIDE stride;  
  stride.add_addr (addr, size, id, flag);

  if ( id != dep->a() )
    sd3_stride_insert (Top (), id, &stride);

  else
  {
    if ( (kind & 0x1) && (kind & 0x2) )   // loop-carried RAW dependence
    {    
      SD3_STRIDE_SET  strides;
      Top ()->check_loop_independent_killed (stride, strides) ;
      for ( SD3_STRIDE_SET::iterator iter = strides.begin(); iter != strides.end(); ++iter )
      {
        SD3_STRIDE &new_stride = const_cast<SD3_STRIDE&>(*iter);  
        if ( Top ()->compute_loop_carried_edge (new_stride, dep) )
        {
          /* caught the dependence, print and exit */
          __edge_print_exit();      
        }
      }
    }      
    
    else if ( dep->kind() == LOOP_INDEPENDENT )
    {    
      if ( Top ()->compute_loop_independent_edge (stride, dep) )
      {
        /* caught the dependence, print and exit */
        __edge_print_exit();      
      }
    }
    
    else if ( dep->kind() == LOOP_CARRIED )
    {    
      if ( Top ()->compute_loop_carried_edge (stride, dep) )
      {
        /* caught the dependence, print and exit */
        __edge_print_exit();      
      }
    }
    
  }
      
  
}




/* entry-loop enter/exit; loc - loop global id; s-
*/

extern "C" void
__edge_print_edge (int loop_id, int entry, DEPENDENCY *dep)
{


  if (entry) 
  {
    if ( Empty() ) 
    {
      lentry *loop = new lentry;
      loop->loop_id = loop_id;
      loop->iter_no = 1;
      Push (loop);
    }

    /* New iteration in same loop_id */
    else
    {
      if ( dep->kind() == LOOP_CARRIED ) 
      {
        // Top ()->merge_pending_table ();        
        std::map <int, SD3_STRIDE_SET>::iterator iter =  Top()->pending_stride_table.begin ();
        for (; iter != Top()->pending_stride_table.end (); ++iter)
        {
          if ( iter->first != dep->b() )
            continue;
          SD3_STRIDE_SET  &history = Top()->history_stride_table [iter->first];
          Top()->merge_stride (history, iter->second);
        }
        
        /* Clear pending table */
        Top()->pending_stride_table.clear();
      }
      else
        sd3_clear_loop (Top ());
        
      Top ()->iter_no++;
    }
  }
  
  else 
  {
    lentry *loop = Top ();

    if (loop->loop_id == loop_id) 
    {

      // loop_infos[loop_id].itercount += loop->iter_no;
    
      sd3_clear_loop (loop);
      Pop ();
    }
  }

  return;
}


