/* hash_insert.c
 * function called from profile.c 
 * arguments: char * for address
 * <initial test>
 * 1. check if address already inserted
 * 2. if not, insert. otherwise increment the value to indicate the collisions
 * 
 */

#ifndef _HASH_INSERT_H_
#define _HASH_INSERT_H_


#include <string.h>		/* strcpy */
#include <stdlib.h>		/* malloc */
#include <stdio.h>		/* printf */
#include "libprofile.h"
#include <unistd.h>



void
record_dependence (const DEPENDENCY & new_dep, LOOP_DEPENDENCICES & deps,
		   bool check_redundancy)
{
  if ( new_dep.type() != dep_RAR )
    deps.Insert (new_dep, check_redundancy);
}


// reset old_dep to be accomodated for new state
static void
adjust_dependence (DEPENDENCY & old_dep, const DEPENDENCY & new_dep)
{


  bool succ_rw = false;		// read --false, write -- true

  switch (new_dep.type ())
    {
    case dep_WAR:
      succ_rw = true;
      break;

    case dep_WAW:
      succ_rw = true;
      break;
    }


  if (succ_rw)			// write
    {
      if (old_dep.type () == dep_RAW)
	old_dep.set_type (dep_WAW);

      else if (old_dep.type () == dep_WAR)
	old_dep.set_type (dep_WAR);

      else if (old_dep.type () == dep_WAW)
	old_dep.set_type (dep_WAW);
      else
	old_dep.set_type (dep_WAR);
    }
  else				// read
    {
      if (old_dep.type () == dep_RAW)
	old_dep.set_type (dep_RAW);

      else if (old_dep.type () == dep_WAR)
	old_dep.set_type (dep_RAR);

      else if (old_dep.type () == dep_WAW)
	old_dep.set_type (dep_RAW);
      else
	old_dep.set_type (dep_RAR);
    }

  old_dep.set_a (new_dep.a ());


  // If the old dependence is out of the current loop, then the result dependence must 
  // be the same with old_dep
  int old_sp = Contains (old_dep.loop_id ());
  int new_sp = Contains (new_dep.loop_id ());

  if (old_sp && old_sp < new_sp)
    {
      return;
    }

  else if (old_sp == new_sp)
    {
      if (new_dep.kind () == LOOP_CARRIED)
	old_dep.set_kind (LOOP_CARRIED);
    }

  else
    {
      old_dep.set_kind (new_dep.kind ());
      old_dep.set_loop_id (new_dep.loop_id ());
    }

}


static void
transfer_dependence (hash_entry * hash, const DEPENDENCY & new_dep)
{

  bool succ_rw = false;		// read --false, write -- true
  bool adjust = true;

  switch (new_dep.type ())
    {
    case dep_WAR:
      succ_rw = true;
      break;

    case dep_WAW:
      succ_rw = true;
      break;
    }

  // loop-independent RAW
  if (new_dep.kind () == LOOP_INDEPENDENT && new_dep.type () == dep_RAW)
    return;

  MEMORY_OPERATION_INFO *info = mem_info_hash[new_dep.a ()];

  for (DEPENDENCY_VEC::iterator iter = hash->dep_list.begin ();
       iter != hash->dep_list.end (); ++iter)
    {
      DEPENDENCY & dep = *iter;
      adjust_dependence (dep, new_dep);

      if (dep.kind () == LOOP_CARRIED)
	{
	  LOOP_DEPENDENCICES & true_carried = info->true_dependencies.second;
	  record_dependence (dep, true_carried, true);
	}
      else
	{
	  LOOP_DEPENDENCICES & true_indepdent = info->true_dependencies.first;
	  record_dependence (dep, true_indepdent, true);
	}

      // loop-independent RAW
      if (dep.kind () == LOOP_INDEPENDENT && dep.type () == dep_RAW)
	return;
    }

}


static void
check_dependence_locally (hash_entry * hash, lentry * loop,
			  const MEMORY_OPERATION & pred,
			  const MEMORY_OPERATION & succ)
{

  /* record dependence */
  enum dep_type type;

  if (pred.rwflag () == 1 && succ.rwflag () == 0)
    type = dep_RAW;

  else if (pred.rwflag () == 0 && succ.rwflag () == 1)
    type = dep_WAR;

  else if (pred.rwflag () == 1 && succ.rwflag () == 1)
    type = dep_WAW;

  else
    type = dep_RAR;

  enum dep_kind kind;

  if (pred.iterno () == succ.iterno ())	/* loop-independent */
    kind = LOOP_INDEPENDENT;
  else
    kind = LOOP_CARRIED;

  DEPENDENCY new_dep (succ.id (), pred.id (), kind, type, loop->loop_id);


  MEMORY_OPERATION_INFO *info = mem_info_hash[succ.id ()];

  if (kind == LOOP_CARRIED)
    {
      LOOP_DEPENDENCICES & true_carried = info->true_dependencies.second;
      record_dependence (new_dep, true_carried, true);
    }
  else
    {
      LOOP_DEPENDENCICES & true_indepdent = info->true_dependencies.first;
      record_dependence (new_dep, true_indepdent, true);
    }


  /* Transfer dependencies */
  transfer_dependence (hash, new_dep);
  hash->dep_list.Insert (new_dep);
  hash->dep_list.Unique ();

}


static void
check_dependence_globally (hash_entry * hash, lentry * loop, PTR addr,
			   const MEMORY_OPERATION & succ)
{

  int depth = Length ();

  if (depth <= 1)
    return;

  hash_entry *s = NULL;
  lentry *father = NULL;

  while (--depth)
    {
      father = Elem (depth);
      if (father == loop)
	continue;
      HASH_FIND_PTR (father->hashaddr, &addr, s);
      if (s)
	break;
    }

  if (!s)
    return;


  /* check dependence */
  const MEMORY_OPERATION & pred = s->memop;

  /* record dependence */
  enum dep_type type;

  if (pred.rwflag () == 1 && succ.rwflag () == 0)
    type = dep_RAW;

  else if (pred.rwflag () == 0 && succ.rwflag () == 1)
    type = dep_WAR;

  else if (pred.rwflag () == 1 && succ.rwflag () == 1)
    type = dep_WAW;

  else
    type = dep_RAR;


  enum dep_kind kind;

  if (pred.iterno () == father->iter_no)	/* loop-independent */
    kind = LOOP_INDEPENDENT;
  else
    kind = LOOP_CARRIED;


  DEPENDENCY new_dep (succ.id (), pred.id (), kind, type, father->loop_id);
  MEMORY_OPERATION_INFO *info = mem_info_hash[succ.id ()];

  if (kind == LOOP_CARRIED)
    {
      LOOP_DEPENDENCICES & true_carried = info->true_dependencies.second;
      record_dependence (new_dep, true_carried, true);
    }
  else
    {
      LOOP_DEPENDENCICES & true_indepdent = info->true_dependencies.first;
      record_dependence (new_dep, true_indepdent, true);
    }

  /* Transfer dependencies */
  hash->dep_list.splice (hash->dep_list.end (), s->dep_list);
  transfer_dependence (hash, new_dep);
  hash->dep_list.Insert (new_dep);
  hash->dep_list.Unique ();

}


const MEMORY_BLOCK &
get_mem_block (PTR p)
{
  MEMORY_BLOCK block_p (p, p, 0);
  MEMOP_BLOCK_MAP::iterator iter_p = mem_block_map.find (block_p);
  if (iter_p == mem_block_map.end ())
    return null_block;
  else
    return *iter_p;
}

const MEMORY_BLOCK &
get_mem_block (int id)
{
  return *mem_block_vec[id];
}


bool
same_mem_block_p (PTR p, PTR q)
{
  MEMORY_BLOCK block_p (p, p, 0);
  MEMOP_BLOCK_MAP::iterator iter_p = mem_block_map.find (block_p);
  if (iter_p == mem_block_map.end ())
    return false;


  MEMORY_BLOCK block_q (q, q, 0);
  MEMOP_BLOCK_MAP::iterator iter_q = mem_block_map.find (block_q);
  if (iter_q == mem_block_map.end ())
    return false;

  return iter_p == iter_q;
}





void
address_hash_insert (lentry * loop, PTR addr, const MEMORY_OPERATION & memop)
{

  /* hash into address table */
  struct hash_entry *s;
  HASH_FIND_PTR (loop->hashaddr, &addr, s);

  if (s)
  {
    /* check dependence */
    check_dependence_locally (s, loop, s->memop, memop);
    s->memop = memop;
  }
  else
  {
    s = new hash_entry;
    s->addr = addr;
    s->memop = memop;
    HASH_ADD_PTR (loop->hashaddr, addr, s);
    /* check dependence against history */
    check_dependence_globally (s, loop, addr, memop);
  }

}






void
transfer_addr_hash (lentry * father, lentry * loop)
{

  hash_entry *&father_addr_hash = father->hashaddr;
  hash_entry *child_addr_hash = loop->hashaddr;
  hash_entry *iter;

  for (iter = child_addr_hash; iter != NULL;
       iter = (hash_entry *) (iter->hh.next))
    {
      PTR addr = iter->addr;
      MEMORY_OPERATION & memop = iter->memop;
      memop.set_iterno (father->iter_no);

      struct hash_entry *s;
      HASH_FIND_PTR (father_addr_hash, &addr, s);

      if (s)
	{
	  s->memop = iter->memop;
	  s->dep_list.clear ();
	  s->dep_list.splice (s->dep_list.end (), iter->dep_list);
	}
      else
	{
	  s = new hash_entry;
	  s->addr = iter->addr;
	  s->memop = iter->memop;
	  s->dep_list.splice (s->dep_list.end (), iter->dep_list);
	  HASH_ADD_PTR (father_addr_hash, addr, s);
	}
    }

}



void
transfer_loop (lentry * father, lentry * loop)
{
  if (loop->kind == pk_not_profile)
    return;

  count_time_start (2);
  transfer_addr_hash (father, loop);
  count_time_end (2);
}



static void
clear_hash (lentry * loop)
{

  struct hash_entry *cur_addr = NULL;
  struct hash_entry *hashaddr = loop->hashaddr;

  ave_dep_num = 0;
  addr_count = 0;
  while (hashaddr)
  {
    ++addr_count;
    cur_addr = hashaddr;	/* copy pointer to first item     */
    ave_dep_num += cur_addr->dep_list.size ();
    cur_addr->dep_list.clear ();
    HASH_DEL (hashaddr, cur_addr);	/* delete; hashaddr advances to next */
    delete cur_addr;
  }
}



static void
clear_loop (lentry * loop)
{
  clear_hash (loop);
}

#endif
