/* Set operations on pointers
   Copyright (C) 2004, 2007 Free Software Foundation, Inc.

This file is part of GCC.

GCC is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free
Software Foundation; either version 3, or (at your option) any later
version.

GCC is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License
along with GCC; see the file COPYING3.  If not see
<http://www.gnu.org/licenses/>.  */

#ifndef POINTER_SET_H
#define POINTER_SET_H

struct pointer_set_t;
struct pointer_set_t *pointer_set_create (void);
void pointer_set_destroy (struct pointer_set_t *pset);

int pointer_set_contains (const struct pointer_set_t *pset, const void *p);
int pointer_set_insert (struct pointer_set_t *pset, const void *p);
void pointer_set_traverse (const struct pointer_set_t *,
			   bool (*) (const void *, void *),
			   void *);


/* A pointer map is represented the same way as a pointer_set, so
   the hash code is based on the address of the key, rather than
   its contents.  Null keys are a reserved value.  Deletion is not
   supported (yet).  There is no mechanism for user control of hash
   function, equality comparison, initial size, or resizing policy.  */

struct pointer_map_t
{
  size_t log_slots;
  size_t n_slots;		/* n_slots = 2^log_slots */
  size_t n_elements;

  const void **keys;
  void **values;
};

typedef struct pointer_map_t *pointer_map_ptr;

struct pointer_map_t *pointer_map_create (void);
void pointer_map_destroy (struct pointer_map_t *pmap);

void **pointer_map_contains (const struct pointer_map_t *pmap, const void *p);
void **pointer_map_insert (struct pointer_map_t *pmap, const void *p);
void pointer_map_traverse (const struct pointer_map_t *,
			   bool (*) (const void *, void **, void *), void *);
struct pointer_map_t *pointer_map_copy (const struct pointer_map_t *pmap);
void **pointer_map_delete (struct pointer_map_t *pmap, const void *p);

#define POINTER_SET_FOR_EACH(pset,elt,i)         \
  for (i = 0, elt = pset->slots[i]; i < pset->n_slots && elt; ++i, elt = pset->slots[i])

#define POINTER_MAP_FOR_EACH(pmap,key,value,i)   \
  for (i = 0, key = pmap->keys[i], value = pmap->values[i]; i < pmap->n_slots && key; ++i, key = pmap->keys[i], value = pmap->values[i])

#define POINTER_MAP_FOR_EACH_SLOT(pmap,key,slot,i)       \
  for (i = 0, key = pmap->keys[i], slot = &pmap->values[i]; i < pmap->n_slots && key; ++i, key = pmap->keys[i], slot = &pmap->values[i])

#endif  /* POINTER_SET_H  */
