/* Profile Generation of Data Dependencies
   Copyright (C) 2006 Free Software Foundation, Inc.
   Contributed by Georges-Andre Silber <Georges-Andre.Silber@ensmp.fr>.

This file is part of GCC.

GCC is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free
Software Foundation; either version 2, or (at your option) any later
version.

GCC is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License
along with GCC; see the file COPYING.  If not, write to the Free
Software Foundation, 51 Franklin Street, Fifth Floor, Boston, MA
02110-1301, USA.  */

/* Edited by HTYU
 * This file instruments the code to collect all data dependencies.
 * Additional information like distance vectors, memory references may
 * also be collected.

   The pass fipa-parallel-loops uses the dependence information collected
   for compiler optimizations such as automatic parallelization, privitization,
   loop optimizations etc.
*/

#include "ipa-check-dependence.h"

extern "C"
{
#include "config.h"
#include "system.h"
#include "coretypes.h"
#include "tm.h"
#include "ggc.h"
#include "tree.h"
#include "target.h"
#include "rtl.h"
#include "basic-block.h"
#include "diagnostic.h"
#include "tree-flow.h"
#include "tree-dump.h"
#include "timevar.h"
#include "cfgloop.h"
#include "expr.h"
#include "optabs.h"
#include "tree-chrec.h"
#include "tree-data-ref.h"
#include "tree-scalar-evolution.h"
#include "tree-pass.h"
#include "langhooks.h"
#include "coverage.h"
#include "output.h"
#include "tree-vectorizer.h"
#include "tm_p.h"
#include "output.h"
#include "flags.h"
#include "function.h"
#include "toplev.h"
#include "cfglayout.h"
#include "pointer-set.h"
#include "gimple.h"

extern void verify_flow_info (void);

}

#include <string>

typedef unsigned long long UINT64;

/* global declarations */
int num_assignment, total_assignment, num_copy_stmt, num_const_stmt,
  num_arith_stmt, num_global_ref;
void initialize_stats (void);
void process_statement (tree stmt);
void printstatistics (void);
void compute_local_consts (void);
void compute_global_consts (void);
void insert_stmt (tree * bsi, char defbuf[100], int deflen, char usebuf[100],
		  int uselen, int lineno);
void fill_buf (tree stmt, int defuse);

char vardef_buf[100];
char varuse_buf[100];
long vardef_len;
long varuse_len;
FILE *fpw, *fpr;
char fnname[100];
char filename[100];

static GTY (()) tree malloc_profiler_fn_type;
static GTY (()) tree range_profiler_fn_type;
static GTY (()) tree address_profiler_fn_type;
static GTY (()) tree edge_profiler_fn_type;
static GTY (()) tree exit_profiler_fn_type;


static GTY (()) tree SD3_address_profiler_fn_type;
static GTY (()) tree SD3_edge_profiler_fn_type;
static GTY (()) tree SD3_exit_profiler_fn_type;

static GTY (()) tree edge_address_profiler_fn_type;
static GTY (()) tree edge_edge_profiler_fn_type;
static GTY (()) tree edge_exit_profiler_fn_type;
static GTY (()) tree edge_start_profiler_fn_type;
static GTY (()) tree edge_open_fn_type;
static GTY (()) tree edge_close_fn_type;
static GTY (()) tree edge_read_fn_type;
static GTY (()) tree edge_write_fn_type;
static GTY (()) tree edge_lseek_fn_type;
static GTY (()) tree edge_fopen_fn_type;
static GTY (()) tree edge_fclose_fn_type;
static GTY (()) tree edge_fread_fn_type;
static GTY (()) tree edge_fwrite_fn_type;
static GTY (()) tree edge_fseek_fn_type;
static GTY (()) tree edge_ftell_fn_type;
static GTY (()) tree edge_fflush_fn_type;
static GTY (()) tree edge_ferror_fn_type;
static GTY (()) tree edge_cleanerror_fn_type;
static GTY (()) tree edge_feof_fn_type;
static GTY (()) tree edge_fputc_fn_type;
static GTY (()) tree edge_fgetc_fn_type;
static GTY (()) tree edge_fgets_fn_type;
static GTY (()) tree edge_fputs_fn_type;
static GTY (()) tree edge_fscanf_fn_type;
static GTY (()) tree edge_fprintf_fn_type;
static GTY (()) tree edge_vfprintf_fn_type;
static GTY (()) tree edge_printf_fn_type;
static GTY (()) tree edge_puts_fn_type;
static GTY (()) tree edge_putchar_fn_type;
static GTY (()) tree edge_rewind_fn_type;
static GTY (()) tree edge_malloc_fn_type;
static GTY (()) tree edge_calloc_fn_type;
static GTY (()) tree edge_realloc_fn_type;
static GTY (()) tree edge_free_fn_type;

static GTY (()) tree pairwise_address_profiler_fn_type;
static GTY (()) tree pairwise_edge_profiler_fn_type;
static GTY (()) tree pairwise_exit_profiler_fn_type;

static GTY (()) tree privatize_store_profiler_fn_type;
static GTY (()) tree privatize_load_profiler_fn_type;
static GTY (()) tree privatize_entry_profiler_fn_type;
static GTY (()) tree privatize_exit_profiler_fn_type;
static GTY (()) tree privatize_end_profiler_fn_type;

static GTY (()) tree hash_store_profiler_fn_type;
static GTY (()) tree hash_load_profiler_fn_type;
static GTY (()) tree shadow_store_profiler_fn_type;
static GTY (()) tree shadow_load_profiler_fn_type;
static GTY (()) tree shadow_entry_profiler_fn_type;
static GTY (()) tree shadow_exit_profiler_fn_type;
static GTY (()) tree shadow_end_profiler_fn_type;
static GTY (()) tree shadow_begin_profiler_fn_type;


static GTY (()) tree shadow_store_profiler_fn;
static GTY (()) tree shadow_load_profiler_fn;
static GTY (()) tree shadow_entry_profiler_fn;
static GTY (()) tree shadow_exit_profiler_fn;
static GTY (()) tree shadow_end_profiler_fn;
static GTY (()) tree shadow_begin_profiler_fn;


static GTY (()) tree access_dimension_type;
static GTY (()) tree loop_info_table_type;
static GTY (()) tree loop_info_decl;
static int profile_count = 0;

static const char *malloc_profiler_name = "__print_malloc";
static const char *address_profiler_name = "__print_address";
static const char *range_profiler_name = "__print_range";
static const char *edge_profiler_name = "__print_edge";
static const char *exit_profiler_name = "__print_exit";

static const char *SD3_address_profiler_name = "__sd3_print_address";
static const char *SD3_edge_profiler_name = "__sd3_print_edge";
static const char *SD3_exit_profiler_name = "__sd3_print_exit";

static const char *pairwise_address_profiler_name = "__pairwise_print_address";
static const char *pairwise_edge_profiler_name = "__pairwise_print_edge";
static const char *pairwise_exit_profiler_name = "__pairwise_print_exit";

static const char *edge_address_profiler_name = "__edge_print_address";
static const char *edge_edge_profiler_name = "__edge_print_edge";
static const char *edge_exit_profiler_name = "__edge_print_exit";
static const char *edge_start_profiler_name = "__edge_profile_initialize";
static const char *edge_open_name = "__edge_open";
static const char *edge_close_name = "__edge_close";
static const char *edge_read_name = "__edge_read";
static const char *edge_write_name = "__edge_write";
static const char *edge_lseek_name = "__edge_lseek";
static const char *edge_fopen_name = "__edge_fopen";
static const char *edge_fclose_name = "__edge_fclose";
static const char *edge_fread_name = "__edge_fread";
static const char *edge_fwrite_name = "__edge_fwrite";
static const char *edge_fseek_name = "__edge_fseek";
static const char *edge_ftell_name = "__edge_ftell";
static const char *edge_fflush_name = "__edge_fflush";
static const char *edge_ferror_name = "__edge_ferror";
static const char *edge_cleanerror_name = "__edge_cleanerror";
static const char *edge_feof_name = "__edge_feof";
static const char *edge_fgetc_name = "__edge_fgetc";
static const char *edge_fputc_name = "__edge_fputc";
static const char *edge_fgets_name = "__edge_fgets";
static const char *edge_fputs_name = "__edge_fputs";
static const char *edge_fscanf_name = "__edge_fscanf";
static const char *edge_fprintf_name = "__edge_fprintf";
static const char *edge_printf_name = "__edge_printf";
static const char *edge_vfprintf_name = "__edge_vfprintf";
static const char *edge_puts_name = "__edge_puts";
static const char *edge_putchar_name = "__edge_putchar";
static const char *edge_rewind_name = "__edge_rewind";
static const char *edge_malloc_name = "__edge_malloc";
static const char *edge_calloc_name = "__edge_calloc";
static const char *edge_realloc_name = "__edge_realloc";
static const char *edge_free_name = "__edge_free";

static const char *privatize_store_profiler_name = "__privatize_print_store";
static const char *privatize_load_profiler_name = "__privatize_print_load";
static const char *privatize_entry_profiler_name = "__privatize_print_entry";
static const char *privatize_exit_profiler_name = "__privatize_print_exit";
static const char *privatize_end_profiler_name = "__privatize_profile_finalize";

static const char *shadow_privatizable_store_profiler_name = "__shadow_privatizable_print_store";
static const char *shadow_privatizable_load_profiler_name = "__shadow_privatizable_print_load";
static const char *shadow_privatizable_entry_profiler_name = "__shadow_privatizable_print_entry";
static const char *shadow_privatizable_exit_profiler_name = "__shadow_privatizable_print_exit";
static const char *shadow_privatizable_end_profiler_name = "__shadow_privatizable_finalize";
static const char *shadow_privatizable_begin_profiler_name = "__shadow_privatizable_initialize";

static const char *hash_bitwise_store_profiler_name = "__hash_bitwise_print_store";
static const char *hash_bitwise_load_profiler_name = "__hash_bitwise_print_load";
static const char *hash_dependence_store_profiler_name = "__hash_dependence_print_store";
static const char *hash_dependence_load_profiler_name = "__hash_dependence_print_load";
static const char *shadow_dependence_store_profiler_name = "__shadow_dependence_print_store";
static const char *shadow_dependence_load_profiler_name = "__shadow_dependence_print_load";
static const char *shadow_dependence_end_profiler_name = "__shadow_dependence_finalize";
static const char *shadow_dependence_begin_profiler_name = "__shadow_dependence_initialize";
static const char *shadow_dependence_enter_profiler_name = "__shadow_dependence_print_enter";
static const char *shadow_dependence_exit_profiler_name = "__shadow_dependence_print_exit";

static const char *shadow_malloc_name = "__shadow_malloc";
static const char *shadow_calloc_name = "__shadow_calloc";
static const char *shadow_realloc_name = "__shadow_realloc";

using namespace std;

static std::set < const char* > func_set;
std::set < int > loop_set;
static std::set < int > alias_set;
static std::set < int > edge_set;
static std::set < int > mem_set; 
static loop_p cur_loop=NULL; // current loop to be profiled


static int checking_kind=0;  // 0x1 - loop-independent ,  0x2 - loop-carried
static ipa_data_dependency objdep;
static tree objdep_type;
static tree objdep_var;

static bool strided = true;

static bool load_library_func_info_p = false;


typedef std::map< std::string, std::string > string_map;

static string_map  funcs_to_replace;
static set<string> replaced_funcs;


static void load_replaced_func_info()
{
 funcs_to_replace["__ctype_b_loc"] = "__ctype_b_loc_wrapper" ;
 replaced_funcs.insert("__ctype_b_loc_wrapper"); 
 //funcs_to_replace["main"] = "exec_main" ;
 //replaced_funcs.insert("exec_main");
}

extern "C" 
const char*  is_function_to_replace(const char *name)
{
  if ( !load_library_func_info_p )
  {
    load_replaced_func_info ();
    load_library_func_info_p = true;
  }

  string_map::iterator iter = funcs_to_replace.find(name);
  if ( iter!= funcs_to_replace.end() )
    return iter->second.c_str();
  return NULL;
}

bool is_replaced_function(const char *name)
{
  if ( !load_library_func_info_p )
  {
    load_replaced_func_info ();
    load_library_func_info_p = true;
  }
  return replaced_funcs.find(name) != replaced_funcs.end();
}


bool profiling_library_p = false; 
static set<string> profiling_libs;
  
static void load_profiling_library_info()
{
  profiling_libs.insert("bitvector.h");
  profiling_libs.insert("bitvector.cxx");
  profiling_libs.insert("hash_insert.cxx");
  profiling_libs.insert("libalias.cxx");
  profiling_libs.insert("libconsequtive.cxx");
  profiling_libs.insert("libcount.cxx");
  profiling_libs.insert("libdefs.cxx");
  profiling_libs.insert("libedge.cxx");
  profiling_libs.insert("libmemory.c");
  profiling_libs.insert("libmempool.h");
  profiling_libs.insert("libpairwise.cxx");
  profiling_libs.insert("libprivatize.cxx");
  profiling_libs.insert("libsd3.cxx");
  profiling_libs.insert("libshadow.cxx");
  profiling_libs.insert("libwrapper.h");
  profiling_libs.insert("libprofile.h");
  profiling_libs.insert("libprofile.cxx");
  profiling_libs.insert("thread_pool.h");
  profiling_libs.insert("uthash.h");
  profiling_libs.insert("utlist.h");  
  profiling_libs.insert("char_traits.h");  
  profiling_libs.insert("atomicity.h");  
  profiling_libs.insert("hashtable_policy.h");  
  profiling_libs.insert("stl_set.h");  
  profiling_libs.insert("hash_insert.h");  
  profiling_libs.insert("stl_tree.h");  
  profiling_libs.insert("vector.tcc");  
  profiling_libs.insert("stl_list.h");  
  profiling_libs.insert("hashtable.h");  
  profiling_libs.insert("basic_string.h");  
  profiling_libs.insert("new_allocator.h");  
  profiling_libs.insert("interval_set_algo.hpp");  
  profiling_libs.insert("stl_algobase.h");  
  profiling_libs.insert("thread_pool.h");    
  profiling_libs.insert("stl_iterator.h");    
  profiling_libs.insert("interval.hpp");    
  profiling_libs.insert("stl_uninitialized.h");   
  
  

}

bool is_profiling_library(const char *name)
{
  if ( !profiling_library_p )
  {
    load_profiling_library_info ();
    profiling_library_p = true;
  }

  return profiling_libs.find(name) != profiling_libs.end();
}



static TREE_MAP  type_profile_map;
static TREE_SET  new_type_set;

extern "C" int is_profile_structre(tree t)
{
	return new_type_set.find(t) != new_type_set.end();
}

extern "C" void add_profile_structre(tree t)
{
  new_type_set.insert(t);
}


extern "C" tree hash_profile_type(tree type)
{
  TREE_MAP::iterator iter = type_profile_map.find(type);
  if ( iter != type_profile_map.end() )
    return iter->second;
  return NULL_TREE;
}

extern "C"  void add_profile_type(tree type, tree newtype)
{
  type_profile_map[type] = newtype;
  add_profile_structre(newtype);
}


/* This function inserts NEW_DECL to varpool.  */

/* Create a new global variable of type TYPE.  */
tree
ipa_add_new_external_global (tree type, tree name)
{
  tree new_decl;
  struct varpool_node *new_node;

  new_decl = build_decl (UNKNOWN_LOCATION, VAR_DECL, name, type);
  TREE_READONLY (new_decl) = 0;
  TREE_STATIC (new_decl) = 1;
  TREE_PUBLIC (new_decl) = 1;
  TREE_USED (new_decl) = 1;
  DECL_COMDAT (new_decl) = 1;
  DECL_CONTEXT (new_decl) = NULL_TREE;
  DECL_ABSTRACT (new_decl) = 0;
  DECL_ARTIFICIAL (new_decl) = 1;
  lang_hooks.dup_lang_specific_decl (new_decl);
  create_var_ann (new_decl);
  new_node = varpool_node (new_decl);
  new_node->externally_visible = true;
  new_node->force_output = 1;
  varpool_mark_needed_node (new_node);
  varpool_finalize_decl (new_decl);

  return new_node->decl;
}



/* 
  class DEPENDENCY
	{
    int _a;  
    int _b;    
    int _loop_id;
    dep_type _type;
    dep_kind _kind;
	};
*/
static tree create_fields_of_dependency (void)
{

  tree id1 = get_identifier ("a");
  tree fld_decl_1 = build_decl (UNKNOWN_LOCATION, FIELD_DECL, id1, integer_type_node);

  tree id2 = get_identifier ("b");
  tree fld_decl_2 = build_decl (UNKNOWN_LOCATION, FIELD_DECL, id2, integer_type_node);

  tree id3 = get_identifier ("loop_id");
  tree fld_decl_3 = build_decl (UNKNOWN_LOCATION, FIELD_DECL, id3, integer_type_node);

  tree id4 = get_identifier ("type");
  tree fld_decl_4 = build_decl (UNKNOWN_LOCATION, FIELD_DECL, id4, integer_type_node);

  tree id5 = get_identifier ("kind");
  tree fld_decl_5 = build_decl (UNKNOWN_LOCATION, FIELD_DECL, id5, integer_type_node);

  TREE_CHAIN (fld_decl_1) = fld_decl_2;
  TREE_CHAIN (fld_decl_2) = fld_decl_3;
  TREE_CHAIN (fld_decl_3) = fld_decl_4;
  TREE_CHAIN (fld_decl_4) = fld_decl_5;

  return fld_decl_1;
}



/* 

  class DEPENDENCY
	{
    int _a;  
    int _b;    
    int _loop_id;
    dep_type _type;
    dep_kind _kind;
	};

*/

static tree
build_dependency_type ()
{

  tree attributes = NULL_TREE;
  tree ref = 0;
  tree x;
  tree fields = create_fields_of_dependency ();
  tree name;

  name = get_identifier ("DEPENDENCY");
  ref = make_node (RECORD_TYPE);
  TYPE_SIZE (ref) = 0;
  decl_attributes (&ref, attributes, (int) ATTR_FLAG_TYPE_IN_PLACE);
  TYPE_PACKED (ref) = 0;
  for (x = fields; x; x = TREE_CHAIN (x))
  {
    DECL_CONTEXT (x) = ref;
    DECL_PACKED (x) |= TYPE_PACKED (ref);
  }
  TYPE_FIELDS (ref) = fields;
  layout_type (ref);
  TYPE_NAME (ref) = name;

  return ref;
}







/* 
	
  class DEPENDENCY
	{
    int _a;  
    int _b;    
    int _loop_id;
    dep_type _type;
    dep_kind _kind;
	};
	
*/


static tree
build_initializer_for_dependency (ipa_data_dependency &dep)
{

  tree vals, cur, last, fld;

  /* Field a */
  fld = TYPE_FIELDS (objdep_type);
  cur = make_node (TREE_LIST);
  TREE_PURPOSE (cur) = fld;
  TREE_VALUE (cur) = build_int_cst_type (integer_type_node, dep.source());
  last = cur;
  vals = cur;

  /* Field b */
  fld = DECL_CHAIN (fld);
  cur = make_node (TREE_LIST);
  TREE_PURPOSE (cur) = fld;
  TREE_VALUE (cur) = build_int_cst_type (integer_type_node, dep.sink());
  TREE_CHAIN (last) = cur;
  last = cur;

  /* Field loop_id */
  fld = DECL_CHAIN (fld);
  cur = make_node (TREE_LIST);
  TREE_PURPOSE (cur) = fld;
  TREE_VALUE (cur) = build_int_cst_type (integer_type_node, dep.loop());
  TREE_CHAIN (last) = cur;
  last = cur;

  /* Field type */
  fld = DECL_CHAIN (fld);
  cur = make_node (TREE_LIST);
  TREE_PURPOSE (cur) = fld;
  TREE_VALUE (cur) = build_int_cst_type (integer_type_node, dep.type());
  TREE_CHAIN (last) = cur;
  last = cur;

  /* Field kind */
  fld = DECL_CHAIN (fld);
  cur = make_node (TREE_LIST);
  TREE_PURPOSE (cur) = fld;
  TREE_VALUE (cur) = build_int_cst_type (integer_type_node, dep.kind());
  TREE_CHAIN (last) = cur;
  last = cur;

  return build_constructor_from_list (objdep_type, vals);
}



/*
	see also in libprofile.cxx

	void __print_malloc  (PTR 	old_addr, PTR new_addr,	int 	size,  	char	flag);
	void __print_address	(PTR	 addr, 	int 	size,  	char	flag,  int 	id) ;
	void __print_range  (char 	flag,  int 	id,  ACCESS_FUNCTION	*access_fn, ... );
	void __print_edge( int loop_id, 	int entry );
	void __print_exit();
*/

static void
initialize_profile_prototype ()
{

  malloc_profiler_fn_type = build_function_type_list (void_type_node, ptr_type_node,	/* void *old_addr */
						      ptr_type_node,	/* void *new_addr */
						      integer_type_node,	/* int size */
						      char_type_node,	/* char flag */
						      NULL_TREE);


  address_profiler_fn_type = build_function_type_list (void_type_node, 
                                                      ptr_type_node,	/* void *addr */
                                    						       integer_type_node,	/* int size */
                                    						       char_type_node,	/* char flag */
                                    						       integer_type_node,
                                    						       NULL_TREE);
  
  SD3_address_profiler_fn_type = build_function_type_list (void_type_node, 
                                                      ptr_type_node,	/* void *addr */
                                    						       integer_type_node,	/* int size */
                                    						       char_type_node,	/* char flag */
                                    						       integer_type_node,
                                    						       NULL_TREE);

  pairwise_address_profiler_fn_type = build_function_type_list (void_type_node, 
                                                      ptr_type_node,	/* void *addr */
                                    						       integer_type_node,	/* int size */
                                    						       char_type_node,	/* char flag */
                                    						       integer_type_node,
                                    						       NULL_TREE);

  range_profiler_fn_type = build_function_type_list (void_type_node, char_type_node,	/* char flag */
						     integer_type_node,
						     ptr_type_node,
						     NULL_TREE);


  edge_profiler_fn_type = build_function_type_list (void_type_node, char_type_node,	/* char flag */
						    integer_type_node,	/* int entry */
						    NULL_TREE);

  SD3_edge_profiler_fn_type = build_function_type_list (void_type_node, char_type_node,	/* char flag */
						    integer_type_node,	/* int entry */
						    NULL_TREE);

  pairwise_edge_profiler_fn_type = build_function_type_list (void_type_node, char_type_node,	/* char flag */
						    integer_type_node,	/* int entry */
						    NULL_TREE);

  exit_profiler_fn_type = build_function_type_list (void_type_node,
						    NULL_TREE);

  SD3_exit_profiler_fn_type = build_function_type_list (void_type_node,
						    NULL_TREE);

  pairwise_exit_profiler_fn_type = build_function_type_list (void_type_node,
						    NULL_TREE);



  edge_address_profiler_fn_type = build_function_type_list (void_type_node, 
                                                      ptr_type_node,	/* void *addr */
                                    						       integer_type_node,	/* int size */
                                    						       char_type_node,	/* char flag */
                                    						       integer_type_node, /* dr id*/
                                                      integer_type_node, /* checking kind */
                                                      ptr_type_node, /* void *dep */
                                    						       NULL_TREE);

  edge_edge_profiler_fn_type = build_function_type_list (void_type_node, char_type_node, /* char flag */
                integer_type_node,  /* int entry */
                integer_type_node,  /* checking loop */
                ptr_type_node, /* void *dep */
                NULL_TREE);


  edge_exit_profiler_fn_type = build_function_type_list (void_type_node,
						    NULL_TREE);

  edge_start_profiler_fn_type = build_function_type_list (void_type_node,
						    NULL_TREE);

  // void __edge_open(int fd, char* fpath, int flags)
  edge_open_fn_type = build_function_type_list (void_type_node, integer_type_node, /* int fd */
                                                  ptr_type_node,  /* char* fpath */
                                                  integer_type_node,  /* int flags */
                                                  NULL_TREE);

  // void __edge_close(int fd)
  edge_close_fn_type = build_function_type_list (void_type_node, integer_type_node, /* FILE *fp */
                                                  NULL_TREE);
  
  /* 
     long __edge_read (int fd, char * ptr, long numbytes)
    */
  edge_read_fn_type = build_function_type_list (long_integer_type_node, integer_type_node, /* int fd */
                                                  ptr_type_node,  /* char * ptr */
                                                  long_integer_type_node,  /* int numbytes */
                                                  NULL_TREE);
  
  // ssize_t  __edge_write (int fd, char * ptr, size_t numbytes)  
  edge_write_fn_type = build_function_type_list (long_integer_type_node, integer_type_node, /* int fd */
                                                  ptr_type_node,  /* char * ptr */
                                                  size_type_node,  /* int numbytes */
                                                  NULL_TREE);
  
  // long __edge_lseek (int fd, long position, int startpoint)
  edge_lseek_fn_type = build_function_type_list (long_integer_type_node, integer_type_node, /* int fd */
                                                  long_integer_type_node,  /* int position */
                                                  integer_type_node,  /* int startpoint */
                                                  NULL_TREE);

  /* void __edge_fopen(FILE *fp, char* fpath, char* mode) */
  edge_fopen_fn_type = build_function_type_list (void_type_node, ptr_type_node, /* FILE *fp */
                                                  ptr_type_node,  /* char* fpath */
                                                  ptr_type_node,  /* char* mode */
                                                  NULL_TREE);

  /* void __edge_fclose(FILE *fp)   */
  edge_fclose_fn_type = build_function_type_list (void_type_node, ptr_type_node, /* FILE *fp */
                                                  NULL_TREE);

  /* size_t __edge_fread ( char * ptr, size_t size, size_t count, FILE * stream ) */
  edge_fread_fn_type = build_function_type_list (size_type_node, ptr_type_node, /* char * ptr */
                                                  size_type_node, /* size_t size */
                                                  size_type_node,  /* size_t count */
                                                  ptr_type_node,  /* FILE * stream */
                                                  NULL_TREE);

  /* size_t __edge_fwrite ( const char * ptr, size_t size, size_t count, FILE * stream ) */
  edge_fwrite_fn_type = build_function_type_list (size_type_node, const_ptr_type_node, /* char * ptr */
                                                  size_type_node, /* size_t size */
                                                  size_type_node,  /* size_t count */
                                                  ptr_type_node,  /* FILE * stream */
                                                  NULL_TREE);

  // int __edge_fseek ( FILE * stream, long offset, int origin )
  edge_fseek_fn_type = build_function_type_list (integer_type_node, ptr_type_node, /* FILE * stream */
                                                long_integer_type_node,  /* long offset */
                                                integer_type_node,  /* int origin */
                                                NULL_TREE);
  // long _edge_ftell ( FILE * stream )
  edge_ftell_fn_type = build_function_type_list (long_integer_type_node, ptr_type_node, /* FILE * stream */
                                                 NULL_TREE);

  // void _edge_rewind ( FILE * stream )
  edge_rewind_fn_type = build_function_type_list (void_type_node, ptr_type_node, /* FILE * stream */
                                                 NULL_TREE);

  // int _edge_flush ( FILE * stream )
  edge_fflush_fn_type = build_function_type_list (integer_type_node, ptr_type_node, /* FILE * stream */
                                                 NULL_TREE);

  // int _edge_ferror ( FILE * stream )
  edge_ferror_fn_type = build_function_type_list (integer_type_node, ptr_type_node, /* FILE * stream */
                                               NULL_TREE);
  
  // void _edge_cleanerror ( FILE * stream )
  edge_cleanerror_fn_type = build_function_type_list (void_type_node, ptr_type_node, /* FILE * stream */
                                               NULL_TREE);

  // int _edge_feof ( FILE * stream )
  edge_feof_fn_type = build_function_type_list (integer_type_node, ptr_type_node, /* FILE * stream */
                                               NULL_TREE);

  // int __edge_fgetc (FILE * stream )
  edge_fgetc_fn_type = build_function_type_list (integer_type_node, ptr_type_node, /* FILE * stream */
                                               NULL_TREE);

  
  // char * __edge_fgets ( char * str, int num, FILE * stream )
  edge_fgets_fn_type = build_function_type_list (ptr_type_node, ptr_type_node, /* char * str */
                                                 integer_type_node,
                                                 ptr_type_node, /* FILE * stream */
                                               NULL_TREE);
  
  // int __edge_fputs ( char * str, FILE * stream )  
  edge_fputs_fn_type = build_function_type_list (integer_type_node, ptr_type_node, /* char * str */
                                                  ptr_type_node, /* FILE * stream */
                                                 NULL_TREE);

  // int __edge_fputc ( int character, FILE * stream )
  edge_fputc_fn_type = build_function_type_list (integer_type_node, integer_type_node, /* int character */
                                                ptr_type_node, /* FILE * stream */
                                               NULL_TREE);
  // int __edge_fscanf ( FILE * stream, const char * format, ... )
  edge_fscanf_fn_type = build_varargs_function_type_list (integer_type_node, ptr_type_node, /* FILE * stream */
                                                  const_ptr_type_node, /* const char * format */
                                                  NULL_TREE);
  
  // int __edge_fprintf ( FILE * stream, const char * format, ... )
  edge_fprintf_fn_type = build_varargs_function_type_list (integer_type_node, ptr_type_node, /* FILE * stream */
                                                  const_ptr_type_node, /* const char * format */
                                                  NULL_TREE);
  
  // int __edge_fprintf ( const char * format, ... )
  edge_printf_fn_type = build_varargs_function_type_list (integer_type_node, 
                                                  const_ptr_type_node, /* const char * format */
                                                  NULL_TREE);

  // int __edge_vfprintf ( FILE * stream, const char * format, va_list argList )
  edge_vfprintf_fn_type = build_varargs_function_type_list (integer_type_node, ptr_type_node, /* FILE * stream */
                                                  const_ptr_type_node, /* const char * format */
                                                  va_list_type_node,  /* va_list argList */
                                                  NULL_TREE);

  // int  __edge_puts ( const char * str )
  edge_puts_fn_type = build_function_type_list (integer_type_node, ptr_type_node, /*  const char * str */
                                               NULL_TREE);
  // int  __edge_putchar ( int character )
  edge_putchar_fn_type = build_function_type_list (integer_type_node, integer_type_node, /*  int character */
                                               NULL_TREE);


  // void * malloc ( size_t size )
  edge_malloc_fn_type = build_function_type_list (ptr_type_node, size_type_node, /*  int character */
                                                 NULL_TREE);

  // void * calloc ( size_t num, size_t size );
  edge_calloc_fn_type = build_function_type_list (ptr_type_node, size_type_node, /*  int character */
                                                  size_type_node, NULL_TREE);

  // void * realloc ( void * ptr, size_t size );
  edge_realloc_fn_type = build_function_type_list (ptr_type_node, ptr_type_node, /*  int character */
                                                  size_type_node, NULL_TREE);

  // void free ( void * ptr );
  edge_free_fn_type = build_function_type_list (void_type_node, ptr_type_node, /*  int character */
                                                NULL_TREE);




  // void  __privatize_print_store (PTR addr, int size, int id)
  privatize_store_profiler_fn_type = build_function_type_list (void_type_node, 
                                                    ptr_type_node,  /* void *addr */
                                                     integer_type_node, /* int size */
                                                     integer_type_node, /* dr id*/
                                                     NULL_TREE);

  privatize_load_profiler_fn_type = build_function_type_list (void_type_node, 
                                                  ptr_type_node,  /* void *addr */
                                                   integer_type_node, /* int size */
                                                   integer_type_node, /* dr id*/
                                                   NULL_TREE);

  // void __privatize_print_entry (int loop_id)
  privatize_entry_profiler_fn_type = build_function_type_list (void_type_node, 
                                                    integer_type_node,  /* checking loop */
                                                    NULL_TREE);

  privatize_exit_profiler_fn_type = build_function_type_list (void_type_node, 
                                                  integer_type_node,  /* checking loop */
                                                  NULL_TREE);

  privatize_end_profiler_fn_type = build_function_type_list (void_type_node, NULL_TREE);


  shadow_store_profiler_fn_type = build_function_type_list (void_type_node, 
                                                    ptr_type_node,  /* void *addr */
                                                     integer_type_node, /* int size */
                                                     integer_type_node, /* dr id*/
                                                     NULL_TREE);

  shadow_load_profiler_fn_type = build_function_type_list (void_type_node, 
                                                  ptr_type_node,  /* void *addr */
                                                   integer_type_node, /* int size */
                                                   integer_type_node, /* dr id*/
                                                   NULL_TREE);

  // void __privatize_print_entry (int loop_id)
  shadow_entry_profiler_fn_type = build_function_type_list (void_type_node, 
                                                    integer_type_node,  /* checking loop */
                                                    NULL_TREE);

  shadow_exit_profiler_fn_type = build_function_type_list (void_type_node, 
                                                  integer_type_node,  /* checking loop */
                                                  NULL_TREE);

  shadow_end_profiler_fn_type = build_function_type_list (void_type_node, NULL_TREE);

  shadow_begin_profiler_fn_type = build_function_type_list (void_type_node, 
                                                  integer_type_node,  /* int mem_num */
                                                  NULL_TREE);

#if 0
  for (struct cgraph_node *node = cgraph_nodes; node; node = node->next)
  {
    
  	switch_to_context (node->decl);

    switch (flag_profiling_type)
    {
      case PROFILING_PRIVATIZABLE :
      
        if ( strcmp( cgraph_node_name(node), shadow_privatizable_store_profiler_name) == 0 )
        {
          DECL_DECLARED_INLINE_P(node->decl) = 1;
          DECL_UNINLINABLE (node->decl) = 0;
          node->local.inlinable = 1;
          shadow_store_profiler_fn = node->decl;
        }
        else if ( strcmp(cgraph_node_name(node), shadow_privatizable_load_profiler_name) ==0 )
        {
          DECL_DECLARED_INLINE_P(node->decl) = 1;
          DECL_UNINLINABLE (node->decl) = 0;
          node->local.inlinable = 1;
          shadow_load_profiler_fn = node->decl;
        }
        else if ( strcmp(cgraph_node_name(node),shadow_privatizable_begin_profiler_name)==0 )
        {
          DECL_UNINLINABLE (node->decl) = 0;
          node->local.inlinable = 1;
          shadow_begin_profiler_fn = node->decl;
        }
        else if ( strcmp(cgraph_node_name(node), shadow_privatizable_end_profiler_name) ==0 )
        {
          DECL_UNINLINABLE (node->decl) = 0;
          node->local.inlinable = 1;
          shadow_end_profiler_fn = node->decl;
        }

        break;

       case PROFILING_DEPENDENCE :
         if ( strcmp( cgraph_node_name(node), shadow_dependence_store_profiler_name) == 0 )
         {
           //DECL_DECLARED_INLINE_P(node->decl) = 1;
           //DECL_UNINLINABLE (node->decl) = 0;
           //node->local.inlinable = 1;
           shadow_store_profiler_fn = node->decl;
         }
         else if ( strcmp(cgraph_node_name(node), shadow_dependence_load_profiler_name) ==0 )
         {
          // DECL_DECLARED_INLINE_P(node->decl) = 1;
           //DECL_UNINLINABLE (node->decl) = 0;
           //node->local.inlinable = 1;
           shadow_load_profiler_fn = node->decl;
         }
         else if ( strcmp(cgraph_node_name(node), shadow_dependence_begin_profiler_name)==0 )
         {
           DECL_UNINLINABLE (node->decl) = 0;
           node->local.inlinable = 1;
           shadow_begin_profiler_fn = node->decl;
         }
         else if ( strcmp(cgraph_node_name(node), shadow_dependence_end_profiler_name) ==0 )
         {
           DECL_UNINLINABLE (node->decl) = 0;
           node->local.inlinable = 1;
           shadow_end_profiler_fn = node->decl;
         }
         break;
         
       default:
          break;
    };

    if ( strcmp(cgraph_node_name(node),shadow_privatizable_entry_profiler_name) ==0 )
    {
     // DECL_UNINLINABLE (node->decl) = 0;
      //node->local.inlinable = 1;
      shadow_entry_profiler_fn = node->decl;
    }
    else if ( strcmp(cgraph_node_name(node), shadow_privatizable_exit_profiler_name)==0 )
    {
      //DECL_UNINLINABLE (node->decl) = 0;
      //node->local.inlinable = 1;
      shadow_exit_profiler_fn = node->decl;
    }
     

  	switch_off_context ();
  }
#endif

}


/* Specify which dependence edge to profile */
void 
specify_loops_to_profile()
{

  loop_set.clear();
  loop_set.insert(-1); // profile all loops

  char *loopinfo = NULL;

  find_candidate_loops (loop_set);

  if ( flag_profile_loop )
    loop_set.insert( flag_profile_loop );

  if ( flag_profile_loops )
  {
      
    FILE *fp = fopen ( flag_profile_loops , "r");
    gcc_assert (fp);    

    char *line = NULL;
    size_t len = 0;
    int bytes_read;
    char str[100];

    while ((bytes_read = getline (&line, &len, fp)) != -1)
    {
      sscanf (line, "%s", str);

      if (strcmp (str, "LOOP") == 0 )
      {       
        // ignore "LOOP", go on reading loop_id
        int loop_id;
        sscanf (line, "%*s %d", &loop_id);
        loop_set.insert( loop_id );
      }      
    }
  }

  if (flag_profile_loop_srcinfo)
  {
    std::string buf;
    std::string file;
    int startline=0;
    int endline=0;
    for (int i=0; i < strlen(flag_profile_loop_srcinfo); i++)
    {
      if (flag_profile_loop_srcinfo[i] != ',' && flag_profile_loop_srcinfo[i] !=':')
        buf += flag_profile_loop_srcinfo[i];
      else if (flag_profile_loop_srcinfo[i] == ':')
      {
        file = buf;      
        buf.clear();
      }
      else
      {
        startline = atoi( buf.c_str() );
        buf.clear();
      }
    }
    endline = atoi( buf.c_str() );

    bool found=false;
    loop_p candidate = NULL;
    struct cgraph_node *node;
    for (node = cgraph_nodes; node; node = node->next)
    {

      if (found)
        break;

      if (!valid_function_node_p (node))
        continue;    


      switch_to_context(node->decl);
      
 
      if ( file.compare(DECL_SOURCE_FILE (current_function_decl)) != 0 )
      {
        switch_off_context();
        continue;
      }

      
      loop_iterator li;
      loop_p loop = NULL;
      FOR_EACH_LOOP (li, loop, LI_FROM_INNERMOST)
      {
      
        if ( loop_linenum(loop) == startline )
        {
          candidate = loop;
          found = true;
          break;
        }        
        else if ( loop_linenum(loop) == endline )
        {
          candidate = loop;
          found = true;
          break;
        }        

        else if ( loop_startline(loop) >= startline && loop_endline(loop) <=endline )
        {
          if ( !candidate )
            candidate = loop;
          else if ( loop_startline(candidate) >= loop_startline(loop) && loop_endline(candidate) <= loop_endline(loop) )
            candidate = loop;                        
        }              
        
      }


      switch_off_context();
      
    }

    gcc_assert(candidate);
    loop_set.insert( loop_uid (candidate) );
    
  }

  if ( loop_set.size() > 1 )
    loop_set.erase(-1);

  
  if (!(loop_set.size() == 1 && *loop_set.begin() >0 ))
  {
    printf("Please specify a loop, like xxx.c:startline,endline");
    gcc_assert (false);
  }
}

/* Specify which dependence edge to profile */
static void 
specify_functions_to_profile()
{

  char *funcinfo = NULL;

  if ( flag_profile_fn )
    func_set.insert( flag_profile_fn );

  if ( flag_profile_fns )
  {
      
    FILE *fp = fopen ( flag_profile_loops , "r");
    gcc_assert (fp);    

    char *line = NULL;
    size_t len = 0;
    int bytes_read;
    char str[100];

    while ((bytes_read = getline (&line, &len, fp)) != -1)
    {
      sscanf (line, "%s", str);

      if (strcmp (str, "FUNC") == 0 )
      {       
        // ignore "FUNC", go on reading funtion name
        sscanf (line, "%*s %s", str);
        func_set.insert( str );
      }      
    }
  }
}


/* Specify which alias class to profile */
static void 
specify_alias_class_to_profile()
{

  strided = true;

  if ( flag_profile_aliasclass != 0 )
    alias_set.insert (flag_profile_aliasclass);  
  
}


/* Specify which dependence edge to profile */
static void 
specify_dep_edge_to_profile()
{

  strided = true;
  char *edgeinfo = NULL;

  if ( flag_profile_dep_file )
    edgeinfo = flag_profile_dep_file;
  else
    edgeinfo = "edge.info";

  if ( flag_profile_depedge != 0 )
  {
      
    FILE *fp = fopen ( edgeinfo , "r");
    gcc_assert (fp);    

    char *line = NULL;
    size_t len = 0;
    int bytes_read;
    int a, b, loop_id, kind;
    char str[100];
    bool profile_all = true;
    bool force_stride = false;
    bool force_point = false;
    bool read_until = false;
    ipa_data_dependency tmp_indep;
    ipa_data_dependency tmp_carried;

    while ((bytes_read = getline (&line, &len, fp)) != -1)
    {
      sscanf (line, "%s", str);

      if (strcmp (str, "CLASS") == 0 || strcmp (str, "EDGE") == 0)
      {
        if ( read_until )
          break;
        
        // ignore "CLASS", go on reading class_id
        int class_id;
        sscanf (line, "%*s %d", &class_id);
        if ( class_id == flag_profile_depedge )
          read_until = true;    
      }

      
      if ( !read_until )
        continue;

      
      if (strcmp (str, "LOOP") == 0)
      {
        /* Try to recognize the form :     LOOP 86  WAW    576 --> 562  MAY LOOP INDEPENDENT     */
        char dep_kind[100];
        sscanf (line, "%*s %d %*s %d %*s %d %*s %*s %s", &loop_id, &a, &b, dep_kind );
        edge_set.insert (a) ;
        edge_set.insert (b) ;
        if (strcmp (dep_kind, "INDEPENDENT") == 0)
        {
          checking_kind |= 0x1;
          tmp_indep.set_source(a);
          tmp_indep.set_sink(b);        
          tmp_indep.set_kind(id_loop_independent);
          tmp_indep.set_loop(loop_id);        
        }
        else if (strcmp (dep_kind, "CARRIED") == 0)
        {
          checking_kind |= 0x2;
          tmp_carried.set_source(a);
          tmp_carried.set_sink(b);        
          tmp_carried.set_kind(id_loop_carried);
          tmp_carried.set_loop(loop_id);        
        }
        else
          gcc_assert (false);     

      } 
      
    }
     
    fclose (fp);

    if ( checking_kind & 0x2 )
      objdep = tmp_carried;
    else
      objdep = tmp_indep;        

    objdep_type = build_dependency_type ();

    /* Build variable declare */
    objdep_var = ipa_add_new_external_global (objdep_type, get_identifier ("objdep"));
    add_referenced_var (objdep_var);

    tree objdep_init = build_initializer_for_dependency (objdep);
    DECL_INITIAL (objdep_var) = objdep_init;
    insert_global_to_varpool (objdep_var);

  }
      
}



static void read_mem_info( std::set<int> loop_set,  std::set<int> &mem_set )
{
  if ( loop_set.empty() )
    return;


  FILE *meminfo = fopen ("memop.info", "r");
  if (!meminfo)
  {
    fprintf (stderr, "No file memop.info found!\n" );
    gcc_assert(false);
  }

  char *line = NULL;
  size_t len = 0;
  int bytes_read;
  int a, b, loop_id, kind;
  char str[200];
  while ((bytes_read = getline (&line, &len, meminfo)) != -1)
  {
    sscanf (line, "%s", str);

    if (strcmp (str, "LOOP") == 0)
    {
      char must[10];
      /* Try to recognize the form :     LOOP 86  WAW    576 --> 562  MAY LOOP INDEPENDENT     */
      char kind[100];
      char type[100];
      sscanf (line, "%*s %d %s %d %*s %d %s %*s %s", &loop_id, type, &a, &b, must, kind );
      if ( strcmp(must, "MUST") ==0 && strcmp(kind, "CARRIED") == 0 )
        continue;

      if ( loop_set.find(loop_id) != loop_set.end() )
      {
        mem_set.insert( a );
        mem_set.insert( b );
      }
    }   
  }

  fclose(meminfo);

}


/* New call was added, make goto call edges if neccesary.  */

void
ipa_add_abnormal_goto_call_edges (gimple_stmt_iterator gsi)
{
  gimple stmt = gsi_stmt (gsi);

  if (!stmt_can_make_abnormal_goto (stmt))
    return;
  if (!gsi_end_p (gsi))
    split_block (gimple_bb (stmt), stmt);
  make_abnormal_goto_edges (gimple_bb (stmt), true);
}




tree
dr_var_name (ipa_data_reference * dr, const char *prefix)
{
  tree name;
  int id = dr->uid ();
  char a[30];
  sprintf (a, "%s_%d", prefix, id);
  name = get_identifier (a);
  return name;
}

void
update_new_call_edge (struct cgraph_node *node)
{

  basic_block bb;
  FOR_EACH_BB (bb)
  {
    int bb_freq = compute_call_stmt_bb_frequency (current_function_decl, bb);
    for (gimple_stmt_iterator gsi = gsi_start_bb (bb); !gsi_end_p (gsi); gsi_next (&gsi))
    {
      gimple stmt = gsi_stmt (gsi);
      if (!is_gimple_call (stmt))
        continue;

      struct cgraph_edge *e = cgraph_edge (node, stmt);
      
      tree decl = gimple_call_fndecl (stmt);
      if (!decl)
        continue;
      
      struct cgraph_node *callee_node = cgraph_node (decl);
      if (e == NULL)
        e = cgraph_create_edge (node, callee_node, stmt, bb->count, bb_freq, bb->loop_depth);      
      
      e->frequency = bb_freq;        
    }
  }

}





/* Generate the address taken node of a memory */
tree
gimple_build_addr_expr (gimple_stmt_iterator * gsi, tree mem)
{

  tree addr = build_int_cst (0, 0);
  tree byte = build_int_cst (0, 8);;
  tree offset;

  switch (TREE_CODE (mem))
  {
    case PARM_DECL:
    case VAR_DECL:
      addr = gimplify_build1 (gsi, ADDR_EXPR, build_pointer_type (TREE_TYPE (mem)), mem);
      break;

    case SSA_NAME:
      addr = gimple_build_addr_expr (gsi, SSA_NAME_VAR (mem));
      break;

    case INDIRECT_REF:
      addr = TREE_OPERAND (mem, 0);
      addr = force_gimple_operand_gsi (gsi, addr, true, NULL, true, GSI_SAME_STMT);
      break;

    case MEM_REF:
      offset = 	build1_stat (CONVERT_EXPR, size_type_node, TREE_OPERAND (mem, 1));
      addr = gimplify_build2 (gsi, POINTER_PLUS_EXPR, build_pointer_type (TREE_TYPE (mem)),
                      			 TREE_OPERAND (mem, 0), offset);
      break;

    case ARRAY_REF:
      addr = gimple_build_addr_expr (gsi, TREE_OPERAND (mem, 0));
      offset =	build1_stat (CONVERT_EXPR, size_type_node, TREE_OPERAND (mem, 1));
      offset = 	gimplify_build2 (gsi, MULT_EXPR, size_type_node, offset, array_ref_element_size (mem));
      offset = build1_stat (CONVERT_EXPR, size_type_node, offset);
      addr = gimplify_build2 (gsi, POINTER_PLUS_EXPR, build_pointer_type (TREE_TYPE (mem)), addr, offset);
      break;

    case COMPONENT_REF:
      addr = gimple_build_addr_expr (gsi, TREE_OPERAND (mem, 0));
      offset = byte_position (TREE_OPERAND (mem, 1));
      offset = build1_stat (CONVERT_EXPR, size_type_node, offset);
      addr =
	gimplify_build2 (gsi, POINTER_PLUS_EXPR, ptr_type_node, addr, offset);
      break;

    case BIT_FIELD_REF:
      addr = gimple_build_addr_expr (gsi, TREE_OPERAND (mem, 0));
      offset =
	gimplify_build2 (gsi, TRUNC_DIV_EXPR, size_type_node,
			 TREE_OPERAND (mem, 2), byte);
      offset = build1_stat (CONVERT_EXPR, size_type_node, offset);
      addr =
	gimplify_build2 (gsi, POINTER_PLUS_EXPR, ptr_type_node, addr, offset);
      break;

    default:
      break;


    }


  return addr;

}






static void
insert_exit (gimple_stmt_iterator * gsi)
{

  /*
     see also in libprofile.cxx   
     void __print_exit  ();
   */

  tree tree_printf_profiler_fn = build_fn_decl (exit_profiler_name, exit_profiler_fn_type);
  gimple call = gimple_build_call (tree_printf_profiler_fn, 0);
  gsi_insert_before (gsi, call, GSI_SAME_STMT);
  ipa_add_abnormal_goto_call_edges (*gsi);

}



static void
insert_malloc (gimple_stmt_iterator * gsi, tree old_addr, tree new_addr,
	       tree size, int flag)
{

  /*
     see also in libprofile.cxx   
     void __print_malloc  (PTR    old_addr, PTR new_addr, int     size,   char    flag);
   */

  if (!new_addr)
    return;

  tree tree_printf_profiler_fn = build_fn_decl (malloc_profiler_name, malloc_profiler_fn_type);

  tree flag_t = build_int_cst_type (char_type_node, flag);
  gimple call = gimple_build_call (tree_printf_profiler_fn, 4, old_addr, new_addr, size,
                        		       flag_t);

  gsi_insert_after (gsi, call, GSI_NEW_STMT);
  ipa_add_abnormal_goto_call_edges (*gsi);

}



/* Clear profile information attached to gimple
	 If DEFUSE=1, then REF is a write. */
static void
insert_address (gimple_stmt_iterator * gsi, tree ref, int opnd, int defuse)
{
  gimple call;
  tree addr, size, loc, flag;
  tree tree_printf_profiler_fn;


  /* Do not need to profile temporary variables */
  if (TREE_CODE (ref) == VAR_DECL && DECL_ARTIFICIAL (ref))
    return;

  if (TREE_CODE (ref) == SSA_NAME)
    return;


  gimple stmt = gsi_stmt (*gsi);
  ipa_data_reference *dr = tree_data_reference (stmt, opnd);

  if (!dr)
    return;

  if ( flag_ipa_expose_reference && !dr->need_profile ())
    return;


  if (alias_set.find(-1) == alias_set.end() && 
      alias_set.find(dr->class_id()) == alias_set.end())
    return;

	profile_count++;
  

  flag = build_int_cst_type (char_type_node, defuse);
  loc = build_int_cst_type (integer_type_node, dr->uid ());

//  if (dr->pattern ().kind () == pk_no_pattern)
  if ( !strided )
  {
    /* If not pattern, it is a point */
    tree_printf_profiler_fn = build_fn_decl (address_profiler_name, address_profiler_fn_type);
    /*
       see also in libprofile.cxx   
       void __print_address (PTR     addr,  int     size,   char    flag,  int      id) ;
     */

    switch (TREE_CODE (ref))
    {
      case COMPONENT_REF:
      case ARRAY_REF:
      case INDIRECT_REF:
      case MEM_REF:
      case VAR_DECL:
        addr = gimple_build_addr_expr (gsi, ref);
        break;
      default:
        return;
        break;
    }

    gcc_assert (addr);

    size = TYPE_SIZE (TREE_TYPE (ref));
    call = gimple_build_call (tree_printf_profiler_fn, 4, addr, size, flag, loc);

  }
  else
  {
    tree_printf_profiler_fn = build_fn_decl (SD3_address_profiler_name, SD3_address_profiler_fn_type);
    /*
       see also in libprofile.cxx   
       void __print_address (PTR     addr,  int     size,   char    flag,  int      id) ;
     */

    switch (TREE_CODE (ref))
    {
      case COMPONENT_REF:
      case ARRAY_REF:
      case INDIRECT_REF:
      case MEM_REF:
      case VAR_DECL:
        addr = gimple_build_addr_expr (gsi, ref);
        break;
      default:
        return;
        break;
    }

    gcc_assert (addr);

    size = TYPE_SIZE (TREE_TYPE (ref));
    call = gimple_build_call (tree_printf_profiler_fn, 4, addr, size, flag, loc);

  }


  gsi_insert_before (gsi, call, GSI_SAME_STMT);
  ipa_add_abnormal_goto_call_edges (*gsi);

}


static void
insert_edge (int entry, int loop_id, edge e)
{

  tree ent = build_int_cst_type (integer_type_node, entry);
  tree lnum = build_int_cst_type (unsigned_type_node, loop_id);


  /* see also in libprofile.cxx
     void __print_edge( int loop_id,      // The global loop id 
     int entry            // 1- loop entry , 0- loop exit 
     )                                                    */

  tree tree_printf_profiler_fn =
    build_fn_decl (edge_profiler_name, edge_profiler_fn_type);

  gimple call =
    gimple_build_call (tree_printf_profiler_fn, 2, lnum, ent);

  gimple_seq seq = gimple_seq_alloc ();
  gimple_seq_add_stmt (&seq, call);
  gsi_insert_seq_on_edge (e, seq);

}



/* Clear profile information attached to gimple
	 If DEFUSE=1, then REF is a write. */
static void
insert_address_SD3 (gimple_stmt_iterator * gsi, tree ref, int opnd, int defuse)
{
  gimple call;
  tree addr, size, loc, flag;
  tree tree_printf_profiler_fn;


  gimple stmt = gsi_stmt (*gsi);
  ipa_data_reference *dr = tree_data_reference (stmt, opnd);

  if (!dr)
    return;
  
  if ( !is_shadow_profile_candidate(dr, ref) )
    return;

  if (dr->is_read)
  {
    if (!bitmap_bit_p(loop_extra_info(cur_loop)->dr_read, dr->uid ()))
      return;    
  }
  else if (!bitmap_bit_p(loop_extra_info(cur_loop)->dr_write, dr->uid ()))
      return;    


  flag = build_int_cst_type (char_type_node, defuse);
  loc = build_int_cst_type (integer_type_node, dr->uid ());

  
  {
    tree_printf_profiler_fn = build_fn_decl (SD3_address_profiler_name, SD3_address_profiler_fn_type);
    /*
       see also in libprofile.cxx   
       void __print_address (PTR     addr,  int     size,   char    flag,  int      id) ;
     */

    switch (TREE_CODE (ref))
    {
      case COMPONENT_REF:
      case ARRAY_REF:
      case INDIRECT_REF:
      case MEM_REF:
      case VAR_DECL:
        addr = gimple_build_addr_expr (gsi, ref);
        break;
      default:
        return;
        break;
    }

    gcc_assert (addr);

    size = TYPE_SIZE (TREE_TYPE (ref));
    call = gimple_build_call (tree_printf_profiler_fn, 4, addr, size, flag, loc);

  }


  gsi_insert_before (gsi, call, GSI_SAME_STMT);
  ipa_add_abnormal_goto_call_edges (*gsi);

}

static void
insert_edge_SD3 (int entry, int loop_id, edge e)
{

  if (loop_id != *(loop_set.begin()))
    return;

  tree ent = build_int_cst_type (integer_type_node, entry);
  tree lnum = build_int_cst_type (unsigned_type_node, loop_id);


  /* see also in libprofile.cxx
     void __print_edge( int loop_id,      // The global loop id 
     int entry            // 1- loop entry , 0- loop exit 
     )                                                    */

  tree tree_printf_profiler_fn = build_fn_decl (SD3_edge_profiler_name, SD3_edge_profiler_fn_type);

  gimple call = gimple_build_call (tree_printf_profiler_fn, 2, lnum, ent);

  gimple_seq seq = gimple_seq_alloc ();
  gimple_seq_add_stmt (&seq, call);
  gsi_insert_seq_on_edge (e, seq);

}


static void
insert_exit_SD3 (gimple_stmt_iterator * gsi)
{

  /*
     see also in libprofile.cxx   
     void __print_exit  ();
   */

  tree tree_printf_profiler_fn = build_fn_decl (SD3_exit_profiler_name, SD3_exit_profiler_fn_type);
  gimple call = gimple_build_call (tree_printf_profiler_fn, 0);
  gsi_insert_before (gsi, call, GSI_SAME_STMT);
  ipa_add_abnormal_goto_call_edges (*gsi);

}



void
profile_IO_operation (struct cgraph_node *node)
{

  /* print malloc */
  basic_block bb;
  FOR_EACH_BB (bb)
  {

    gimple_seq seq = bb_seq (bb);
    for (gimple_stmt_iterator gsi = gsi_start_bb (bb); !gsi_end_p (gsi); gsi_next (&gsi))
    {

      int flag;
      tree size;
      tree old_addr;
      tree new_addr;
      gimple stmt = gsi_stmt (gsi);

      switch (gimple_code (stmt))
      {

        case GIMPLE_CALL:
        {
          /* the arguments are to be profiled!!! */
          tree decl = gimple_call_fndecl (stmt);

          if (decl)
        	{
            const char* name = fn_decl_name (decl);
            tree tree_edge_fn;
            gimple call;

            if ( strcmp(name, "fopen") == 0 )
            {
              tree_edge_fn = build_fn_decl (edge_fopen_name, edge_fopen_fn_type);                
              call = gimple_build_call (tree_edge_fn, 3, gimple_call_lhs (stmt),  
                                         gimple_call_arg (stmt, 0), gimple_call_arg (stmt, 1));
              gsi_insert_after(&gsi, call, GSI_NEW_STMT);
              break;
            } 
            else if ( strcmp(name, "open") == 0 )
            {
              tree_edge_fn = build_fn_decl (edge_open_name, edge_open_fn_type);                
              call = gimple_build_call (tree_edge_fn, 3, gimple_call_lhs (stmt),  
                                         gimple_call_arg (stmt, 0), gimple_call_arg (stmt, 1));
              gsi_insert_after(&gsi, call, GSI_NEW_STMT);
              break;
            } 
            else if ( strcmp(name, "fclose") == 0 )
            {
              tree_edge_fn = build_fn_decl (edge_fclose_name, edge_fclose_fn_type);                 
              call = gimple_build_call (tree_edge_fn, 1, gimple_call_arg (stmt, 0));
              gsi_insert_before(&gsi, call, GSI_SAME_STMT);
              break;
    	      }             
            else if ( strcmp(name, "close") == 0 )
            {
              tree_edge_fn = build_fn_decl (edge_close_name, edge_close_fn_type);                 
              call = gimple_build_call (tree_edge_fn, 1, gimple_call_arg (stmt, 0));
              gsi_insert_before(&gsi, call, GSI_SAME_STMT);
              break;
    	      }             

            /* The same declaration with __edge_ version and need to replace the original */
            
        	  if (DECL_FUNCTION_CODE (decl) == BUILT_IN_FWRITE )
              tree_edge_fn = build_fn_decl (edge_fwrite_name, edge_fwrite_fn_type);         
        	  else if (DECL_FUNCTION_CODE (decl) == BUILT_IN_VFPRINTF)
              tree_edge_fn = build_fn_decl (edge_vfprintf_name, edge_vfprintf_fn_type); 
        	  else if (DECL_FUNCTION_CODE (decl) == BUILT_IN_FPRINTF)
              tree_edge_fn = build_fn_decl (edge_fprintf_name, edge_fprintf_fn_type); 
        	  else if (DECL_FUNCTION_CODE (decl) == BUILT_IN_PRINTF)
              tree_edge_fn = build_fn_decl (edge_printf_name, edge_printf_fn_type); 
        	  else if (DECL_FUNCTION_CODE (decl) == BUILT_IN_FSCANF)
              tree_edge_fn = build_fn_decl (edge_fscanf_name, edge_fscanf_fn_type); 
            else if (DECL_FUNCTION_CODE (decl) == BUILT_IN_FPUTC )
              tree_edge_fn = build_fn_decl (edge_fputc_name, edge_fputc_fn_type); 
            else if (DECL_FUNCTION_CODE (decl) == BUILT_IN_PUTS)
              tree_edge_fn = build_fn_decl (edge_puts_name, edge_puts_fn_type);                       
            else if (DECL_FUNCTION_CODE (decl) == BUILT_IN_FPUTS)
              tree_edge_fn = build_fn_decl (edge_fputs_name, edge_fputs_fn_type);             
            else if (DECL_FUNCTION_CODE (decl) == BUILT_IN_PUTCHAR)
              tree_edge_fn = build_fn_decl (edge_putchar_name, edge_putchar_fn_type);             
            else if ( strcmp(name, "fread") == 0 )
              tree_edge_fn = build_fn_decl (edge_fread_name, edge_fread_fn_type); 
            else if ( strcmp(name, "fseek") == 0 )
              tree_edge_fn = build_fn_decl (edge_fseek_name, edge_fseek_fn_type); 
            else if ( strcmp(name, "ftell") == 0 )
              tree_edge_fn = build_fn_decl (edge_ftell_name, edge_ftell_fn_type); 
            else if ( strcmp(name, "fflush") == 0 )
              tree_edge_fn = build_fn_decl (edge_fflush_name, edge_fflush_fn_type); 
            else if ( strcmp(name, "ferror") == 0 )
              tree_edge_fn = build_fn_decl (edge_ferror_name, edge_ferror_fn_type); 
            else if ( strcmp(name, "cleanerror") == 0 )
              tree_edge_fn = build_fn_decl (edge_cleanerror_name, edge_cleanerror_fn_type); 
            else if ( strcmp(name, "feof") == 0 )
              tree_edge_fn = build_fn_decl (edge_feof_name, edge_feof_fn_type); 
            else if ( strcmp(name, "fgetc") == 0 )
              tree_edge_fn = build_fn_decl (edge_fgetc_name, edge_fgetc_fn_type); 
            else if ( strcmp(name, "getc") == 0 )
              tree_edge_fn = build_fn_decl (edge_fgetc_name, edge_fgetc_fn_type); 
            else if ( strcmp(name, "_IO_getc") == 0 )
              tree_edge_fn = build_fn_decl (edge_fgetc_name, edge_fgetc_fn_type); 
            else if ( strcmp(name, "fgets") == 0 )
              tree_edge_fn = build_fn_decl (edge_fgets_name, edge_fgets_fn_type); 
            else if ( strcmp(name, "lseek") == 0 )
              tree_edge_fn = build_fn_decl (edge_lseek_name, edge_lseek_fn_type); 
            else if ( strcmp(name, "write") == 0 )
              tree_edge_fn = build_fn_decl (edge_write_name, edge_write_fn_type); 
            else if ( strcmp(name, "read") == 0 )
              tree_edge_fn = build_fn_decl (edge_read_name, edge_read_fn_type); 
            else if ( strcmp(name, "rewind") == 0 )
              tree_edge_fn = build_fn_decl (edge_rewind_name, edge_rewind_fn_type); 
        	  else if (DECL_FUNCTION_CODE (decl) == BUILT_IN_MALLOC)
              tree_edge_fn = build_fn_decl (edge_malloc_name, edge_malloc_fn_type); 
        	  else if (DECL_FUNCTION_CODE (decl) == BUILT_IN_CALLOC)
              tree_edge_fn = build_fn_decl (edge_calloc_name, edge_calloc_fn_type); 
        	  else if (DECL_FUNCTION_CODE (decl) == BUILT_IN_REALLOC)
              tree_edge_fn = build_fn_decl (edge_realloc_name, edge_realloc_fn_type); 
        	  else if (DECL_FUNCTION_CODE (decl) == BUILT_IN_FREE)
              tree_edge_fn = build_fn_decl (edge_free_name, edge_free_fn_type); 

            

            else 
              break;



            int nargs = gimple_call_num_args (stmt);
            VEC(tree, heap) *vargs = VEC_alloc (tree, heap, nargs);
            for (int i = 0; i <nargs ; i++)
            {
              tree arg = gimple_call_arg (stmt, i);
              VEC_quick_push (tree, vargs, arg);
            }
            
            call = gimple_build_call_vec (tree_edge_fn, vargs);
            VEC_free (tree, heap, vargs);
            
            tree retval = gimple_call_lhs (stmt);
            if (retval)
              gimple_set_lhs (call, retval);   
            
            gsi_insert_after(&gsi, call, GSI_SAME_STMT);  
            struct cgraph_edge *edge = cgraph_edge (node, stmt);
            cgraph_remove_edge (edge);
            gsi_remove (&gsi, true);
             
            
             

        	}


          break;

        }

      default:
        break;
      }

    }				// gsi

  }

  rewrite_into_loop_closed_ssa (NULL, TODO_update_ssa);
  update_new_call_edge (node);

}


void
profile_malloc (struct cgraph_node *node, ProfileMalloc func)
{

  /* print malloc */
  basic_block bb;
  FOR_EACH_BB (bb)
  {

    gimple_seq seq = bb_seq (bb);
    for (gimple_stmt_iterator gsi = gsi_start_bb (bb); !gsi_end_p (gsi); gsi_next (&gsi))
    {

    	int flag;
    	tree size;
    	tree old_addr;
    	tree new_addr;
    	gimple stmt = gsi_stmt (gsi);

    	switch (gimple_code (stmt))
  	  {

    	  case GIMPLE_CALL:
  	    {
  	      /* the arguments are to be profiled!!! */
  	      tree decl = gimple_call_fndecl (stmt);

  	      if (decl)
      		{
      		  switch (DECL_FUNCTION_CODE (decl))
    		    {
      		    case BUILT_IN_MALLOC:
      		    case BUILT_IN_CALLOC:
    		      {
          			size = gimple_call_arg (stmt, 0);
          			old_addr = build_int_cst (0, 0);
          			new_addr = gimple_call_lhs (stmt);
          			func (&gsi, old_addr, new_addr, size, 0);
          			break;
    		      }

      		    case BUILT_IN_REALLOC:
    		      {
          			size = gimple_call_arg (stmt, 1);
          			old_addr = gimple_call_arg (stmt, 0);
          			new_addr = gimple_call_lhs (stmt);
          			func (&gsi, old_addr, new_addr, size, 1);
          			break;
    		      }

      		    case BUILT_IN_FREE:
    		      {
          			size = build_int_cst (0, 0);
          			old_addr = gimple_call_arg (stmt, 0);
          			new_addr = build_int_cst (0, 0);
          			func (&gsi, old_addr, new_addr, size, 2);
          			break;
    		      }
      		    default:
      		      break;

    		    }

      		}


  	      break;

  	    }

  	  default:
  	    break;
  	  }

          }				// gsi

  }

  // insert print_malloc for all the global symbol table entries
  if (strcmp (cgraph_node_name (node), "main") == 0)
    {
      // TODO:

    }


  rewrite_into_loop_closed_ssa (NULL, TODO_update_ssa);
  update_new_call_edge (node);

}








/* Insert profiling statements to collect memory address read/write
	 iterating over each loop and going through the loop body and conditional statements;
	 inserting print statement before/after a certain line num..the arguments for
	 the print function (defined in test.c) is the L.H.S integer type variable
	 the vairable can be of any type.. */

void
profile_bb (struct cgraph_node *node, ProfileBB func)
{

  func_extra_info_t *finfo = func_extra_info (node);
  basic_block bb;
  FOR_EACH_BB (bb)
  {

    gimple_seq seq = bb_seq (bb);

    if ( gimple_seq_empty_p(seq) )
      continue;

    func (bb);

  }			
  
  /* END - Insert profiling statements to collect memory address read/write */

  rewrite_into_loop_closed_ssa (NULL, TODO_update_ssa);
  update_new_call_edge (node);


}




/* Insert profiling statements to collect memory address read/write
	 iterating over each loop and going through the loop body and conditional statements;
	 inserting print statement before/after a certain line num..the arguments for
	 the print function (defined in test.c) is the L.H.S integer type variable
	 the vairable can be of any type.. */

void
profile_memop (struct cgraph_node *node, ProfileMemop func)
{

  func_extra_info_t *finfo = func_extra_info (node);
  basic_block bb;
  FOR_EACH_BB (bb)
  {

    for (gimple_stmt_iterator gsi = gsi_start_bb (bb); !gsi_end_p (gsi); gsi_next (&gsi))
    {
    	tree lval = NULL;
    	tree rval1 = NULL;
    	tree rval2 = NULL;
    	gimple stmt = gsi_stmt (gsi);

    	switch (gimple_code (stmt))
  	  {
    	  case GIMPLE_COND:
    	    rval1 = gimple_cond_lhs (stmt);
    	    rval2 = gimple_cond_rhs (stmt);
    	    func (&gsi, rval1, 0, 0);
    	    func (&gsi, rval2, 1, 0);
    	    break;

    	  case GIMPLE_ASSIGN:

    	    lval = gimple_assign_lhs (stmt);
    	    func (&gsi, lval, 0, 1);

    	    if (gimple_assign_rhs_class (stmt) == GIMPLE_BINARY_RHS)
          {
          	rval1 = gimple_assign_rhs1 (stmt);
          	rval2 = gimple_assign_rhs2 (stmt);
          }
    	    else
          {
          	rval1 = gimple_assign_rhs1 (stmt);
          }

    	    if (rval1 != NULL)
    	      func (&gsi, rval1, 1, 0);
    	    if (rval2 != NULL)
    	      func (&gsi, rval2, 2, 0);

    	    break;

    	  case GIMPLE_CALL:
  	    {
  	      /* the arguments are to be profiled!!! */
  	      tree decl = gimple_call_fndecl (stmt);
  	      if (decl)
      		{
      		  const char *name = lang_hooks.decl_printable_name (decl, 2);
      		  if (strcmp (name, edge_profiler_name) == 0)
      		    break;
      		  else if (strcmp (name, malloc_profiler_name) == 0)
      		    break;
      		  else if (strcmp (name, address_profiler_name) == 0)
      		    break;
      		  else if (strcmp (name, range_profiler_name) == 0)
      		    break;
      		}


  	      for (int i = 0; i < gimple_call_num_args (stmt); i++)
      		{
      		  rval1 = gimple_call_arg (stmt, i);
      		  func (&gsi, rval1, i + 3, 0);
      		}
  	      break;

  	    }

    	  case GIMPLE_SWITCH:
          rval1 = gimple_switch_index (stmt);
          func (&gsi, rval1, 1, 0);
          break;

    	  case GIMPLE_RETURN:
          rval1 = gimple_return_retval (stmt);
          if (rval1)
            func (&gsi, rval1, 1, 0);
           break;
          
        case GIMPLE_DEBUG:
        case GIMPLE_PREDICT:
        case GIMPLE_LABEL:
        CASE_GIMPLE_UPP:
    	    break;


    	  default:
    	    printf("MAIN:UNKNOWN TREE CODE %d\n",gimple_code(stmt));
          gcc_unreachable();
    	    break;
  	  }

    }				//bsi

  }				//bb
  /* END - Insert profiling statements to collect memory address read/write */

  rewrite_into_loop_closed_ssa (NULL, TODO_update_ssa);
  update_new_call_edge (node);


}





/* Insert profiling statements to collect memory address read/write
	 iterating over each loop and going through the loop body and conditional statements;
	 inserting print statement before/after a certain line num..the arguments for
	 the print function (defined in test.c) is the L.H.S integer type variable
	 the vairable can be of any type.. */

void
profile_stmt (struct cgraph_node *node, ProfileStmt func)
{

  func_extra_info_t *finfo = func_extra_info (node);
  basic_block bb;
  FOR_EACH_BB (bb)
  {
    for (gimple_stmt_iterator gsi = gsi_start_bb (bb); !gsi_end_p (gsi); gsi_next (&gsi))
    {
    	gimple stmt = gsi_stmt (gsi);

    	if (!gimple_has_location (stmt))
  	  {
  	    /* printf("the statment has no location\n"); */
  	    continue;
  	  }
      func (&gsi); 	  
    }			

  }				//bb
  /* END - Insert profiling statements to collect memory address read/write */

  rewrite_into_loop_closed_ssa (NULL, TODO_update_ssa);
  update_new_call_edge (node);


}


/* Inserting loop entry/exit on edges */
void
profile_edge (struct cgraph_node *node, ProfileLoopedge func)
{

  loop_iterator li;
  unsigned int i;
  edge e;
  edge_iterator ei;
  struct loop *loop;

  FOR_EACH_LOOP (li, loop, LI_FROM_INNERMOST)
  {

    /* step2: find loop entry and insert entry statements   */

    basic_block header = loop->header;
    basic_block entry;

    if (single_succ_p (header))
    {
      entry = single_succ (header);
      gimple_stmt_iterator gsi = gsi_start_bb (entry);
      func (1, loop_uid (loop), single_succ_edge (header));
    }
    else
    {
      FOR_EACH_EDGE (e, ei, header->succs)
      {
        if (e->flags & EDGE_LOOP_EXIT)
          continue;

        VEC (edge, heap) * exits = get_loop_exit_edges (loop);
        edge exit_e;
        bool is_exit = false;
        for (i = 0; VEC_iterate (edge, exits, i, exit_e); i++)
        {
          if (exit_e == e)
          	is_exit = true;
        }

        if (!is_exit)
        {
          entry = e->dest;
          gimple_stmt_iterator gsi = gsi_start_bb (entry);
          func (1, loop_uid (loop), e);
        }
        
        VEC_free (edge, heap, exits);

      }

    }


    /* step2: find loop exits and insert exit statements    */
    VEC (edge, heap) * exits = get_loop_exit_edges (loop);
    for (i = 0; VEC_iterate (edge, exits, i, e); i++)
    {
      int flg = 0;
      basic_block bb = e->dest;
      gimple_stmt_iterator gsi = gsi_after_labels (bb);
      func (0, loop_uid (loop), e);
     }

    VEC_free (edge, heap, exits);
  }

  gsi_commit_edge_inserts ();
  rewrite_into_loop_closed_ssa (NULL, TODO_update_ssa);

  update_new_call_edge (node);

}

/* Inserting loop entry/exit on edges */
void
profile_loop_entry (struct cgraph_node *node, ProfileLoopheader func)
{

  loop_iterator li;
  unsigned int i;
  edge e;
  edge_iterator ei;
  struct loop *loop;

  FOR_EACH_LOOP (li, loop, LI_FROM_INNERMOST)
  {
    /* find loop entry and insert entry statements   */
    basic_block header = loop->header;
    gimple_stmt_iterator gsi = gsi_start_bb (header);
    func (loop_uid (loop), &gsi);
  }

  gsi_commit_edge_inserts ();
  rewrite_into_loop_closed_ssa (NULL, TODO_update_ssa);

  update_new_call_edge (node);

}




/* Inserting loop entry/exit on edges */
void
profile_loop_exit_edge (struct cgraph_node *node, ProfileLoopedge func)
{

  loop_iterator li;
  unsigned int i;
  edge e;
  edge_iterator ei;
  struct loop *loop;

  FOR_EACH_LOOP (li, loop, LI_FROM_INNERMOST)
  {
    /* find loop exits and insert exit statements    */
    VEC (edge, heap) * exits = get_loop_exit_edges (loop);
    for (i = 0; VEC_iterate (edge, exits, i, e); i++)
    {
      int flg = 0;
      basic_block bb = e->dest;
      gimple_stmt_iterator gsi = gsi_after_labels (bb);
      func (0, loop_uid (loop), e);
     }

    VEC_free (edge, heap, exits);
  }

  gsi_commit_edge_inserts ();
  rewrite_into_loop_closed_ssa (NULL, TODO_update_ssa);

  update_new_call_edge (node);

}


/* Inserting print_exit on exit points */
void
profile_exit (struct cgraph_node *node, ProfileExit func)
{

  bool main_p = false;
  if (strcmp (cgraph_node_name (node), "main") == 0)
    main_p = true;
  else if (strcmp (cgraph_node_name (node), "exec_main") == 0)
    main_p = true;


  basic_block bb;
  FOR_EACH_BB (bb)
  {

    gimple_seq seq = bb_seq (bb);
    for (gimple_stmt_iterator gsi = gsi_start_bb (bb); !gsi_end_p (gsi); gsi_next (&gsi))
      {

	int flag;
	tree size;
	tree old_addr;
	tree new_addr;
	gimple stmt = gsi_stmt (gsi);

	switch (gimple_code (stmt))
	  {
	  case GIMPLE_RETURN:
	    if (main_p)
	      func (&gsi);
	    break;

	  case GIMPLE_CALL:
	    {
	      /* the arguments are to be profiled!!! */
	      tree decl = gimple_call_fndecl (stmt);

	      if (decl)
		{
		  switch (DECL_FUNCTION_CODE (decl))
		    {
		    case BUILT_IN_EXIT:
		    case BUILT_IN_ABORT:
		      func (&gsi);
		      break;

		    default:
		      break;

		    }

		}

	      break;

	    }

	  default:
	    break;
	  }

      }				// gsi

  }

  gsi_commit_edge_inserts ();
  rewrite_into_loop_closed_ssa (NULL, TODO_update_ssa);

  update_new_call_edge (node);

}






/* Inserting print_initialize on exit points */
void
profile_begin(struct cgraph_node *node, ProfileExit func)
{

  bool main_p = false;
  if (strcmp (cgraph_node_name (node), "main") == 0)
    main_p = true;
  else if (strcmp (cgraph_node_name (node), "exec_main") == 0)
    main_p = true;

  if (!main_p)
    return;

  
  basic_block bb = ENTRY_BLOCK_PTR;
  gimple_seq seq = bb_seq (bb);

  if ( gimple_seq_empty_p(seq) )
  {
    bb = single_succ(bb);
    seq = bb_seq (bb);
  }

  gimple_stmt_iterator gsi = gsi_start_bb (bb);
  func(&gsi);


  rewrite_into_loop_closed_ssa (NULL, TODO_update_ssa);

  update_new_call_edge (node);

}




/* Inserting print_initialize on exit points */
void
profile_start (struct cgraph_node *node)
{

  bool main_p = false;
  if (strcmp (cgraph_node_name (node), "main") == 0)
    main_p = true;
  else if (strcmp (cgraph_node_name (node), "exec_main") == 0)
    main_p = true;

  if (!main_p)
    return;

  
  basic_block bb = ENTRY_BLOCK_PTR;
  gimple_seq seq = bb_seq (bb);

  if ( gimple_seq_empty_p(seq) )
  {
    bb = single_succ(bb);
    seq = bb_seq (bb);
  }

  gimple_stmt_iterator gsi = gsi_start_bb (bb);
  tree tree_printf_profiler_fn = build_fn_decl (edge_start_profiler_name, edge_start_profiler_fn_type);
  gimple call = gimple_build_call (tree_printf_profiler_fn, 0);
  gsi_insert_before (&gsi, call, GSI_SAME_STMT);
  ipa_add_abnormal_goto_call_edges (gsi);


  rewrite_into_loop_closed_ssa (NULL, TODO_update_ssa);

  update_new_call_edge (node);

}



/* Function executed by the pass for each function.  */

static unsigned int
ipa_profile_dependence (void)
{  

  /* Construct data structure and SSA form for each loop,
     initialize function extra information
     Assign an ID to each bb
  */
  {
    FILE *fp = fopen ("loop.info", "w");
    ipa_ssa_loop_init (fp);
    fclose (fp);
  }

  specify_loops_to_profile ();
  if (!(loop_set.size() == 1 && *loop_set.begin() >0 ))
  {
    printf("Please specify a loop, like xxx.c:startline,endline");
    gcc_assert (false);
  }

  /* Read count info */
  if ( flag_ipa_profile_dependence >=3 )  // O2
    read_count_infor_and_remove_bb ();

  {
    FILE *fp = fopen ("bb.info", "w");
    ipa_dump_basic_blocks (fp);
    fclose (fp);
  }

  /* Assign an ID to each gimple statement */
  {
    FILE *fp = fopen ("stmt.info", "w");
    ipa_gimple_id_assignment (fp);
    fclose (fp);
  }

  /* Dump call graph */
  {
    FILE *fp = fopen ("callsite.info", "w");
		ipa_dump_call_sites (fp);
    fclose (fp);
  }

  /* Find potential malloc functions */
  if (1)
  {
    FILE *fp = fopen ("malloc.info", "w");
    recognize_mem_allocate_functions(fp);
    fclose (fp);
  }

  ipa_points_to_analysis(); 

  /* Use points-to result to complement indirect call edges  */
  cgraph_resolve_indirect_edges ();

  /* Create data references to record each memory access */
  dr_hash = htab_create (100, hash_dr_id, eq_dr_id, hash_dr_del);
  ipa_create_data_references (NULL);

  /* compute reference set */
  fprintf (stderr, "start computing reference sets \n");
  htab_traverse (dr_hash, compute_reference_set, NULL); 

  if (0)
  {
    extern int  count_valid_dr (void **slot, void *data ATTRIBUTE_UNUSED);
  	htab_traverse (dr_hash, count_valid_dr, NULL);    
    extern int dr_count;
    extern int valid_dr_count;
    FILE *fp = fopen ("drcount.info", "w");
    fprintf(fp, "Total dr=%d,  valid dr=%d\n", dr_count, valid_dr_count);
    fclose (fp);    
    //return 0;
  }
    



  if ( flag_ipa_profile_dependence >=3 )  // O2
  {
    /* based on points-to analysis */
    FILE *fp = fopen ("aliasclass.info", "w");
    ipa_classify_memory_operations(fp);
    //ipa_partition_memory_operations();
    fclose (fp);
  }


  /* Dump dr */
  if (1)
  {
    FILE *fp = fopen ("memop.info", "w");
		ipa_dump_data_references (fp);
    fclose (fp);
  }

  /* Dump source info */
  if (1)
  {
    FILE *fp = fopen ("source.info", "w");
    ipa_dump_source_info (fp);
    fclose (fp);
  }
   
  
  /* Perform interprocedural mod-ref analysis */
  if (1)
  {
  	printf ("start ipa mod ref analysis \n");
    ipa_mod_ref_analysis ();
  }

  /* compute access pattern */
  if (0)
  {
	  printf ("start compute_access_pattern \n");
    htab_traverse (dr_hash, compute_access_pattern, NULL);
  }

  formal_checked = BITMAP_ALLOC (NULL);
  formal_checking = BITMAP_ALLOC (NULL);
  slice_hash = htab_create (100, hash_slice_id, eq_slice_id, hash_slice_del);

#if 1
  /* Bottom-up analyze data references */
  if (flag_ipa_expose_reference || flag_ipa_profile_dependence >=4 ) // O3
  {
    FILE *fp = fopen ("expose.info", "w");
    struct cgraph_node **order = XCNEWVEC (struct cgraph_node *, cgraph_n_nodes);
    int order_pos = ipa_utils_reduced_inorder (order, true, true, NULL);
    for (int i = 0; i < order_pos ; i++)
    {
      struct cgraph_node *node = order[i];
      if (!valid_function_node_p (node))
        continue;

      /* Find recursive cycles */
      std::set < cgraph_node_ptr > scc;
      scc.insert (node);
      struct ipa_dfs_info *w_info = (struct ipa_dfs_info *) node->aux;
      struct cgraph_node *next = w_info->next_cycle;
      while (next)
      {
        w_info = (struct ipa_dfs_info *) next->aux;
        if (valid_function_node_p(next))
          scc.insert (next);
        next = w_info->next_cycle;
      }

      /* Perform Exposed-references Analysis */
      printf ("Completed %d of %d	: Now ", i, order_pos);
      ipa_expose_reference_analysis (scc, fp);
      
      /* Collect data references to be profiled */
      collect_data_references_to_be_profiled (scc);
      release_resouces_of_callees (scc);

    }

    fclose (fp);
    free(order);

  }

  else if ( flag_ipa_profile_dependence >=3 )  // O2
  {
    ipa_collect_data_dependencies (true);
  }
  else if ( flag_ipa_profile_dependence >=2 )  // O1
  {
    ipa_collect_data_dependencies (false);
  }
#endif

  BITMAP_FREE(formal_checked);
  BITMAP_FREE(formal_checking);

  if (0)
  {
    cur_loop = ipa_get_loop(*loop_set.begin());
    FILE *fp = fopen("memop.info", "w");
    ipa_collect_data_references_for_loop(cur_loop, fp);    
    fclose(fp);
  }

  if (1)
  {
    FILE *fp = fopen ("dep.num", "w");
    fprintf(fp, "Number of dependences  = %d \n", 
    	         (int)(loop_carried_dependencies.size() + 
                    loop_independent_dependencies.size() ) );
    fprintf(fp, "Number of may dependences  = %d \n", 
    	         (int)(loop_carried_dependencies.size() + 
                    loop_independent_dependencies.size() - must_loop_carried_dependencies.size()
                    -must_loop_independent_dependencies.size()) );
    
    fprintf(fp, "Number of loop-indep dependences = %d \n", 
    	         (int)loop_independent_dependencies.size());

    fprintf(fp, "Number of loop-carried dependences = %d \n", 
    	         (int)loop_carried_dependencies.size());
    
    fprintf(fp, "Number of MUST loop-carried dependences = %d \n", 
    	         (int)must_loop_carried_dependencies.size());

    fprintf(fp, "Number of MUST loop-independent dependences = %d \n", 
    	         (int)must_loop_independent_dependencies.size());
    fclose(fp);

  }

  if (1)
  {
    FILE *fp = fopen ("edge.info", "w");
    ipa_partition_dependence_edges(fp);
    fclose (fp);
  }

  /* update instruction count of alias class with loop specified */
  if (0)
  {
    FILE *fp = fopen ("loopaliasclass.info", "w");
    ipa_update_alias_class_for_loop(cur_loop, fp);
    fclose (fp);
  }

  delete_points_to_sets ();
  

  /* Step 2. Instrument to each procedure */
  if (1)
  {
    printf ("start instrumenting for dependence profiling \n");

    initialize_profile_prototype ();
    read_mem_info ( loop_set, mem_set );


    /* Clear profile information attached to gimple */
    struct cgraph_node *node;
    for (node = cgraph_nodes; node; node = node->next)
    {
    	if (!valid_function_node_p (node))
    	  continue;
    	switch_to_context (node->decl);

     	/* Insert profiling statements to collect memory address read/write */
     	profile_memop (node, insert_address_SD3);

    	/* Inserting loop entry/exit on edges */
    	profile_edge (node, insert_edge_SD3);

      //    	profile_malloc (node, insert_malloc);

    	profile_exit (node, insert_exit_SD3);

    	switch_off_context ();
    }

  }

  /* Cleanup. */
  {
    struct cgraph_node *node;
    for (node = cgraph_nodes; node; node = node->next)
    {
      /* Get rid of the aux information.      */
      if (node->aux)
      {
        free (node->aux);
        node->aux = NULL;
      }
    }
  }

  ipa_ssa_loop_done ();

	if (dump_file)
	{
		fprintf (dump_file, "profile_count = %d\n", profile_count);
	}


  htab_delete (slice_hash);
  htab_delete (dr_hash);
  htab_delete (loop_hash);
  
  printf ("ipa profile dependence done\n");

  return 0;
}

static bool
gate_ipa_profile_dependence (void)
{
  return flag_ipa_profile_dependence > 1;
}



struct simple_ipa_opt_pass pass_profile_dependence = 
{
  {
    SIMPLE_IPA_PASS,
    "profile-dependence",	/* name */
    gate_ipa_profile_dependence,	/* gate */
    ipa_profile_dependence,	/* execute */
    NULL,			/* sub */
    NULL,			/* next */
    0,				/* static_pass_number */
    TV_TREE_PROFILE_DEPENDENCE,	/* tv_id */
    PROP_ssa,			/* properties_required */
    0,				/* properties_provided */
    0,				/* properties_destroyed */
    0,				/* todo_flags_start */
    TODO_dump_func | TODO_update_ssa	/* todo_flags_finish */
  }
};




/* Clear profile information attached to gimple
	 If DEFUSE=1, then REF is a write. */
static void
insert_address_edge(gimple_stmt_iterator * gsi, tree ref, int opnd, int defuse)
{
  gimple call;
  tree addr, size, loc, flag, kind;
  tree tree_printf_profiler_fn;


  /* Do not need to profile temporary variables */
  if (TREE_CODE (ref) == VAR_DECL && DECL_ARTIFICIAL (ref))
    return;

  if (TREE_CODE (ref) == SSA_NAME)
    return;


  gimple stmt = gsi_stmt (*gsi);
  ipa_data_reference *dr = tree_data_reference (stmt, opnd);

  if (!dr)
    return;

  if ( edge_set.find(dr->uid()) == edge_set.end() )
    return;


  flag = build_int_cst_type (char_type_node, defuse);
  loc = build_int_cst_type (integer_type_node, dr->uid ());
  kind = build_int_cst_type (integer_type_node, checking_kind); 

  tree objdep_addr = gimplify_build1 (gsi, ADDR_EXPR, build_pointer_type (objdep_type), objdep_var);
  TREE_ADDRESSABLE (objdep_var) = 1;

//  if (dr->pattern ().kind () == pk_no_pattern)
  {
    /* If not pattern, it is a point */
    tree_printf_profiler_fn = build_fn_decl (edge_address_profiler_name, edge_address_profiler_fn_type);
    /*
       see also in libprofile.cxx   
       addr - address;
          size - the bit size of memory to be accessed
          flag - read/write, if address read (0) or written (1) ; 
          id - memory operation id or slice id; 
          kind -- 0x1 - loop-independent ,  0x2 - loop-carried
      
       
       extern "C" void
       __edge_print_address (PTR addr, int size, char flag, int id, int kind, DEPENDENCY *dep)
       
     */

    switch (TREE_CODE (ref))
    {
      case COMPONENT_REF:
      case ARRAY_REF:
      case INDIRECT_REF:
      case MEM_REF:
      case VAR_DECL:
        addr = gimple_build_addr_expr (gsi, ref);
        break;
      default:
        return;
        break;
    }

    gcc_assert (addr);

    size = TYPE_SIZE (TREE_TYPE (ref));
    call = gimple_build_call (tree_printf_profiler_fn, 6, addr, size, flag, loc, kind, objdep_addr);

  }
 


  gsi_insert_before (gsi, call, GSI_SAME_STMT);
  ipa_add_abnormal_goto_call_edges (*gsi);

}






static void
insert_edge_edge (int entry, int loop_id, edge e)
{

  if (loop_id != objdep.loop())
    return;

  /* see also in libedge.cxx
     __edge_print_edge (int loop_id, int entry, DEPENDENCY *dep)
     )                                                    */

  tree tree_printf_profiler_fn =  build_fn_decl (edge_edge_profiler_name, edge_edge_profiler_fn_type);

  tree ent = build_int_cst_type (integer_type_node, entry);
  tree lnum = build_int_cst_type (unsigned_type_node, loop_id);

  gimple_seq seq = gimple_seq_alloc ();

  tree objdep_addr = NULL;

  gimple call = gimple_build_call (tree_printf_profiler_fn, 3, lnum, ent, objdep_addr);
  
  gimple_seq_add_stmt (&seq, call);

  gimple_stmt_iterator gsi = gsi_last (seq);
  objdep_addr = gimplify_build1 (&gsi, ADDR_EXPR, build_pointer_type (objdep_type), objdep_var);
  TREE_ADDRESSABLE (objdep_var) = 1;

  gimple_call_set_arg(call, 2, objdep_addr);
    
  gsi_insert_seq_on_edge (e, seq);

}

static void
insert_exit_edge (gimple_stmt_iterator * gsi)
{

  /*
     see also in libprofile.cxx   
     void __print_exit  ();
   */

  tree tree_printf_profiler_fn = build_fn_decl (edge_exit_profiler_name, edge_exit_profiler_fn_type);
  gimple call = gimple_build_call (tree_printf_profiler_fn, 0);
  gsi_insert_before (gsi, call, GSI_SAME_STMT);
  ipa_add_abnormal_goto_call_edges (*gsi);

}


/* Function executed by the pass for each function.  */

static unsigned int
ipa_profile_depedge (void)
{

  /* Construct data structure and SSA form for each loop,
     initialize function extra information
  */
  ipa_ssa_loop_init (NULL);

  /* Read count info */
  read_count_infor_and_remove_bb ();

  /* Assign an ID to each gimple statement */
  ipa_gimple_id_assignment (NULL);

  /* Use points-to result to complement indirect call edges  */
  cgraph_resolve_indirect_edges ();


  /* Create data references to record each memory access */
  dr_hash = htab_create (100, hash_dr_id, eq_dr_id, hash_dr_del);
  ipa_create_data_references (NULL);


  /* Step 2. Instrument to each procedure */

  {
    initialize_profile_prototype ();

    for (struct cgraph_node *node = cgraph_nodes; node; node = node->next)
    {
    	if (!valid_function_node_p (node))
    	  continue;
    	switch_to_context (node->decl);

      /* Specify which alias class to profile */
      specify_dep_edge_to_profile ();   

    	switch_off_context ();

      break;
    }


    strided = true;

    /* Clear profile information attached to gimple */
    for (struct cgraph_node *node = cgraph_nodes; node; node = node->next)
    {
    	if (!valid_function_node_p (node))
    	  continue;
    	switch_to_context (node->decl);

     	/* Insert profiling statements to collect memory address read/write */      
     	profile_memop (node, insert_address_edge);

    	/* Inserting loop entry/exit on edges */
    	profile_edge (node, insert_edge_edge);

      profile_IO_operation (node);

      profile_start (node);

    	profile_exit (node, insert_exit_edge);

    	switch_off_context ();
    }

    /* Cleanup. */
    for (struct cgraph_node *node = cgraph_nodes; node; node = node->next)
    {
      /* Get rid of the aux information.      */
      if (node->aux)
      {
	      free (node->aux);
	      node->aux = NULL;
      }
    }

  }

  ipa_ssa_loop_done ();

  htab_delete (dr_hash);

  if (0)
  for (struct cgraph_node *node = cgraph_nodes; node; node = node->next)
  {
    if ( strcmp( cgraph_node_name(node), "main" ) == 0 )
    {
      tree name = get_identifier ("exec_main")  ;      
      DECL_NAME (node->decl) = name;
      SET_DECL_ASSEMBLER_NAME (node->decl, name);
      break;
    }
  }
  
  return 0;
}


static bool
gate_ipa_profile_depedge (void)
{
  return flag_ipa_profile_depedge != 0;
}

struct simple_ipa_opt_pass pass_profile_depedge = 
{
  {
    SIMPLE_IPA_PASS,
    "profile-depedge",	/* name */
    gate_ipa_profile_depedge,	/* gate */
    ipa_profile_depedge,	/* execute */
    NULL,			/* sub */
    NULL,			/* next */
    0,				/* static_pass_number */
    TV_TREE_PROFILE_DEPENDENCE,	/* tv_id */
    PROP_ssa,			/* properties_required */
    0,				/* properties_provided */
    0,				/* properties_destroyed */
    0,				/* todo_flags_start */
    TODO_dump_func | TODO_update_ssa	/* todo_flags_finish */
  }
};




/* Function executed by the pass for each function.  */

static unsigned int
ipa_profile_SD3 (void)
{
  {
    FILE *fp = fopen ("libfunc.info", "w");
    find_library_calls (fp);
    fclose (fp);
  }

  /* Construct data structure and SSA form for each loop,
     initialize function extra information
  */
  {
    FILE *fp = fopen ("loop.info", "w");
    ipa_ssa_loop_init (fp);
    fclose (fp);
  }


  specify_loops_to_profile ();

  /* Assign an ID to each gimple statement */
  {
    FILE *fp = fopen ("stmt.info", "w");
    ipa_gimple_id_assignment (fp);
    fclose (fp);
  }

  if (0)
  {
    ipa_points_to_analysis();   
    /* Use points-to result to complement indirect call edges  */
    cgraph_resolve_indirect_edges ();
  }


  /* Create data references to record each memory access */
  dr_hash = htab_create (100, hash_dr_id, eq_dr_id, hash_dr_del);
  ipa_create_data_references (NULL);

  {
    cur_loop = ipa_get_loop(*loop_set.begin());
    FILE *fp = fopen("loopmemop.info", "w");
    ipa_collect_data_references_for_loop(cur_loop, fp);    
    fclose(fp);
  }

  /* Step 2. Instrument to each procedure */

  {
    printf ("start instrumenting for dependence profiling \n");

    initialize_profile_prototype ();

    /* Clear profile information attached to gimple */
    struct cgraph_node *node;
    for (node = cgraph_nodes; node; node = node->next)
    {
    	if (!valid_function_node_p (node))
    	  continue;
    	switch_to_context (node->decl);

    	/* Insert profiling statements to collect memory address read/write */
    	profile_memop (node, insert_address_SD3);

    	/* Inserting loop entry/exit on edges */
    	profile_edge (node, insert_edge_SD3);
      
    	profile_exit (node, insert_exit_SD3);

    	switch_off_context ();
    }

    /* Cleanup. */
    for (node = cgraph_nodes; node; node = node->next)
    {
      /* Get rid of the aux information.      */
      if (node->aux)
      {
	      free (node->aux);
	      node->aux = NULL;
      }
    }

    printf ("ipa profile dependence done\n");
  }

  ipa_ssa_loop_done ();


  htab_delete (dr_hash);
  htab_delete (loop_hash);

  return 0;
}


static bool
gate_ipa_profile_SD3 (void)
{
  return flag_ipa_profile_SD3 != 0 || flag_ipa_profile_dependence == 1 ;
}

struct simple_ipa_opt_pass pass_profile_SD3 = 
{
  {
    SIMPLE_IPA_PASS,
    "profile-sd3",	/* name */
    gate_ipa_profile_SD3,	/* gate */
    ipa_profile_SD3,	/* execute */
    NULL,			/* sub */
    NULL,			/* next */
    0,				/* static_pass_number */
    TV_TREE_PROFILE_DEPENDENCE,	/* tv_id */
    PROP_ssa,			/* properties_required */
    0,				/* properties_provided */
    0,				/* properties_destroyed */
    0,				/* todo_flags_start */
    TODO_dump_func | TODO_update_ssa	/* todo_flags_finish */
  }
};






/* Clear profile information attached to gimple
	 If DEFUSE=1, then REF is a write. */
static void
insert_address_AC (gimple_stmt_iterator * gsi, tree ref, int opnd, int defuse)
{
  gimple call;
  tree addr, size, loc, flag;
  tree tree_printf_profiler_fn;


  /* Do not need to profile temporary variables */
  if (TREE_CODE (ref) == VAR_DECL && DECL_ARTIFICIAL (ref))
    return;

  if (TREE_CODE (ref) == SSA_NAME)
	{
		if (TREE_CODE (SSA_NAME_VAR(ref)) == VAR_DECL && DECL_ARTIFICIAL (SSA_NAME_VAR(ref)))
		  return;
	}


  gimple stmt = gsi_stmt (*gsi);
  ipa_data_reference *dr = tree_data_reference (stmt, opnd);

  if (!dr)
    return;

  if ( !dr->need_profile ())
    return;

  if (alias_set.find(-1) == alias_set.end() && 
      alias_set.find(dr->class_id()) == alias_set.end())
    return;


  flag = build_int_cst_type (char_type_node, defuse);
  loc = build_int_cst_type (integer_type_node, dr->uid ());

  
  {
    tree_printf_profiler_fn = build_fn_decl (SD3_address_profiler_name, SD3_address_profiler_fn_type);
    /*
       see also in libprofile.cxx   
       void __print_address (PTR     addr,  int     size,   char    flag,  int      id) ;
     */

    switch (TREE_CODE (ref))
    {
      case COMPONENT_REF:
      case ARRAY_REF:
      case INDIRECT_REF:
      case MEM_REF:
      case VAR_DECL:
        addr = gimple_build_addr_expr (gsi, ref);
        break;
      default:
        return;
        break;
    }

    gcc_assert (addr);

    size = TYPE_SIZE (TREE_TYPE (ref));
    call = gimple_build_call (tree_printf_profiler_fn, 4, addr, size, flag, loc);

  }


  gsi_insert_before (gsi, call, GSI_SAME_STMT);
  ipa_add_abnormal_goto_call_edges (*gsi);

}



static bool
gate_ipa_profile_AC (void)
{
  return flag_ipa_profile_aliasclass != 0;
}


static void 
read_data_reference_information()
{

  FILE *meminfo = fopen ("memop.info", "r");
  if (!meminfo)
  {
    fprintf (stderr, "No file memop.info found!\n" );
    abort() ;
  }

  ipa_data_reference* dr = NULL;
  char *line = NULL;
  size_t len = 0;
  int bytes_read;
  int a, b, loop_id, kind;
  char str[200];
  while ((bytes_read = getline (&line, &len, meminfo)) != -1)
  {
    sscanf (line, "%s", str);

    if (strcmp (str, "MEMOP") == 0)
    {
      // ignore "MEMOP", go on reading memop_id
      int mem_id;
      sscanf (line, "%*s %d", &mem_id);
      dr = ipa_get_data_ref (mem_id);
      gcc_assert(dr);
    }
    
    else if (strcmp (str, "CLASS") == 0)
    {
      // ignore "MEMOP", go on reading memop_id
      int class_id;
      sscanf (line, "%*s %d", &class_id);
      dr->set_class_id(class_id);
    }

    else if (strcmp (str, "LOOP") == 0)
    {
      char must[10];
      /* Try to recognize the form :     LOOP 86  WAW    576 --> 562  MAY LOOP INDEPENDENT     */
      char kind[100];
      char type[100];
      sscanf (line, "%*s %d %*s %d %*s %d %s", &loop_id, &a, &b, must );
      if ( strcmp(must, "MUST") ==0 )
        continue;

      dr->set_need_profile();

      ipa_data_reference *tmp_dr = ipa_get_data_ref (b);
      gcc_assert(tmp_dr);
      tmp_dr->set_need_profile();      

    } 
   
  }

  fclose(meminfo);


}


/* Function executed by the pass for each function.  */

static unsigned int
ipa_profile_AC (void)
{

  /* Construct data structure and SSA form for each loop,
     initialize function extra information
  */
  ipa_ssa_loop_init (NULL);

  /* Read count info */
  read_count_infor_and_remove_bb ();

  /* Assign an ID to each gimple statement */
  ipa_gimple_id_assignment (NULL);

  /* Use points-to result to complement indirect call edges  */
  cgraph_resolve_indirect_edges ();


  /* Create data references to record each memory access */
  dr_hash = htab_create (100, hash_dr_id, eq_dr_id, hash_dr_del);
  ipa_create_data_references (NULL);

  /* Dump dr */
  {
	  FILE *fp = fopen ("memop-class.info", "w");
	  ipa_dump_data_references (fp);
	  fclose (fp);
  }

  /* Read count info */
  read_data_reference_information ();


  /* Step 2. Instrument to each procedure */

  {
    initialize_profile_prototype ();

    for (struct cgraph_node *node = cgraph_nodes; node; node = node->next)
    {
      if (!valid_function_node_p (node))
        continue;
      switch_to_context (node->decl);

      /* Specify which alias class to profile */
      specify_alias_class_to_profile ();   

      switch_off_context ();

      break;
    }


    strided = true;

    /* Clear profile information attached to gimple */
    for (struct cgraph_node *node = cgraph_nodes; node; node = node->next)
    {
      if (!valid_function_node_p (node))
        continue;
      switch_to_context (node->decl);

      /* Insert profiling statements to collect memory address read/write */
      
      profile_memop (node, insert_address_AC);

      /* Inserting loop entry/exit on edges */
      profile_edge (node, insert_edge_SD3);

      profile_IO_operation (node);

      profile_start (node);

      profile_exit (node, insert_exit_edge);

      switch_off_context ();
    }

    /* Cleanup. */
    for (struct cgraph_node *node = cgraph_nodes; node; node = node->next)
    {
      /* Get rid of the aux information.      */
      if (node->aux)
      {
        free (node->aux);
        node->aux = NULL;
      }
    }

  }

  ipa_ssa_loop_done ();

  htab_delete (dr_hash);

  return 0;
}




struct simple_ipa_opt_pass pass_profile_aliasclass = 
{
  {
    SIMPLE_IPA_PASS,
    "profile-aliasclass",	/* name */
    gate_ipa_profile_AC,	/* gate */
    ipa_profile_AC,	/* execute */
    NULL,			/* sub */
    NULL,			/* next */
    0,				/* static_pass_number */
    TV_TREE_PROFILE_DEPENDENCE,	/* tv_id */
    PROP_ssa,			/* properties_required */
    0,				/* properties_provided */
    0,				/* properties_destroyed */
    0,				/* todo_flags_start */
    TODO_dump_func | TODO_update_ssa	/* todo_flags_finish */
  }
};


static void
insert_edge_pairwise (int entry, int loop_id, edge e)
{

  tree ent = build_int_cst_type (integer_type_node, entry);
  tree lnum = build_int_cst_type (unsigned_type_node, loop_id);


  /* see also in libprofile.cxx
     void __print_edge( int loop_id,      // The global loop id 
     int entry            // 1- loop entry , 0- loop exit 
     )                                                    */

  tree tree_printf_profiler_fn = build_fn_decl (pairwise_edge_profiler_name, pairwise_edge_profiler_fn_type);

  gimple call = gimple_build_call (tree_printf_profiler_fn, 2, lnum, ent);

  gimple_seq seq = gimple_seq_alloc ();
  gimple_seq_add_stmt (&seq, call);
  gsi_insert_seq_on_edge (e, seq);

}


static void
insert_exit_pairwise (gimple_stmt_iterator * gsi)
{

  /*
     see also in libprofile.cxx   
     void __print_exit  ();
   */

  tree tree_printf_profiler_fn = build_fn_decl (pairwise_exit_profiler_name, pairwise_exit_profiler_fn_type);
  gimple call = gimple_build_call (tree_printf_profiler_fn, 0);
  gsi_insert_before (gsi, call, GSI_SAME_STMT);
  ipa_add_abnormal_goto_call_edges (*gsi);

}




/* Clear profile information attached to gimple
	 If DEFUSE=1, then REF is a write. */
static void
insert_address_pairwise (gimple_stmt_iterator * gsi, tree ref, int opnd, int defuse)
{
  gimple call;
  tree addr, size, loc, flag;
  tree tree_printf_profiler_fn;


  /* Do not need to profile temporary variables */
  if (TREE_CODE (ref) == VAR_DECL && DECL_ARTIFICIAL (ref))
    return;

  if (TREE_CODE (ref) == SSA_NAME)
	{
		if (TREE_CODE (SSA_NAME_VAR(ref)) == VAR_DECL && DECL_ARTIFICIAL (SSA_NAME_VAR(ref)))
		  return;
	}


  gimple stmt = gsi_stmt (*gsi);
  ipa_data_reference *dr = tree_data_reference (stmt, opnd);

  if (!dr)
    return;

  flag = build_int_cst_type (char_type_node, defuse);
  loc = build_int_cst_type (integer_type_node, dr->uid ());

  
  {
    tree_printf_profiler_fn = build_fn_decl (pairwise_address_profiler_name, pairwise_address_profiler_fn_type);
    /*
       see also in libprofile.cxx   
       void __print_address (PTR     addr,  int     size,   char    flag,  int      id) ;
     */

    switch (TREE_CODE (ref))
    {
      case COMPONENT_REF:
      case ARRAY_REF:
      case INDIRECT_REF:
      case MEM_REF:
      case VAR_DECL:
        addr = gimple_build_addr_expr (gsi, ref);
        break;
      default:
        return;
        break;
    }

    gcc_assert (addr);

    size = TYPE_SIZE (TREE_TYPE (ref));
    call = gimple_build_call (tree_printf_profiler_fn, 4, addr, size, flag, loc);

  }


  gsi_insert_before (gsi, call, GSI_SAME_STMT);
  ipa_add_abnormal_goto_call_edges (*gsi);

}



/* Function executed by the pass for each function.  */

static unsigned int
ipa_profile_pairwise (void)
{

  /* Construct data structure and SSA form for each loop,
     initialize function extra information
  */
  {
    FILE *fp = fopen ("loop.info", "w");
    ipa_ssa_loop_init (fp);
    fclose (fp);
  }


  /* Assign an ID to each gimple statement */
  {
    FILE *fp = fopen ("stmt.info", "w");
    ipa_gimple_id_assignment (fp);
    fclose (fp);
  }

  /* Create data references to record each memory access */
  dr_hash = htab_create (100, hash_dr_id, eq_dr_id, hash_dr_del);
  ipa_create_data_references (NULL);


  /* Step 2. Instrument to each procedure */

  {
    printf ("start instrumenting for dependence profiling \n");

    initialize_profile_prototype ();

    /* Clear profile information attached to gimple */
    struct cgraph_node *node;
    for (node = cgraph_nodes; node; node = node->next)
    {
    	if (!valid_function_node_p (node))
    	  continue;
    	switch_to_context (node->decl);

    	/* Insert profiling statements to collect memory address read/write */
    	profile_memop (node, insert_address_pairwise);

    	/* Inserting loop entry/exit on edges */
    	profile_edge (node, insert_edge_pairwise);

//    	profile_malloc (node, insert_malloc);

    	profile_exit (node, insert_exit_pairwise);

    	switch_off_context ();
    }

    /* Cleanup. */
    for (node = cgraph_nodes; node; node = node->next)
    {
      /* Get rid of the aux information.      */
      if (node->aux)
      {
	      free (node->aux);
	      node->aux = NULL;
      }
    }

    printf ("ipa profile dependence done\n");
  }

  ipa_ssa_loop_done ();


  htab_delete (dr_hash);

  return 0;
}



static bool
gate_ipa_profile_pairwise (void)
{
  return flag_ipa_profile_pairwise != 0;
}

struct simple_ipa_opt_pass pass_profile_pairwise = 
{
  {
    SIMPLE_IPA_PASS,
    "profile-pairwise",	/* name */
    gate_ipa_profile_pairwise,	/* gate */
    ipa_profile_pairwise,	/* execute */
    NULL,			/* sub */
    NULL,			/* next */
    0,				/* static_pass_number */
    TV_TREE_PROFILE_DEPENDENCE,	/* tv_id */
    PROP_ssa,			/* properties_required */
    0,				/* properties_provided */
    0,				/* properties_destroyed */
    0,				/* todo_flags_start */
    TODO_dump_func | TODO_rebuild_alias | TODO_update_ssa	/* todo_flags_finish */
  }
};




/* Clear profile information attached to gimple
	 If DEFUSE=1, then REF is a write. */
static void
insert_address_privatizable (gimple_stmt_iterator * gsi, tree ref, int opnd, int def)
{
  gimple call;
  tree addr, size, loc, flag;
  tree tree_printf_profiler_fn;


  /* Do not need to profile temporary variables */
  if (TREE_CODE (ref) == SSA_NAME)
     return;

  gimple stmt = gsi_stmt (*gsi);
  ipa_data_reference *dr = tree_data_reference (stmt, opnd);

  if (!dr)
    return;
 
  if (dr->is_read)
  {
    if (!bitmap_bit_p(loop_extra_info(cur_loop)->dr_read, dr->uid ()))
      return;    
  }
  else if (!bitmap_bit_p(loop_extra_info(cur_loop)->dr_write, dr->uid ()))
      return;    


  loc = build_int_cst_type (integer_type_node, dr->uid ());

  
  {
    if (dr->is_read)
      tree_printf_profiler_fn = build_fn_decl (privatize_load_profiler_name, privatize_load_profiler_fn_type);
    else
      tree_printf_profiler_fn = build_fn_decl (privatize_store_profiler_name, privatize_store_profiler_fn_type);
      
    /*
       see also in libprivatize.cxx   
       void __privatize_print_store (PTR addr, int size, int id);
     */

    switch (TREE_CODE (ref))
    {
      case COMPONENT_REF:
      case ARRAY_REF:
      case INDIRECT_REF:
      case MEM_REF:
      case VAR_DECL:
        addr = gimple_build_addr_expr (gsi, ref);
        break;
      default:
        return;
        break;
    }

    gcc_assert (addr);

    size = TYPE_SIZE (TREE_TYPE (ref));
    call = gimple_build_call (tree_printf_profiler_fn, 3, addr, size, loc);

  }


  gsi_insert_before (gsi, call, GSI_SAME_STMT);
  ipa_add_abnormal_goto_call_edges (*gsi);

}


// flag -1 realloc
static void
insert_malloc_privatizable (gimple_stmt_iterator * gsi, tree old_addr, tree new_addr,
	       tree size, int flag)
{

  /*
     see also in libprofile.cxx   
     void __print_malloc  (PTR    old_addr, PTR new_addr, int     size,   char    flag);
   */

  if (!new_addr)
    return;

  if (flag != 1 )
    return;

  tree tree_printf_profiler_fn = build_fn_decl (privatize_load_profiler_name, privatize_load_profiler_fn_type);
  tree loc = build_int_cst_type (integer_type_node, 0);

  gimple call = gimple_build_call (tree_printf_profiler_fn, 3, old_addr, size, loc);

  gsi_insert_after (gsi, call, GSI_NEW_STMT);
  ipa_add_abnormal_goto_call_edges (*gsi);

  tree_printf_profiler_fn = build_fn_decl (privatize_store_profiler_name, privatize_store_profiler_fn_type);
  loc = build_int_cst_type (integer_type_node, 0);
  call = gimple_build_call (tree_printf_profiler_fn, 3, new_addr, size, loc);
  gsi_insert_after (gsi, call, GSI_NEW_STMT);
  ipa_add_abnormal_goto_call_edges (*gsi);

}




static void
insert_edge_privatizable  (int entry, int loop_id, edge e)
{

  if (loop_id != *(loop_set.begin()))
    return;

  /* see also in libprivatize.cxx
     __privatize_print_entry (int loop_id)
     )                                                    */

  tree tree_printf_profiler_fn;

  if (entry)
    tree_printf_profiler_fn =  build_fn_decl (privatize_entry_profiler_name, privatize_entry_profiler_fn_type);
  else  
    tree_printf_profiler_fn =  build_fn_decl (privatize_exit_profiler_name, privatize_exit_profiler_fn_type);

  tree lnum = build_int_cst_type (unsigned_type_node, loop_id);

  gimple_seq seq = gimple_seq_alloc ();

  gimple call = gimple_build_call (tree_printf_profiler_fn, 1, lnum);
  
  gimple_seq_add_stmt (&seq, call);
    
  gsi_insert_seq_on_edge (e, seq);

}

static void
insert_exit_privatizable (gimple_stmt_iterator * gsi)
{

  /*
     see also in libprivatize.cxx   
     void __privatize_profile_finalize  ();
   */

  tree tree_printf_profiler_fn = build_fn_decl (privatize_end_profiler_name, privatize_end_profiler_fn_type);
  gimple call = gimple_build_call (tree_printf_profiler_fn, 0);
  gsi_insert_before (gsi, call, GSI_SAME_STMT);
  ipa_add_abnormal_goto_call_edges (*gsi);

}







static char profile_type=0;

/* We need ref because dr-ref may be modifed and no longer be valid*/
bool is_shadow_profile_candidate(ipa_data_reference_p dr, tree ref)
{
  
  if ( dr->is_virtual() )
    return false;


  bool toprofile=false;
  /* Do not need to profile temporary variables */
  if (TREE_CODE (ref) == COMPONENT_REF)
  {
    tree base = TREE_OPERAND (ref, 0);
    tree field = TREE_OPERAND (ref, 1);
    std::string name = SHADOW_PROFILING_DATA_NAME;
    if ( name == IDENTIFIER_POINTER (DECL_NAME (field)) )
    {
     
      /* Do not need to profile temporary and stack variables */
      if (TREE_CODE (base) == PARM_DECL)
        toprofile = false;
      
      else if (TREE_CODE (base) == VAR_DECL && DECL_ARTIFICIAL (base))
        toprofile = false;
     
      else if (TREE_CODE (base) == SSA_NAME && DECL_ARTIFICIAL (SSA_NAME_VAR(base)) )
        toprofile = false;

      else if (TREE_CODE (base) == SSA_NAME )
        toprofile = false;

      else if (TREE_CODE (base) == VAR_DECL && !TREE_ADDRESSABLE (base))
        toprofile = false;

      else if ( is_local_reference(base)  && is_gimple_scalar(base) )
        toprofile = false;        

 //     else if ( is_local_reference(base) )
 //       toprofile = false;        

      else if (TREE_READONLY (base))
        toprofile = false;

      else if (TYPE_READONLY (TREE_TYPE(base)))
        toprofile = false;

      else if (TREE_READONLY (ref))
        toprofile = false;

      else if (TYPE_READONLY (TREE_TYPE(ref)))
        toprofile = false;

      else
        toprofile = true;
    }
  }

  return toprofile;
  
}



bool is_profile_candidate(ipa_data_reference_p dr)
{

  if ( dr->is_virtual() )
    return true;

  tree ref = DR_REF(dr);

  bool toprofile=false;
  /* Do not need to profile temporary variables */
  {
    tree base = ref;  
    
    if (TREE_CODE (base) == VAR_DECL && DECL_ARTIFICIAL (base))
      toprofile = false;
   
    else if (TREE_CODE (base) == SSA_NAME && DECL_ARTIFICIAL (SSA_NAME_VAR(base)) )
      toprofile = false;

    else if (TREE_CODE (base) == SSA_NAME )
      toprofile = false;

    else if (TREE_CODE (base) == VAR_DECL && !TREE_ADDRESSABLE (base))
      toprofile = false;

    else if ( is_local_reference(base)  && is_gimple_scalar(base) )
      toprofile = false;        

    else if ( is_local_reference(base) )
      toprofile = false;        

    else if (TREE_READONLY (base))
      toprofile = false;


    else
      toprofile = true;
  }

  return toprofile;
  
}


/* Clear profile information attached to gimple
	 If DEFUSE=1, then REF is a write. */
static void
insert_address_hash (gimple_stmt_iterator * gsi, tree ref, int opnd, int def)
{
  gimple call;
  tree addr, size, loc, flag;
  tree tree_printf_profiler_fn;
  bool toprofile= false;
  bool bytewise=1;
  

  gimple stmt = gsi_stmt (*gsi);
  ipa_data_reference *dr = tree_data_reference (stmt, opnd);

  if (!dr)
    return;

  if ( !is_profile_candidate(dr) )
    return;  

  if ( cpunum > 1 )
  {
    if (alias_set.find(dr->uid()) == alias_set.end() ) 
      return;
  }
  else 
  {    
    if (dr->is_read)
    {
      if (!bitmap_bit_p(loop_extra_info(cur_loop)->dr_read, dr->uid ()))
        return;    
    }
    else if (!bitmap_bit_p(loop_extra_info(cur_loop)->dr_write, dr->uid ()))
        return;    
  }

  loc = build_int_cst_type (integer_type_node, dr->slice_id());

  switch (flag_profiling_type)
  {
    case PROFILING_PRIVATIZABLE :
        break;
    case PROFILING_DEPENDENCE:

      {
        if (dr->is_read)
          if (bytewise)
            tree_printf_profiler_fn = build_fn_decl (hash_bitwise_load_profiler_name, shadow_load_profiler_fn_type);
          else
            tree_printf_profiler_fn = build_fn_decl (hash_dependence_load_profiler_name, shadow_load_profiler_fn_type);
        else
          if (bytewise)
            tree_printf_profiler_fn = build_fn_decl (hash_bitwise_store_profiler_name, shadow_store_profiler_fn_type);
          else
            tree_printf_profiler_fn = build_fn_decl (hash_dependence_store_profiler_name, shadow_store_profiler_fn_type);
      }
      break;

    default:
        return;
  };
  
  {
      
    /*
       see also in libprivatize.cxx   
       void __privatize_print_store (PTR addr, int size, int id);
     */

    switch (TREE_CODE (ref))
    {
      case COMPONENT_REF:
      case ARRAY_REF:
      case INDIRECT_REF:
      case MEM_REF:
      case VAR_DECL:
        addr = gimple_build_addr_expr (gsi, ref);
        break;
      default:
        return;
        break;
    }

    gcc_assert (addr);

    size = TYPE_SIZE_UNIT (TREE_TYPE (ref));
    call = gimple_build_call (tree_printf_profiler_fn, 3, addr, size, loc);
  }


  gimple_set_block (call, gimple_block(gsi_stmt(*gsi)));
  gsi_insert_before (gsi, call, GSI_SAME_STMT);
  ipa_add_abnormal_goto_call_edges (*gsi);

}



static unsigned int shadow_type_count = 0;

static int
shadow_type_hash_marked_p (const void *p)
{
	return 1;
}

static GTY ((if_marked ("shadow_type_hash_marked_p"), param_is (struct type_hash)))
     htab_t shadow_type_hash;

#if 1

void
hash_add_shadow_type(tree type, tree shadow_type)
{
	add_profile_type(type, shadow_type);

  struct type_hash *h;
  void **loc;

	++shadow_type_count;
  h = ggc_alloc_type_hash ();
  h->hash = shadow_type_count;
  h->type = shadow_type;
  loc = htab_find_slot_with_hash (shadow_type_hash, h, shadow_type_count, INSERT);
  *loc = (void *)h;
}


/*
  Given a TYPE, promote it to a structure with additional profiling fields 
  struct {
    TYPE data;
    char memops[SHADOW_PROFILING_SHADOW_LENGTH];
  }
*/
static tree
shadow_promote_scalar_type(tree scalar_type)
{
	static int count=0;

  std::string buf, str;
  Get_type_name (scalar_type, buf);
  str = "{";
  str += buf;
  str += "}";     
    
  tree name = get_identifier (str.c_str());
  tree ref = make_node (RECORD_TYPE);
  tree attributes = NULL_TREE;
  TYPE_SIZE (ref) = 0;
  TYPE_PACKED (ref) = 0;
  decl_attributes (&ref, attributes, (int) ATTR_FLAG_TYPE_IN_PLACE);
	int orig_quals = TYPE_QUALS (strip_array_types (scalar_type));
	set_type_quals(ref, orig_quals); 


  tree fields = create_fields_of_profile_structure (scalar_type);
	tree x;
  for (x = fields; x; x = TREE_CHAIN (x))
  {
    DECL_CONTEXT (x) = ref;
    DECL_PACKED (x) |= TYPE_PACKED (ref);
  }


  TYPE_FIELDS (ref) = fields;
  layout_type (ref);
  TYPE_NAME (ref) = name;
	gcc_assert(TYPE_NAME (ref));

	hash_add_shadow_type( scalar_type, ref);

  return ref;


}



static tree shadow_promote_type(tree type, bool *promotable)
{
  if (is_profile_structre(type))
    return type;

	tree ret = hash_profile_type(type) ;
	if (ret)
		return ret;
  
  if ( types_compatible_p(type, char_type_node) )
    return type;
  if ( types_compatible_p(type, signed_char_type_node) )
    return type;
  if ( types_compatible_p(type, unsigned_char_type_node) )
    return type;  

  switch ( TREE_CODE(type) )
  {

    case INTEGER_TYPE:
    case REAL_TYPE:
    case ENUMERAL_TYPE:
      return shadow_promote_scalar_type(type); 

    case POINTER_TYPE:
    {
      tree type_pointed = shadow_promote_type(TREE_TYPE(type),NULL);
      tree type_pointer;
      if ( TREE_TYPE(type) == type_pointed )
        type_pointer = type;
      else 
        type_pointer = build_pointer_type(type_pointed);
      tree newtype = shadow_promote_scalar_type(type_pointer); 
      hash_add_shadow_type( type, newtype );
      return newtype;
    }

    case ARRAY_TYPE :
    {
      tree etype = shadow_promote_type(TREE_TYPE(type), NULL);
      if ( TREE_TYPE(type) == etype )
        return type;
      tree newtype = build_array_type( etype, TYPE_DOMAIN(type) );      
      hash_add_shadow_type( type, newtype );
      return newtype;
    }
  
    case RECORD_TYPE :
    {      
      tree field = TYPE_FIELDS (type);
      if ( !field || in_system_header_at(DECL_SOURCE_LOCATION(field)) ||  
           DECL_SOURCE_LOCATION(field) == BUILTINS_LOCATION )
        return type;

      tree x;
      tree fields = NULL_TREE;
      tree last_fld;

      tree ref = make_node (RECORD_TYPE);      
      hash_add_shadow_type( type, ref );

      for (field = TYPE_FIELDS (type); field; field = DECL_CHAIN (field))
      {
        gcc_assert(TREE_CODE (field) == FIELD_DECL);
        tree newtype = shadow_promote_type(TREE_TYPE(field),NULL);
        tree fld_decl = build_decl (DECL_SOURCE_LOCATION(field), FIELD_DECL, DECL_NAME(field), newtype);
        hash_add_shadow_type( field, fld_decl );        
        if (!fields)
          last_fld = fields = fld_decl;
        else
        {
          TREE_CHAIN (last_fld) = fld_decl;        
          last_fld = fld_decl;
        }
      }


      std::string buf, str;
      Get_type_name (type, buf);
      str = "{";
      str += buf;
      str += "}";           
      tree name = get_identifier (str.c_str());
      decl_attributes (&ref, TYPE_ATTRIBUTES(type), (int) ATTR_FLAG_TYPE_IN_PLACE);
      for (x = fields; x; x = TREE_CHAIN (x))
      {
        DECL_CONTEXT (x) = ref;
        DECL_PACKED (x) |= TYPE_PACKED (ref);
      }
      TYPE_FIELDS (ref) = fields;
      layout_type (ref);
      TYPE_NAME (ref) = name;
      return ref;
    }

    case FUNCTION_TYPE :
    {
      tree newtype = copy_node (type);
      /* Release the link from TYPE_POINTER_TO */      
      TYPE_POINTER_TO (newtype) = NULL;

      for (tree p = TYPE_ARG_TYPES (newtype); p; p = TREE_CHAIN (p))
      {
        tree arg_type = TREE_VALUE (p);
        TREE_VALUE (p) = shadow_promote_type (arg_type, NULL);       
      }

      tree rettype = TREE_TYPE (newtype);
      TREE_TYPE (newtype) = shadow_promote_type (rettype, NULL);

      return newtype;      
    }

    case VOID_TYPE: 
    case UNION_TYPE :
      return type;
        
    default:      
      gcc_assert(false);
      break;

  }

}



static tree
shadow_promote_init(tree shadowtype, tree datainit)
{
	tree vals, cur, last, fld;
	tree fld2_value = NULL_TREE, fld2_last;
	int i;
	
	/* Field data */
	fld = TYPE_FIELDS (shadowtype);
	cur = make_node (TREE_LIST);
	TREE_PURPOSE (cur) = fld;
	TREE_VALUE (cur) = datainit;
	last = cur;
	vals = cur;
	//return build_constructor_from_list (shadowtype, vals);

	/* Field info */
	fld = DECL_CHAIN (fld);
	cur = make_node (TREE_LIST);
	TREE_PURPOSE (cur) = fld;
	//TREE_VALUE (cur) = null_node;
	TREE_CHAIN (last) = cur;
	last = cur;


	/* Build constructors for char[SHADOW_PROFILING_SHADOW_LENGTH] */
	for (i=0; i < SHADOW_PROFILING_SHADOW_LENGTH; i++ )
	{ 	 
		tree vec_val = make_node (TREE_LIST);
		TREE_PURPOSE (vec_val) = build_int_cst_type (unsigned_type_node, i);
		TREE_VALUE (vec_val) = build_int_cst_type (char_type_node, 0);
		if ( !fld2_value ) 
				fld2_value = vec_val;
		else 
				TREE_CHAIN (fld2_last) = vec_val; 		
		fld2_last = vec_val;
	} 	 

	TREE_VALUE (cur) = build_constructor_from_list (TREE_TYPE(fld), fld2_value);			

	return build_constructor_from_list (shadowtype, vals);
}

static tree shadow_promote_value(tree value, tree shadowtype, bool *promotable)
{
  if ( types_compatible_p(TREE_TYPE(value),shadowtype) )
    return value;

  if ( TREE_CODE (value) == CONSTRUCTOR )
  {
    unsigned int i;
    tree val, field;
    if ( TREE_CODE(shadowtype) == RECORD_TYPE || TREE_CODE(shadowtype) == UNION_TYPE )
    {
      field = TYPE_FIELDS (shadowtype);
      FOR_EACH_CONSTRUCTOR_VALUE (CONSTRUCTOR_ELTS(value), i, val)
      {
        /* promote val to shadow struct */      
        tree newval = shadow_promote_value(val, TREE_TYPE(field), NULL);        
        CONSTRUCTOR_REPLACE_ELT (CONSTRUCTOR_ELTS(value), i, field, newval);
        field = TREE_CHAIN(field);
      }
    }
    
    else if ( TREE_CODE(shadowtype) == ARRAY_TYPE )
    {
      FOR_EACH_CONSTRUCTOR_ELT (CONSTRUCTOR_ELTS(value), i, field, val)
      {
        /* promote val to shadow struct */      
        tree newval = shadow_promote_value(val, TREE_TYPE(shadowtype), NULL);        
        CONSTRUCTOR_REPLACE_ELT (CONSTRUCTOR_ELTS(value), i, field, newval);
      }
    }

    TREE_TYPE(value) = shadowtype;
    return value;
  }
  else
    return shadow_promote_init (shadowtype, value);

}

static void 
do_shadow_promote_declaration()
{

	shadow_type_hash = htab_create_ggc (100, type_hash_hash, type_hash_eq, 0);

  /* Promote for global variables */
  struct varpool_node *var;
  for (var = varpool_nodes; var; var = var->next)
  {
    tree decl = var->decl;

    if (in_system_header_at(DECL_SOURCE_LOCATION(decl)) || DECL_SOURCE_LOCATION(decl) == BUILTINS_LOCATION  )
      continue;
    
    tree type = shadow_promote_type(TREE_TYPE(decl),NULL);

    if (!is_profile_structre(type))
      continue;

    TREE_TYPE(decl) = type; 
    relayout_decl(decl);

    /* promote initializer */
    if ( DECL_INITIAL (decl) )
    {
      tree init = DECL_INITIAL (decl);
      DECL_INITIAL (decl) = shadow_promote_value ( init, TREE_TYPE(decl), NULL );
    }  
  }


  /* Promote for local variables and formal parameters */  
  struct cgraph_node *node;
  for (node = cgraph_nodes; node; node = node->next)
  {
    if (!valid_function_node_p (node))
      continue;

    if (in_system_header_at(DECL_SOURCE_LOCATION(node->decl)) )
      continue;

    switch_to_context (node->decl);

    /* Promote formal parameters and return val */
    if ( !is_main_procedure(node) )
    {
      tree newtype = shadow_promote_type (TREE_TYPE(node->decl), NULL);
      TREE_TYPE(node->decl) = newtype;
    
  		for (tree parm = DECL_ARGUMENTS (node->decl); parm; parm = DECL_CHAIN (parm))
      {  
        tree type = TREE_TYPE(parm);
        type = shadow_promote_type (TREE_TYPE(parm), NULL);
        TREE_TYPE(parm) = type; 
        DECL_ARG_TYPE(parm) = type;  
        relayout_decl(parm);        
      }

      tree ret = DECL_RESULT (node->decl);
      TREE_TYPE(ret) = TREE_TYPE (newtype); 
      relayout_decl(ret);    
    }   
            
    
    /* Promote local variables */
    unsigned ix;
    tree var;
    struct function *dsf = DECL_STRUCT_FUNCTION (node->decl);

    FOR_EACH_LOCAL_DECL (dsf, ix, var)
    {
      if ( TREE_CODE(var) != VAR_DECL )
        continue;
      
      if ( TREE_STATIC(var) )
        continue;

      tree type = TREE_TYPE(var);

      if ( !is_gimple_reg (var) )
      {  
        type = shadow_promote_type (type, NULL);
        TREE_TYPE(var) = type; 
        relayout_decl(var);
      }
      else if (TREE_CODE(type) == POINTER_TYPE)
      {
        tree type_pointed = TREE_TYPE(type);
        type_pointed = shadow_promote_type (type_pointed, NULL); 
        tree type_pointer = build_pointer_type(type_pointed);
        TREE_TYPE(var) = type_pointer;         
      }
    }

    switch_off_context ();
    
  }


  
  
}

static void
shadow_update_type(tree ref)
{
  switch( TREE_CODE(ref) )
  {

    case ADDR_EXPR:
    {
      tree base = TREE_OPERAND(ref, 0);
      shadow_update_type(base);
      TREE_TYPE(ref) = build_pointer_type(TREE_TYPE(base));
      break;
    }

    case SSA_NAME:
    {
      TREE_TYPE(ref) = TREE_TYPE(SSA_NAME_VAR(ref));
      break;
    }

    case ARRAY_REF:
    {
      tree base = TREE_OPERAND(ref, 0);
      shadow_update_type(base);
      TREE_TYPE(ref) = TREE_TYPE(TREE_TYPE(base));
      break;
    }

		case COMPONENT_REF:
    {
      tree base = TREE_OPERAND(ref, 0);
      shadow_update_type(base);

      tree type = TREE_TYPE(base);
      
      tree field = TREE_OPERAND(ref, 1);
      tree newfield = hash_profile_type(field) ;
      if (!newfield)
        newfield = field;
      TREE_OPERAND(ref, 1) = newfield;
      TREE_TYPE(ref) = TREE_TYPE(newfield);
      break;
    }

    case MEM_REF :    
    {
      tree base = TREE_OPERAND(ref, 0);
      tree ofst = TREE_OPERAND(ref, 1);
      tree type = TREE_TYPE (ref);
      tree base_type = TREE_TYPE (base);
      tree type_pointed = TREE_TYPE (TREE_TYPE(base));

      enum { T_UNKNOWN, T_DEREF, T_DEREF_ARR } kind;
      if ( type == type_pointed )
        kind = T_DEREF;
      else if ( TREE_CODE(type_pointed) == ARRAY_TYPE && type == TREE_TYPE(type_pointed))
        kind = T_DEREF_ARR;
      else if ( is_profile_structre(type_pointed) && hash_profile_type(type) == type_pointed )
        kind = T_DEREF;
      else 
        kind = T_UNKNOWN;
      
      shadow_update_type(base);    
      type_pointed = TREE_TYPE (TREE_TYPE(base));
      if ( kind == T_DEREF_ARR )
        type_pointed = TREE_TYPE (type_pointed);
       
      if ( kind != T_UNKNOWN && AGGREGATE_TYPE_P(type_pointed) )
      {
        /* Since we may have promoted the structure */
        TREE_TYPE(ref) = type_pointed;
      }
      break;
    }    

    case SIZEOF_EXPR:
    {      
      /* What if the sizeof concerns a system library expression */
      tree type = TREE_OPERAND(ref, 0);
      type = shadow_promote_type (type, NULL);
      TREE_OPERAND(ref, 0) = type;
      break;
    }

    case CONSTRUCTOR:
    {
      unsigned i;
      tree val;
      FOR_EACH_CONSTRUCTOR_VALUE (CONSTRUCTOR_ELTS(ref), i, val)
        shadow_update_type (val);
      break;
    }


    case VAR_DECL:
    case PARM_DECL:
    case RESULT_DECL:
    case FUNCTION_DECL :
    case INTEGER_CST:
    case REAL_CST:
    case STRING_CST:
      break;

    default:
      gcc_assert(false);
      break;
  }
}

static tree
shadow_promote_ref(gimple_stmt_iterator * gsi, tree ref, int opnd)
{
  if (!gsi)
    return ref;

  tree type = TREE_TYPE(ref);
  if ( !is_profile_structre(type) )
    return ref;

  if ( TREE_CODE(type) != RECORD_TYPE )
    return ref;
  
  // promote to a component_ref  
  gimple stmt = gsi_stmt (*gsi);
  tree field = TYPE_FIELDS (type);        
  /*
  int orig_quals = TYPE_QUALS (strip_array_types (type));
  TYPE_READONLY (TREE_TYPE(field)) = (orig_quals & TYPE_QUAL_CONST) != 0;
  TYPE_VOLATILE (TREE_TYPE(field)) = (orig_quals & TYPE_QUAL_VOLATILE) != 0;
  TYPE_RESTRICT (TREE_TYPE(field)) = (orig_quals & TYPE_QUAL_RESTRICT) != 0;
  TYPE_ADDR_SPACE (TREE_TYPE(field)) = DECODE_QUAL_ADDR_SPACE (orig_quals);
  */
  tree newref = build3 (COMPONENT_REF, TREE_TYPE(field), ref, field,  NULL );
  newref = force_gimple_operand_gsi (gsi, newref, false, NULL, true, GSI_SAME_STMT);  
  gimple_set_op( stmt, opnd, newref );
  return newref;

}



static tree 
shadow_promote_reference(gimple_stmt_iterator * gsi, tree ref, int opnd)
{
  if (!ref)
    return ref;
 

  switch( TREE_CODE(ref) )
  {
    case SSA_NAME :
      shadow_update_type(ref);
      return ref;
      
    case PARM_DECL :
    case VAR_DECL :     
		case COMPONENT_REF:
    case ARRAY_REF:
    case MEM_REF :
    {
      shadow_update_type(ref);
      return shadow_promote_ref(gsi, ref, opnd);
    }
 
    case ADDR_EXPR :
    {
      tree base = TREE_OPERAND(ref, 0);
      shadow_update_type(base);
      return ref;
    }

    case SIZEOF_EXPR:
    {
      shadow_update_type(ref);
      return ref;
    }

    case CONSTRUCTOR :
        /* Usually the constructor is empty as it must appear in a local initialization */      
    case INTEGER_CST :
    case REAL_CST :
    case STRING_CST :
      return ref;
      

    default:
      gcc_unreachable();
      break;
  }

  return ref;

}


/* Generate copy instructions after gsi, from src to dst, without copying tag field */
static void
shadow_copy_structure(gimple_stmt_iterator * gsi, tree src, tree dst)
{

  //mark_symbols_for_renaming (assign);

}




static void
shadow_promote_funccall(gimple_stmt_iterator * gsi)
{

  /* Exception : Malloc, calloc, Realloc */
  gimple stmt = gsi_stmt (*gsi);
  tree fn = gimple_call_fn (stmt);
  tree fndecl = gimple_call_fndecl (stmt);
  tree fntype = TREE_TYPE (TREE_TYPE (fn));

  if ( fndecl )
  {
    cgraph_node_ptr node = cgraph_get_node(fndecl);
    if (!node || !valid_function_node_p (node))
    {
      for (int i = 0; i < gimple_call_num_args (stmt); i++)
        shadow_promote_reference (gsi, gimple_call_arg (stmt, i), i + 3); 
      shadow_promote_reference (gsi, gimple_call_lhs(stmt), 0);         

      /* Replace some system calls with profiler-built-in call */
      if ( strcmp(cgraph_node_name(node), "__ctype_b_loc") == 0 )
      {
        tree tree_replacing = build_fn_decl ("__ctype_b_loc_wrapper", fntype);      
        gimple_call_set_fndecl (stmt, tree_replacing);
        struct cgraph_edge *edge = cgraph_edge (cgraph_node(current_function_decl), stmt);
        if (edge)
          cgraph_remove_edge (edge);
      }
      
      mark_symbols_for_renaming (stmt);
      return;
    }
  }


  shadow_update_type (fn);
  fntype = TREE_TYPE (TREE_TYPE (fn));
  tree ret_type = TREE_TYPE (fntype);

  
  tree p = TYPE_ARG_TYPES (fntype); 
  for (int i = 0; i < gimple_call_num_args (stmt); i++)
  {

    tree rval = gimple_call_arg (stmt, i);
    shadow_update_type(rval);    
    tree type = TREE_TYPE(rval);

    if (!p)
    {
      gcc_assert (!is_profile_structre (type));
      continue;
    }

    tree arg_type = TREE_VALUE (p);    

    // Make assignment from actual parameter to tmp
    if (is_profile_structre (arg_type) && is_profile_structre (type))
    {
      if ( arg_type != type )
      {
        // New tmp structure        
        tree decl = create_tmp_var (arg_type, NULL);
        
        // Field data
        tree lhs = build3 (COMPONENT_REF, TREE_TYPE(TYPE_FIELDS(arg_type)), decl, TYPE_FIELDS(arg_type), NULL );        
        tree rhs = build3 (COMPONENT_REF, TREE_TYPE(TYPE_FIELDS(type)), rval, TYPE_FIELDS(type), NULL );
        lhs = force_gimple_operand_gsi (gsi, lhs, false, NULL, true, GSI_SAME_STMT);  
        rhs = force_gimple_operand_gsi (gsi, lhs, true, NULL, true, GSI_SAME_STMT);  
        gimple assign = gimple_build_assign (lhs, rhs);
        gsi_insert_before (gsi, assign, GSI_SAME_STMT);       
        mark_symbols_for_renaming (assign);       
        gimple_call_set_arg (stmt, i, decl);
        mark_symbols_for_renaming (stmt);
      }
    }
    else if ( is_profile_structre (arg_type) )
    {
      // New tmp structure        
      tree decl = create_tmp_var (arg_type, NULL);
      
      // Field data
      tree lhs = build3 (COMPONENT_REF, TREE_TYPE(TYPE_FIELDS(arg_type)), decl, TYPE_FIELDS(arg_type), NULL );        
      lhs = force_gimple_operand_gsi (gsi, lhs, false, NULL, true, GSI_SAME_STMT);  
      if ( !types_compatible_p (TREE_TYPE(lhs), TREE_TYPE(rval)) ) 
        rval =	build1_stat (CONVERT_EXPR, TREE_TYPE(lhs), rval);     
      rval = force_gimple_operand_gsi (gsi, rval, true, NULL, true, GSI_SAME_STMT);        
      gimple assign = gimple_build_assign (lhs, rval);
      gsi_insert_before (gsi, assign, GSI_SAME_STMT);       
      mark_symbols_for_renaming (assign);       
      gimple_call_set_arg (stmt, i, decl);
      mark_symbols_for_renaming (stmt);
    }
    else if ( is_profile_structre (type) )
    {
      // New tmp structure        
      tree decl = create_tmp_var (arg_type, NULL);
      // Field data
      tree rhs = build3 (COMPONENT_REF, TREE_TYPE(TYPE_FIELDS(type)), rval, TYPE_FIELDS(type), NULL );
      rhs = force_gimple_operand_gsi (gsi, rhs, true, NULL, true, GSI_SAME_STMT);  
      gimple assign = gimple_build_assign (decl, rhs);
      gsi_insert_before (gsi, assign, GSI_SAME_STMT);       
      mark_symbols_for_renaming (assign);       
      gimple_call_set_arg (stmt, i, decl);
      mark_symbols_for_renaming (stmt);
    }    

    if (p)
      p = TREE_CHAIN (p);
  }


  // handle return value
  tree lval = gimple_call_lhs (stmt);  
  if (!lval)
    return ;
  
  shadow_update_type(lval);
  tree type = TREE_TYPE(lval);

  
  if (is_profile_structre (ret_type) && is_profile_structre (type))
  {
    ;
    //tree decl = create_tmp_var (ret_type, NULL);
    //gimple_call_set_lhs (stmt, decl);
    //mark_symbols_for_renaming (stmt);

    // Make assignment from tmp to actual parameter
    //shadow_copy_structure (gsi, decl, lval);
  }
  else if (is_profile_structre (ret_type) )
  {
    tree decl = create_tmp_var (ret_type, NULL);
    gimple_call_set_lhs (stmt, decl);
    mark_symbols_for_renaming (stmt);

    tree field = TYPE_FIELDS (ret_type);        
    tree rhs = build3 (COMPONENT_REF, TREE_TYPE(field), decl, field,  NULL );
    rhs = force_gimple_operand_gsi (gsi, rhs, true, NULL, false, GSI_NEW_STMT);  
    gimple assign = gimple_build_assign (lval, rhs);
    gsi_insert_after (gsi, assign, GSI_NEW_STMT);  
    mark_symbols_for_renaming (assign);
  }
  else if (is_profile_structre (type) )
  {
    tree decl = create_tmp_var (ret_type, NULL);
    gimple_call_set_lhs (stmt, decl);
    mark_symbols_for_renaming (stmt);

    tree field = TYPE_FIELDS (type);        
    tree lhs = build3 (COMPONENT_REF, TREE_TYPE(field), lval, field,  NULL );        
    lhs = force_gimple_operand_gsi (gsi, lhs, false, NULL, true, GSI_SAME_STMT);  
    gimple assign = gimple_build_assign (lhs, decl);
    gsi_insert_after (gsi, assign, GSI_NEW_STMT);       
    mark_symbols_for_renaming (assign);       
  }
}





static void
shadow_promote_return(gimple_stmt_iterator * gsi)
{

  gimple stmt = gsi_stmt (*gsi);
  
  tree rval = gimple_return_retval(stmt);
  if (!rval)
    return;
  
  shadow_update_type(rval);
  tree type = TREE_TYPE(rval);
  tree ret_type = TREE_TYPE (TREE_TYPE(current_function_decl));
    
       
  if (is_profile_structre (ret_type) && is_profile_structre (type))
  {
    ;
    //tree decl = create_tmp_var (ret_type, NULL);
    //gimple_call_set_lhs (stmt, decl);
    //mark_symbols_for_renaming (stmt);

    // Make assignment from tmp to actual parameter
    //shadow_copy_structure (gsi, decl, lval);
  }
  else if (is_profile_structre (ret_type) )
  {
    tree decl = DECL_RESULT (current_function_decl);
    gimple_return_set_retval (stmt, decl);
    mark_symbols_for_renaming (stmt);

    tree field = TYPE_FIELDS (ret_type);        
    tree lhs = build3 (COMPONENT_REF, TREE_TYPE(field), decl, field,  NULL );
    lhs = force_gimple_operand_gsi (gsi, lhs, false, NULL, false, GSI_NEW_STMT);  
    gimple assign = gimple_build_assign (lhs, rval);
    gsi_insert_before (gsi, assign, GSI_SAME_STMT);  
    mark_symbols_for_renaming (assign);
  }
   else if (is_profile_structre (type) )
   {
     tree decl = create_tmp_var (ret_type, NULL);
     gimple_return_set_retval (stmt, decl);
     mark_symbols_for_renaming (stmt);
   
     tree field = TYPE_FIELDS (type);        
     tree rhs = build3 (COMPONENT_REF, TREE_TYPE(field), decl, field,  NULL );        
     rhs = force_gimple_operand_gsi (gsi, rhs, true, NULL, true, GSI_SAME_STMT);  
     gimple assign = gimple_build_assign (decl, rhs);
     gsi_insert_before (gsi, assign, GSI_SAME_STMT);  
     mark_symbols_for_renaming (assign);       
   }
     

}



static void
shadow_forward_argv(struct cgraph_node *node)
{

  tree argc = DECL_ARGUMENTS (node->decl); 
  tree argv = DECL_CHAIN (argc);

  /* New argv_S */    
  tree type = shadow_promote_type (TREE_TYPE(argv), NULL);
  tree field = TYPE_FIELDS (type); 
  tree name = get_identifier ("argv_S");      
  tree argv_S = build_decl (UNKNOWN_LOCATION, VAR_DECL, name, type);
  gimple_add_tmp_var (argv_S);
  add_referenced_var (argv_S);


  edge e = split_block (ENTRY_BLOCK_PTR, NULL);  
  basic_block bb = e->dest;

  /* argv_S.xdatax = alloca (argc * sizeof(type_pointed(typeof(argv_S.pointer))) ) */
  gimple_stmt_iterator gsi = gsi_last_bb (bb);
  tree size = build1 (SIZEOF_EXPR, size_type_node,TREE_TYPE(TREE_TYPE(field))); 
  tree num = build1 (CONVERT_EXPR, size_type_node, argc);
  size = build2 (MULT_EXPR, size_type_node, num, size);
  size = force_gimple_operand_gsi (&gsi, size, true, NULL, true, GSI_SAME_STMT);  
  gimple alloc_call = gimple_build_call ( built_in_decls[BUILT_IN_ALLOCA], 1, size);

  tree tmp_lhs = build_decl (UNKNOWN_LOCATION, VAR_DECL, NULL, TREE_TYPE(field));   
  DECL_ARTIFICIAL (tmp_lhs) = 1;
  gimple_add_tmp_var (tmp_lhs);
  add_referenced_var (tmp_lhs);
  gsi_insert_before (&gsi, alloc_call, GSI_SAME_STMT);  
  gimple_call_set_lhs (alloc_call, tmp_lhs);  
  mark_symbols_for_renaming (alloc_call);
  
  tree lhs = build3 (COMPONENT_REF, TREE_TYPE(field), argv_S, field,  NULL );
  lhs = force_gimple_operand_gsi (&gsi, lhs, false, NULL, true, GSI_SAME_STMT);  
  gimple assign = gimple_build_assign (lhs, tmp_lhs);
  gsi_insert_before (&gsi, assign, GSI_SAME_STMT);       
  mark_symbols_for_renaming (assign);
  

  

  /* Copy the argv array to argv_S.xdatax */    
  gcc_assert (single_succ_p(bb));
  e = single_succ_edge(bb);  
  tree idx_type = TREE_TYPE(argc);
  tree initial_value = build_int_cst_type (idx_type, 0);
  tree stride = build_int_cst_type (idx_type, 1);
  tree upper_bound = argc;  
  tree iv, iv_before, iv_after;
  iv = create_tmp_var (idx_type, "arg_i");
  loop_p loop = create_empty_loop_on_edge (e, initial_value, stride, upper_bound, iv, &iv_before, &iv_after,
                                           bb->loop_father);
  
  gsi = gsi_last_bb (loop->header);

  // argv_S.xdatax[i].xdatax
  tree type_pointed = TREE_TYPE( TREE_TYPE(field) );      
  tree offset = build1(CONVERT_EXPR, size_type_node, iv_before);  
  size = build1 (SIZEOF_EXPR, size_type_node, type_pointed); 
  offset = build2 (MULT_EXPR, size_type_node, offset, size);
  lhs = build3 (COMPONENT_REF, TREE_TYPE(field), argv_S, field,  NULL );  // argv_S.pointer
  lhs = build2 (POINTER_PLUS_EXPR, TREE_TYPE(field), lhs, offset);
  lhs = build_simple_mem_ref(lhs);      // argv_S.pointer[i]
  tree pointer_field = TYPE_FIELDS (type_pointed);
  lhs = build3 (COMPONENT_REF, TREE_TYPE(pointer_field), lhs, pointer_field,  NULL );  // argv_S.pointer[i].pointer
  
  // argv[i]
  type_pointed = TREE_TYPE( TREE_TYPE(argv) );      
  offset = build1 (CONVERT_EXPR, size_type_node, iv_before);
  size = build1 (SIZEOF_EXPR, size_type_node, type_pointed); 
  offset = build2 (MULT_EXPR, size_type_node, offset, size);
  tree rhs = build2 (POINTER_PLUS_EXPR, TREE_TYPE(argv), argv, offset);
  rhs = build_simple_mem_ref(rhs);     
 
  lhs = force_gimple_operand_gsi (&gsi, lhs, false, NULL, true, GSI_SAME_STMT);  
  rhs = force_gimple_operand_gsi (&gsi, rhs, true, NULL, true, GSI_SAME_STMT);  
  assign = gimple_build_assign (lhs, rhs);
  gsi_insert_before (&gsi, assign, GSI_SAME_STMT);       
  mark_symbols_for_renaming (assign);


  /* Forward every apperance of argv to argv_S.pointer */
  
  tree newargv = build3 (COMPONENT_REF, TREE_TYPE(field), argv_S, field,  NULL );  // argv_S.pointer
  basic_block bb1;
  FOR_EACH_BB(bb1)  {    

    if ( bb1 == bb || bb1 == loop->header)
      continue;
    for (gimple_stmt_iterator gsi = gsi_start_bb (bb1); !gsi_end_p (gsi); gsi_next (&gsi))
    {
      tree lval = NULL;
      tree rval1 = NULL;
      tree rval2 = NULL;
      gimple stmt = gsi_stmt (gsi);
      tree tmpargv;
      switch (gimple_code (stmt))
      {
        case GIMPLE_COND:
        {
          lval = gimple_cond_lhs (stmt);
          rval1 = gimple_cond_rhs (stmt);
          if ( lval == argv )
          {
            tmpargv = force_gimple_operand_gsi (&gsi, newargv, true, NULL, true, GSI_SAME_STMT);  
            gimple_cond_set_lhs (stmt, tmpargv);
            mark_symbols_for_renaming (stmt);
          }
          if ( rval1 == argv )
          {
            tmpargv = force_gimple_operand_gsi (&gsi, newargv, true, NULL, true, GSI_SAME_STMT);  
            gimple_cond_set_rhs (stmt, tmpargv);
            mark_symbols_for_renaming (stmt);
          }
          break;
        }
  
        case GIMPLE_SWITCH:
        {
          lval = gimple_switch_index (stmt);
          if ( lval == argv )
          {
            tmpargv = force_gimple_operand_gsi (&gsi, newargv, true, NULL, true, GSI_SAME_STMT);  
            gimple_switch_set_index (stmt, tmpargv);
            mark_symbols_for_renaming (stmt);
          }
          break;
        }
  
        case GIMPLE_ASSIGN:
        {
          lval = gimple_assign_lhs (stmt);
          if (gimple_assign_rhs_class (stmt) == GIMPLE_BINARY_RHS)
          {
            rval1 = gimple_assign_rhs1 (stmt);
            rval2 = gimple_assign_rhs2 (stmt);
          }
          else
          {
            rval1 = gimple_assign_rhs1 (stmt);
          }
        
          if ( lval == argv )
          {
            tmpargv = force_gimple_operand_gsi (&gsi, newargv, false, NULL, true, GSI_SAME_STMT);  
            gimple_assign_set_lhs (stmt, tmpargv);
            mark_symbols_for_renaming (stmt);
          }
          if ( rval1 == argv )
          {
            tmpargv = force_gimple_operand_gsi (&gsi, newargv, true, NULL, true, GSI_SAME_STMT);  
            gimple_assign_set_rhs1(stmt, tmpargv);
            mark_symbols_for_renaming (stmt);
          }
          if ( rval2 == argv )
          {
            tmpargv = force_gimple_operand_gsi (&gsi, newargv, true, NULL, true, GSI_SAME_STMT);  
            gimple_assign_set_rhs2(stmt, tmpargv);
            mark_symbols_for_renaming (stmt);
          }
          break;
        }
  
        case GIMPLE_CALL:
        {            
          for (int i = 0; i < gimple_call_num_args (stmt); i++)
          {
            rval1 = gimple_call_arg (stmt, i);
            if ( rval1 == argv )
            {
              tmpargv = force_gimple_operand_gsi (&gsi, newargv, true, NULL, true, GSI_SAME_STMT);  
              gimple_call_set_arg (stmt, i, tmpargv);
              mark_symbols_for_renaming (stmt);
            }
          }
          break;        
        }
  
        CASE_GIMPLE_UPP:
        case GIMPLE_RETURN :  
        case GIMPLE_LABEL :
        case GIMPLE_PREDICT :
        case GIMPLE_DEBUG:
          break;
  
        default:
          printf("MAIN:UNKNOWN TREE CODE %d\n",gimple_code(stmt));
          gcc_assert(false);
          break;
      }
  
  
    }    
  }

}



static void 
do_shadow_promote_reference()
{

  /* Promote each global initializer */
  struct varpool_node *var;
  for (var = varpool_nodes; var; var = var->next)
  {
    tree decl = var->decl;

    if (in_system_header_at(DECL_SOURCE_LOCATION(decl)) || DECL_SOURCE_LOCATION(decl) == BUILTINS_LOCATION  )
      continue;

    if ( DECL_INITIAL (decl) )
    {
      tree init = DECL_INITIAL (decl);
      shadow_update_type (init) ;
    }  
  }

  /* Promote each statement */ 
  struct cgraph_node *node;
  for (node = cgraph_nodes; node; node = node->next)
  {
    if (!valid_function_node_p (node))
      continue;

    switch_to_context(node->decl);
    basic_block bb;
    FOR_EACH_BB(bb)
    {    
      for (gimple_stmt_iterator gsi = gsi_start_phis (bb); !gsi_end_p (gsi); gsi_next (&gsi))
      {
        gimple phi = gsi_stmt (gsi);
        shadow_update_type (gimple_phi_result (phi));
        for (int i = 0; i < gimple_phi_num_args (phi); i++)
        {
          tree strippedrhs = PHI_ARG_DEF (phi, i);
          STRIP_NOPS (strippedrhs);        
          shadow_update_type (strippedrhs);            
        }
      }      
      
      for (gimple_stmt_iterator gsi = gsi_start_bb (bb); !gsi_end_p (gsi); gsi_next (&gsi))
      {
      	tree lval = NULL;
      	tree rval1 = NULL;
      	tree rval2 = NULL;
      	gimple stmt = gsi_stmt (gsi);

      	switch (gimple_code (stmt))
    	  {
      	  case GIMPLE_COND:
          {
            lval = gimple_cond_lhs (stmt);
            rval1 = gimple_cond_rhs (stmt);
            shadow_promote_reference (&gsi, lval, 0);
            shadow_promote_reference (&gsi, rval1, 1);     
            break;
          }

      	  case GIMPLE_SWITCH:
          {
            lval = gimple_switch_index (stmt);
            shadow_promote_reference (&gsi, lval, 0);
            break;
          }

      	  case GIMPLE_ASSIGN:

            lval = gimple_assign_lhs (stmt);
      	    if (gimple_assign_rhs_class (stmt) == GIMPLE_BINARY_RHS)
            {
            	rval1 = gimple_assign_rhs1 (stmt);
            	rval2 = gimple_assign_rhs2 (stmt);
            }
      	    else
            {
            	rval1 = gimple_assign_rhs1 (stmt);
            }

            shadow_promote_reference (&gsi, lval, 0);
            shadow_promote_reference (&gsi, rval1, 1);
            shadow_promote_reference (&gsi, rval2, 2);       
      	    break;

      	  case GIMPLE_CALL:
    	    {            
            shadow_promote_funccall (&gsi);
    	      break;

    	    }

          case GIMPLE_RETURN :
          {
            shadow_promote_return (&gsi);
            break;
          }

          case GIMPLE_DEBUG:
          case GIMPLE_LABEL :
          case GIMPLE_PREDICT :
          CASE_GIMPLE_UPP:
            break;

      	  default:
      	    printf("MAIN:UNKNOWN TREE CODE %d\n",gimple_code(stmt));
            gcc_assert(false);
      	    break;
    	  }


      }    
    }
    
    /* Forward the arugment argv of the main procdure */
    if (is_main_procedure(node))
      shadow_forward_argv (node);

    update_ssa (TODO_update_ssa);
    switch_off_context();
    
  }

  ipa_update_loop_info ();

}

static void 
do_shadow_promotion()
{
  do_shadow_promote_declaration();
  
  do_shadow_promote_reference();    

}

#endif


static std::set<int> size_set;

/* Clear profile information attached to gimple
	 If DEFUSE=1, then REF is a write. */
static void
insert_address_shadow (gimple_stmt_iterator * gsi, tree ref, int opnd, int def)
{
  gimple call;
  tree addr, size, loc, flag;
  tree tree_printf_profiler_fn;
  bool toprofile= false;
  bool shadow = 0;
  bool bytewise=0;
  

  gimple stmt = gsi_stmt (*gsi);
  ipa_data_reference *dr = tree_data_reference (stmt, opnd);

  if (!dr)
    return;

  if ( !is_profile_candidate(dr) )
    return;
 
  shadow = is_shadow_profile_candidate(dr, ref);


  if ( cpunum > 1 )
  {
    if (alias_set.find(dr->uid()) == alias_set.end() ) 
      return;
  }
  else 
  {    
    if (dr->is_read)
    {
      if (!bitmap_bit_p(loop_extra_info(cur_loop)->dr_read, dr->uid ()))
        return;    
    }
    else if (!bitmap_bit_p(loop_extra_info(cur_loop)->dr_write, dr->uid ()))
        return;    
  }

  loc = build_int_cst_type (integer_type_node, dr->slice_id());

  switch (flag_profiling_type)
  {
    case PROFILING_PRIVATIZABLE :
        break;
    case PROFILING_DEPENDENCE:

      if (shadow)
      {
        if (dr->is_read)
          tree_printf_profiler_fn = build_fn_decl (shadow_dependence_load_profiler_name, shadow_load_profiler_fn_type);
        else
          tree_printf_profiler_fn = build_fn_decl (shadow_dependence_store_profiler_name, shadow_store_profiler_fn_type);
      }
      else
      {
        if (dr->is_read)
          if (bytewise)
            tree_printf_profiler_fn = build_fn_decl (hash_bitwise_load_profiler_name, shadow_load_profiler_fn_type);
          else
            tree_printf_profiler_fn = build_fn_decl (hash_dependence_load_profiler_name, shadow_load_profiler_fn_type);
        else
          if (bytewise)
            tree_printf_profiler_fn = build_fn_decl (hash_bitwise_store_profiler_name, shadow_store_profiler_fn_type);
          else
            tree_printf_profiler_fn = build_fn_decl (hash_dependence_store_profiler_name, shadow_store_profiler_fn_type);
      }
      break;

    default:
        return;
  };
  
  {
      
    /*
       see also in libprivatize.cxx   
       void __privatize_print_store (PTR addr, int size, int id);
     */

    switch (TREE_CODE (ref))
    {
      case COMPONENT_REF:
      case ARRAY_REF:
      case INDIRECT_REF:
      case MEM_REF:
      case VAR_DECL:
        addr = gimple_build_addr_expr (gsi, ref);
        break;
      default:
        return;
        break;
    }

    gcc_assert (addr);

    size = TYPE_SIZE_UNIT (TREE_TYPE (ref));
    call = gimple_build_call (tree_printf_profiler_fn, 3, addr, size, loc);
    size_set.insert(tree_low_cst(size, 1));
  }


  gimple_set_block (call, gimple_block(gsi_stmt(*gsi)));
  gsi_insert_before (gsi, call, GSI_SAME_STMT);
  ipa_add_abnormal_goto_call_edges (*gsi);

}







static void
insert_loop_header  (int loop_id, gimple_stmt_iterator * gsi)
{

  if (loop_id != *(loop_set.begin()))
    return;

  /* see also in libprivatize.cxx
     __privatize_print_entry (int loop_id)
     )                                                    */

  tree tree_printf_profiler_fn;

  switch (flag_profiling_type)
  {
    case PROFILING_PRIVATIZABLE :
    case PROFILING_DEPENDENCE:
    
        tree_printf_profiler_fn = build_fn_decl (shadow_dependence_enter_profiler_name, shadow_entry_profiler_fn_type); 
      break;

    default:
        return;
  };


  tree lnum = build_int_cst_type (unsigned_type_node, loop_id);

  gimple_seq seq = gimple_seq_alloc ();

  gimple call = gimple_build_call (tree_printf_profiler_fn, 1, lnum);

  
  gimple_set_block (call, gimple_block(gsi_stmt(*gsi)));
  gsi_insert_before (gsi, call, GSI_SAME_STMT);
  ipa_add_abnormal_goto_call_edges (*gsi);  

}



static void
insert_edge_shadow  (int entry, int loop_id, edge e)
{

  if (loop_id != *(loop_set.begin()))
    return;

  /* see also in libprivatize.cxx
     __privatize_print_entry (int loop_id)
     )                                                    */

  tree tree_printf_profiler_fn;

  switch (flag_profiling_type)
  {
    case PROFILING_PRIVATIZABLE :
    case PROFILING_DEPENDENCE:
    
      if (entry)
        tree_printf_profiler_fn = build_fn_decl (shadow_dependence_enter_profiler_name, shadow_entry_profiler_fn_type); 
      else  
        tree_printf_profiler_fn = build_fn_decl (shadow_dependence_exit_profiler_name, shadow_exit_profiler_fn_type);
      break;

    default:
        return;
  };


  tree lnum = build_int_cst_type (unsigned_type_node, loop_id);

  gimple_seq seq = gimple_seq_alloc ();

  gimple call = gimple_build_call (tree_printf_profiler_fn, 1, lnum);
  
  gimple_seq_add_stmt (&seq, call);
    
  gsi_insert_seq_on_edge (e, seq);

}


static void
insert_begin_shadow (gimple_stmt_iterator * gsi)
{

  /*
     see also in libprivatize.cxx   
     void __privatize_profile_finalize  ();
   */

  tree tree_printf_profiler_fn;

  switch (flag_profiling_type)
  {
    case PROFILING_PRIVATIZABLE :
    case PROFILING_DEPENDENCE:
    
        tree_printf_profiler_fn = build_fn_decl (shadow_dependence_begin_profiler_name, shadow_begin_profiler_fn_type); 
        break;
        
     default:
        return;
  };

  
  tree lnum = build_int_cst_type (unsigned_type_node, loop_dr_count+1);

//  tree lnum = build_int_cst_type (unsigned_type_node, bitmap_count_bits(loop_extra_info(cur_loop)->dr_read) +
//                                  bitmap_count_bits(loop_extra_info(cur_loop)->dr_write));

  gimple call = gimple_build_call (tree_printf_profiler_fn, 1, lnum);
  gimple_set_block (call, gimple_block(gsi_stmt(*gsi)));
  gsi_insert_before (gsi, call, GSI_SAME_STMT);
  ipa_add_abnormal_goto_call_edges (*gsi);

}



static void
insert_exit_shadow (gimple_stmt_iterator * gsi)
{

  /*
     see also in libprivatize.cxx   
     void __privatize_profile_finalize  ();
   */

  tree tree_printf_profiler_fn;

  switch (flag_profiling_type)
  {
    case PROFILING_PRIVATIZABLE :
    case PROFILING_DEPENDENCE:
    
        tree_printf_profiler_fn = build_fn_decl (shadow_dependence_end_profiler_name, shadow_end_profiler_fn_type);
        break;
        
     default:
        return;
  };

  gimple call = gimple_build_call (tree_printf_profiler_fn, 0);
  gimple_set_block (call, gimple_block(gsi_stmt(*gsi)));
  gsi_insert_before (gsi, call, GSI_SAME_STMT);
  ipa_add_abnormal_goto_call_edges (*gsi);

}


static void
insert_virtual_address (gimple_stmt_iterator gsi, int opnd, int def)
{
  gimple call;
  tree addr, size, loc, flag;
  tree tree_printf_profiler_fn;
  bool toprofile= false;
  bool shadow = 0;
  bool bytewise=0;
  

  gimple stmt = gsi_stmt (gsi);
  ipa_data_reference *dr = tree_data_reference (stmt, opnd);
  tree ref = DR_REF(dr);

  if (!dr)
    return;

  if ( cpunum > 1 )
  {
    if (alias_set.find(dr->uid()) == alias_set.end() ) 
      return;
  }
  else
  {    
    if (dr->is_read)
    {
      if (!bitmap_bit_p(loop_extra_info(cur_loop)->dr_read, dr->uid ()))
        return;    
    }
    else if (!bitmap_bit_p(loop_extra_info(cur_loop)->dr_write, dr->uid ()))
        return;    
  }

  loc = build_int_cst_type (integer_type_node, dr->slice_id());

  switch (flag_profiling_type)
  {
    case PROFILING_PRIVATIZABLE :
        break;
    case PROFILING_DEPENDENCE:

      {
        if (dr->is_read)
          if (bytewise)
            tree_printf_profiler_fn = build_fn_decl (hash_bitwise_load_profiler_name, shadow_load_profiler_fn_type);
          else
            tree_printf_profiler_fn = build_fn_decl (hash_dependence_load_profiler_name, shadow_load_profiler_fn_type);
        else
          if (bytewise)
            tree_printf_profiler_fn = build_fn_decl (hash_bitwise_store_profiler_name, shadow_store_profiler_fn_type);
          else
            tree_printf_profiler_fn = build_fn_decl (hash_dependence_store_profiler_name, shadow_store_profiler_fn_type);
      }
      break;

    default:
        return;
  };
  

  addr = ref; //force_gimple_operand_gsi (&gsi, ref, true, NULL, true, GSI_SAME_STMT);  
  size = build_int_cst_type (size_type_node, 1);
  call = gimple_build_call (tree_printf_profiler_fn, 3, addr, size, loc);


  gimple_set_block (call, gimple_block(gsi_stmt(gsi)));
  gsi_insert_after (&gsi, call, GSI_NEW_STMT);
  ipa_add_abnormal_goto_call_edges (gsi);

}


void
shadow_profile_library_call (struct cgraph_node *node)
{

  basic_block bb;
  FOR_EACH_BB (bb)
  {

    gimple_seq seq = bb_seq (bb);
    for (gimple_stmt_iterator gsi = gsi_start_bb (bb); !gsi_end_p (gsi); gsi_next (&gsi))
    {

      int flag;
      tree size;
      tree old_addr;
      tree new_addr;
      gimple stmt = gsi_stmt (gsi);

      switch (gimple_code (stmt))
      {

        case GIMPLE_CALL:
        {

          tree decl = gimple_call_fndecl(stmt);         
          if (!decl)
            break;

          const char* name = fn_decl_name (decl);

          if ( strcmp(name, "fopen") == 0 || strcmp(name, "open") == 0)
          {
            insert_virtual_address (gsi, 1, false);    
          } 
          else if ( strcmp(name, "fclose") == 0 || strcmp(name, "close") == 0)
          {
            insert_virtual_address (gsi, 2, true);    
            insert_virtual_address (gsi, 1, false);    
          }         
          else if (DECL_FUNCTION_CODE (decl) == BUILT_IN_FWRITE ||
                    DECL_FUNCTION_CODE (decl) == BUILT_IN_FPRINTF || 
                    DECL_FUNCTION_CODE (decl) == BUILT_IN_FSCANF ||
                    DECL_FUNCTION_CODE (decl) == BUILT_IN_VFPRINTF || 
                    DECL_FUNCTION_CODE (decl) == BUILT_IN_PRINTF ||
                    DECL_FUNCTION_CODE (decl) == BUILT_IN_FPUTC ||
                    DECL_FUNCTION_CODE (decl) == BUILT_IN_PUTS ||
                    DECL_FUNCTION_CODE (decl) == BUILT_IN_FPUTS ||
                    DECL_FUNCTION_CODE (decl) == BUILT_IN_PUTCHAR ||
              strcmp(name, "fread") == 0 || strcmp(name, "fseek") == 0 || strcmp(name, "ftell") == 0 || 
              strcmp(name, "fflush") == 0 || strcmp(name, "ferror") == 0 || strcmp(name, "feof") == 0 ||
              strcmp(name, "fgetc") == 0 || strcmp(name, "getc") == 0 || strcmp(name, "_IO_getc") == 0 ||
              strcmp(name, "rewind") == 0 || strcmp(name, "fgets") == 0 || strcmp(name, "ungetc") == 0 ||
              strcmp(name, "lseek") == 0 || strcmp(name, "write") == 0 || strcmp(name, "read") == 0 )
          {
            insert_virtual_address (gsi, 2, true);    
            insert_virtual_address (gsi, 1, false);    
          }

        }

      default:
        break;
      }

    }				// gsi

  }

  rewrite_into_loop_closed_ssa (NULL, TODO_update_ssa);
  update_new_call_edge (node);

}



/* */
static void
initialize_local_variables (struct cgraph_node *node)
{

	unsigned ix;
  tree var;
  struct function *dsf = DECL_STRUCT_FUNCTION (node->decl);

  basic_block bb = ENTRY_BLOCK_PTR;
  gimple_seq seq = bb_seq (bb);

  if ( gimple_seq_empty_p(seq) )
  {
    bb = single_succ(bb);
    seq = bb_seq (bb);
    if ( gimple_seq_empty_p(seq) )
    {
      gimple_stmt_iterator gsi = gsi_last_bb (bb);
      gsi_insert_after (&gsi, gimple_build_nop(), GSI_NEW_STMT);
    }
  }



  tree tree_fn = implicit_built_in_decls[BUILT_IN_MEMSET];  
  tree zero = build_int_cst_type (unsigned_type_node, 0);

  gimple call;
  tree addr, size;

  FOR_EACH_LOCAL_DECL (dsf, ix, var)
  {
    if (TREE_CODE (var) == PARM_DECL)
      continue;
    if ( DECL_ARTIFICIAL(var) )
      continue;
    if ( TREE_STATIC (var) || DECL_EXTERNAL (var) )
      continue;
    if ( !TREE_ADDRESSABLE(var) )
      continue;

    gimple_stmt_iterator gsi = gsi_start_bb (bb);
    size =  TYPE_SIZE_UNIT(TREE_TYPE(var));        
    size = force_gimple_operand_gsi (&gsi, size, true, NULL, true, GSI_SAME_STMT);  
    addr = gimple_build_addr_expr (&gsi, var);
    call = gimple_build_call (tree_fn, 3, addr, zero, size);
    gimple_set_block (call, gimple_block(gsi_stmt(gsi)));
    gsi_insert_before (&gsi, call, GSI_SAME_STMT);
    ipa_add_abnormal_goto_call_edges (gsi);
  }

  // TODO: clear tag fields of formal parameters
 
  rewrite_into_loop_closed_ssa (NULL, TODO_update_ssa);
  update_new_call_edge (node);


}



void
remove_free_call (struct cgraph_node *node)
{

  basic_block bb;
  FOR_EACH_BB (bb)
  {

    gimple_seq seq = bb_seq (bb);
    for (gimple_stmt_iterator gsi = gsi_start_bb (bb); !gsi_end_p (gsi); )
    {

      int flag;
      tree size;
      tree old_addr;
      tree new_addr;
      gimple stmt = gsi_stmt (gsi);

      switch (gimple_code (stmt))
      {

        case GIMPLE_CALL:
        {
          /* the arguments are to be profiled!!! */
          tree decl = gimple_call_fndecl (stmt);

          if (decl)
        	{
            const char* name = fn_decl_name (decl);
        	  if (DECL_FUNCTION_CODE (decl) == BUILT_IN_FREE)
            { 
              struct cgraph_edge *edge = cgraph_edge (node, stmt);
              cgraph_remove_edge (edge);
              gsi_remove (&gsi, true);
              break;
            }             
         	}
        }

        default:
          gsi_next (&gsi);
          break;
      }

    }				// gsi

  }

  rewrite_into_loop_closed_ssa (NULL, TODO_update_ssa);
  update_new_call_edge (node);

}



void
replace_system_call (struct cgraph_node *node)
{

  basic_block bb;
  FOR_EACH_BB (bb)
  {

    gimple_seq seq = bb_seq (bb);
    for (gimple_stmt_iterator gsi = gsi_start_bb (bb); !gsi_end_p (gsi); gsi_next (&gsi))
    {

      int flag;
      tree size;
      tree old_addr;
      tree new_addr;
      gimple stmt = gsi_stmt (gsi);

      switch (gimple_code (stmt))
      {

        case GIMPLE_CALL:
        {
          /* the arguments are to be profiled!!! */
          tree decl = gimple_call_fndecl (stmt);

          if (decl)
        	{
            const char* name = fn_decl_name (decl);
            tree tree_edge_fn;
            gimple call;
                      

            /* The same declaration with profiling version and need to replace the original */
            
        	  if (DECL_FUNCTION_CODE (decl) == BUILT_IN_MALLOC)
              tree_edge_fn = build_fn_decl (edge_malloc_name, edge_malloc_fn_type); 
        	  else if (DECL_FUNCTION_CODE (decl) == BUILT_IN_CALLOC)
              tree_edge_fn = build_fn_decl (edge_calloc_name, edge_calloc_fn_type); 
        	  else if (DECL_FUNCTION_CODE (decl) == BUILT_IN_REALLOC)
              tree_edge_fn = build_fn_decl (edge_realloc_name, edge_realloc_fn_type); 
        	  else if (DECL_FUNCTION_CODE (decl) == BUILT_IN_FREE)
              tree_edge_fn = build_fn_decl (edge_free_name, edge_free_fn_type);   
        	  else if (DECL_FUNCTION_CODE (decl) == BUILT_IN_FREE)
              tree_edge_fn = build_fn_decl (edge_free_name, edge_free_fn_type);   
            else 
              break;

            int nargs = gimple_call_num_args (stmt);
            VEC(tree, heap) *vargs = VEC_alloc (tree, heap, nargs);
            for (int i = 0; i <nargs ; i++)
            {
              tree arg = gimple_call_arg (stmt, i);
              VEC_quick_push (tree, vargs, arg);
            }
            
            call = gimple_build_call_vec (tree_edge_fn, vargs);
            VEC_free (tree, heap, vargs);
            
            tree retval = gimple_call_lhs (stmt);
            if (retval)
              gimple_set_lhs (call, retval);   
            
            gsi_insert_after(&gsi, call, GSI_SAME_STMT);  
            struct cgraph_edge *edge = cgraph_edge (node, stmt);
            cgraph_remove_edge (edge);
            gsi_remove (&gsi, true);
             
            
             

        	}


          break;

        }

      default:
        break;
      }

    }				// gsi

  }

  rewrite_into_loop_closed_ssa (NULL, TODO_update_ssa);
  update_new_call_edge (node);

}

static void
shadow_specify_cpu(int cpu, const char *memoptasksinfo)
{
  /* Reading aliasclass.info, fill taskpool 
     MEMOP UID %8d  CPU  %4d         
  */
  FILE *fp = fopen(memoptasksinfo, "r"); 
  
  if (!fp)
    return;

  char *line = NULL;
  size_t len = 0;
  int bytes_read;
  while ((bytes_read = getline (&line, &len, fp)) != -1)
  {
    char str[100];
    sscanf (line, "%s", str);
  
    if (strcmp (str, "MEMOP") == 0)
    {
      int          uid;
      int     cpuid;  // The total load/store number of all executions 
      sscanf (line, "%*s %*s %d %*s %d", &uid, &cpuid);
      if (cpuid == cpu )
        alias_set.insert(uid);
    }
  }
  fclose(fp);


 // 

}

typedef long double FLOAT128;

class CPU
{
public:
  int             id;
  std::list<int>  tasks;
  long double     weight;

  CPU(int cpu_id) :
    id(cpu_id),
    tasks(),
    weight(0)
  {}

  bool operator<(const CPU &cpu) const
  {
    return weight < cpu.weight;
  }
};

void partition_tasks(int cpunum, FILE* fp)
{
  
  typedef std::pair<long double, int> TASK;
  typedef std::vector<TASK>  TASK_POOL;  

  typedef std::vector<CPU>  CPU_POOL;

  TASK_POOL task_pool;
  CPU_POOL  cpu_pool;
  FLOAT128 total_instruction_count = 0;


  for (ALIAS_CLASS_MAP::iterator iter = alias_class_map.begin(); iter != alias_class_map.end(); ++iter )
  {
    ALIAS_CLASS_INFO &info = iter->second;  
    TASK task(info.instruction_count, iter->first);
    task_pool.push_back(task);
    total_instruction_count += info.instruction_count;
  }  

  
  gcc_assert( !task_pool.empty() );

  // sort task_pool
  std::sort (task_pool.begin(), task_pool.end()); // from small to large

  FLOAT128  threshold = total_instruction_count / cpunum ;

  fprintf(fp, "%d tasks\n", task_pool.size() );
  fprintf(fp, "%d cpus\n", cpunum );
  fprintf(fp, "Threshold %Lf\n\n", threshold );

  // initialize cpu pool
  for (int i = 1; i <= cpunum; i++)
  {
    CPU cpu(i);
    cpu_pool.push_back(cpu);
  }

  while ( !task_pool.empty() )
  {
    for (int i = 0; i < cpunum; i++)
    {
      if ( task_pool.empty() )
        break;
      CPU  &cpu = cpu_pool[i];
      if ( cpu.weight >= threshold)
        continue;
      TASK task = *task_pool.rbegin();    // the largest weight
      task_pool.pop_back();
      cpu.tasks.push_back(task.second);
      cpu.weight += task.first;
    }
    // sort cpu pool
    std::sort (cpu_pool.begin(), cpu_pool.end()); // from small to large    
  }


  std::map<int, int> task_map;
  for (int i = 0; i < cpunum; i++)
  {
    CPU  &cpu = cpu_pool[i];    
    fprintf(fp, "CPU %-4d       WEIGHT   %Lf  \n", cpu.id, cpu.weight );
    //fprintf(fp, "CPU %4d  TASKS  ", cpu.id );
    for ( std::list<int>::iterator iter = cpu.tasks.begin(); iter != cpu.tasks.end(); ++iter)
    {
      task_map[*iter] = cpu.id;
      //fprintf(fp, "%d ", *iter);      
    }
  }
  fprintf(fp, "\n\n=================================================================\n\n");


  /* Output tasks in form of memops */

  unsigned int i;
  bitmap_iterator bi;
  EXECUTE_IF_SET_IN_BITMAP (loop_extra_info(cur_loop)->dr_read, 0, i, bi)
  {
    ipa_data_reference *dr = ipa_get_data_ref(i);  
    fprintf(fp, "MEMOP UID %8d  CPU  %4d\n", dr->uid(), task_map[dr->class_id()] );
  }
  
  EXECUTE_IF_SET_IN_BITMAP (loop_extra_info(cur_loop)->dr_write, 0, i, bi)
  {
    ipa_data_reference *dr = ipa_get_data_ref(i);  
    fprintf(fp, "MEMOP UID %8d  CPU  %4d\n", dr->uid(), task_map[dr->class_id()] );
  }  
  
}

extern "C" void
symee_init_alias (void)
{
  if (!in_lto_p || flag_wpa)
    fprintf (stderr,
             "error: bootstraping alias analysis only run in simple lto mode\n"
             "       or it will generate wrong alias information\n");

  {
    FILE *fp = fopen ("libfunc.info", "w");
    find_library_calls (fp);
    fclose (fp);
  }

  /* Construct data structure and SSA form for each loop,
     initialize function extra information
  */
  {
    FILE *fp = fopen ("loop.info", "w");
    ipa_ssa_loop_init (fp);
    fclose (fp);
  }


  //specify_loops_to_profile ();

  /* Assign an ID to each gimple statement */
  {
    FILE *fp = fopen ("stmt.info", "w");
    ipa_gimple_id_assignment (fp);
    fclose (fp);
  }

  if ( flag_profiling_opt >= PROFILING_O2 )
    read_count_infor_and_remove_bb ();

  /* Create data references to record each memory access */
  dr_hash = htab_create (100, hash_dr_id, eq_dr_id, hash_dr_del);
  ipa_create_data_references (NULL);


  /* Find potential malloc functions */
  if (1 || flag_profiling_opt >= PROFILING_O1 )
    {
      FILE *fp = fopen ("malloc.info", "w");
      recognize_mem_allocate_functions(fp);
      fclose (fp);    
      ipa_points_to_analysis();   
      /* Use points-to result to complement indirect call edges  */
      cgraph_resolve_indirect_edges ();
    }

  /* compute reference set */
  fprintf (stderr, "start computing reference sets \n");
  htab_traverse (dr_hash, compute_reference_set, NULL);
    
  /* based on points-to analysis */
  FILE *fp = fopen ("aliasclass.info", "w");
  ipa_classify_memory_operations(fp);
  fclose (fp);    

}


extern "C" void
symee_fini_alias ()
{
  struct cgraph_node *node;
  /* Cleanup. */
  for (node = cgraph_nodes; node; node = node->next)
    {
      /* Get rid of the aux information.      */
      if (node->aux)
        {
          free (node->aux);
          node->aux = NULL;
        }
    }
  ipa_ssa_loop_done ();
  htab_delete (dr_hash);
  htab_delete (loop_hash);
}


/* Function executed by the pass for each function.  */

static unsigned int
ipa_hash_profiling (void)
{
  {
    FILE *fp = fopen ("libfunc.info", "w");
    find_library_calls (fp);
    fclose (fp);
  }

  /* Construct data structure and SSA form for each loop,
     initialize function extra information
  */
  {
    FILE *fp = fopen ("loop.info", "w");
    ipa_ssa_loop_init (fp);
    fclose (fp);
  }

  canonicalize_gimple_call ();

  specify_loops_to_profile ();

  /* Assign an ID to each gimple statement */
  {
    FILE *fp = fopen ("stmt.info", "w");
    ipa_gimple_id_assignment (fp);
    fclose (fp);
  }


  /* Find potential malloc functions */
  //if ( flag_profiling_opt >= PROFILING_O1 )
  {
    read_mem_allocate_func("malloc.info");
    ipa_points_to_analysis();   
    /* Use points-to result to complement indirect call edges  */
    cgraph_resolve_indirect_edges ();
  }

  
  if ( flag_profiling_opt >= PROFILING_O2 )
    read_count_infor_and_remove_bb ();

  /* Create data references to record each memory access */
  dr_hash = htab_create (100, hash_dr_id, eq_dr_id, hash_dr_del);
  ipa_create_data_references (NULL);


  if ( flag_profiling_opt >= PROFILING_O1 )
  {
    /* compute reference set */
    printf ("start computing reference sets \n");
    htab_traverse (dr_hash, compute_reference_set, NULL);
  }

 
  {
    FILE *fp = fopen ("memop.info", "w");
    ipa_dump_data_references (fp);
    fclose (fp);
  }

  if ( flag_profiling_opt >= PROFILING_O1 )
  {
    /* based on points-to analysis */
    FILE *fp = fopen ("aliasclass.info", "w");
    ipa_classify_memory_operations(fp);
    fclose (fp);    
  }

  cur_loop = ipa_get_loop(*loop_set.begin());
  if (0)
  {
    FILE *fp = fopen("loopmemop.info", "w");
    ipa_collect_data_references_for_loop(cur_loop, fp);    
    fclose(fp);
  }
  else
  {
    FILE *fp = fopen("loopmemop.info", "w");
    dump_data_references_for_loop (cur_loop, fp);
    fclose(fp);
  }

  /* update instruction count of alias class with loop specified */
  if ( flag_profiling_opt >= PROFILING_O2 )
  {
    FILE *fp = fopen ("loopaliasclass.info", "w");
    ipa_update_alias_class_for_loop(cur_loop, fp);
    fclose (fp);
  }

  if ( flag_profiling_opt >= PROFILING_O1 )
  {
    FILE *fp = fopen ("looppartition.info", "w");
    partition_tasks(cpunum, fp);
    fclose(fp);
  }

  if ( flag_profiling_opt >= PROFILING_O1 )
    delete_points_to_sets ();  

  /* Step 2. Instrument to each procedure */

  {
    printf ("start instrumenting for dependence profiling \n");

    initialize_profile_prototype ();

    if ( cpunum > 1 )
      shadow_specify_cpu( cpuid, "looppartition.info");      

    /* Clear profile information attached to gimple */
    struct cgraph_node *node;
    for (node = cgraph_nodes; node; node = node->next)
    {
      if (!valid_function_node_p (node))
        continue;
      switch_to_context (node->decl);

      /* Insert profiling statements to collect memory address read/write */
      profile_memop (node, insert_address_hash);

      /* Inserting loop entry/exit on edges */
      profile_loop_entry (node, insert_loop_header);
      profile_loop_exit_edge (node, insert_edge_shadow);

      remove_free_call(node);

      profile_begin (node, insert_begin_shadow);

      profile_exit (node, insert_exit_shadow);

      shadow_profile_library_call (node);

      switch_off_context ();
    }

    /* Cleanup. */
    for (node = cgraph_nodes; node; node = node->next)
    {
      /* Get rid of the aux information.      */
      if (node->aux)
      {
        free (node->aux);
        node->aux = NULL;
      }
    }

    printf ("ipa profile dependence done\n");
  }

  dump_all_nodes ("dump-hash");
  update_cgraph ();   

  ipa_ssa_loop_done ();


  htab_delete (dr_hash);

  return 0;
}




static bool
gate_ipa_hash_profiling (void)
{
  return flag_ipa_hash_profiling;
}

struct simple_ipa_opt_pass pass_hash_profiling = 
{
  {
    SIMPLE_IPA_PASS,
    "hash-profiling",	/* name */
    gate_ipa_hash_profiling,	/* gate */
    ipa_hash_profiling,	/* execute */
    NULL,			/* sub */
    NULL,			/* next */
    0,				/* static_pass_number */
    TV_TREE_PROFILE_DEPENDENCE,	/* tv_id */
    PROP_ssa,			/* properties_required */
    0,				/* properties_provided */
    0,				/* properties_destroyed */
    0,				/* todo_flags_start */
    TODO_dump_func | TODO_rebuild_alias | TODO_update_ssa	/* todo_flags_finish */
  }
};

/* Function executed by the pass for each function.  */

static unsigned int
ipa_shadow_profiling (void)
{
  dr_hash = NULL;

  {
    FILE *fp = fopen ("libfunc.info", "w");
    find_library_calls (fp);
    fclose (fp);
  }

  /* Construct data structure and SSA form for each loop,
     initialize function extra information
  */
  {
    FILE *fp = fopen ("loop.info", "w");
    ipa_ssa_loop_init (fp);
    fclose (fp);
  }

  canonicalize_gimple_call ();

  specify_loops_to_profile ();

  /* Assign an ID to each gimple statement */
  {
    FILE *fp = fopen ("stmt.info", "w");
    ipa_gimple_id_assignment (fp);
    fclose (fp);
  }


  /* Find potential malloc functions */
  //if ( flag_profiling_opt >= PROFILING_O1 )
  {
    read_mem_allocate_func("malloc.info");
    ipa_points_to_analysis(); 
    /* Use points-to result to complement indirect call edges  */
    cgraph_resolve_indirect_edges ();
    ipa_convert_icall ();
  }

  
  if ( flag_profiling_opt >= PROFILING_O2 )
    read_count_infor_and_remove_bb ();

  /* Create data references to record each memory access */
  dr_hash = htab_create (100, hash_dr_id, eq_dr_id, hash_dr_del);
  ipa_create_data_references (NULL);


  if ( flag_profiling_opt >= PROFILING_O1 || flag_ipa_pta_use_bootstrap)
  {
    /* compute reference set */
    fprintf (stderr, "start computing reference sets \n");
    htab_traverse (dr_hash, compute_reference_set, NULL);
  }

 
  {
    FILE *fp = fopen ("memop.info", "w");
		ipa_dump_data_references (fp);
    fclose (fp);
  }

  if ( flag_profiling_opt >= PROFILING_O1 )
  {
    /* based on points-to analysis */
    FILE *fp = fopen ("aliasclass.info", "w");
    ipa_classify_memory_operations(fp);
    fclose (fp);    
  }

  cur_loop = ipa_get_loop(*loop_set.begin());

  if (0)
  {
    FILE *fp = fopen("loopmemop.info", "w");
    ipa_collect_data_references_for_loop(cur_loop, fp);    
    fclose(fp);
  }
  else
  {
    FILE *fp = fopen("loopmemop.info", "w");
    dump_data_references_for_loop (cur_loop, fp);
    fclose(fp);
  }

  /* update instruction count of alias class with loop specified */
  if ( flag_profiling_opt >= PROFILING_O2 )
  {
    FILE *fp = fopen ("loopaliasclass.info", "w");
    ipa_update_alias_class_for_loop(cur_loop, fp);
    fclose (fp);
  }

  if ( flag_profiling_opt >= PROFILING_O1 )
  {
    FILE *fp = fopen ("looppartition.info", "w");
    partition_tasks(cpunum, fp);
    fclose(fp);
  }

  //added by kobill
  if(flag_ipa_pta_use_bootstrap)
	  ipa_collect_data_dependencies (false);

  if ( flag_profiling_opt >= PROFILING_O1 )
    delete_points_to_sets ();  

  /* Step 2. Instrument to each procedure */

  {
    printf ("start instrumenting for dependence profiling \n");

    initialize_profile_prototype ();

    dump_all_nodes ("dump-shadow0");

    do_shadow_promotion();

    if ( cpunum > 1 )
      shadow_specify_cpu( cpuid, "looppartition.info");      

    /* Clear profile information attached to gimple */
    struct cgraph_node *node;
    for (node = cgraph_nodes; node; node = node->next)
    {
    	if (!valid_function_node_p (node))
    	  continue;
    	switch_to_context (node->decl);

      /* Insert profiling statements to collect memory address read/write */
      profile_memop (node, insert_address_shadow);

      /* Inserting loop entry/exit on edges */
      profile_loop_entry (node, insert_loop_header);
      profile_loop_exit_edge (node, insert_edge_shadow);

      replace_system_call(node);

      profile_begin (node, insert_begin_shadow);

      profile_exit (node, insert_exit_shadow);

      shadow_profile_library_call (node);

      initialize_local_variables(node);

    	switch_off_context ();
    }

    /* Cleanup. */
    for (node = cgraph_nodes; node; node = node->next)
    {
      /* Get rid of the aux information.      */
      if (node->aux)
      {
	      free (node->aux);
	      node->aux = NULL;
      }
    }

    printf ("ipa profile dependence done\n");
  }

  dump_all_nodes ("dump-shadow");
  update_cgraph ();    

  ipa_ssa_loop_done ();


  htab_delete (dr_hash);

  return 0;
}

static bool
gate_ipa_shadow_profiling (void)
{
  return flag_ipa_shadow_profiling;
}


struct simple_ipa_opt_pass pass_shadow_profiling = 
{
  {
    SIMPLE_IPA_PASS,
    "shadow-profiling",	/* name */
    gate_ipa_shadow_profiling,	/* gate */
    ipa_shadow_profiling,	/* execute */
    NULL,			/* sub */
    NULL,			/* next */
    0,				/* static_pass_number */
    TV_TREE_PROFILE_DEPENDENCE,	/* tv_id */
    PROP_ssa,			/* properties_required */
    0,				/* properties_provided */
    0,				/* properties_destroyed */
    0,				/* todo_flags_start */
    TODO_update_ssa	/* todo_flags_finish TODO_rebuild_alias*/
  }
};



/* Function executed by the pass for each function.  */

static unsigned int
ipa_shadow_test (void)
{

  /* Construct data structure and SSA form for each loop,
     initialize function extra information
  */
  {
    FILE *fp = fopen ("loop.info", "w");
    ipa_ssa_loop_init (fp);
    fclose (fp);
  }


  do_shadow_promotion();  

  ipa_ssa_loop_done ();

 
  /* Cleanup. */
  for (struct cgraph_node *node = cgraph_nodes; node; node = node->next)
  {
    /* Get rid of the aux information.      */
    if (node->aux)
    {
      free (node->aux);
      node->aux = NULL;
    }
  }

  return 0;
}


static bool
gate_ipa_shadow_test (void)
{
  return flag_ipa_shadow_test;
}

struct simple_ipa_opt_pass pass_shadow_test = 
{
  {
    SIMPLE_IPA_PASS,
    "shadow-test",  /* name */
    gate_ipa_shadow_test, /* gate */
    ipa_shadow_test,  /* execute */
    NULL,     /* sub */
    NULL,     /* next */
    0,        /* static_pass_number */
    TV_TREE_PROFILE_DEPENDENCE, /* tv_id */
    PROP_ssa,     /* properties_required */
    0,        /* properties_provided */
    0,        /* properties_destroyed */
    0,        /* todo_flags_start */
    TODO_dump_func | TODO_update_ssa  /* todo_flags_finish TODO_rebuild_alias*/
  }
};


bool expr_is_structural(tree ref)
{
  tree type = TREE_TYPE(ref);
  return TREE_CODE(type) == RECORD_TYPE;
}

basic_block flatten_structural_assignment(basic_block bb, tree lhs, tree rhs, const location_t location);

/*
  Flatten the assignment from rhs to lhs.
  Return the last bb after flattening
*/
basic_block flatten_assignment(basic_block bb, tree lhs, tree rhs, const location_t location)
{
  gimple_stmt_iterator gsi = gsi_last_bb (bb);
  tree type = TREE_TYPE(lhs);
  
  switch ( TREE_CODE(type) )
  {
    case INTEGER_TYPE:
    case REAL_TYPE:
    case ENUMERAL_TYPE:
    case POINTER_TYPE:
    {
      lhs = force_gimple_operand_gsi (&gsi, lhs, false, NULL, true, GSI_SAME_STMT);  
      rhs = force_gimple_operand_gsi (&gsi, rhs, true, NULL, true, GSI_SAME_STMT);  
      gimple assgin = gimple_build_assign (lhs, rhs);
      gimple_set_location( assgin,  location);          
      gsi_insert_before (&gsi, assgin, GSI_SAME_STMT);       
      break;
    }

    case RECORD_TYPE:
    case UNION_TYPE :
    {
      bb = flatten_structural_assignment (bb, lhs, rhs, location);
      break;
    }
    
    case ARRAY_TYPE:
    {
      edge e = split_block (bb, NULL);
      bb = e->dest;
      tree elem_type = TREE_TYPE(type);
      tree domain_type = TYPE_DOMAIN (type);
      tree idx_type = TREE_TYPE(TYPE_MAX_VALUE(domain_type));
      tree initial_value = TYPE_MIN_VALUE(domain_type);
      tree stride = build_int_cst_type (idx_type, 1);
      tree upper_bound = TYPE_MAX_VALUE(domain_type);
      tree iv, iv_before, iv_after;
      iv = create_tmp_var (idx_type, "flatten_i");
      loop_p loop = create_empty_loop_on_edge (e, initial_value, stride, upper_bound, iv, &iv_before, &iv_after,
                                 bb->loop_father);
    
      // flatten_array_assignment (after, lhs, rhs, location);
      tree lhs_elem = build4 (ARRAY_REF, elem_type, lhs, iv_before, NULL_TREE, NULL_TREE);
      tree rhs_elem = build4 (ARRAY_REF, elem_type, rhs, iv_before, NULL_TREE, NULL_TREE);
      flatten_assignment (loop->header, lhs_elem, rhs_elem, location);
      break; 
    }
    
    default:      
      gcc_assert(false);
      break;

  }

  return bb;
}


/*

  struct {
    int a;
    struct {
      int b;
      int c[100];
    } d;
    int e;
  } s, t;

  s = t; 
    ==>
  s.a = t.a;
  s.d.b = t.d.b;
  for (int i=0; i<100; i++)
    s.d.c[i] = t.d.c[i];
  s.e = t.e;   

  Return the last bb after flattening

*/

basic_block flatten_structural_assignment(basic_block bb, tree lhs, tree rhs, const location_t location)
{
  gimple_stmt_iterator gsi = gsi_last_bb (bb);

  // split bb

  tree type = TREE_TYPE(lhs);

  for (tree field = TYPE_FIELDS (type); field; field = DECL_CHAIN (field))
  {
    gcc_assert(TREE_CODE (field) == FIELD_DECL);
    tree fld_type = TREE_TYPE(field);

    lhs = unshare_expr (lhs);
    rhs = unshare_expr (rhs);
    
    tree lhs_fld = build3 (COMPONENT_REF, fld_type, lhs, field,  NULL );
    tree rhs_fld = build3 (COMPONENT_REF, fld_type, rhs, field,  NULL );       
    bb = flatten_assignment (bb, lhs_fld, rhs_fld, location);     
  }

  return bb;
}

unsigned int
ipa_flatten_structural_assignments()
{

  ipa_ssa_loop_init (NULL);


  struct cgraph_node *node;
  for (node = cgraph_nodes; node; node = node->next)
  {
    if (!valid_function_node_p (node))
      continue;

    switch_to_context(node->decl);
    basic_block bb;
    FOR_EACH_BB(bb)
    {    
            
      for (gimple_stmt_iterator gsi = gsi_start_bb (bb); !gsi_end_p (gsi); gsi_next (&gsi))
      {
      	tree lval = NULL;
      	gimple stmt = gsi_stmt (gsi);
        //fold_stmt_inplace (stmt);
        bool split = false;
      	switch (gimple_code (stmt))
    	  {
      	  case GIMPLE_ASSIGN:
          {
            tree lhs = gimple_assign_lhs (stmt);
            if ( expr_is_structural(lhs) )  
            {
              gcc_assert(gimple_assign_rhs_class (stmt) == GIMPLE_SINGLE_RHS);
              tree rhs = gimple_assign_rhs1 (stmt);
              gcc_assert(expr_is_structural(rhs));

              // split bb 
              edge e = split_block (bb, stmt);
              basic_block dest = e->dest;
              remove_edge (e);

              basic_block new_bb = create_empty_bb (bb);
              new_bb->loop_depth = bb->loop_depth;
              new_bb->loop_father = bb->loop_father;
              make_edge (bb, new_bb, EDGE_FALLTHRU);
              make_edge (new_bb, dest, EDGE_FALLTHRU);
              
              flatten_structural_assignment(new_bb, lhs, rhs, gimple_location(stmt));
              gsi_remove (&gsi, true);
              split = true;
            }

      	    break;
          }

      	  default:
      	    break;
    	  }

        if (split)
          break;

      }    
    }
    switch_off_context();
  }

  /* Cleanup. */
  for (struct cgraph_node *node = cgraph_nodes; node; node = node->next)
  {
    /* Get rid of the aux information.      */
    if (node->aux)
    {
      free (node->aux);
      node->aux = NULL;
    }
  }

  ipa_ssa_loop_done ();

  
  return 0;
}


static bool
gate_ipa_flatten_structural_assignments (void)
{
  return flag_flatten_structural_assignments;
}

      


struct simple_ipa_opt_pass pass_flatten_structural_assignments = 
{
  {
    SIMPLE_IPA_PASS,
    "flatten-structcopy",  /* name */
    gate_ipa_flatten_structural_assignments, /* gate */
    ipa_flatten_structural_assignments,  /* execute */
    NULL,     /* sub */
    NULL,     /* next */
    0,        /* static_pass_number */
    TV_TREE_PROFILE_DEPENDENCE, /* tv_id */
    PROP_ssa,     /* properties_required */
    0,        /* properties_provided */
    0,        /* properties_destroyed */
    0,        /* todo_flags_start */
    TODO_dump_func | TODO_update_ssa  /* todo_flags_finish TODO_rebuild_alias*/
  }
};



static void 
emit_sizeof(gimple_stmt_iterator * gsi, tree ref, int opnd)
{
  if (!ref)
    return; 

  if ( TREE_CODE(ref) == SIZEOF_EXPR)
  {    
    gimple stmt = gsi_stmt (*gsi);
    tree type = TREE_OPERAND(ref, 0);
    tree size = size_in_bytes (type);
    size = force_gimple_operand_gsi (gsi, size, false, NULL, true, GSI_SAME_STMT);  
    gimple_set_op( stmt, opnd, size );
    mark_symbols_for_renaming (stmt);
  }
  else if (TREE_CODE_CLASS (TREE_CODE (ref)) == tcc_unary)
  {    
    gimple stmt = gsi_stmt (*gsi);
    tree op = TREE_OPERAND(ref, 0);
    if ( TREE_CODE(op) == SIZEOF_EXPR)
    {    
      tree type = TREE_OPERAND(op, 0);
      tree size = size_in_bytes (type);
      size = force_gimple_operand_gsi (gsi, size, false, NULL, true, GSI_SAME_STMT);  
      TREE_OPERAND(ref, 0) = size;
      mark_symbols_for_renaming (stmt);
    }
  }
  else if (TREE_CODE_CLASS (TREE_CODE (ref)) == tcc_binary)
  {    
    gimple stmt = gsi_stmt (*gsi);
    for (int i=0; i<2; i++)
    {
      tree op = TREE_OPERAND(ref, i);
      if ( TREE_CODE(op) == SIZEOF_EXPR)
      {    
        tree type = TREE_OPERAND(op, 0);
        tree size = size_in_bytes (type);
        size = force_gimple_operand_gsi (gsi, size, false, NULL, true, GSI_SAME_STMT);  
        TREE_OPERAND(ref, i) = size;
        mark_symbols_for_renaming (stmt);
      }
    }
  }
}


static unsigned int
ipa_emit_sizeof()
{

  ipa_ssa_loop_init (NULL);


  struct cgraph_node *node;
  for (node = cgraph_nodes; node; node = node->next)
  {

    if (!valid_function_node_p (node))
      continue;

    switch_to_context(node->decl);
    basic_block bb;
    FOR_EACH_BB(bb)
    {          
      
      for (gimple_stmt_iterator gsi = gsi_start_bb (bb); !gsi_end_p (gsi); gsi_next (&gsi))
      {
      	tree lval = NULL;
      	tree rval1 = NULL;
      	tree rval2 = NULL;
      	gimple stmt = gsi_stmt (gsi);

      	switch (gimple_code (stmt))
    	  {
      	  case GIMPLE_COND:
          {
            lval = gimple_cond_lhs (stmt);
            rval1 = gimple_cond_rhs (stmt);
            emit_sizeof (&gsi, lval, 0);
            emit_sizeof (&gsi, rval1, 1);     
            break;
          }

      	  case GIMPLE_SWITCH:
          {
            lval = gimple_switch_index (stmt);
            emit_sizeof (&gsi, lval, 0);
            break;
          }

      	  case GIMPLE_ASSIGN:

            lval = gimple_assign_lhs (stmt);
      	    if (gimple_assign_rhs_class (stmt) == GIMPLE_BINARY_RHS)
            {
            	rval1 = gimple_assign_rhs1 (stmt);
            	rval2 = gimple_assign_rhs2 (stmt);
            }
      	    else
            {
            	rval1 = gimple_assign_rhs1 (stmt);
            }

            emit_sizeof (&gsi, lval, 0);
            emit_sizeof (&gsi, rval1, 1);
            emit_sizeof (&gsi, rval2, 2);       
      	    break;

      	  case GIMPLE_CALL:
    	    {            
            for (int i = 0; i < gimple_call_num_args (stmt); i++)
        		{
        		  rval1 = gimple_call_arg (stmt, i);
              emit_sizeof (&gsi, rval1, i+3);
        		}
            
            lval = gimple_call_lhs (stmt);
            emit_sizeof (&gsi, lval, 0);            
    	      break;

    	    }

          case GIMPLE_RETURN :
          {
            rval1 = gimple_return_retval(stmt);
            emit_sizeof (&gsi, lval, 0);
            break;
          }

          case GIMPLE_DEBUG:
            lval = gimple_debug_bind_get_var (stmt);
            emit_sizeof (&gsi, lval, 0);            
            rval1 = gimple_debug_bind_get_value (stmt);
            emit_sizeof (&gsi, rval1, 1);
            break;
            
          case GIMPLE_LABEL :
          case GIMPLE_PREDICT :
          case GIMPLE_NOP:
            break;

          CASE_GIMPLE_UPP:
            break;

      	  default:
      	    printf("MAIN:UNKNOWN TREE CODE %d\n",gimple_code(stmt));
            gcc_assert(false);
      	    break;
    	  }


      }    
    }
    switch_off_context();

  }

  /* Cleanup. */
  for (struct cgraph_node *node = cgraph_nodes; node; node = node->next)
  {
    /* Get rid of the aux information.      */
    if (node->aux)
    {
      free (node->aux);
      node->aux = NULL;
    }
  }

  ipa_ssa_loop_done ();

  
  return 0;
}

      
static bool
gate_ipa_emit_sizeof (void)
{
  return flag_keep_sizeof;
}


struct simple_ipa_opt_pass pass_emit_sizeof = 
{
  {
    SIMPLE_IPA_PASS,
    "emit-sizeof",  /* name */
    gate_ipa_emit_sizeof, /* gate */
    ipa_emit_sizeof,  /* execute */
    NULL,     /* sub */
    NULL,     /* next */
    0,        /* static_pass_number */
    TV_TREE_PROFILE_DEPENDENCE, /* tv_id */
    PROP_ssa,     /* properties_required */
    0,        /* properties_provided */
    0,        /* properties_destroyed */
    0,        /* todo_flags_start */
    TODO_dump_func | TODO_update_ssa  /* todo_flags_finish TODO_rebuild_alias*/
  }
};


unsigned int
remove_upp_directives()
{

  struct cgraph_node *node;
  for (node = cgraph_nodes; node; node = node->next)
  {

    if (!valid_function_node_p (node))
      continue;

    switch_to_context(node->decl);
    basic_block bb;
    FOR_EACH_BB(bb)
    {          
      
      for (gimple_stmt_iterator gsi = gsi_start_bb (bb); !gsi_end_p (gsi); )
      {
        gimple stmt = gsi_stmt (gsi);

        if (is_gimple_upp(stmt))
          gsi_remove (&gsi, true);
        else
          gsi_next (&gsi);

      }    
    }
    switch_off_context();

  }

  /* Cleanup. */
  for (struct cgraph_node *node = cgraph_nodes; node; node = node->next)
  {
    /* Get rid of the aux information.      */
    if (node->aux)
    {
      free (node->aux);
      node->aux = NULL;
    }
  }


  
  return 0;
}


static bool      
gate_remove_upp_directives(void)
{
  return false;
}


struct simple_ipa_opt_pass pass_remove_upp_directives = 
{
  {
    SIMPLE_IPA_PASS,
    "remove-upp",  /* name */
    gate_remove_upp_directives, /* gate */
    remove_upp_directives,  /* execute */
    NULL,     /* sub */
    NULL,     /* next */
    0,        /* static_pass_number */
    TV_TREE_PROFILE_DEPENDENCE, /* tv_id */
    PROP_ssa,     /* properties_required */
    0,        /* properties_provided */
    0,        /* properties_destroyed */
    0,        /* todo_flags_start */
    TODO_dump_func | TODO_update_ssa  /* todo_flags_finish TODO_rebuild_alias*/
  }
};
      
