
#include "ipa-check-dependence.h"

extern "C"
{
#include "stdlib.h"
#include "config.h"
#include "system.h"
#include "coretypes.h"
#include "timevar.h"
#include "flags.h"
#include "tree.h"

#include "cgraph.h"
#include "basic-block.h"
#include "cfgloop.h"
#include "function.h"
#include "tree-flow.h"
#include "tree-pass.h"
#include "tree-scalar-evolution.h"

#include "ipa-utils.h"
#include "langhooks.h"
}


htab_t callsite_hash;


struct callsite_t
{
  gimple call;
  exposed_ref_summary *references;
};


/* Compares database elements E1 and E2.  */

static int
eq_callsite (const void *e1, const void *e2)
{
  const struct callsite_t *elt1 = (const struct callsite_t *) e1;
  const struct callsite_t *elt2 = (const struct callsite_t *) e2;

  return elt1->call == elt2->call;
}

/* Computes a hash function for element ELT.  */

static hashval_t
hash_callsite (const void *elt)
{
  const struct callsite_t *const callsite = (const struct callsite_t *) elt;
  gimple call = callsite->call;
  return htab_hash_pointer (call);
}

/* Free the element E.  */

static void
hash_callsite_del (void *e)
{
  free (e);
}


int
list_size (const symbolic_range_list & list)
{
  return list.size ();
}

void
print_range_list (const symbolic_range_list & list)
{
  list.print (stderr);
}

void
print_range (symbolic_range_list::const_iterator & iter)
{
  iter->print (stderr);
}

void
print_range (symbolic_range_list::iterator & iter)
{
  iter->print (stderr);
}

void
print_range (const symbolic_range & range)
{
  range.print (stderr);
}

/* DEBUG Functions */
void
print_gimple (const gimple g)
{
  debug_gimple_stmt (g);
}

void
print_tree (tree g)
{
  debug_generic_stmt (g);
}


/* Generate the address taken node of a memory */
tree
tree_build_address_expr (tree mem)
{

  tree addr;
  tree offset;
  tree byte = build_int_cst (0, 8);
  mem = tree_strip_nops (mem);
  mem = unshare_expr (mem);
  bool addressable = false;

  switch (TREE_CODE (mem))
  {
    case VAR_DECL:
    case PARM_DECL:
    case RESULT_DECL:
    case FUNCTION_DECL:
      addressable = TREE_ADDRESSABLE (mem);
      addr = build_addr (mem, current_function_decl);
      TREE_ADDRESSABLE (mem) = addressable;
      break;

    case SSA_NAME:
      addr = tree_build_address_expr (SSA_NAME_VAR (mem));
      break;

    case COMPONENT_REF:
      addr = tree_build_address_expr (TREE_OPERAND (mem, 0));
      offset = byte_position (TREE_OPERAND (mem, 1));
      addr = build2_stat (PLUS_EXPR, size_type_node, addr, offset);
      break;

    case BIT_FIELD_REF:
      addr = tree_build_address_expr (TREE_OPERAND (mem, 0));
      offset = fold_build2 (TRUNC_DIV_EXPR, size_type_node, TREE_OPERAND (mem, 2), byte);
      addr = build2_stat (PLUS_EXPR, size_type_node, addr, offset);
      break;

    case MEM_REF:
      addr = build2_stat (PLUS_EXPR, size_type_node, (TREE_OPERAND (mem, 0)),  (TREE_OPERAND (mem, 1)));
      break;

    case ARRAY_REF:
      addr = tree_build_address_expr (TREE_OPERAND (mem, 0));
      offset = build2_stat (MULT_EXPR, size_type_node, (TREE_OPERAND (mem, 1)), array_ref_element_size (mem));
      addr = build2_stat (PLUS_EXPR, size_type_node, addr, offset);
      break;

		case VIEW_CONVERT_EXPR:
    case REALPART_EXPR:
    case IMAGPART_EXPR:
    case COMPLEX_EXPR:
    case MULT_EXPR:
      addr = tree_build_address_expr(TREE_OPERAND (mem, 0));
      break;


    default:
      fprintf (stderr, ("Unsupported Operator\n"));
      gcc_assert (false);
      break;


    }


  return addr;

}





/* Check whether ADDR contains only address of local variables */
static bool
address_contains_only_aritificial_var (tree addr)
{

  if (TREE_CODE (addr) == ADDR_EXPR)
    {
      tree opnd = TREE_OPERAND (addr, 0);
      gcc_assert (TREE_CODE (opnd) == VAR_DECL
		  || TREE_CODE (opnd) == PARM_DECL);
      if (TREE_CODE (opnd) == VAR_DECL)
	return DECL_ARTIFICIAL (opnd);
      else
	return false;
    }

  return false;

}



static bool
check_artificial_range (const symbolic_range & ranges)
{
  if (address_contains_only_aritificial_var (ranges.low ()) &&
      address_contains_only_aritificial_var (ranges.high ()))
    return true;
  return false;
}

extern tree get_base_pointer (tree ref);






/* Check whether ABOVE can contain or partially overlap with BELOW 
   If BELOW is fully contained by ABOVE, return true. 
   After the whole process, BELOW_RANGE either keep its original position or move to 
   the next available position.
 */
static bool
check_and_reduce_symbolic_range (const symbolic_range & above_range,
				 symbolic_range_list::iterator & below_range,
				 symbolic_range_list & below_range_list)
{

  if (above_range.is_read () && !below_range->is_read ())
    return false;

  /* check whether the two range can be aliased */
  if (!ipa_dr_may_alias_p (above_range.dr (), below_range->dr ()))
    return false;

  symbolic_range & b_range = *below_range;

  if (!above_range.is_must ())
    return false;

  gimple stmt_a = above_range.stmt ();
  gimple stmt_b = below_range->stmt ();


  bool must_aliased = false;

  tree ref_a = DR_ORIG_FORM (above_range.dr ());
  tree ref_b = DR_ORIG_FORM (below_range->dr ());
  if (TREE_CODE (ref_a) != TREE_CODE (ref_b))
    return false;

  
  if (TREE_CODE (ref_a) == SSA_NAME)
  {
    if (operand_equal_p (SSA_NAME_VAR(ref_a), SSA_NAME_VAR(ref_b), 0))
      if (dominated_by_p (CDI_DOMINATORS, below_range->bb (), above_range.bb ())
    //        && dominated_by_p (CDI_POST_DOMINATORS, above_range.bb(), below_range->bb()) 
        )
        if (!below_range->is_branch_copy ())
        {
          alias_slice_merge (above_range.dr (), below_range->dr ());
          /* remove and move to the next range */
          below_range = below_range_list.erase (below_range);
          return true;
        }
    return false;
  }

  tree addr_a = get_base_var (ref_a);
  tree addr_b = get_base_var (ref_b);

  must_aliased = operand_equal_p (addr_a, addr_b, 0);


  if (!must_aliased)
    return false;

  tree above_lower_range = above_range.low ();
  tree above_upper_range = above_range.high ();
  tree below_lower_range = below_range->low ();
  tree below_upper_range = below_range->high ();

  if (loop_containing_stmt (stmt_a) == loop_containing_stmt (stmt_b))
    if (dominated_by_p (CDI_DOMINATORS, below_range->bb (), above_range.bb ())
	//        && dominated_by_p (CDI_POST_DOMINATORS, above_range.bb(), below_range->bb()) 
      )
    {
    	{
    	  if (above_range == *below_range && !below_range->is_branch_copy ())
    	    {
    	      alias_slice_merge (above_range.dr (), below_range->dr ());
    	      /* remove and move to the next range */
    	      below_range = below_range_list.erase (below_range);
    	      return true;
    	    }
    	}
    }

  if (above_range.kind () == rk_all)
    {
      /* remove and move to the next range */
      below_range = below_range_list.erase (below_range);
      return true;
    }

  if (above_range.kind () == rk_unknown)
    return false;

  if (below_range->kind () == rk_unknown)
    return false;


  if (below_range->kind () == rk_all)
    {
      if (!below_range->is_read ())
	return false;
      /*
         ========
         =================
         Split and Reduce the symbolic range of BELOW : Replace the lower bound of BELOW by the
         upper bound of ABOVE with the minimu step of upper bound plus 1
       */
      below_range_list.split_symbolic_range (above_range, below_range);
      return false;
    }




  /*
     =========
     ==========
   */
  if (value_less_p (above_upper_range, below_lower_range))
    return false;


  /*
     ========== ........
     =========== .....
   */

  if (value_less_or_equal_p (above_lower_range, below_lower_range))
    {

      /*
         ==============
         ===========
       */
      if (value_less_or_equal_p (below_upper_range, above_upper_range))
	{
	  below_range = below_range_list.erase (below_range);
	  return true;
	}
      else if (value_less_p (above_upper_range, below_upper_range))
	{
	  /*
	     ==========
	     ===========
	     Reduce the symbolic range of BELOW : Replace the lower bound of BELOW by the
	     upper bound of ABOVE with the minimu step of upper bound plus 1
	   */
	  below_range->set_low (unshare_expr (above_upper_range));
	  below_range->inc_lower_bound ();
	  return false;
	}
    }

  /*
     ======== ........
     ============ .....
   */
  else if (value_less_p (below_lower_range, above_lower_range))
    {

      /*
         ==============
         ===========
       */
      if (value_less_or_equal_p (below_upper_range, above_upper_range))
	{
	  below_range->set_high (unshare_expr (above_lower_range));
	  below_range->dec_upper_bound ();
	  return false;
	}
      else if (value_less_p (above_upper_range, below_upper_range))
	{
	  /*
	     ========
	     =================
	     Split and Reduce the symbolic range of BELOW : Replace the lower bound of BELOW by the
	     upper bound of ABOVE with the minimu step of upper bound plus 1
	   */
	  below_range_list.split_symbolic_range (above_range, below_range);
	  return false;

	}
    }
  else
    {
      /* unknown */
//              below_range->set_kind(rk_unknown);
      return false;
    }

}


static bool
dr_is_aritificial_var (ipa_data_reference_p dr)
{

  tree ref = DR_REF (dr);

  if (TREE_CODE (ref) == VAR_DECL && DECL_ARTIFICIAL (DR_REF (dr)))
    return true;

  if (TREE_CODE (ref) == SSA_NAME)
    if (DECL_ARTIFICIAL (SSA_NAME_VAR (ref)))
      return true;

  return false;

}


/* Get or create */
static symbolic_range_list &
dr_get_symbolic_ranges (ipa_data_reference_p dr)
{

  gcc_assert (dr->ref == gimple_op (dr->stmt, dr->opnd_idx));

  if (!dr_allocated_ranges (dr))
    {
      if (!dr_is_aritificial_var (dr))
	{
	  tree type = TREE_TYPE (DR_REF (dr));
	  tree size = TYPE_SIZE (type);
	  tree ref = DR_ORIG_FORM (dr);

	  symbolic_range range;
	  range.set_must (true);
	  range.set_dr (dr);
	  range.set_kind (rk_sure);
	  range.set_low (tree_build_address_expr (ref));
	  range.set_high (unshare_expr (range.low ()));
	  range.set_bb (gimple_bb (DR_STMT (dr)));
	  range.set_step (size);
	  DR_UP_EXP_RANGE (dr).push_back (range);
	}

      dr_set_allocated_ranges (dr);

    }

  return DR_UP_EXP_RANGE (dr);
}




/* Replace formal parameters in symbolic range of summary with coresponding actuals 
   at callsite */
static void
translate_formal_to_actual (gimple call, tree * callee_ref)
{

  int i = 0;
  int n = TREE_OPERAND_LENGTH (*callee_ref);

  tree callee_decl = gimple_call_fndecl (call);

  if (TREE_CODE (*callee_ref) == PARM_DECL)
    {
      /* find corresponding actual parameters */
      tree parm;
      for (parm = DECL_ARGUMENTS (callee_decl); parm;
	   parm = DECL_CHAIN (parm))
	{
	  i++;
	  if (parm == *callee_ref)
	    break;
	}

      tree arg = gimple_call_arg (call, i - 1);
      *callee_ref = unshare_expr (arg);
      return;
    }

  else if (TREE_CODE (*callee_ref) == SSA_NAME)
    {
      /* find corresponding actual parameters */
      tree parm = SSA_NAME_VAR (*callee_ref);
      if (TREE_CODE (*callee_ref) != PARM_DECL)
	return;

      for (parm = DECL_ARGUMENTS (callee_decl); parm;
	   parm = DECL_CHAIN (parm))
	{
	  i++;
	  if (parm == *callee_ref)
	    break;
	}

      tree arg = gimple_call_arg (call, i - 1);
      *callee_ref = unshare_expr (arg);
      return;
    }

  else if (TREE_CODE (*callee_ref) == VAR_DECL)
    {
//              gcc_assert(is_global_var(*callee_ref));
      return;
    }

  n = TREE_OPERAND_LENGTH (*callee_ref);
  for (i = 0; i < n; i++)
    {
      if (TREE_OPERAND (*callee_ref, i))
	translate_formal_to_actual (call, &(TREE_OPERAND (*callee_ref, i)));
    }

}


static void
new_callsite_summary (struct callsite_t *callsite)
{

  gimple call = callsite->call;
  tree callee_decl = gimple_call_fndecl (call);

  callsite->references = new exposed_ref_summary_t;

  /* Adding actual parameters and return value */
  int i;
  int n = gimple_call_num_args (call);
  tree op0;

  for (i = 0; i < n; i++)
  {
    op0 = gimple_call_arg (call, i);
    ipa_data_reference_p dr = tree_data_reference (call, i + 3);
    if (dr)
    {
      symbolic_range_list & tmp1 = dr_get_symbolic_ranges (dr);
      callsite->references->up_read ().Union (tmp1);
    }
  }

  /* return value */
  op0 = gimple_call_lhs (call);
  if (op0)
  {
    ipa_data_reference_p dr = tree_data_reference (call, 0);
    if (dr)
    {
      symbolic_range_list & tmp2 = dr_get_symbolic_ranges (dr);
      callsite->references->up_write ().Union (tmp2);
    }
  }

  std::list < cgraph_node_ptr > callee_list;

  if (callee_decl != NULL_TREE)
  {
    cgraph_node_ptr callee = cgraph_get_node (callee_decl);
    callee_list.push_back (callee);
  }
  else
  {
    /* Handle indirect calls.  */
    cgraph_node_ptr node = cgraph_get_node (current_function_decl);    
    struct cgraph_edge *e = cgraph_edge(node, call);
    for (e = node->indirect_calls; e; e = e->next_callee)
    {
      struct nodeList * callees = e->indirect_info->callees;
      while (callees)
      {
        if ( valid_function_node_p(callees->node) )
        {
          callee_list.push_back (callees->node);
        }
        callees = callees->next;
      }
    }         
  }


  for (std::list < cgraph_node_ptr >::iterator citer = callee_list.begin ();
       citer != callee_list.end (); ++citer)
  {

    cgraph_node_ptr callee = *citer;

    /* for built-in functions, return summary containing formal parameters */
    bool is_build_in = false;
    if (cgraph_function_body_availability (callee) < AVAIL_OVERWRITABLE ||
        DECL_BUILT_IN_CLASS (callee_decl) == BUILT_IN_NORMAL)
      is_build_in = true;

    if (is_build_in)
      continue;

    if ( !valid_function_node_p (callee) )
      continue;

    exposed_ref_summary *callee_summary = func_exposed_refs (callee);

    if (!callee_summary)
      continue;

    callsite->references->copy (*callee_summary);

    /* Translate formal parameters in summary to corresponding actual parameters at callsite */

    for (symbolic_range_list::iterator iter =
         callsite->references->up_read ().begin ();
         iter != callsite->references->up_read ().end (); ++iter)
    {
      iter->set_bb (gimple_bb (call));
      iter->reset_up_expose ();
      iter->reset_down_expose ();
      if (iter->kind () == rk_unknown)
        continue;
      translate_formal_to_actual (call, iter->low_ptr ());
      translate_formal_to_actual (call, iter->high_ptr ());
    }

    for (symbolic_range_list::iterator iter =
         callsite->references->up_write ().begin ();
         iter != callsite->references->up_write ().end (); ++iter)
    {
      iter->set_bb (gimple_bb (call));
      iter->reset_up_expose ();
      iter->reset_down_expose ();
      if (iter->kind () == rk_unknown)
        continue;
      translate_formal_to_actual (call, iter->low_ptr ());
      translate_formal_to_actual (call, iter->high_ptr ());
    }


  }
}



static symbolic_range_list &
get_callsite_symbolic_ranges (gimple call, bool read)
{
  struct callsite_t tmp;
  tmp.call = call;
  tmp.references = NULL;

  struct callsite_t *callsite;
  struct callsite_t **slot =
    (struct callsite_t **) htab_find_slot (callsite_hash, &tmp, INSERT);

  if (*slot)
    callsite = *slot;
  else
  {
    /* create new callsite */
    callsite = XNEW (struct callsite_t);
    callsite->call = call;
    new_callsite_summary (callsite);
    *slot = callsite;
  }


  if (read)
    return callsite->references->up_read ();
  else
    return callsite->references->up_write ();

}


/* Get copy of up or down read ranges */
static void
get_read_symbolic_ranges (gimple stmt, symbolic_range_list & ranges)
{

  tree operand;
  ipa_data_reference_p dr;

  switch (gimple_code (stmt))
    {
    case GIMPLE_COND:
      operand = gimple_cond_lhs (stmt);
      dr = tree_data_reference (stmt, 0);
      if (dr)
	{
	  symbolic_range_list & tmp = dr_get_symbolic_ranges (dr);
	  ranges.Union (tmp);
	}

      operand = gimple_cond_rhs (stmt);
      dr = tree_data_reference (stmt, 1);
      if (dr)
	{
	  symbolic_range_list & tmp = dr_get_symbolic_ranges (dr);
	  ranges.Union (tmp);
	}
      break;

    case GIMPLE_ASSIGN:

      operand = gimple_assign_rhs1 (stmt);
      dr = tree_data_reference (stmt, 1);
      if (dr)
	{
	  symbolic_range_list & tmp = dr_get_symbolic_ranges (dr);
	  ranges.Union (tmp);
	}

      operand = gimple_assign_rhs2 (stmt);
      if (operand && tree_data_reference (stmt, 2))
	{
	  dr = tree_data_reference (stmt, 2);
	  symbolic_range_list & tmp = dr_get_symbolic_ranges (dr);
	  ranges.Union (tmp);
	}

      break;

    case GIMPLE_CALL:
      {
	symbolic_range_list & tmp = get_callsite_symbolic_ranges (stmt, true);
	ranges.Union (tmp);
	break;
      }

    default:
      //              fprintf(stderr, "Unknown operator\n");
      //              gcc_assert(false);
      break;
    }
}



/* Get a copy of up or down write ranges */
static void
get_write_symbolic_ranges (gimple stmt, symbolic_range_list & ranges)
{

  tree operand;
  ipa_data_reference_p dr;

  switch (gimple_code (stmt))
  {
    case GIMPLE_COND:
      break;

    case GIMPLE_ASSIGN:

      operand = gimple_assign_lhs (stmt);
      dr = tree_data_reference (stmt, 0);
      if (dr)
    	{
    	  symbolic_range_list & tmp = dr_get_symbolic_ranges (dr);
    	  ranges.Union (tmp);
    	}

      break;

    case GIMPLE_CALL:
    {
    	symbolic_range_list & tmp = get_callsite_symbolic_ranges (stmt, false);
    	ranges.Union (tmp);
    	break;
    }

    case GIMPLE_RETURN:
      break;

    default:
      //              fprintf(stderr, "Unknown operator\n");
      //              gcc_assert(false);
      break;
    }
}



/* For a[i], return a[1] .. a[N-1] */
static symbolic_range *
reference_domain (tree ref)
{

  symbolic_range *range = new symbolic_range;
  range->set_kind (rk_unknown);

  if (TREE_CODE (ref) == ARRAY_REF)
    {
      tree array_name = TREE_OPERAND (ref, 0);
      tree array_type = TREE_TYPE (array_name);
      tree domain = TYPE_DOMAIN (array_type);
      if (!domain)
	return range;
      tree max = TYPE_MAX_VALUE (domain);
      tree min = TYPE_MIN_VALUE (domain);

      tree low = unshare_expr (ref);
      tree high = unshare_expr (ref);

      TREE_OPERAND (low, 1) = min;
      TREE_OPERAND (high, 1) = max;

      low = tree_build_address_expr (low);
      high = tree_build_address_expr (high);

      range->set_low (low);
      range->set_high (high);
      range->set_kind (rk_sure);

    }

  return range;
}


/* search lhs in expr */
bool
search_in_ref (tree expr, tree lhs)
{

  if (operand_equal_p (expr, lhs, 0))
    return true;

  if (TREE_CODE (expr) == ADDR_EXPR)
    return false;

  bool found = false;
  int i;
  int n = TREE_OPERAND_LENGTH (expr);
  for (i = 0; i < n; i++)
    {
      tree opnd = TREE_OPERAND (expr, i);
      if (!opnd)
	continue;
      if (operand_equal_p (opnd, lhs, 0))
	{
	  return true;
	}
      else if (search_in_ref (opnd, lhs))
	found = true;
    }

  return found;

}

static void
search_and_substitute (tree expr, tree lhs, tree rhs)
{

  if (TREE_CODE (expr) == ADDR_EXPR)
    return;

  int i;
  int n = TREE_OPERAND_LENGTH (expr);
  for (i = 0; i < n; i++)
    {
      tree opnd = TREE_OPERAND (expr, i);
      if (!opnd)
	continue;

      if (operand_equal_p (opnd, lhs, 0))
	{
	  TREE_OPERAND (expr, i) = unshare_expr (rhs);
	}
      // TODO: consider alias here
      else
	search_and_substitute (opnd, lhs, rhs);
    }

}


static void
search_and_substitute (tree * expr, tree lhs, tree rhs)
{
  if (operand_equal_p (*expr, lhs, 0))
    *expr = unshare_expr (rhs);

  else
    search_and_substitute (*expr, lhs, rhs);

}



static void
substitute_ref_by_callsite (gimple call, symbolic_range & range)
{

  /* currently not implemented */

  symbolic_range_list ranges;
  get_write_symbolic_ranges (call, ranges);


  if (range.kind () == rk_sure)
    {
      /* search lhs in bounds */
      for (symbolic_range_list::iterator iter = ranges.begin ();
	   iter != ranges.end (); ++iter)
	{
	  if (search_in_ref (range.low (), DR_REF (iter->dr ())))
	    {
	      range.set_kind (rk_unknown);
	      return;
	    }
	}

      for (symbolic_range_list::iterator iter = ranges.begin ();
	   iter != ranges.end (); ++iter)
	{
	  if (search_in_ref (range.high (), DR_REF (iter->dr ())))
	    {
	      range.set_kind (rk_unknown);
	      return;
	    }
	}

    }

}


static void
substitute_ref_by_stmt (gimple pred, symbolic_range & range)
{

  if (range.kind () == rk_unknown)
    return;

  if (range.kind () == rk_all)
    return;

  if (is_gimple_call (pred))
    {
      substitute_ref_by_callsite (pred, range);
      return;
    }

  if (!is_gimple_assign (pred))
    return;

  tree lhs = gimple_assign_lhs (pred);
  tree rhs = gimple_assign_rhs_to_tree (pred);

  if (range.kind () == rk_sure)
    {
      /* search lhs in bounds */
      search_and_substitute (range.low (), lhs, rhs);
      search_and_substitute (range.high (), lhs, rhs);
    }

  symbolic_range *domain = reference_domain (DR_ORIG_FORM (range.dr ()));

  if (range == *domain)
    range.set_kind (rk_all);

  delete domain;

}


/* copy on write. Substitute variables in dr with the final value of loop */
static void
substitute_ref_by_loop_defs (loop_p loop, loop_p father,
			     symbolic_range & range)
{

  if (range.kind () == rk_unknown)
    return;

  exposed_ref_summary *exposed_refs = loop_exposed_refs (loop);

  for (symbolic_range_list::iterator iter =
       exposed_refs->up_write ().begin ();
       iter != exposed_refs->up_write ().end (); ++iter)
    {
      ipa_data_reference_p dr = iter->dr ();
      tree lhs = DR_REF (dr);

      if (DR_PROC (dr) != loop_func_decl (loop))
	{
	  if (range.kind () == rk_sure)
	    {
	      if (search_in_ref (range.low (), lhs))
		{
		  range.set_kind (rk_unknown);
		  return;
		}
	      if (search_in_ref (range.high (), lhs))
		{
		  range.set_kind (rk_unknown);
		  return;
		}
	    }

	  continue;
	}

      tree ev = analyze_scalar_evolution (loop, lhs);
      if (TREE_CODE (ev) == POLYNOMIAL_CHREC)
	{
	  ev = analyze_scalar_evolution (father, lhs);
	  search_and_substitute (range.low (), lhs, ev);
	  search_and_substitute (range.high (), lhs, ev);
	}
      else
	{
	  if (range.kind () == rk_sure)
	    {
	      if (search_in_ref (range.low (), lhs))
		{
		  range.set_kind (rk_unknown);
		  return;
		}
	      if (search_in_ref (range.high (), lhs))
		{
		  range.set_kind (rk_unknown);
		  return;
		}
	    }
	}
    }

}

static tree
get_final_value_of_induction_var (loop_p loop, tree iv, tree step)
{
  /* Infer final value */
  iv = tree_strip_nops (iv);

  int val = int_cst_value (step);

  if (val != 1 && val != -1)
    return NULL_TREE;

  gimple exit_cond = get_loop_exit_condition (loop);
  if (!exit_cond || gimple_code (exit_cond) != GIMPLE_COND)
    return NULL_TREE;

  /* We want the condition for staying inside loop.  */
  edge exit_edge = single_exit (loop);
  enum tree_code code = gimple_cond_code (exit_cond);
  if (exit_edge->flags & EDGE_TRUE_VALUE)
    code = invert_tree_comparison (code, false);

  switch (code)
    {
    case GT_EXPR:
    case GE_EXPR:
    case NE_EXPR:
    case LT_EXPR:
    case LE_EXPR:
      break;

    default:
      return NULL_TREE;
    }

  tree op0 = gimple_cond_lhs (exit_cond);
  tree op1 = gimple_cond_rhs (exit_cond);
  tree final;

  if (operand_equal_p (op0, iv, 0))
    final = op1;
  else if (operand_equal_p (op1, iv, 0))
    final = op0;
  else
    return NULL_TREE;

  switch (code)
    {
    case GE_EXPR:
    case LE_EXPR:
      break;

    case LT_EXPR:
    case GT_EXPR:
    case NE_EXPR:
      {
	/* POLYNOMIAL_CHREC is + */
	//              if ( POINTER_TYPE_P (TREE_TYPE(iv)) )
	//                      final = build2_stat (MINUS_EXPR, TREE_TYPE(iv), final, step); 

	final = fold_build2 (MINUS_EXPR, TREE_TYPE (iv), final, step);
	break;
      }
    default:
      return NULL_TREE;
    }


  gimple_stmt_iterator bsi;
  for (bsi = gsi_last_bb (loop->header); !gsi_end_p (bsi); gsi_prev (&bsi))
    {
      gimple stmt = gsi_stmt (bsi);
      if (!is_gimple_assign (stmt))
	continue;

      tree lhs = gimple_assign_lhs (stmt);
      tree rhs = gimple_assign_rhs_to_tree (stmt);

      search_and_substitute (&final, lhs, rhs);
    }

  return final;

}



/* substitute loop induction variables in an expression. Return true if all variables
   of EXPR are loop indunction variables */
static bool
substitute_loop_induction_var (loop_p loop, tree * expr, gimple stmt,
			       bool low)
{

  bool result = true;

  if (TREE_CODE (*expr) == ADDR_EXPR)
    return result;

  if (is_gimple_scalar (*expr))
    {
      /* Check if expr is loop-invariant */
      if (expr_invariant_in_loop_p (loop, *expr))
	return true;

      loop_p expr_loop = loop_containing_stmt (stmt);
      if (current_function_decl == loop_func_decl (expr_loop))
	if (!ipa_ref_contains_symbols_defined_in_loop (loop, *expr, stmt))
	  return true;

      tree ev = analyze_scalar_evolution (loop, *expr);
      ev = tree_strip_nops (ev);
      if (TREE_CODE (ev) == POLYNOMIAL_CHREC)
	{
	  tree iv = *expr;
	  tree base = CHREC_LEFT (ev);
	  tree step = CHREC_RIGHT (ev);
	  tree final;

	  if (TREE_CODE (step) != INTEGER_CST)
	    return false;

	  int val = int_cst_value (step);

	  if (val < 0)
	    low = -low;

	  if (low)
	    {
	      *expr = unshare_expr (CHREC_LEFT (ev));
	    }
	  else
	    {
	      final = get_final_value_of_induction_var (loop, iv, step);
	      if (final == NULL_TREE)
		return false;
	      *expr = final;
	    }
	}

      /* recoginize 
         last++; block[last] = (UChar)ch; break; */

      else if (ev == chrec_dont_know)
	result = false;

      return result;
    }



  int i;
  int n = TREE_OPERAND_LENGTH (*expr);
  for (i = 0; i < n; i++)
    {
      if (!TREE_OPERAND (*expr, i))
	continue;

      tree *opnd = &TREE_OPERAND (*expr, i);
      bool min = low;

      /* check coefficient */
      if (TREE_CODE (*expr) == MINUS_EXPR && i == 1)
	min = !low;

      if (!substitute_loop_induction_var (loop, opnd, stmt, min))
	result = false;
    }

  return result;

}

static void
initialize_expose_state (exposed_ref_summary & summary)
{

  for (symbolic_range_list::iterator iter = summary.up_read ().begin ();
       iter != summary.up_read ().end (); ++iter)
    {
      iter->reset_up_expose ();
      iter->reset_down_expose ();
    }

  for (symbolic_range_list::iterator iter = summary.up_write ().begin ();
       iter != summary.up_write ().end (); ++iter)
    {
      iter->reset_up_expose ();
      iter->reset_down_expose ();
    }


}



static void
evaluate_ref_range_for_loop (loop_p loop, symbolic_range & range)
{

  if (range.kind () == rk_unknown)
    return;

  if (range.kind () == rk_all)
    return;

  /* range->kind == rk_sure */
  if (substitute_loop_induction_var
      (loop, range.low_ptr (), range.stmt (), true)
      && substitute_loop_induction_var (loop, range.high_ptr (),
					range.stmt (), false))
    {
      symbolic_range *domain = reference_domain (DR_ORIG_FORM (range.dr ()));

      if (range == *domain)
	range.set_kind (rk_all);

      delete domain;
    }
  else
    {
      range.set_kind (rk_unknown);
    }

}


static void
evaluate_expose_ref_range_for_loop (loop_p loop)
{

  exposed_ref_summary *exposed_refs = loop_exposed_refs (loop);
  int i;

  for (symbolic_range_list::iterator iter = exposed_refs->up_read ().begin ();
       iter != exposed_refs->up_read ().end (); ++iter)
    evaluate_ref_range_for_loop (loop, *iter);

  for (symbolic_range_list::iterator iter =
       exposed_refs->up_write ().begin ();
       iter != exposed_refs->up_write ().end (); ++iter)
    evaluate_ref_range_for_loop (loop, *iter);


  /* check killness */
  for (symbolic_range_list::iterator iter1 =
       exposed_refs->up_write ().begin ();
       iter1 != exposed_refs->up_write ().end (); ++iter1)
  {
    basic_block bb1 = iter1->bb ();

    symbolic_range_list::iterator iter2 = exposed_refs->up_read ().begin ();
    while (iter2 != exposed_refs->up_read ().end ())
    {
      symbolic_range_list::iterator cur = iter2;
      ++iter2;

      basic_block bb2 = cur->bb ();
      if (!dominated_by_p (CDI_DOMINATORS, bb2, bb1))
        continue;

      ipa_data_reference *dr = cur->dr ();
      bool must = check_and_reduce_symbolic_range (*iter1, cur, exposed_refs->up_read ());
      if (ipa_dr_may_alias_p (dr, iter1->dr ()))
      {
        if ( loop->num != 0 )
          enter_loop_independent_dependency (loop_uid (loop), dr, iter1->dr (), must);
        else
        {
          cgraph_node_ptr node = cgraph_get_node (loop_func_decl(loop));
          std::set<loop_p> &inner_loops = func_extra_info (node)->inner_loops;
          if ( !inner_loops.empty() )
          {
            for (std::set<loop_p>::iterator liter = inner_loops.begin(); 
                  liter != inner_loops.end(); ++liter)
              enter_loop_independent_dependency (loop_uid (*liter), dr, iter1->dr (), must);              
          }
            
        }
      }
     }
  }

  for (symbolic_range_list::iterator iter = exposed_refs->up_read ().begin ();
       iter != exposed_refs->up_read ().end (); ++iter)
    iter->set_bb (loop->header);


  for (symbolic_range_list::iterator iter =
       exposed_refs->up_write ().begin ();
       iter != exposed_refs->up_write ().end (); ++iter)
    iter->set_bb (loop->header);


}



/* Check whether ADDR contains only address of local variables */
static bool
address_contains_only_local_var (tree addr)
{

  if (TREE_CODE (addr) == ADDR_EXPR)
    {
      tree opnd = TREE_OPERAND (addr, 0);
      gcc_assert (TREE_CODE (opnd) == VAR_DECL
		  || TREE_CODE (opnd) == PARM_DECL);
      return !is_global_var (opnd);
    }

  return false;

}

static void
remove_local_range (symbolic_range_list & ranges)
{
  /* check whether range contains only the address of local variables if 
     the current scope is function scope */
  symbolic_range_list::iterator iter = ranges.begin ();
  while (iter != ranges.end ())
    {
      if (iter->kind () == rk_unknown)
	{
	  ++iter;
	  continue;
	}
      if (address_contains_only_local_var (iter->low ()) &&
	  address_contains_only_local_var (iter->high ()))
	{
	  iter = ranges.erase (iter);
	}
      else
	++iter;
    }
}

static void 
expose_union(ipa_data_reference *above, ipa_data_reference *below, loop_p loop, FILE *fp)
{
  fprintf (fp, "MEMOP %d equals to MEMOP %d in LOOP %d\n", below->uid(), above->uid(), loop_uid (loop) );
  fprintf (fp, "      FORM MEMOP %d :", below->uid() ); 
  print_generic_expr (fp, DR_ORIG_FORM(below), 0);
  fprintf (fp, " \n"); 
  fprintf (fp, "      FORM MEMOP %d :", above->uid() ); 
  print_generic_expr (fp, DR_ORIG_FORM(above), 0);
  fprintf (fp, " \n");  
}


/* return whether PRED can kill DR fully and dr can kill pred 
	After the whole process, RANGE either keep its original position or move to 
	the next available position.
*/
static bool
check_fully_killed_by_stmt (loop_p loop, gimple pred, gimple ref_stmt,
                          			    symbolic_range_list::iterator & range,
                          			    symbolic_range_list & range_list, FILE *fp)
{

  if (pred == ref_stmt && is_gimple_call (pred))
    return false;

  /* Step 1. check whether pred can kill DR upwards fully */
  bool dr_be_killed = false;

  symbolic_range tmp (*range);
  symbolic_range_list::iterator prev = range;
  symbolic_range_list::iterator cur = range;
  prev--;


  symbolic_range_list up_ranges;
  get_write_symbolic_ranges (pred, up_ranges);

  if (pred != ref_stmt)
  {
    for (symbolic_range_list::iterator iter = up_ranges.begin ();
         iter != up_ranges.end (); ++iter)
    {
      ipa_data_reference *dr = range->dr ();
      bool must = check_and_reduce_symbolic_range (*iter, range, range_list);

      if (ipa_dr_may_alias_p (dr, iter->dr ()))
      {
        if ( loop->num != 0 )
        {
          enter_loop_independent_dependency (loop_uid (loop), dr, iter->dr (), must);          
          if (fp && must)
            expose_union (iter->dr(), dr, loop, fp );
        }
        else
        {
          cgraph_node_ptr node = cgraph_get_node (loop_func_decl(loop));
          std::set<loop_p> &inner_loops = func_extra_info (node)->inner_loops;
          if ( !inner_loops.empty() )
          {
            for (std::set<loop_p>::iterator liter = inner_loops.begin(); 
                  liter != inner_loops.end(); ++liter)
            {
              enter_loop_independent_dependency (loop_uid (*liter), dr, iter->dr (), must);                         
              if (fp && must)
                expose_union (iter->dr(), dr, *liter, fp );
            }
          }            
        }
      }

      if (must)
      {
        dr_be_killed = true;
        break;
      }

    }
  }

  up_ranges.clear ();
  get_read_symbolic_ranges (pred, up_ranges);
  if (pred != ref_stmt)
  {
    prev++;
    if (cur == range)
    {
      for (symbolic_range_list::iterator iter = up_ranges.begin ();
           iter != up_ranges.end (); ++iter)
      {
        ipa_data_reference *dr = cur->dr ();
        bool must = check_and_reduce_symbolic_range (*iter, cur, range_list);

        if (ipa_dr_may_alias_p (dr, iter->dr ()))
        { 
          if ( loop->num != 0 )
          {
            enter_loop_independent_dependency (loop_uid (loop), dr, iter->dr (), must);          
            if (fp && must)
              expose_union (iter->dr(), dr, loop, fp );
          }
          else
          {
            cgraph_node_ptr node = cgraph_get_node (loop_func_decl(loop));
            std::set<loop_p> &inner_loops = func_extra_info (node)->inner_loops;
            if ( !inner_loops.empty() )
            {
              for (std::set<loop_p>::iterator liter = inner_loops.begin(); 
                    liter != inner_loops.end(); ++liter)
              {
                enter_loop_independent_dependency (loop_uid (*liter), dr, iter->dr (), must);                         
                if (fp && must)
                  expose_union (iter->dr(), dr, *liter, fp );
              }
            }
              
          }

        }

        if (must)
      	{
      	  dr_be_killed = true;
      	  break;
      	}
      }
    }
    else
      while (prev != range)
      {
        symbolic_range_list::iterator cur = prev;
        prev++;
        for (symbolic_range_list::iterator iter = up_ranges.begin (); iter != up_ranges.end (); ++iter)
        {
        	ipa_data_reference *dr = cur->dr ();
        	bool must = check_and_reduce_symbolic_range (*iter, cur, range_list);
        	if (ipa_dr_may_alias_p (dr, iter->dr ()))
          {
            if ( loop->num != 0 )
            {
              enter_loop_independent_dependency (loop_uid (loop), dr, iter->dr (), must);          
              if (fp && must)
                expose_union (iter->dr(), dr, loop, fp );
            }
            else
            {
              cgraph_node_ptr node = cgraph_get_node (loop_func_decl(loop));
              std::set<loop_p> &inner_loops = func_extra_info (node)->inner_loops;
              if ( !inner_loops.empty() )
              {
                for (std::set<loop_p>::iterator liter = inner_loops.begin(); 
                      liter != inner_loops.end(); ++liter)
                {
                  enter_loop_independent_dependency (loop_uid (*liter), dr, iter->dr (), must);                         
                  if (fp && must)
                    expose_union (iter->dr(), dr, *liter, fp );
                }
              }
                
            }
          }
 
        	if (must)
      	  {
      	    dr_be_killed = true;
      	    break;
      	  }
        }
      }
  }



  /* Step 2. check whether summary can be killed by DR downwards */
//      check_down_killed (loop, pred, tmp, ref_stmt);  

  return dr_be_killed;
}

/*
	After the whole process, RANGE either keep its original position or move to 
	the next available position.
*/
static bool
check_fully_killed_by_summary (loop_p loop, exposed_ref_summary * summary,
                            			       symbolic_range_list::iterator & range,
                            			       symbolic_range_list & range_list, FILE *fp)
{

  /* Step 1. check whether summary can kill DR upwards fully */
  bool dr_be_killed = false;
  int i;
  symbolic_range tmp (*range);

  for (symbolic_range_list::iterator iter = summary->up_write ().begin ();
       iter != summary->up_write ().end (); ++iter)
  {

    ipa_data_reference *dr = range->dr ();
    bool must = check_and_reduce_symbolic_range (*iter, range, range_list);

    if (ipa_dr_may_alias_p (dr, iter->dr ()))
    {
      if ( loop->num != 0 )
      {
        enter_loop_independent_dependency (loop_uid (loop), dr, iter->dr (), must);          
        if (fp && must)
          expose_union (iter->dr(), dr, loop, fp );
      }
      else
      {
        cgraph_node_ptr node = cgraph_get_node (loop_func_decl(loop));
        std::set<loop_p> &inner_loops = func_extra_info (node)->inner_loops;
        if ( !inner_loops.empty() )
        {
          for (std::set<loop_p>::iterator liter = inner_loops.begin(); 
                liter != inner_loops.end(); ++liter)
          {
            enter_loop_independent_dependency (loop_uid (*liter), dr, iter->dr (), must);                         
            if (fp && must)
              expose_union (iter->dr(), dr, *liter, fp );
          }
        }
          
      }
    }
    
    if (must)
    {
      dr_be_killed = true;
      break;
    }

  }


  return dr_be_killed;

}



/* Check whether these exists a definition of REF on path from A to B. Return true if these is.
	 A dominates B. (A, B]
*/
static bool
check_def_from_A_to_B (basic_block a, basic_block b, tree ref)
{

  if (a == b)
    return false;

  gcc_assert (dominated_by_p (CDI_DOMINATORS, b, a));

  switch (TREE_CODE (ref))
    {

    case VAR_DECL:
    case PARM_DECL:
    case SSA_NAME:
    case INDIRECT_REF:
    case MEM_REF:
    case ARRAY_REF:
    case COMPONENT_REF:
    case BIT_FIELD_REF:
      {
	basic_block cur = b;

	gimple_stmt_iterator gsi;
	gimple stmt;
	for (gsi = gsi_last_bb (cur); !gsi_end_p (gsi); gsi_prev (&gsi))
	  {
	    stmt = gsi_stmt (gsi);
	    if (is_gimple_assign (stmt))
	      {
		tree lhs = gimple_assign_lhs (stmt);
		if (ipa_ref_may_alias_p (lhs, ref))
		  return true;
	      }
	  }


	edge e;
	edge_iterator ei;
	FOR_EACH_EDGE (e, ei, b->preds)
	{
	  if (e->flags & EDGE_DFS_BACK)
	    continue;

	  basic_block pred = e->src;
	  if (check_def_from_A_to_B (a, pred, ref))
	    return true;
	}

      }

      break;

    default:
      return false;
    }


  int i;
  int n = TREE_OPERAND_LENGTH (ref);
  for (i = 0; i < n; i++)
    {
      tree opnd = TREE_OPERAND (ref, i);
      if (!opnd)
	continue;
      if (check_def_from_A_to_B (a, b, opnd))
	return true;
    }

  return false;
}




typedef enum check_from_t
{
  cf_first,			/* check from out of the bb */
  cf_last,			/* check all statements in bb */
  cf_here			/* check from the current statement to first statement */
} check_from;


/* Check whether BB's in_ref can be fully killed under LOOP. If a range can be fully killed,
	 remove the range from in_ref. 
	 Check from above STMT of BB. If STMT is NULL, check from the last stmt of bb */

static void
check_and_substitute_expose_ref_in_bb (loop_p loop, basic_block bb,
                                      				       check_from from, gimple stmt, FILE *fp)
{

  struct bb_extra_info_t *info;
  gimple_stmt_iterator bsi;
  info = bb_extra_info (bb);

  if (from != cf_first)
  {

    if (from == cf_here)
    {
      gcc_assert (stmt);
      bool find_pred = false;
      for (bsi = gsi_last_bb (bb); !gsi_end_p (bsi); gsi_prev (&bsi))
      {
        gimple pred = gsi_stmt (bsi);
        if (pred == stmt)
      	find_pred = true;

        if (find_pred)
      	{
      	  /* check from itself */
      	  symbolic_range_list::iterator iter = info->in_ref.begin ();
      	  while (iter != info->in_ref.end ())
    	    {
    	      symbolic_range_list::iterator cur = iter;
    	      symbolic_range_list::iterator old = iter;
    	      ++iter;

    	      /* After the chekcing, Iter either keep its original position or move to 
    	         the next available position. */
    	      if (!check_fully_killed_by_stmt(loop, pred, stmt, cur, info->in_ref, fp))
        		{
        		  if (pred != stmt)
      		    {
      		      for (symbolic_range_list::iterator iter2 = old; iter2 != iter; ++iter2)
            			substitute_ref_by_stmt (pred, *iter2);
      		    }
        		}
    	    }

      	}
      }
    }

    /* if bb is nested in a loop that is a direct son of this loop */
    else if (bb->loop_father != loop
       && flow_loop_nested_p (loop, bb->loop_father)
       && bb->loop_depth == loop_depth (loop) + 1)
    {
      loop_p inner_loop = bb->loop_father;
      struct loop_extra_info_t *loop_info = loop_extra_info (inner_loop);

      symbolic_range_list::iterator iter = info->in_ref.begin ();
      while (iter != info->in_ref.end ())
      {
        symbolic_range_list::iterator cur = iter;
        symbolic_range_list::iterator old = iter;
        ++iter;

        /* After the chekcing, Iter either keep its original position or move to 
           the next available position. */
        if (!check_fully_killed_by_summary (loop, loop_info->loop_site_exposed_refs, cur, info->in_ref, fp))
      	{
      	  for (symbolic_range_list::iterator iter2 = old;
      	       iter2 != iter; ++iter2)
      	    substitute_ref_by_loop_defs (inner_loop, loop, *iter2);
      	}
      }

      /* jump to loop header, and start checking from loop header */
      bb = inner_loop->header;
    }

    else			/*    from == cf_last */
    {
      for (bsi = gsi_last_bb (bb); !gsi_end_p (bsi); gsi_prev (&bsi))
      {
        gimple pred = gsi_stmt (bsi);
        symbolic_range_list::iterator iter = info->in_ref.begin ();
        while (iter != info->in_ref.end ())
      	{
      	  symbolic_range_list::iterator cur = iter;
      	  symbolic_range_list::iterator old = iter;
      	  ++iter;

      	  /* After the chekcing, Iter either keep its original position or move to 
      	     the next available position. */
      	  if (!check_fully_killed_by_stmt (loop, pred, stmt, cur, info->in_ref, fp))
    	    {
    	      for (symbolic_range_list::iterator iter2 = old; iter2 != iter; ++iter2)
          		substitute_ref_by_stmt (pred, *iter2);
    	    }
      	}
      }
    }

  }



  /* check whether range contains only the address of local variables if 
     the current scope is function scope */
  if (loop == current_loops->tree_root)
    {
      remove_local_range (info->in_ref);
    }

  basic_block idom = get_immediate_dominator (CDI_DOMINATORS, bb);

  /* propagate to predecessor's input */
  /* propagate to idom's input if possible */
  int pred_num = EDGE_COUNT (bb->preds);
  edge e;
  edge_iterator ei;
  FOR_EACH_EDGE (e, ei, bb->preds)
  {
    if (e->flags & EDGE_DFS_BACK)
      continue;

    basic_block pred = e->src;

    /* check data-flow from pred to idom */
    bool has_def = false;
    for (symbolic_range_list::iterator iter = info->in_ref.begin ();
	 iter != info->in_ref.end (); ++iter)
      {
	if (iter->kind () == rk_unknown)
	  continue;
	if (check_def_from_A_to_B (idom, pred, iter->low ()))
	  {
	    has_def = true;
	    break;
	  }
	if (check_def_from_A_to_B (idom, pred, iter->high ()))
	  {
	    has_def = true;
	    break;
	  }
      }

    if (!has_def)
      bb_extra_info (idom)->in_ref.search_and_union (info->in_ref);
    else
      {
	symbolic_range_list tmp_list (info->in_ref);
	for (symbolic_range_list::iterator iter = tmp_list.begin ();
	     iter != tmp_list.end (); ++iter)
	  {
	    if (pred_num > 1)
	      iter->set_branch_copy ();

	    if (iter->kind () == rk_unknown)
	      continue;
	    /* substitute by phi nodes */
	    for (bsi = gsi_start_phis (bb); !gsi_end_p (bsi); gsi_next (&bsi))
	      {
		gimple phi = gsi_stmt (bsi);
		tree result = gimple_phi_result (phi);
		tree opnd = gimple_phi_arg_def (phi, ei.index);
		search_and_substitute (iter->low (), result, opnd);
		search_and_substitute (iter->high (), result, opnd);
	      }
	  }
	bb_extra_info (pred)->in_ref.search_and_union (tmp_list);
      }

  }
}


/* Check whether RANGE is an up-exposed reference inside LOOP from REF_STMT in REF_BB.
   If REF_STMT is NULL, check from the last stmt of REF_BB. */

static void
check_a_expose_ref_in_loop (loop_p loop, const symbolic_range & range,
                              			    basic_block ref_bb, check_from from,
                              			    gimple ref_stmt, FILE *fp)
{

  /* Check the up-expose ref of function */

  int i;
  basic_block bb;
  bool start = false;
  exposed_ref_summary *exposed_refs;

  if (loop == current_loops->tree_root)
  {
    cgraph_node_ptr current_node = cgraph_get_node (current_function_decl);
    exposed_refs = func_extra_info (current_node)->exposed_refs;
  }
  else
    exposed_refs = loop_extra_info (loop)->exposed_refs;



  /* get loop body in post order */
  basic_block *loop_blocks;
  loop_blocks = get_loop_body_in_custom_order (loop, bb_post_order_comparator);


  for (i = 0; i < loop->num_nodes; i++)
  {
    bb = loop_blocks[i];

    if (bb == ref_bb)
    {
      start = true;
      bb_extra_info (bb)->in_ref.push_back (range);
    }

    /* start checking from REF_BB */
    if (start)
    {

      check_and_substitute_expose_ref_in_bb (loop, bb, from, ref_stmt, fp);

      from = cf_last;

      struct bb_extra_info_t *bb_info = bb_extra_info (bb);
      if (bb == loop->header)
      {
        /* set must */
        bool must = dominated_by_p (CDI_DOMINATORS, loop->latch, range.bb ());
        must = range.is_must () && must;
        for (symbolic_range_list::iterator riter = bb_info->in_ref.begin ();
               riter != bb_info->in_ref.end (); ++riter)
        	riter->set_must (must);

        /* add range to loop's up-exposed refs */
        if (range.is_read ())
        	exposed_refs->up_read ().search_and_union (bb_info->in_ref);
        else
        	exposed_refs->up_write ().search_and_union (bb_info->in_ref);
      }
    }

  }

  FOR_ALL_BB (bb) 
    bb_extra_info (bb)->in_ref.clear ();

}




static void
print_expose_refs_of_loop (FILE * fp, loop_p loop)
{

//      if (loop != current_loops->tree_root)
//              return;

  exposed_ref_summary *exposed_refs;
  cgraph_node_ptr current_node = cgraph_get_node (current_function_decl);

  if (loop == current_loops->tree_root)
    {
      exposed_refs = func_extra_info (current_node)->exposed_refs;
    }
  else
    exposed_refs = loop_extra_info (loop)->exposed_refs;


  fprintf (fp,
	   "=============================================================\n");
  fprintf (fp, "FUNCTION : %s  %d", cgraph_node_name (current_node),
	   current_node->uid);

  if (loop != current_loops->tree_root)
    fprintf (fp, "    LOOP : %d, ", loop_uid (loop));

  fprintf (fp, "\n");

  int i;

  fprintf (fp, "UP-EXPOSE REFERENCES : \n");
  exposed_refs->up_read ().print (fp);
  exposed_refs->up_write ().print (fp);


  fprintf (fp,
	   "=============================================================\n\n\n");

}


static void
check_exposed_refs_for_loop (loop_p loop, FILE * fp)
{

  int i;
  basic_block bb;

  /* get loop body in post order */
  basic_block *loop_blocks;
  loop_blocks = get_loop_body_in_custom_order (loop, bb_post_order_comparator);
  loop_p inner_loop = NULL;

  exposed_ref_summary *exposed_refs;

  if (loop == current_loops->tree_root)
  {
    cgraph_node_ptr current_node = cgraph_get_node (current_function_decl);
    exposed_refs = func_extra_info (current_node)->exposed_refs;
  }
  else
    exposed_refs = loop_extra_info (loop)->exposed_refs;


  bitmap visited = BITMAP_ALLOC (NULL);

  for (i = 0; i < loop->num_nodes; i++)
  {
    bb = loop_blocks[i];
    //printf("bb %d\n", bb->index);

    if (bb->loop_depth == loop_depth (loop) + 1)
    {
      /* the first time to meet this loop */
      inner_loop = bb->loop_father;

      if (!bitmap_bit_p (visited, inner_loop->num)
          && bb == inner_loop->header)
      {

        bitmap_set_bit (visited, inner_loop->num);

        exposed_ref_summary *inner_exposed_refs = loop_extra_info (inner_loop)->loop_site_exposed_refs;

        if (loop == current_loops->tree_root)
      	{
      	  remove_local_range (inner_exposed_refs->up_read ());
      	  remove_local_range (inner_exposed_refs->up_write ());
      	}


        /* Check the up-expose ref of inner loop from loop header */
        for (symbolic_range_list_t::const_iterator iter = inner_exposed_refs->up_read ().begin ();
        	    iter != inner_exposed_refs->up_read ().end (); ++iter)
      	{
      	  check_a_expose_ref_in_loop (loop, *iter, inner_loop->header, cf_first, NULL, fp);
      	}


        for (symbolic_range_list_t::const_iterator iter =  inner_exposed_refs->up_write ().begin ();
        	   iter != inner_exposed_refs->up_write ().end (); ++iter)
      	{
      	  check_a_expose_ref_in_loop (loop, *iter, inner_loop->header, cf_first, NULL, fp);
      	}



      }
      continue;
    }

    else if (bb->loop_depth > loop_depth (loop) + 1)
      continue;

    gimple_stmt_iterator bsi;
    for (bsi = gsi_last_bb (bb); !gsi_end_p (bsi); gsi_prev (&bsi))
    {
      gimple stmt = gsi_stmt (bsi);

      
      if (loop_uid(loop) == 83 )
      {
        FILE *fp = fopen("83stmt", "a");
        fprintf(fp, "%d    ", gimple_uid(stmt));
        print_gimple_stmt(fp, stmt, 0, 0);
        fclose(fp);
      }
      
      int j;

      symbolic_range_list ranges;

      /* get up-write ranges */
      get_write_symbolic_ranges (stmt, ranges);
      if (loop == current_loops->tree_root)
        remove_local_range (ranges);

      for (symbolic_range_list_t::const_iterator iter = ranges.begin ();
           iter != ranges.end (); ++iter)
      {
        check_a_expose_ref_in_loop (loop, *iter, bb, cf_here, stmt, fp);
      }

      ranges.clear ();
      get_read_symbolic_ranges (stmt, ranges);
      if (loop == current_loops->tree_root)
        remove_local_range (ranges);
      for (symbolic_range_list_t::const_iterator iter = ranges.begin ();
           iter != ranges.end (); ++iter)
      {
        check_a_expose_ref_in_loop (loop, *iter, bb, cf_here, stmt, fp);
      }

#if 0
      /* check down-exposed */
      ranges.clear ();
      get_write_symbolic_ranges (stmt, ranges, false);
      if (loop == current_loops->tree_root)
        remove_local_range (ranges);
      exposed_refs->down_write ().search_and_union (ranges);

      ranges.clear ();
      get_read_symbolic_ranges (stmt, ranges, false);
      if (loop == current_loops->tree_root)
        remove_local_range (ranges);
      exposed_refs->down_read ().search_and_union (ranges);
#endif
    }

  }


  BITMAP_FREE (visited);


  /* make a copy of summary as loop-site summary */

  if (loop != current_loops->tree_root)
  {
    evaluate_expose_ref_range_for_loop (loop);
    struct loop_extra_info_t *info = loop_extra_info (loop);
    info->loop_site_exposed_refs =
    new exposed_ref_summary (*info->exposed_refs);
    initialize_expose_state (*info->loop_site_exposed_refs);
  }

  //if (fp)
    //print_expose_refs_of_loop (fp, loop);

}



void
check_expose_refs_for_function (cgraph_node_ptr node, FILE * fp)
{

  switch_to_context (node->decl);

  printf ("Analyzing function %s\n", cgraph_node_name (node));

  callsite_hash = htab_create (20, hash_callsite, eq_callsite, hash_callsite_del);

  loop_iterator li;
  loop_p inner_loop = NULL;

  FOR_EACH_LOOP (li, inner_loop, LI_FROM_INNERMOST)
  {
    struct loop_extra_info_t *linfo = loop_extra_info (inner_loop);
    linfo->exposed_refs = new_summary ();
    check_exposed_refs_for_loop (inner_loop, fp);
  }

  struct func_extra_info_t *finfo = func_extra_info (node);
  if (finfo->in_loop())
    check_exposed_refs_for_loop (current_loops->tree_root, fp);


  FOR_EACH_LOOP (li, inner_loop, LI_FROM_INNERMOST)
  {
    struct loop_extra_info_t *finfo = loop_extra_info (inner_loop);
    finfo->loop_site_exposed_refs->clear ();
    finfo->loop_site_exposed_refs = NULL;
  }

  basic_block bb;
  FOR_EACH_BB (bb)
  {
    gimple_stmt_iterator gsi;
    for (gsi = gsi_start_bb (bb); !gsi_end_p (gsi); gsi_next (&gsi))
    {
      gimple stmt = gsi_stmt (gsi);
      if (!is_gimple_call (stmt))
        continue;

      struct callsite_t tmp;
      tmp.call = stmt;
      tmp.references = NULL;

      struct callsite_t *callsite;
      callsite = (struct callsite_t *) htab_find (callsite_hash, &tmp);

      if (callsite)
        callsite->references->clear ();
    }

  }

  htab_delete (callsite_hash);

  int callsite_count = 0;
  struct cgraph_edge *callsite;

  /* Check from each callsite */
  for (callsite = node->callers; callsite; callsite = callsite->next_caller)
    ++callsite_count;

  if (callsite_count > 1)
    {
      exposed_ref_summary *exposed_refs = func_exposed_refs (node);
      for (symbolic_range_list::iterator iter =
	   exposed_refs->up_read ().begin ();
	   iter != exposed_refs->up_read ().end (); ++iter)
	iter->set_must (false);
    }

  switch_off_context ();
}


void
union_expose_references_for_recursion (std::set < cgraph_node_ptr > scc,
				       exposed_ref_summary & summary)
{
  for (std::set < cgraph_node_ptr >::iterator siter = scc.begin ();
       siter != scc.end (); ++siter)
    {
      exposed_ref_summary *exposed_refs = func_exposed_refs (*siter);
      symbolic_range_list ranges;

      /* union up-read */
      ranges.Union (exposed_refs->up_read ());
      for (symbolic_range_list::iterator riter = ranges.begin ();
	   riter != ranges.end (); ++riter)
	{
	  if (riter->kind () == rk_sure)
	    riter->set_kind (rk_unknown);
	}

      summary.up_read ().search_and_union (ranges);
      ranges.clear ();


      /* union up-write */
      ranges.Union (exposed_refs->up_write ());
      for (symbolic_range_list::iterator riter = ranges.begin ();
	   riter != ranges.end (); ++riter)
	{
	  if (riter->kind () == rk_sure)
	    riter->set_kind (rk_unknown);
	}

      summary.up_write ().search_and_union (ranges);

    }
}

unsigned int
ipa_expose_reference_analysis (const std::set < cgraph_node_ptr > &scc,
                            			       FILE * fp)
{

  /* Initialize for exposed-reference analysis */
  for (std::set < cgraph_node_ptr >::iterator siter = scc.begin ();
       siter != scc.end (); ++siter)
  {
    struct cgraph_node *cur = *siter;
    struct func_extra_info_t *finfo = func_extra_info (cur);
    finfo->exposed_refs = new_summary ();
  }

  /* Perform Exposed-references Analysis */
  for (std::set < cgraph_node_ptr >::iterator siter = scc.begin ();
       siter != scc.end (); ++siter)
  {
    struct cgraph_node *cur = *siter;
    check_expose_refs_for_function (cur, fp);
  }

  /* Union expose-reference results for recursion */
  if (scc.size () > 1)
  {
    exposed_ref_summary summary;
    union_expose_references_for_recursion (scc, summary);
    for (std::set < cgraph_node_ptr >::iterator siter = scc.begin ();
         siter != scc.end (); ++siter)
    {
      struct cgraph_node *cur = *siter;
      exposed_ref_summary *exposed_refs = func_exposed_refs (*siter);
      exposed_refs->clear ();
      exposed_refs->copy (summary);
    }
  }

  return 0;

}


void
release_resouces_of_callees (const std::set < cgraph_node_ptr > &scc)
{
  for (std::set < cgraph_node_ptr >::iterator siter = scc.begin ();
       siter != scc.end (); ++siter)
    {
      release_resouces_of_callees (*siter);
    }
}



#if 0


typedef struct
{
  struct loop *loop;
  bool tag;
} temp_pair;


/* DEF is a SSA_NAME.  REF is a non-aliased decl. 
	 Suppose DEF is aliased ith REF.  DATA points to the 
   a loop and a tag. Check whether the DEF defines REF in loop. 
   If yes, set tag to be true. If tag is true, return */
static bool
check_overlap_in_loop_p (ao_ref * ref, tree def, void *data)
{
  temp_pair *p = (temp_pair *) data;
  gimple def_stmt;
  struct loop *def_loop;

  if (p->tag)
    return true;

  def_stmt = SSA_NAME_DEF_STMT (def);

  if (!def_stmt)
    return false;

  /* If p->nest is NULL, it means checking over the whole function body */
  if (p->nest == NULL)
    {
      p->tag = true;
      return true;
    }

  def_loop = loop_containing_stmt (def_stmt);
  if (p->nest == def_loop || flow_loop_nested_p (p->nest, def_loop))
    p->tag = true;

  return p->tag;

}



/* the range is (header, cur] */
bool
inside_loop_from_header_to_cur_p (loop_p loop, gimple cur_stmt, gimple stmt)
{

  basic_block *bb = gimple_bb (stmt);
  basic_block *cur_bb = gimple_bb (cur_stmt);

  gcc_assert (cur_stmt != stmt);

  if (bb == loop->header)
    return false;

  if (!dominated_by_p (CDI_DOMINATORS, bb, loop->header)
      return false; if (bb == cur_bb)
      {
      gimple_stmt_iterator gsi;
      for (gsi = gsi_last_bb (bb); !gsi_end_p (gsi); gsi_prev (&gsi))
      {
      if (cur_stmt == gsi_stmt (gsi))
      return true; if (stmt == gsi_stmt (gsi)) return false;}
      }

      struct extra_flow_info * info = get_extra_info (cfun);
      int i; int tmp; for (i = 0; i < info->n_blocks_inverted; i++)
      {
      tmp = info->postorder_inverted[i]; if (tmp == bb->index) return true;}

      return false;}


      static bool
      ssa_check_def_above_ref_inside_iteration (loop_p loop, gimple def_stmt,
						gimple ref_stmt)
      {

      if (!inside_loop_from_header_to_cur_p (loop, ref_stmt, def_stmt)
	  return false;
	  if (is_gimple_assign (def_stmt))
	  return true; if (gimple_code (def_stmt) == GIMPLE_PHI)
	  {
	  /* check each phi operand */
	  int i; for (i = 0; i < gimple_phi_num_args (def_stmt); i++)
	  {
	  tree opnd = gimple_phi_arg_def (def_stmt, i);
	  gimple def_opnd = SSA_NAME_DEF_STMT (opnd);
	  if (!ssa_check_def_above_ref_inside_iteration
	      (loop, def_opnd, def_stmt)) return false;}
	  }

	  return true;}

	  static bool must_alias_p (tree a, tree b)
	  {

	  if (TREE_CODE (a) != TREE_CODE (b))
	  return false;
	  if (operand_equal_p (a, b, 0)) return true; switch (TREE_CODE (a))
	  {
case ARRAY_REF:
	  {

	  if (!must_alias_p (TREE_OPERAND (a, 0), TREE_OPERAND (b, 0)))
	  return false;
	  /* We may compare the eventual address in the future */
	  return address_equal_p (a, b);}

case MEM_REF:
case COMPONENT_REF:
case INDIRECT_REF:
return address_equal_p (a, b); default:
	  return false;}

	  }


	  typedef struct
	  {
	  struct loop * loop; gimple ref_stmt; bool tag;} check_def;

/* ssa_check_def_above_ref_inside_iteration 
	WALKER is called with REF, the current vdef and DATA.  If WALKER
	returns true the walk is stopped, otherwise it continues.
*/
	  static bool must_ailas_def_in_iteration_p (ao_ref * ref, tree def,
						     void *data)
	  {
	  check_def * p = (check_def *) data; gimple def_stmt; if (!p->tag)	/* define has been out of loop already, stop checking */
	  return true; def_stmt = SSA_NAME_DEF_STMT (def); if (!def_stmt)	/* found a define out of loop already, stop checking */
	  {
	  p->tag = false; return true;}

	  if (!inside_loop_from_header_to_cur_p
	      (p->loop, p->ref_stmt, def_stmt)) return false;
	  /* check must alias */
	  if (must_alias_p (def, ref->ref)) return true; return false;	/* go on searching */
	  }

	  void IPA_check_control_dependence (void)
	  {

/* Build invocation graph */
	  struct invocation_node_info * info =
	  new_invocation_info (callee_decl, stmt);
	  int v_callee = add_vertex (invocation_graph, info);
	  add_edge (invocation_graph, v_caller, v_callee);}


	  struct invocation_vertex_info
	  {
	  int index;		/*  the index to graph vertices array */
	  tree func_decl;	/*  the callee's declaration */
	  tree call_site;	/*  the callsite in caller */
	  };

/* Computes a hash function for invocation node.  */
	  static hashval_t hash_invocation_vertex_info (const void *elt)
	  {
	  const struct invocation_vertex_info * const rvi =
	  (const struct invocation_vertex_info *)elt;
	  tree decl = rvi->func_decl;
	  tree call = rvi->call_site; return htab_hash_pointer (stmt);}

/* Compares database elements E1 and E2.  */

	  static int eq_stmt_vertex_info (const void *e1, const void *e2)
	  {
	  const struct rdg_vertex_info * elt1 =
	  (const struct rdg_vertex_info *)e1;
	  const struct rdg_vertex_info * elt2 =
	  (const struct rdg_vertex_info *)e2;
	  return elt1->stmt == elt2->stmt;}

/* Free the element E.  */

	  static void hash_stmt_vertex_del (void *e)
	  {
	  free (e);}

/* Returns a hash code for P.  */

	  static hashval_t hash_node (const void *p)
	  {
	  const struct cgraph_node * n = (const struct cgraph_node *)p;
	  return (hashval_t) DECL_UID (n->decl);}


/* Returns nonzero if P1 and P2 are equal.  */

	  static int eq_node (const void *p1, const void *p2)
	  {
	  const struct cgraph_node * n1 = (const struct cgraph_node *)p1;
	  const struct cgraph_node * n2 = (const struct cgraph_node *)p2;
	  return DECL_UID (n1->decl) == DECL_UID (n2->decl);}


	  static
	  struct invocation_vertex_info *
	  new_invocation_info (tree func_decl, tree call_site)
	  {

	  struct invocation_vertex_info * v =
	  XNEW (struct invocation_vertex_info); v->func_decl = func_decl;
	  v->call_site = call_site; v->call_site_bb = bb; return v;}


	  static struct graph * invocation_graph;	/* the invocation graph for each loop */
	  int v_caller = -1;	/* temporary record the current caller */
	  int v_root = -1;	/* temporary record the current caller */
/* Free the invocation graph.  */
	  static void free_invocation_graph (struct graph *ivg)
	  {
	  int i;
	  for (i = 0; i < ivg->n_vertices; i++)
	  free (ivg->vertices[i].data);
	  htab_delete (ivg->indices); free_graph (ivg);}





/* Search the indirect data references in LOOP, and record the information into
   DATAREFS.  Returns chrec_dont_know when failing to analyze a
   difficult case, returns NULL_TREE otherwise.

   TODO: This function should be made smarter so that it can handle address
   arithmetic as if they were array accesses, etc.  */

	  tree
	  IPA_find_data_references_in_loop (struct loop *loop,
					    VEC (ipa_data_reference_p,
						 heap) ** datarefs)
	  {
	  basic_block bb, *bbs;
	  unsigned int i;
	  bbs = get_loop_body_in_dom_order (loop);
	  for (i = 0; i < loop->num_nodes; i++)
	  {
	  bb = bbs[i];
	  if (IPA_find_data_references_in_bb (loop, bb, datarefs) ==
	      chrec_dont_know)
	  {
	  free (bbs); return chrec_dont_know;}
	  }
	  free (bbs); return NULL_TREE;}


/* return true when the two memory references are exactly the same */
	  static int
	  memory_equal_p (const ipa_data_reference_p ref0,
			  const ipa_data_reference_p ref1)
	  {
	  tree mem0 = DR_REF (ref0); tree mem1 = DR_REF (ref1); if (mem0 == mem1)	/* VAR_DECL included */
	  return true;
	  STRIP_NOPS (mem0);
	  STRIP_NOPS (mem1);
	  if (TREE_CODE (mem0) != TREE_CODE (mem1))
	  return false; /**/ if (DR_PROC (ref0) == DR_PROC (ref1))
	  {

	  /* switch context */
	  switch_to_context (DR_PROC (ref0));
	  if (TREE_CODE (mem0) == SSA_NAME)
	  {
	  return SSA_NAME_VAR (mem0) == SSA_NAME_VAR (mem1);}
	  switch_off_context ();}
	  else
	  {

	  /* check global variable reference */

	  }

	  }


/* check the data dependence relation between data accesses A and
   B.  If there is a dependence between them. */

	  static struct ipa_data_dependence_relation *
	  IPA_check_data_dependence_relation (struct loop *nest,
					      ipa_data_reference_p a,
					      ipa_data_reference_p b,
					      VEC (loop_p, heap) * loop_nest)
	  {

	  /* First decide the lexicographic order */



	  struct ipa_data_dependence_relation * res;
	  unsigned int i;
	  res = XNEW (struct data_dependence_relation);
	  DDR_A (res) = a;
	  DDR_B (res) = b;
	  DDR_LOOP_NEST (res) = NULL;
	  DDR_REVERSED_P (res) = false;
	  DDR_SUBSCRIPTS (res) = NULL;
	  DDR_DIR_VECTS (res) = NULL;
	  DDR_DIST_VECTS (res) = NULL; if (a == NULL || b == NULL)
	  {
	  DDR_ARE_DEPENDENT (res) = chrec_known; return res;}



	  /* Check whether loop-independent data dependence exists
	     1. Check the two data references can be aliased or not

	     If not sure, check whether the data dependence can change among loop iterations
	     1. check whether loop-invariant
	     2. If loop-variant, check how it varies as possible as I can.
	   */



	  /* When the references are exactly the same, don't spend time doing
	     the data dependence tests, just initialize the ddr and return.  */

	  if (memory_equal_p (DR_REF (a), DR_REF (b)))
	  {
	  DDR_AFFINE_P (res) = true;
	  DDR_ARE_DEPENDENT (res) = NULL_TREE;
	  DDR_SUBSCRIPTS (res) =
	  VEC_alloc (subscript_p, heap, DR_NUM_DIMENSIONS (a));
	  DDR_LOOP_NEST (res) = loop_nest; DDR_INNER_LOOP (res) = 0;
	  DDR_SELF_REFERENCE (res) = true; compute_self_dependence (res);
	  return res;}


	  /* If the data references do not alias, then they are independent.  */
	  if (!IPA_dr_may_alias_p (a, b))
	  {
	  DDR_ARE_DEPENDENT (res) = chrec_known; return res;}


	  /* Check loop-carried data dependence */








	  /* Check it flow-insensitively */


	  /* If the references do not access the same object, we do not know
	     whether they alias or not.  */
	  if (!operand_equal_p (DR_BASE_OBJECT (a), DR_BASE_OBJECT (b), 0))
	  {
	  DDR_ARE_DEPENDENT (res) = chrec_dont_know; return res;}

	  /* If the base of the object is not invariant in the loop nest, we cannot
	     analyze it.  TODO -- in fact, it would suffice to record that there may
	     be arbitrary dependences in the loops where the base object varies.  */
	  if (loop_nest
	      &&
	      !object_address_invariant_in_loop_p (VEC_index
						   (loop_p, loop_nest, 0),
						   DR_BASE_OBJECT (a)))
	  {
	  DDR_ARE_DEPENDENT (res) = chrec_dont_know; return res;}

	  gcc_assert (DR_NUM_DIMENSIONS (a) == DR_NUM_DIMENSIONS (b));
	  DDR_AFFINE_P (res) = true;
	  DDR_ARE_DEPENDENT (res) = NULL_TREE;
	  DDR_SUBSCRIPTS (res) =
	  VEC_alloc (subscript_p, heap, DR_NUM_DIMENSIONS (a));
	  DDR_LOOP_NEST (res) = loop_nest; DDR_INNER_LOOP (res) = 0;
	  DDR_SELF_REFERENCE (res) = false; for (i = 0;
						 i < DR_NUM_DIMENSIONS (a);
						 i++)
	  {
	  struct subscript * subscript;
	  subscript = XNEW (struct subscript);
	  SUB_CONFLICTS_IN_A (subscript) = conflict_fn_not_known ();
	  SUB_CONFLICTS_IN_B (subscript) = conflict_fn_not_known ();
	  SUB_LAST_CONFLICT (subscript) = chrec_dont_know;
	  SUB_DISTANCE (subscript) = chrec_dont_know;
	  VEC_safe_push (subscript_p, heap, DDR_SUBSCRIPTS (res), subscript);}

	  return res;}



/* Compute in DEPENDENCE_RELATIONS the data dependence graph for all
   the data references in DATAREFS, in the LOOP_NEST.   
		* Checking loop-independent data dependence is used for code motioning
		* Checking loop-carried data dependence is used for parallelization
 */

	  void
	  IPA_compute_all_dependences (VEC (ipa_data_reference_p, heap) *
				       datarefs, VEC (ddr_p,
						      heap) **
				       dependence_relations, VEC (loop_p,
								  heap) *
				       loop_nest)
	  {
	  struct data_dependence_relation * ddr;
	  struct ipa_data_reference * a, *b;
	  unsigned int i, j;
	  for (i = 0; VEC_iterate (ipa_data_reference_p, datarefs, i, a); i++)
	  for (j = i + 1; VEC_iterate (ipa_data_reference_p, datarefs, j, b);
	       j++) if (!DR_IS_READ (a) || !DR_IS_READ (b))
	  {

	  /* Let the statement containing a is in front of that containing 
	     b in lexicographic level */


	  ddr = IPA_check_data_dependence_relation (a, b, loop_nest);
	  VEC_safe_push (ddr_p, heap, *dependence_relations, ddr);
	  if (loop_nest)
	  compute_affine_dependence (ddr, VEC_index (loop_p, loop_nest, 0));}

	  if (compute_self_and_rr)
	  for (i = 0; VEC_iterate (data_reference_p, datarefs, i, a); i++)
	  {
	  ddr = initialize_data_dependence_relation (a, a, loop_nest);
	  VEC_safe_push (ddr_p, heap, *dependence_relations, ddr);
	  compute_self_dependence (ddr);}
	  }



/* Returns true when the data dependences have been computed, false otherwise.
   Given a loop nest LOOP, the following vectors are returned:
   DATAREFS is initialized to all the array elements contained in this loop,
   DEPENDENCE_RELATIONS contains the relations between the data references.
   */

	  bool
	  IPA_compute_data_dependences_for_loop (struct loop *loop,
						 VEC (ipa_data_reference_p,
						      heap) ** datarefs,
						 VEC (ddr_p,
						      heap) **
						 dependence_relations)
	  {
	  bool res = true;
	  VEC (loop_p, heap) * vloops = VEC_alloc (loop_p, heap, 3);
	  memset (&dependence_stats, 0, sizeof (dependence_stats));
	  find_loop_nest (loop, &vloops);
	  /* build_empty_invocation_graph () */
	  invocation_graph = new_graph (10);
	  invocation_graph->indices = htab_create (10, hash_stmt_vertex_info,
						   eq_stmt_vertex_info,
						   hash_stmt_vertex_del);
	  struct invocation_node_info * v =
	  new_invocation_info (current_function_decl, 0, 0);
	  v_caller = add_vertex (invocation_graph, v); v_root = v_caller;
	  if (IPA_find_data_references_in_loop (loop, datarefs) ==
	      chrec_dont_know)
	  {
	  struct data_dependence_relation * ddr;
	  /* Insert a single relation into dependence_relations: chrec_dont_know.  */
	  ddr = initialize_data_dependence_relation (NULL, NULL, vloops);
	  VEC_safe_push (ddr_p, heap, *dependence_relations, ddr);
	  res = false;}
	  else
	  {

	  IPA_compute_all_dependences (*datarefs, dependence_relations,
				       vloops);}


	  /* Output dependence file */


	  if (dump_file && (dump_flags & TDF_STATS))
	  {
	  fprintf (dump_file, "Dependence tester statistics:\n");
	  fprintf (dump_file, "Number of dependence tests: %d\n",
		   dependence_stats.num_dependence_tests);
	  fprintf (dump_file,
		   "Number of dependence tests classified dependent: %d\n",
		   dependence_stats.num_dependence_dependent);
	  fprintf (dump_file,
		   "Number of dependence tests classified independent: %d\n",
		   dependence_stats.num_dependence_independent);
	  fprintf (dump_file, "Number of undetermined dependence tests: %d\n",
		   dependence_stats.num_dependence_undetermined);
	  fprintf (dump_file, "Number of subscript tests: %d\n",
		   dependence_stats.num_subscript_tests);
	  fprintf (dump_file, "Number of undetermined subscript tests: %d\n",
		   dependence_stats.num_subscript_undetermined);
	  fprintf (dump_file, "Number of same subscript function: %d\n",
		   dependence_stats.num_same_subscript_function);
	  fprintf (dump_file, "Number of ziv tests: %d\n",
		   dependence_stats.num_ziv);
	  fprintf (dump_file, "Number of ziv tests returning dependent: %d\n",
		   dependence_stats.num_ziv_dependent);
	  fprintf (dump_file,
		   "Number of ziv tests returning independent: %d\n",
		   dependence_stats.num_ziv_independent);
	  fprintf (dump_file, "Number of ziv tests unimplemented: %d\n",
		   dependence_stats.num_ziv_unimplemented);
	  fprintf (dump_file, "Number of siv tests: %d\n",
		   dependence_stats.num_siv);
	  fprintf (dump_file, "Number of siv tests returning dependent: %d\n",
		   dependence_stats.num_siv_dependent);
	  fprintf (dump_file,
		   "Number of siv tests returning independent: %d\n",
		   dependence_stats.num_siv_independent);
	  fprintf (dump_file, "Number of siv tests unimplemented: %d\n",
		   dependence_stats.num_siv_unimplemented);
	  fprintf (dump_file, "Number of miv tests: %d\n",
		   dependence_stats.num_miv);
	  fprintf (dump_file, "Number of miv tests returning dependent: %d\n",
		   dependence_stats.num_miv_dependent);
	  fprintf (dump_file,
		   "Number of miv tests returning independent: %d\n",
		   dependence_stats.num_miv_independent);
	  fprintf (dump_file, "Number of miv tests unimplemented: %d\n",
		   dependence_stats.num_miv_unimplemented);}

	  free_invocation_graph (invocation_graph); return res;}


/* Entry point.  Analyze all the data references
   and the dependence relations in LOOP and its callee body.

   The data references are computed first.

   A relation on these nodes is represented by a complete graph.  Some
   of the relations could be of no interest, thus the relations can be
   computed on demand.

   In the following function we compute all the relations.  

   It is possible to ask only for a part of the graph, avoiding to
   compute the whole dependence graph.  The computed dependences are
   stored in a knowledge base (KB) such that later queries don't
   recompute the same information.  The implementation of this KB is
   transparent to the optimizer, and thus the KB can be changed with a
   more efficient implementation, or the KB could be disabled.  */


	  static void IPA_analyze_all_data_dependences (struct loop *loop)
	  {

	  unsigned int i;
	  int nb_data_refs = 100;
	  VEC (ipa_data_reference_p, heap) * datarefs =
	  VEC_alloc (ipa_data_reference_p, heap, nb_data_refs);
	  VEC (ddr_p, heap) * dependence_relations =
	  VEC_alloc (ddr_p, heap, nb_data_refs * nb_data_refs);
	  /* Compute DDs on the whole function.  */
	  IPA_compute_data_dependences_for_loop (loop, &datarefs,
						 &dependence_relations);
	  if (dump_file)
	  {
	  dump_data_dependence_relations (dump_file, dependence_relations);
	  fprintf (dump_file, "\n\n");
	  if (dump_flags & TDF_DETAILS)
	  dump_dist_dir_vectors (dump_file, dependence_relations);
	  if (dump_flags & TDF_STATS)
	  {
	  unsigned nb_top_relations = 0;
	  unsigned nb_bot_relations = 0;
	  unsigned nb_chrec_relations = 0;
	  struct data_dependence_relation * ddr;
	  for (i = 0; VEC_iterate (ddr_p, dependence_relations, i, ddr); i++)
	  {
	  if (chrec_contains_undetermined (DDR_ARE_DEPENDENT (ddr)))
	  nb_top_relations++;
	  else
	  if (DDR_ARE_DEPENDENT (ddr) == chrec_known) nb_bot_relations++;
	  else
	  nb_chrec_relations++;}

	  gather_stats_on_scev_database ();}
	  }

	  free_dependence_relations (dependence_relations);
	  free_data_refs (datarefs);}



	  static void ipa_check_data_dependence (void)
	  {

	  struct cgraph_node * node;
	  struct cgraph_node ** order =
	  XCNEWVEC (struct cgraph_node *, cgraph_n_nodes);
	  int order_pos = cgraph_postorder (order); int i;
	  /* collect data references */
	  for (i = 0; i < order_pos; i++)
	  {
	  node = order[i];
	  if (cgraph_function_body_availability (node) < AVAIL_OVERWRITABLE
	      || node->global.inlined_to)
	  continue;
	  if (bitmap_bit_p (analyzed_functions, DECL_UID (node->decl)))
	  continue;
	  switch_to_context (node->decl);
	  fprintf (stdout, "<function name=\"%s\">\n",
		   lang_hooks.decl_printable_name (current_function_decl, 2));
//              calculate_dominance_info (CDI_DOMINATORS);
	  FOR_EACH_LOOP (li, loop, LI_ONLY_OUTERMOST)
	  ipa_collect_profiling_data_references (loop, 0);
	  switch_off_context ();}




	  /* Find all of the indirect data references contained in loop interprocedurally */
	  ipa_find_data_references_in_loop (loop, &indirect_datarefs,
					    call_level);
	  /* symbolic range propagation */
	  /* expose reference analysis */
	  if (flag_ipa_expose_reference) ipa_expose_reference_analysis ();
	  else
	  {
	  /* initialize expose-ref set as all exposed */
	  }




	  /* data dependence checking */

	  }


	  static bool gate_ipa_check_data_dep (void)
	  {
	  return (flag_ipa_check_dependence != 0);}

	  struct gimple_opt_pass pass_ipa_check_dependence =
	  {
	  {
	  GIMPLE_PASS, "ipa_check_dependence",	/* name */
	  gate_ipa_check_data_dep,	/* gate */
	  IPA_check_data_dependence,	/* execute */
	  NULL,			/* sub */
	  NULL,			/* next */
	  0,			/* static_pass_number */
	  TV_IPA_CHECK_DEPENDENCE,	/* tv_id */
	  0,			/* properties_required */
	  0,			/* properties_provided */
	  0,			/* properties_destroyed */
	  0,			/* todo_flags_start */
	  0			/* todo_flags_finish */
	  }
	  };
#endif
