/* header file for dependence profiling. -*- C++ -*- */

#ifndef GCC_IPA_DEPENDENCE_CHECKING
#define GCC_IPA_DEPENDENCE_CHECKING

#include <list>
#include <vector>
#include <set>
#include <map>
#include <string>
#include <algorithm>
#include <iostream>




#include "gmp.h"

extern "C"
{
#include "vec.h"
#include "config.h"
#include "system.h"
#include "coretypes.h"
#include "timevar.h"
#include "flags.h"
#include "tree.h"
#include "cgraph.h"
#include "basic-block.h"
#include "cfgloop.h"
#include "function.h"
#include "tree-flow.h"
#include "tree-pass.h"
#include "tree-pretty-print.h"
#include "tree-scalar-evolution.h"
#include "tree-vectorizer.h"

#include "ipa-utils.h"
#include "langhooks.h"
#include "tree-data-ref.h"

#include "tree-pretty-print.h"
#include "gimple-pretty-print.h"

#include "yices_c.h"
}

typedef std::set<tree> TREE_SET;
typedef std::map<tree, tree> TREE_MAP;
typedef std::set<gimple> GIMPLE_SET;
typedef std::set<cgraph_node_ptr> NODE_SET;
typedef std::set<cgraph_node_ptr> IPA_NODE_SET;
typedef std::map<cgraph_node_ptr, int> IPA_NODE_INT_MAP;
typedef std::map<cgraph_node_ptr, cgraph_node_ptr> IPA_NODE_NODE_MAP;
typedef std::set<basic_block> BB_SET;

class gsi_less_p
{
public:
	bool operator()(const gimple_stmt_iterator &g1, const gimple_stmt_iterator &g2) const
	{
		if (g1.ptr < g2.ptr)
			return true;
		else if (g1.ptr > g2.ptr)
			return false;
		if (g1.seq < g2.seq)
			return true;
		else if (g1.seq > g2.seq)
			return false;
		if (g1.bb < g2.bb)
			return true;
		else if (g1.bb > g2.bb)
			return false;

		return false;

	}
};

typedef std::set<gimple_stmt_iterator, gsi_less_p> GSI_SET;


extern int loop_num;
extern int bb_num;

#define BZERO(b,len) (memset((b), '\0', (len)), (void) 0)

static inline void
switch_to_context( tree func_decl )
{
  push_cfun (DECL_STRUCT_FUNCTION (func_decl));
  current_function_decl = func_decl;
}


static inline void
switch_off_context (void)
{
  pop_cfun ();
  if ( cfun )
    current_function_decl = cfun->decl;
  else
    current_function_decl = NULL_TREE;
}


static inline void
print_indents (FILE *fp, int indent)
{
  for (int i=0; i<indent; i++)
    fprintf (fp, " " );
}


static inline tree
tree_strip_nops (tree exp)
{
  while (TREE_CODE(exp) == NOP_EXPR || TREE_CODE(exp) ==CONVERT_EXPR)
    exp = TREE_OPERAND (exp, 0);
  return exp;
}


// if not scalar, return NULL_TREE, else return the scalar

static inline tree
is_gimple_scalar(tree mem)
{
  mem = tree_strip_nops (mem);
  switch (TREE_CODE(mem))
    {
    case PARM_DECL:
    case VAR_DECL:
    case SSA_NAME:
		case RESULT_DECL:
      return mem;

    case MEM_REF:
      {
        tree base = TREE_OPERAND(mem,0);
        tree offset = TREE_OPERAND(mem,1);
        if ( int_cst_value(offset) != 0 )
          return NULL_TREE;
        if ( TREE_CODE(base) == ADDR_EXPR )
          {
            return TREE_OPERAND(base,0);
          }
        return NULL_TREE;

      }

    case COMPONENT_REF:
      return is_gimple_scalar (TREE_OPERAND (mem,0));


    default:
      return NULL_TREE;
    }
}



// reference a local variable or array

static inline bool
is_local_reference(tree mem)
{
  mem = tree_strip_nops (mem);
  switch (TREE_CODE(mem))
  {
	  case PARM_DECL:
	  case SSA_NAME:
		case RESULT_DECL:
	    return true;
		
		case VAR_DECL:
		{
      if ( DECL_ARTIFICIAL (mem) )
        return true;
      
      if (TREE_STATIC (mem))
        return false;

      if (TREE_PUBLIC (mem))
        return false;      

			return true;
		}

	  case MEM_REF:
    {
      tree base = TREE_OPERAND(mem,0);
      tree offset = TREE_OPERAND(mem,1);
      if ( TREE_CODE(base) == ADDR_EXPR )
          return is_local_reference (TREE_OPERAND (base,0));
      return false;
    }

	  case COMPONENT_REF:
	    return is_local_reference (TREE_OPERAND (mem,0));

		case ARRAY_REF:
			return is_local_reference (TREE_OPERAND (mem,0));

  default:
    return false;
  }
}


static inline void
free_reference_set (struct ptr_info_def *pi)
{
  if (pi->pt.vars)
    BITMAP_FREE (pi->pt.vars);
  XDELETE (pi);
}


/* Data dependence type.  */

enum ipa_dep_type {
  /* Read After Write (RAW).  */
  id_flow_dd = 0x1,

  /* Write After Read (WAR).  */
  id_anti_dd = 0x2,

  /* Write After Write (WAW).  */
  id_output_dd = 0x4,

  /* Read After Read (RAR).  */
  id_input_dd = 0x8

};


/* Data dependence kind.  */
enum ipa_dep_kind {
  /* Loop-independent.  */
  id_loop_independent = 0x1,

  /* Loop-carried.  */
  id_loop_carried = 0x2
};


struct ipa_data_reference;


/* A ipa_data_dependence_relation represents a relation from
   statement A to B, i.e. B depends on A
   For a loop-independent relation, A is ahead of B in lexicographic order.
   For a loop-carried relation, B is in the iteration after which A is in.    */

class ipa_data_dependency
{
private:
  int _a;               /* dr id, source */
  int _b;               /* dr id, sink */
  int _loop_id; 	/* The global id of the loop inside which the dependence occurs */
  ipa_dep_kind _kind;
  ipa_dep_type _type;
	int _distance;
  bool _must;

public:

  ipa_data_dependency() {BZERO(this,sizeof(ipa_data_dependency));}
	ipa_data_dependency (const ipa_data_dependency &d)
	{
		memcpy (this, &d, sizeof(ipa_data_dependency));
	}

  ipa_data_dependency(int a, int b, ipa_dep_kind kind,
                      ipa_dep_type type, int loop, int distance=0) :
    _a(a), _b(b), _kind(kind), _type(type), _loop_id(loop), _must(false), _distance(distance)
			{}

  bool operator<(const ipa_data_dependency &r) const
  {
    if (_a < r._a)
      return true;
		else if (_a > r._a)
			return false;

		// _a == r._a here
    if (_b < r._b)
      return true;
    if (_b > r._b)
      return false;

		// _a == r._a && _b == r._b here		
    if (_kind < r._kind)
      return true;
    if (_kind > r._kind)
      return false;
		
    if (_type < r._type)
      return true;
    if (_type > r._type)
      return false;
		
    if (_loop_id < r._loop_id)
      return true;
    if (_loop_id > r._loop_id)
      return false;
		
    if (_distance < r._distance)
      return true;
    if (_distance > r._distance)
      return false;

    return false;
  }

  int source() const { return _a;}
  int sink() const { return _b;}
  int loop() const { return _loop_id;}
  ipa_dep_kind kind() const { return _kind;}
  ipa_dep_type type() const { return _type;}
  bool must() const { return _must;}
	int distance() const { return _distance; }

  void set_source(int a) {_a = a;}
  void set_sink(int b) {_b = b;}
  void set_kind(ipa_dep_kind k) {_kind = k;}
  void set_type(ipa_dep_type t) {_type = t;}
  void set_loop(int loop) {_loop_id = loop;}
  void set_must(bool m) {_must = m;}
	void set_distance(int d) { _distance = d;}

  void print(FILE *fp, int indent) const
  {
    print_indents (fp, indent);
    fprintf(fp, "LOOP %d  ", _loop_id);

    switch (_type )
      {
        /* Read After Write (RAW). */
      case id_flow_dd:
        fprintf(fp, "RAW  ");
        break;

        /* Write After Read (WAR). */
      case id_anti_dd:
        fprintf(fp, "WAR  ");
        break;

        /* Write After Write (WAW). */
      case id_output_dd:
        fprintf(fp, "WAW  ");
        break;

      default:
        fprintf(fp, "ERROR : ");
        break;
      }

    fprintf(fp, "%d --> %d", _a, _b  );

			if ( _must )
				fprintf(fp, " 	MUST");
			else
				fprintf(fp, " 	MAY");

    if ( _kind == id_loop_independent )
      fprintf(fp, "   LOOP INDEPENDENT");
    else
      fprintf(fp, "   LOOP CARRIED ");

		fprintf(fp, "distance %d", _distance);
		fprintf(fp, "\n");
		
  }
};


typedef std::set< ipa_data_dependency > data_dependence_set;
typedef std::list< ipa_data_dependency > data_dependence_list;

extern bool value_equal_p(tree a, tree b);


static inline void
free_copied_tree(tree t)
{
  if (!t)
    return;

  enum tree_code code = TREE_CODE (t);
  enum tree_code_class kind = TREE_CODE_CLASS (code);

  switch (code)
    {
    case SAVE_EXPR :
    case TARGET_EXPR :
    case BIND_EXPR :
    case BLOCK :
      return;
    default:
      break;
    }

  switch (kind)
    {
    case tcc_type :
    case tcc_declaration :
    case tcc_constant :
      return;
    default:
      break;
    }

  if (!(IS_EXPR_CODE_CLASS (kind)
        || code == TREE_LIST
        || code == TREE_VEC
        || code == TYPE_DECL
        || code == OMP_CLAUSE))
    return;


  int i;
  int n = TREE_OPERAND_LENGTH(t);
  for (i=0; i < n; i++)
    {
      tree opnd = TREE_OPERAND (t,i);
      if (opnd)
        free_copied_tree (opnd);
    }

  if (code != STATEMENT_LIST)
    ggc_free(t);

}


enum range_kind_t {
  rk_unknown,   /* unknown range, we are not sure */
  rk_all,       /* the whole range */
  rk_sure       /* a sure range */
};

enum range_flag_t {
  rf_branch_copy = 0x1, /* symbolic is copy due to branches,
                           not original dedicated to a DR */
  rf_up_expose = 0x2,
  rf_down_expose = 0x4,
  rf_must = 0x8         /* whether it is a must definition
                           under current environment*/
};


/* [low, high]
   if both low and high are NULL_TREE, the symbolic range is known
*/

class GTY(()) symbolic_range
{
 private:
  range_kind_t	_kind;
  int _flags;
  tree GTY(()) _low;		/* address form */
  tree GTY(()) _high;
  tree GTY(()) _step;		/* bit size of element */
  tree GTY(()) _predicate;      /* The control predicate under which the range has been evaluated */
  basic_block _bb;      	/* The basic block in which the range has been evaluated */
  ipa_data_reference *_dr;	/* The data reference it is associated with */

 public:

  symbolic_range () { BZERO(this, sizeof(symbolic_range)); }
  ~symbolic_range ()
  {
    free_copied_tree (_low);
    free_copied_tree (_high);
    free_copied_tree (_step);
    free_copied_tree (_predicate);
    _low = NULL; _high = NULL; _step = NULL; _predicate = NULL;
  }

  symbolic_range (const symbolic_range &r)
  {
    _kind = r._kind;
    _low = unshare_expr (r._low);
    _high = unshare_expr (r._high);
    _step = unshare_expr(r._step);
    _predicate = unshare_expr(r._predicate);
    _dr = r._dr;
    _bb = r._bb;
    _flags = r._flags;
  }

  symbolic_range& operator=(const symbolic_range &r)
  {
    _kind = r._kind;
    _low = unshare_expr (r._low);
    _high = unshare_expr (r._high);
    _step = unshare_expr(r._step);
    _predicate = unshare_expr(r._predicate);
    _dr = r._dr;
    _bb = r._bb;
    _flags = r._flags;
  }

  ipa_data_reference* dr() const { return _dr; }
  void set_dr(ipa_data_reference* r) { _dr = r;}

  gimple stmt() const ;

  range_kind_t kind() const { return _kind; }
  void set_kind(range_kind_t  k) { _kind = k;}

  tree low() const { return _low; }
  void set_low(tree  r) { _low = r;}

  tree high() const { return _high; }
  void set_high(tree r) { _high = r;}

  tree*	low_ptr() { return &_low; }
  tree*	high_ptr() { return &_high; }

  tree step() const { return _step; }
  void set_step(tree r) { _step = r;}

  basic_block bb() const { return _bb; }
  void	set_bb(basic_block r) { _bb = r;}

  bool is_must() const { return _flags & rf_must; }
  void set_must(bool r)
  {
    if (r )
      _flags |= rf_must;
    else
      _flags &= ~rf_must;
  }

  bool is_branch_copy() const { return _flags & rf_branch_copy; }
  void set_branch_copy() { _flags |= rf_branch_copy; }
  void reset_branch_copy() { _flags &= ~rf_branch_copy;}

  bool is_up_expose() const { return _flags & rf_up_expose; }
  void set_up_expose() { _flags |= rf_up_expose; }
  void reset_up_expose() { _flags &= ~rf_up_expose;}

  bool is_down_expose()	{ return _flags & rf_down_expose; }
  void set_down_expose() { _flags |= rf_down_expose; }
  void reset_down_expose() { _flags &= ~rf_down_expose;	}

  bool is_read() const;

  symbolic_range* copy() const
  {
    symbolic_range*	result = new symbolic_range(*this);
    return result;
  }

  /* Invoke SMT Solver YICES to prove
     whether the runtime value of a equals to that of b*/
  bool operator==(const symbolic_range &range) const;

  /* Do not invoke SMT Solver YICES to prove
     whether the runtime value of a equals to that of b*/
  bool equals_to(const symbolic_range &range) const
  {
    if ( _kind == rk_unknown && range._kind == rk_unknown )
      {
        if ( _dr == range._dr )
          return true;
        else
          return false;
      }
    else if ( _kind == rk_sure && range._kind == rk_sure)
      {
        return operand_equal_p (_low, range._low, 0) && operand_equal_p (_high, range._high, 0);
      }

    return false;
  }

  bool contains(const symbolic_range &range) const
  {
    if ( _kind == rk_all )
      {
        if ( _dr == range._dr )
          return true;
        else
          return false;
      }

    return false;
  }

  /* increase the lower bound of a range by step */
  void inc_lower_bound()
  {
    _low = build2_stat (PLUS_EXPR, size_type_node, _low, _step);
  }

  /* decrease the upper bound of a range by step */
  void dec_upper_bound()
  {
    _high = build2_stat (MINUS_EXPR, size_type_node, _high, _step);
  }

  void print(FILE *fp) const;
};


typedef  std::list< symbolic_range >  symbolic_range_list_t;


class symbolic_range_list : public symbolic_range_list_t
{
  bool _changed;

public:

  bool changed() { return _changed; }
  void set_changed() { _changed = true; }
  void reset_changed() { _changed = false; }
  void Union( const symbolic_range_list &s) { insert(end(), s.begin(), s.end()); }

  bool search_dr(ipa_data_reference *dr) const
  {
    for ( const_iterator iter = begin(); iter != end(); ++iter )
      {
        if (iter->dr() == dr)
          return true;
      }
    return false;
  }

  bool search_ref(tree ref, bool read) const;

  bool search_and_insert(const symbolic_range &src)
  {
    for ( const_iterator iter = begin(); iter != end(); ++iter )
      {
        if (iter->equals_to(src) || iter->contains(src))
          return false;
      }
    push_back(src);
    return true;
  }

  bool search_and_union( const symbolic_range_list &s)
  {
    bool result =false;
    for ( const_iterator siter = s.begin(); siter != s.end(); ++siter )
      {
        bool tmp = search_and_insert (*siter);
        if (tmp)
          result = true;
      }
    if (result)
      _changed = true;
    return result;
  }


  /*
    ========
    =================
    Split and Reduce the symbolic range of BELOW : Replace the lower bound of BELOW by the
    upper bound of ABOVE with the minimu step of upper bound plus 1.
    below moves to the range after the new_range
  */

  void split_symbolic_range(const symbolic_range &above_range, iterator &below)
  {
    symbolic_range &below_range = *below;
    symbolic_range *new_range = below_range.copy();

    tree above_lower_range = above_range.low();
    tree above_upper_range = above_range.high();

    below_range.set_high (unshare_expr (above_lower_range));
    below_range.dec_upper_bound();

    new_range->set_low (unshare_expr (above_upper_range));
    new_range->inc_lower_bound();

    below = insert (below, *new_range);
    ++below;
    XDELETE (new_range);
  }

  void 	print(FILE *fp=stderr) const
  {
    int i=0;
    for ( const_iterator iter = begin(); iter != end(); ++iter )
      {
        i++;
        fprintf(fp, "[%d] ", i);
        iter->print (fp);
      }
  }

};


class ipa_data_reference;

typedef  std::list< ipa_data_reference* > data_reference_list_t;
typedef  std::list< ipa_data_reference* > data_reference_set_t;


/* Contains data dereferences that must access the same memory and occur together */

class alias_slice : public data_reference_list_t
{
  int _id;
  int _write : 1;

public:

  alias_slice (int id) { _id = id;  _write = 0; }

  ipa_data_reference* representative() const
  {
    if ( empty() )
      return NULL;
    return back();
  }

  bool is_write() const { return _write; }

  int  id() const { return _id; }
  void Union ( ipa_data_reference *dr );

  void print(FILE *fp) const;
};


extern htab_t	slice_hash;


struct slice_hash_entry
{
  int slice_id;
  alias_slice* slice;
};


extern void alias_slice_merge( ipa_data_reference *a, ipa_data_reference *b );


extern int dr_count;

class id_set : public std::set <int>
{
};

enum pattern_kind {
  pk_no_pattern,
  pk_irregular,
  pk_symbolic,
  pk_const
};


class access_dimension
{
private:
  pattern_kind _kind;
  int _offset;          // the byte size if the access if a field access
  int _size;		// the size of	access
  tree _step;		// the stride
  tree _var; 		// the index variable

public:

  access_dimension () :
    _kind(pk_irregular),
    _offset(0),
    _size(0)
  {
    _step = build_int_cst(0 , 1);
    _var = build_int_cst(0 , 0);
  }

  pattern_kind kind() const { return _kind; }
  tree var() const { return _var; }
  tree step() const { return _step; }
  int offset() const { return _offset; }
  int size() const { return _size; }

  void set_kind(pattern_kind k) { _kind = k; }
  void set_var(tree v) { _var = v; }
  void set_step(tree s) { _step = s; }
  void set_offset(int o) { _offset = o;	}
  void set_size(int s) { _size = s; }

  void print(FILE *fp)
  {
    switch (_kind)
      {
      case pk_no_pattern:
        fprintf (fp, "POINT" );
        return;

      case pk_irregular:
        fprintf (fp, "IRREGULAR" );
        return;

      case pk_symbolic :
        {
          fprintf (fp, "SYMBOLIC" );
          fprintf (fp, "		INDEX	" );
          print_generic_expr (fp, _var, 0);
          fprintf (fp, "		STEP	" );
          print_generic_expr (fp, _step, 0);
          fprintf (fp, "		OFFSET	%d", _offset);
          fprintf (fp, "		SIZE	%d", _size);
          return;
        }
      case pk_const :
        {
          fprintf (fp, "CONST" );
          fprintf (fp, "		INDEX	" );
          print_generic_expr (fp, _var, 0);
          fprintf (fp, "		OFFSET	%d", _offset);
          fprintf (fp, "		SIZE	%d", _size);
          return;
        }
      default:
        gcc_assert(false);
        break;
      }
  }
};


typedef std::vector<access_dimension> access_dimension_vec;


// a+ dim[i1] + dim[i2] + ... + dim[iN]
// [a][i1][i2]...[iN]
// Scalar A can be represented as &A[0]

class access_pattern
{
private:
  access_dimension_vec	_dimensions;
  pattern_kind	_kind;

public:
  access_pattern () :
    _kind (pk_no_pattern)
  {}

  const access_dimension_vec&  dimensions() const  { return _dimensions; }
  access_dimension_vec&  dimensions() { return _dimensions; }

  pattern_kind kind() const { return _kind; }
  void set_kind(pattern_kind k) { _kind = k; }

  void 	print(FILE *fp, int indent);
};


enum dr_loop_flags_t {
  DF_LOOP_UNKNOWN_EXPOSE        = 0x1,	/* UP_EXPOSE unchecked yet */
  DF_LOOP_UP_EXPOSE             = 0x2,
  DF_LOOP_DOWN_EXPOSE           = 0x4,
  DF_LOOP_INVARIANT             = 0x8
};


enum dr_flags_t {
  DF_NEED_PROFILE       = 0x1,  /* whether need to be Profiled or not */
  DF_PROFILE_ANYTIME 	= 0x2,  /* Profile for anytime */
  DF_PROC_UP_EXPOSE 	= 0x4,
  DF_PROC_DOWN_EXPOSE	= 0x8,
  DF_PROC_UNKNOWN_EXPOSE= 0x10, /* UP_EXPOSE unchecked yet */
  DF_AllOCATED_RANGES	= 0x20, /* has already allocated symbolic ranges */
  DF_INNER_MOST         = 0x40, /* successfully analyzed innermost behavior */
  DF_REGULAR_ACCESS 	= 0x80,	/* whether the access partern is regular */
  DF_IN_LOOP            = 0x100,/* whether the dr is inside a loop */
	DF_INVALID					= 0x200,/* whether the dr needs to be analyzed  */
	DF_FLOW_DEP					= 0x400,	/* whether the dr has flow dependence with previous dr	*/
	DF_OUTPUT_DEP				= 0x800,	/* whether the dr has output dependence with previous dr	*/
  DF_VIRTUAL  			  = 0x1000,	/* true if the dr represents an internal data strucutre of a library rountine	*/
};


/* 0 - Need not to be Profiled ; 1 - Need be Profiled; -1 Profile for anytime */
#define DF_PROFILE_ALL DF_NEED_PROFILE | DF_PROFILE_ANYTIME


class ipa_data_reference
{
public:

  /* A pointer to the statement that contains this DR.  */
  gimple stmt;

  /* A pointer to the memory reference.  */
  tree ref;

  /* Auxiliary info specific to a pass.  */
  void *aux;

  /* True when the data reference is in RHS of a stmt.  */
  bool is_read;

  /* Behavior of the memory reference in the innermost loop.  */
  struct innermost_loop_behavior innermost;

  /* Subscripts of this data reference.  */
  struct indices indices;

  /* Alias information for the data reference.  */
  struct dr_alias alias;

  /* Matrix representation for the data access functions.  */
  struct access_matrix *access_matrix;

  /* The index of operands in stmt */
  int opnd_idx;

  /* The declaration of procedure that contains this DR.  */
  tree func_decl;

  /* The original ref expression in before gimpilfication */
  tree  original_form;

  /* possible loop-carried dependencies */
  data_dependence_set _loop_carried_deps;

  /* possible loop-independent dependencies */
  data_dependence_set _loop_independent_deps;

  /* The header of up-exposed reference ranges out of the innermost scope */
  symbolic_range_list up_exposed_ranges;

  /* The current value range */
  symbolic_range_list _value_ranges;

  /* The current access range, i.e. address range */
  symbolic_range_list _access_ranges;

  /* The access pattern */
  access_pattern	_pattern;

  /* The reference set if dr is not a scalar. Scalar reference itself */
  ptr_info_def	 *_reference;

  int _flags;

  int	_slice_id;

  int	_uid;

  int _class_id;	// the alias class id based on alias profiling

  int _variant_loop; // most inner loop containing dr in which the base address of dr may changed

	long double _count; // execution count

  ipa_data_reference()
  {
    _uid = ++dr_count;
    _slice_id = _uid;
    _class_id = 0;
    _flags = 0;
    _reference = NULL;
		_count = 0;
    BZERO (&indices, sizeof(struct indices))	;
    BZERO (&innermost, sizeof(struct innermost_loop_behavior))	;
    BZERO (&alias, sizeof(struct dr_alias))	;
  }

  ~ipa_data_reference();

  int  uid () const { return _uid; }

  int  slice_id () const { return _slice_id; }
  void set_slice_id( int i) { _slice_id = i; }

  int  variant_loop () const { return _variant_loop; }
  void set_variant_loop( int i) { _variant_loop = i; }

  ptr_info_def*	reference() const  { return _reference; }

  symbolic_range_list &access_ranges() { return _access_ranges;}


  int  class_id () const { return _class_id; }
  void set_class_id( int i) { _class_id = i; }

  long double  count () const { return _count; }
  void set_count( long double i) { _count = i; }

  data_dependence_set&	loop_independent_deps() { return _loop_independent_deps;}
  data_dependence_set&	loop_carried_deps() { return _loop_carried_deps;}
  access_pattern&		pattern() { return _pattern; }

  bool in_loop() const 	{ return _flags & DF_IN_LOOP; }

  void set_in_loop( )	{ _flags |= DF_IN_LOOP;}

  void reset_in_loop( )	{ _flags &= ~DF_IN_LOOP;}

  bool  invalid () const { return _flags & DF_INVALID; }
  void  set_invalid( ) { _flags |= DF_INVALID;}
  void  reset_invalid( ) { _flags &= ~DF_INVALID;}

  bool  is_virtual () const { return _flags & DF_VIRTUAL; }
  void  set_virtual( ) { _flags |= DF_VIRTUAL;}
  void  reset_virtual( ) { _flags &= ~DF_VIRTUAL;}

	bool	need_profile () const { return _flags & DF_NEED_PROFILE; }
	void	set_need_profile( ) { _flags |= DF_NEED_PROFILE;}
	void	reset_need_profile( ) { _flags &= ~DF_NEED_PROFILE;}

	bool	has_flow_dep () const { return _flags & DF_FLOW_DEP; }
	void	set_flow_dep( ) { _flags |= DF_FLOW_DEP;}
	void	reset_flow_dep( ) { _flags &= ~DF_FLOW_DEP;}

	bool	has_output_dep () const { return _flags & DF_OUTPUT_DEP; }
	void	set_output_dep( ) { _flags |= DF_OUTPUT_DEP;}
	void	reset_output_dep( ) { _flags &= ~DF_OUTPUT_DEP;}

  bool  analyze_innermost();

  void  compute_index_pattern( tree index, gimple index_stmt, access_dimension &dimesion );

  bool	compute_base_pattern( tree base, gimple base_stmt );

  void 	compute_access_pattern( tree mem );

  void	compute_reference_set ();

  void print(FILE *fp, int indent=0);
};


#define DR_PROC(DR)                (DR)->func_decl
#define DR_ADDR_RANGES(DR)         (DR)->address_ranges
#define DR_ORIG_FORM(DR)           (DR)->original_form
#define DR_UP_EXP_RANGE(DR)        (DR)->up_exposed_ranges
#define DR_OPND_IDX(DR)      		   (DR)->opnd_idx


typedef struct ipa_data_reference *ipa_data_reference_p;
DEF_VEC_P(ipa_data_reference_p);
DEF_VEC_ALLOC_P (ipa_data_reference_p, heap);

extern htab_t	dr_hash;

struct dr_hash_entry
{
  int id;
  ipa_data_reference* dr;
};

typedef std::list<ipa_data_reference*> data_reference_list;
typedef std::set<ipa_data_reference*> data_reference_set;


static inline bool
dr_profile_anytime(ipa_data_reference_p dr)
{
  return dr->_flags & DF_PROFILE_ANYTIME;
}


static inline void
dr_set_profile_anytime(ipa_data_reference_p dr)
{
  dr->_flags |= DF_PROFILE_ANYTIME;
}


static inline void
dr_reset_profile_anytime(ipa_data_reference_p dr)
{
  dr->_flags &= ~DF_PROFILE_ANYTIME;
}


static inline bool
dr_inner_most_p(ipa_data_reference_p dr)
{
  return dr->_flags & DF_INNER_MOST;
}


static inline void
dr_set_inner_most_p(ipa_data_reference_p dr)
{
  dr->_flags |= DF_INNER_MOST;
}


static inline bool
dr_proc_up_expose(ipa_data_reference_p dr)
{
  return dr->_flags & DF_PROC_UP_EXPOSE;
}


static inline void
dr_set_proc_up_expose(ipa_data_reference_p dr)
{
  dr->_flags |= DF_PROC_UP_EXPOSE;
}


static inline void
dr_reset_proc_up_expose(ipa_data_reference_p dr)
{
  dr->_flags &= ~DF_PROC_UP_EXPOSE;
}


static inline bool
dr_allocated_ranges(ipa_data_reference_p dr)
{
  return dr->_flags & DF_AllOCATED_RANGES;
}


static inline void
dr_set_allocated_ranges(ipa_data_reference_p dr)
{
  dr->_flags |= DF_AllOCATED_RANGES;
}


extern void ipa_free_data_ref (ipa_data_reference_p dr);

inline ipa_data_reference* ipa_get_data_ref(int dr_uid)
{
  dr_hash_entry dr_entry;
  dr_entry.id = dr_uid;      
  dr_hash_entry **slot = (dr_hash_entry **) htab_find_slot (dr_hash, &dr_entry, NO_INSERT);
  gcc_assert (*slot);
  ipa_data_reference* drb = (**slot).dr;
  return drb;
}



inline bool
symbolic_range::is_read() const
{
  return DR_IS_READ(dr());
}


inline gimple
symbolic_range::stmt() const
{
  return DR_STMT(dr());
}


/* Invoke SMT Solver YICES to prove whether the runtime value of a equals to that of b*/
inline bool
symbolic_range::operator==(const symbolic_range &range) const
{
  if ( _kind == rk_unknown && range._kind == rk_unknown )
  {
    if ( _dr == range._dr )
      return true;
    else
      return false;
  }
  else if ( _kind == rk_all && range._kind == rk_all )
  {
    tree ref_a = DR_ORIG_FORM(_dr);
    tree ref_b = DR_ORIG_FORM(range._dr);
    return operand_equal_p(ref_a, ref_b, 0);
  }
  else if ( _kind == rk_sure && range._kind == rk_sure)
  {  
    return value_equal_p(_low, range._low) && value_equal_p(_high, range._high);
  }

  return false;
}


inline void
symbolic_range::print( FILE *fp ) const
{
  gimple stmt = DR_STMT(_dr);
  tree ref = DR_ORIG_FORM(_dr);


  fprintf (fp, "MEMOP %d\n", _dr->uid());
  fprintf (fp, "REF : ");
  print_generic_expr(fp, ref, 0);


  if ( DR_IS_READ(_dr) )
    fprintf (fp, "	 READ  ");
  else
    fprintf (fp, "   WRITE : ");

  fprintf (fp, "\n   KIND : %d", _kind);

  fprintf (fp, "   RANGE : ");
  print_generic_expr(fp, _low, 0);

  fprintf (fp, "   TO   ");
  print_generic_expr(fp, _high, 0);

  fprintf (fp, "   STEP   ");
  print_generic_expr(fp, _step, 0);

  fprintf (fp, "   MUST=%d", is_must());

  fprintf (fp, "   STMT %lld", gimple_uid(stmt));
  fprintf (fp, "   LINE %d    ", gimple_lineno(stmt));

  print_gimple_stmt(fp, stmt, 0, 0);
  fprintf (fp, "\n");

}


inline bool
symbolic_range_list::search_ref(tree ref, bool read) const
{
  for ( const_iterator iter = begin(); iter != end(); ++iter )
    {
      if (iter->is_read() != read)
        continue;
      if (operand_equal_p (DR_REF(iter->dr()), ref, 0) )
        return true;
    }
  return false;
}


/* initialized as all zero */
struct tree_extra_info_t
{

};


static inline struct tree_extra_info_t*
tree_extra_info(tree node)
{
  return ((struct tree_extra_info_t*)TREE_PROPERTY(node));
}


/* name is large enough, return the length of name */
#if 0
static int
get_expr_string(tree expr, char *name)
{
  FILE *fpw, *fpr;
  int name_len=0;
  fpw = fopen("tmp_dump", "w");
  fpr = fopen("tmp_dump", "r");
  print_generic_stmt(fpw, expr, 0);
  name_len= ftell(fpw);
  fread (name,1,name_len,fpr);
  fclose(fpw);
  fclose(fpr);
  name_len = name_len-1;
  gcc_assert (name_len);
  name[name_len] = '\0';
  return name_len;
}
#endif

extern "C" int get_expr_string (tree expr, char *name);


static yices_expr
make_var_from_decl_name(yices_context ctx, char *name)
{
  yices_var_decl xdecl;
  yices_expr     x ;
  yices_type ty  = yices_mk_type(ctx, "int");
  xdecl = yices_get_var_decl_from_name(ctx, name);
  if (!xdecl)
    xdecl = yices_mk_var_decl(ctx, name, ty);
  x = yices_mk_var_from_decl(ctx, xdecl);
  return x;
}


/* If taken_addr==1, emit the address of the tree.
   address of variable p :  __addr_p
   dereference of p :  __deref_p
*/

static yices_expr
emit_yices_expr_from_tree(yices_context ctx, tree expr)
{

  yices_type ty  = yices_mk_type(ctx, "int");
  yices_expr     x  ;
  yices_expr     y  ;
  yices_expr args[3];
  const char* addr_prefix = "&";
  int addr_prefix_len = strlen (addr_prefix);
  char name[100];

  switch (TREE_CODE(expr))
    {
    case INTEGER_CST :
      {
        x = yices_mk_num(ctx, int_cst_value (expr));
        return x;
      }
    case SSA_NAME :
    case VAR_DECL :
      {
        get_expr_string (expr, name);
        x = make_var_from_decl_name (ctx, name);
        return x;

      }

    case ADDR_EXPR:
      {
        get_expr_string (expr, name);
        x = make_var_from_decl_name (ctx, name);
        return x;
      }

    case MEM_REF :
      {
        get_expr_string (expr, name);
        x = make_var_from_decl_name (ctx, name);
        return x;
      }

    case ARRAY_REF :
      {
        get_expr_string (expr, name);
        x = make_var_from_decl_name (ctx, name);
        return x;


      }

    case COMPONENT_REF:
      {
        get_expr_string (expr, name);
        x = make_var_from_decl_name (ctx, name);
        return x;
      }

    case PLUS_EXPR :
    case POINTER_PLUS_EXPR:
      {
        x = emit_yices_expr_from_tree(ctx, TREE_OPERAND(expr,0));
        y = emit_yices_expr_from_tree(ctx, TREE_OPERAND(expr,1));
        if (!x || !y)
          return NULL;
        args[0] = x;
        args[1] = y;
        yices_expr	e  = yices_mk_sum (ctx, args, 2);  /* x + y */
        return e;

      }

    case MINUS_EXPR :
      {
        x = emit_yices_expr_from_tree(ctx, TREE_OPERAND(expr,0));
        y = emit_yices_expr_from_tree(ctx, TREE_OPERAND(expr,1));
        if (!x || !y)
          return NULL;
        args[0] = x;
        args[1] = y;
        yices_expr	e  = yices_mk_sub (ctx, args, 2);  /* x - y */
        return e;

      }

    case MULT_EXPR :
      {
        //			return NULL;
        x = emit_yices_expr_from_tree(ctx, TREE_OPERAND(expr,0));
        y = emit_yices_expr_from_tree(ctx, TREE_OPERAND(expr,1));
        if (!x || !y)
          return NULL;
        args[0] = x;
        args[1] = y;
        yices_expr	e  = yices_mk_mul (ctx, args, 2);  /* x * y */
        return e;
      }

    case BIT_XOR_EXPR :
      {
        x = emit_yices_expr_from_tree(ctx, TREE_OPERAND(expr,0));
        y = emit_yices_expr_from_tree(ctx, TREE_OPERAND(expr,1));
        if (!x || !y)
          return NULL;
        yices_expr	e  = yices_mk_bv_xor (ctx, x, y);  /* x ^ y */
        return e;
      }

    case BIT_AND_EXPR :
      {
        x = emit_yices_expr_from_tree(ctx, TREE_OPERAND(expr,0));
        y = emit_yices_expr_from_tree(ctx, TREE_OPERAND(expr,1));
        if (!x || !y)
          return NULL;
        yices_expr	e  = yices_mk_bv_and (ctx, x, y);  /* x & y */
        return e;
      }

    case BIT_IOR_EXPR :
      {
        x = emit_yices_expr_from_tree(ctx, TREE_OPERAND(expr,0));
        y = emit_yices_expr_from_tree(ctx, TREE_OPERAND(expr,1));
        if (!x || !y)
          return NULL;
        yices_expr	e  = yices_mk_bv_or (ctx, x, y);  /* x ^ y */
        return e;
      }

    case NEGATE_EXPR:
      {
        x = yices_mk_num(ctx, 0);
        y = emit_yices_expr_from_tree(ctx, TREE_OPERAND(expr,0));
        if (!x || !y)
          return NULL;
        args[0] = x;
        args[1] = y;
        yices_expr	e  = yices_mk_sub (ctx, args, 2);  /* 0 - x */
        return e;
      }

    case NOP_EXPR:
    case CONVERT_EXPR:
      return emit_yices_expr_from_tree(ctx, TREE_OPERAND(expr,0));

    case EXACT_DIV_EXPR:
    case TRUNC_DIV_EXPR:
    case TRUNC_MOD_EXPR:
    case RSHIFT_EXPR :
    case LSHIFT_EXPR :
    case EQ_EXPR :
		case MAX_EXPR :
		case MIN_EXPR :
      return NULL;



    default:
      fprintf(stderr, "UNKNOWN OPERATOR\n");
      gcc_assert(false);
      break;

    }
}


extern bool search_in_ref(tree expr, tree lhs);


/* check whether symbols in A all appear in B */

inline bool
check_symbol_version(tree a, tree b)
{
  switch ( TREE_CODE(a) )
    {
      /* atomic symbol*/
    case VAR_DECL :
    case PARM_DECL :
    case SSA_NAME :
    case ADDR_EXPR:
    case INDIRECT_REF:
    case MEM_REF :
    case ARRAY_REF:
    case COMPONENT_REF :
      return search_in_ref (b, a) ;

    case INTEGER_CST :
      return true;

    default:
      {
        int i;
        int n = TREE_OPERAND_LENGTH(a);
        for (i=0; i < n; i++)
          {
            tree opnd = TREE_OPERAND (a,i);
            if (!opnd)
              continue;
            if ( !check_symbol_version( opnd, b) )
              return false;
          }

        return true;
      }
    }
}


/* Invoke SMT Solver YICES to prove whether the runtime value of a equals to that of b*/
inline bool
value_equal_p(tree a, tree b)
{
  if ( operand_equal_p (a, b, 0) )
    return true;

  /* if A and B contains different symbols, return false */
  if (!check_symbol_version(a,b) || !check_symbol_version(b,a))
    return false;

  bool result = false;
  yices_context	ctx = yices_mk_context();
  yices_expr     x = emit_yices_expr_from_tree (ctx, a);
  yices_expr     y = emit_yices_expr_from_tree (ctx, b) ;

  if (!x || !y)
    return false;

  yices_expr     e = yices_mk_diseq(ctx, x, y);
  yices_assert(ctx, e);
  if ( yices_check(ctx) == l_false )
    result = true;
  yices_del_context(ctx);
  return result;
}


/* Invoke SMT Solver YICES to prove whether the runtime address of a less than that of b*/
static bool
value_less_p(tree a, tree b)
{
  if ( operand_equal_p (a, b, 0) )
    return false;

  /* if A and B contains different symbols, return false */
  if (!check_symbol_version(a,b) || !check_symbol_version(b,a))
    return false;

  bool result = false;
  yices_context	ctx = yices_mk_context();
  yices_expr		 x		 = emit_yices_expr_from_tree (ctx, a);
  yices_expr		 y		 = emit_yices_expr_from_tree (ctx, b);

  if (!x || !y)
    return false;

  yices_expr     e     = yices_mk_ge(ctx, x, y);
  yices_assert(ctx, e);
  if ( yices_check(ctx) == l_false )
    result = true;
  yices_del_context(ctx);
  return result;
}


/* Invoke SMT Solver YICES to prove whether the runtime address of a less than or equals to that of b*/
static bool
value_less_or_equal_p(tree a, tree b)
{

  if ( operand_equal_p (a, b, 0) )
    return true;

  /* if A and B contains different symbols, return false */
  if (!check_symbol_version(a,b) || !check_symbol_version(b,a))
    return false;

  bool result = false;
  yices_context	ctx = yices_mk_context();
  yices_expr		 x		 = emit_yices_expr_from_tree (ctx, a);
  yices_expr		 y		 = emit_yices_expr_from_tree (ctx, b);

  if (!x || !y)
    return false;

  yices_expr     e     = yices_mk_gt(ctx, x, y);
  yices_assert(ctx, e);
  if ( yices_check(ctx) == l_false )
    result = true;
  yices_del_context(ctx);
  return result;
}


struct loop_info
{
  char file_name[100];
  char func_name[100];
  int  lineno;
};




class exposed_ref_summary
{
private:
  symbolic_range_list _up_read;         /* up-expose read */
  symbolic_range_list _up_write;        /* up-expose write */


public:
  symbolic_range_list& 	up_read()  { return _up_read; }
  symbolic_range_list& 	up_write()  { return _up_write; }

  void copy(const exposed_ref_summary &src)
  {
    _up_read = (src._up_read);
    _up_write = (src._up_write);
  }

  void clear()
  {
    _up_read.clear();
    _up_write.clear();
  }

  void clear_changed()
  {
    _up_read.reset_changed();
    _up_write.reset_changed();
  }

  bool changed()
  {
    if ( _up_read.changed() )
      return true;
    if ( _up_write.changed() )
      return true;
    return false;
  }
};


typedef exposed_ref_summary exposed_ref_summary_t;

inline exposed_ref_summary_t*
new_summary ()
{
  return new exposed_ref_summary_t();
}


enum func_flag {
  FF_MALLOC = 0x1,
  FF_REALLOC = 0x2,
  FF_FREE = 0x4,
	FF_MUST_EXECUTE 		= 0x10,/* whether the function must be executed	*/
	FF_NEW  = 0x20,	  /* newly generated function */
};


struct allocation_routine
{

	int retalias; 	/* The arg id of the original pointer for a realloc */
	int alloc_size; /* The arg id of the size of the allocation site*/

	allocation_routine() :
		retalias(-1),
		alloc_size(-1)		
		{}

};

class func_extra_info_t
{
public:

  /* control dependences (Reverse dominator frointiers) */
  bitmap_head *reverse_dfs;
  int *postorder;               /* The current set of basic blocks
                                   in reverse top order.  */
  int *postorder_inverted;      /* The current set of basic blocks
                                   in reverse top order of inverted CFG.  */
  int n_blocks;                 /* The number of blocks in reverse toporder.	*/
  int n_blocks_inverted;        /* The number of blocks
                                   in reverse toporder of inverted CFG.  */
  exposed_ref_summary *exposed_refs;
  std::set<loop_p> inner_loops; /* The inner most loops which the function may be nested in */

  bitmap	global_read;
  bitmap	global_write;

  int count;
  int callers;		/* number of callers */
  int callees;		/* number of callees */
  int flags;

  bitmap   local_dr_read;   // direct read, without any callee
  bitmap   local_dr_write;

  bitmap   dr_read;   // local read, include callees
  bitmap   dr_write;

	tree		 thread_id;	 // the declartion of thread_id var

	allocation_routine alloc;

  func_extra_info_t ()  :
    inner_loops(),
    exposed_refs(NULL),
    flags(0),
    local_dr_read(NULL),
    local_dr_write(NULL),
    reverse_dfs(NULL)
  {
    dr_read = BITMAP_ALLOC (NULL);
    dr_write = BITMAP_ALLOC (NULL);
  }

	~func_extra_info_t()
	{
		BITMAP_FREE(dr_read);
		BITMAP_FREE(dr_write);

		if (local_dr_read)
			BITMAP_FREE(local_dr_read);
		if (local_dr_write)
			BITMAP_FREE(local_dr_write);

		if (exposed_refs)
			delete exposed_refs;	

		if (reverse_dfs)
			XDELETEVEC(reverse_dfs);

		if (postorder)
			XDELETEVEC(postorder);
		
		if (postorder_inverted)
			XDELETEVEC(postorder_inverted);
		
	}

  bool in_loop() const { return !inner_loops.empty(); }

};


static inline struct func_extra_info_t*
func_extra_info(const cgraph_node_ptr node)
{
  return (struct func_extra_info_t*)(node->extra_info);
}


static inline exposed_ref_summary_t*
func_exposed_refs(cgraph_node_ptr node)
{
  return func_extra_info(node)->exposed_refs;
}


static inline symbolic_range_list&
func_up_read(cgraph_node_ptr node)
{
  return func_extra_info(node)->exposed_refs->up_read();	/* up-expose read */
}


static inline symbolic_range_list&
func_up_write(cgraph_node_ptr node)
{
  return func_extra_info(node)->exposed_refs->up_write();	/* up-expose read */
}

static inline bool	func_must_execute (cgraph_node_ptr node) 
{ 
  return  func_extra_info(node)->flags & FF_MUST_EXECUTE; 
}
static inline void	func_set_must_execute(cgraph_node_ptr node) 
{
  func_extra_info(node)->flags |= FF_MUST_EXECUTE;
}
static inline void	func_reset_must_execute(cgraph_node_ptr node) 
{ 
  func_extra_info(node)->flags &= ~FF_MUST_EXECUTE;
}

static inline tree func_thread_id(cgraph_node_ptr node)
{
	return func_extra_info(node)->thread_id;
}

static inline void	func_set_thread_id(cgraph_node_ptr node, tree thread_id) 
{ 
  func_extra_info(node)->thread_id = thread_id;
}

static inline bool	func_new(cgraph_node_ptr node) 
{ 
  return  func_extra_info(node)->flags & FF_NEW; 
}
static inline void	func_set_new(cgraph_node_ptr node) 
{
  func_extra_info(node)->flags |= FF_NEW;
}
static inline void	func_reset_new(cgraph_node_ptr node) 
{ 
  func_extra_info(node)->flags &= ~FF_NEW;
}


inline void
initialize_func_extra_info(struct cgraph_node *node)
{
  struct func_extra_info_t *info=  new func_extra_info_t();
  struct cgraph_edge *edge;
  info->count = 0;
  for (edge = node->callees; edge; edge = edge->next_callee)
    info->callees++;
  for (edge = node->callers; edge; edge = edge->next_caller)
    info->callers++;
  node->extra_info = (unsigned long long)info;

}

typedef std::vector<ipa_data_reference*> data_reference_vector;

typedef std::vector<data_reference_vector> data_reference_bucket;

typedef std::map<int, data_reference_set> data_reference_class_map;

typedef std::map<ipa_data_reference*, ipa_data_reference*> data_reference_map;

typedef std::map<unsigned long, unsigned long> UINT32_MAP;


class loop_extra_info_t
{
public:
	
  /* True if the loop is heavy.  */
  bool heavy;

  /* The function where the loop is */
  tree func_decl;

  /* Global id, indexed from 1 */
  int  uid;

	/* Iteration number */
	tree iter_var; 

	loop_p loop;

  int  linenum;

  int  startline;
  int  endline;

  exposed_ref_summary_t   *exposed_refs;

  exposed_ref_summary_t	  *loop_site_exposed_refs;

	data_reference_class_map	dr_equivalence_classes;

  bitmap   dr_read;   // local read, include callees
  bitmap   dr_write;

  /* possible loop-carried dependencies */
  data_dependence_set loop_carried_deps;

  /* possible loop-independent dependencies */
  data_dependence_set loop_independent_deps;

  /* profiled loop-carried dependencies */
  data_dependence_set profiled_loop_carried_deps;

  /* profiled loop-independent dependencies */
  data_dependence_set profiled_loop_independent_deps;


  loop_extra_info_t():
    exposed_refs(NULL), loop_site_exposed_refs(NULL), uid(0)
  {
  	linenum = startline = endline = 0;
    dr_read = BITMAP_ALLOC (NULL);
    dr_write = BITMAP_ALLOC (NULL);
  }

		
	~loop_extra_info_t()
	{
		BITMAP_FREE(dr_read);
		BITMAP_FREE(dr_write);
		if (exposed_refs)
			delete exposed_refs;
		if (loop_site_exposed_refs)
			delete loop_site_exposed_refs;
	}

};


static inline struct loop_extra_info_t*
loop_extra_info(loop_p loop)
{
  return (struct loop_extra_info_t*)(loop->extra_info);
}


static inline tree
loop_iter_var(loop_p loop)
{
  return loop_extra_info(loop)->iter_var;
}


static inline void
set_loop_iter_var(loop_p loop, tree i)
{
  loop_extra_info(loop)->iter_var = i;
}

static inline int
loop_uid(loop_p loop)
{
  return loop_extra_info(loop)->uid;
}


static inline void
set_loop_uid(loop_p loop, int i)
{
  loop_extra_info(loop)->uid = i;
}

static inline tree
loop_func_decl(loop_p loop)
{
  return loop_extra_info(loop)->func_decl;
}


static inline void
set_loop_func_decl(loop_p loop, tree f)
{
  loop_extra_info(loop)->func_decl = f;
}


static inline int
loop_linenum(loop_p loop)
{
  return loop_extra_info(loop)->linenum;
}


static inline void
set_loop_linenum(loop_p loop, int i)
{
  loop_extra_info(loop)->linenum = i;
}

static inline int
loop_endline(loop_p loop)
{
  return loop_extra_info(loop)->endline;
}


static inline void
set_loop_endline(loop_p loop, int i)
{
  loop_extra_info(loop)->endline= i;
}

static inline int
loop_startline(loop_p loop)
{
  return loop_extra_info(loop)->startline;
}


static inline void
set_loop_startline(loop_p loop, int i)
{
  loop_extra_info(loop)->startline= i;
}


static inline exposed_ref_summary*
loop_exposed_refs(loop_p loop)
{
  return loop_extra_info(loop)->exposed_refs;
}


static inline symbolic_range_list&
loop_up_read(loop_p loop)
{
  return loop_exposed_refs(loop)->up_read();	/* up-expose read */
}

static inline symbolic_range_list&
loop_up_write(loop_p loop)
{
  return loop_exposed_refs(loop)->up_write();	/* up-expose read */
}



enum bb_extra_flags_t {
	BF_INVALID					= 0x1,/* whether the bb needs to be analyzed  */
	BF_MUST_EXECUTE 		= 0x2,/* whether the bb must be executed	*/	
	BF_PROMOTED 		    = 0x4,/* whether the bb has been promoted for thread space 	*/
};

class bb_extra_info_t
{
public:
	
	int uid;						  /* Global id, indexed from 1 */
  int preorder;         /* pre-order id*/
  int postorder;        /* post-order id */
	long double count;		/* execution number */
	tree decl;						/* function decl */

  /* the current transformed ref from its successor */
  symbolic_range_list in_ref;
  /* the current transformed propagated to its predecessor.
     NULL_TREE - haven't been checked,
     chrec_known - have no expose ref  */
  int flags;

	bb_extra_info_t():
		uid(0),
		flags(0),
		preorder(0),
		postorder(0),
		in_ref(),
		count(0),
		decl(NULL_TREE)
			{}


};


static inline struct bb_extra_info_t*
bb_extra_info(basic_block bb)
{
  return (struct bb_extra_info_t*)(bb->extra_info);
}


static inline int
bb_uid(basic_block bb)
{
  return bb_extra_info(bb)->uid;
}


static inline void
set_bb_uid(basic_block bb, int i)
{
  bb_extra_info(bb)->uid = i;
}


static inline long double
bb_count(basic_block bb)
{
  return bb_extra_info(bb)->count;
}


static inline void
set_bb_count(basic_block bb, long double i)
{
  bb_extra_info(bb)->count = i;
}

static inline tree
bb_func(basic_block bb)
{
  return bb_extra_info(bb)->decl;
}

static inline void
set_bb_func(basic_block bb, tree f)
{
  bb_extra_info(bb)->decl = f;
}


static inline bool	bb_invalid (basic_block bb) 
{ 
  return  bb_extra_info(bb)->flags & BF_INVALID; 
}
static inline void	bb_set_invalid( basic_block bb) 
{
  bb_extra_info(bb)->flags |= BF_INVALID;
}
static inline void	bb_reset_invalid( basic_block bb) 
{ 
  bb_extra_info(bb)->flags &= ~BF_INVALID;
}

static inline bool	bb_must_execute (basic_block bb) 
{ 
  return  bb_extra_info(bb)->flags & BF_MUST_EXECUTE; 
}
static inline void	bb_set_must_execute( basic_block bb) 
{
  bb_extra_info(bb)->flags |= BF_MUST_EXECUTE;
}
static inline void	bb_reset_must_execute( basic_block bb) 
{ 
  bb_extra_info(bb)->flags &= ~BF_MUST_EXECUTE;
}

static inline bool	bb_promted (basic_block bb) 
{ 
  return  bb_extra_info(bb)->flags & BF_PROMOTED; 
}
static inline void	bb_set_promted( basic_block bb) 
{
  bb_extra_info(bb)->flags |= BF_PROMOTED;
}
static inline void	bb_reset_promted( basic_block bb) 
{ 
  bb_extra_info(bb)->flags &= ~BF_PROMOTED;
}


extern bool
ipa_dr_may_alias_p (const ipa_data_reference *a, const ipa_data_reference *b);

extern bool
ipa_dr_may_alias_p (tree a, tree b);


class statement_extra_info
{
  data_reference_vector	_dr_vec;

public:

  statement_extra_info() {}

  data_reference_vector& dr_vec() { return _dr_vec; }
};



static inline statement_extra_info*
gimple_extra_info(gimple g)
{
  return (statement_extra_info*)(g->gsbase.extra_info);
}

static inline void
gimple_set_extra_info(gimple g, statement_extra_info *info)
{
  g->gsbase.extra_info = (unsigned long long)info;
}

static inline ipa_data_reference_p
tree_data_reference(gimple stmt, unsigned int opnd)
{
  statement_extra_info *sinfo = gimple_extra_info(stmt);
	if (!sinfo)
		return NULL;
  data_reference_vector	&dr_vec = sinfo->dr_vec();
  if (dr_vec.size() <= opnd )
    return NULL;
  ipa_data_reference_p  dr =	dr_vec[opnd] ;	
	if ( !dr || dr->invalid() )
		return NULL;

	return dr;
}

static inline bool
is_main_procedure(cgraph_node_ptr node)
{
	return strcmp(cgraph_node_name(node), "main") == 0 ;
}


static inline ipa_data_reference_p
symee_get_ipa_data_reference_for_tree (gimple stmt, tree ref)
{
  for (unsigned int i = 0; i < gimple_num_ops (stmt); i++)
    if (gimple_op (stmt, i) == ref)
      return tree_data_reference (stmt, i);
}

static inline ipa_data_reference_p
symee_get_ipa_data_reference_for_data_reference (struct data_reference *ref)
{
  for (unsigned int i = 0; i < gimple_num_ops (ref->stmt); i++)
    if (gimple_op (ref->stmt, i) == ref->ref)
      return tree_data_reference (ref->stmt, i);
}

extern bool is_profiling_library(const char *name);
extern bool is_replaced_function(const char *name);



inline std::string ExtractFilename( const std::string& path )
{
  return path.substr( path.find_last_of( '/' ) +1 );
}

inline bool
is_profiling_libaray_function(cgraph_node_ptr node)
{
	switch_to_context(node->decl);
	std::string filename = ExtractFilename( DECL_SOURCE_FILE (node->decl));
	bool ret = is_profiling_library(filename.c_str());	
	switch_off_context ();
	return ret;
}

inline bool
valid_function_node_p(cgraph_node_ptr node)
{
  if (!node )
		return false;
	
  if ( cgraph_function_body_availability(node) < AVAIL_OVERWRITABLE )
     //  || node->global.inlined_to )
    return false;

  /* Nodes without a body are not interesting.  */
  if (!gimple_has_body_p (node->decl)  || node->clone_of)
    return false;

	const char *name = cgraph_node_name(node);

	if ( strcmp(name, "atoi")==0 )
		return false;
	if ( strcmp(name, "strtol")==0 )
		return false;
	
	if ( strcmp(name, "atof")==0 )
		return false;
	if ( strcmp(name, "strtod")==0 )
		return false;

	if ( strcmp(name, "__strpbrk_c3")==0 )
		return false;	

	if ( strcmp(name, "strtoul")==0 )
		return false;	
	
	if ( is_profiling_libaray_function(node) )
		return false;

  return true;
}


typedef std::vector<int> ID_VECTOR;
typedef std::map<int, int> ID_MAP;

class ALIAS_CLASS_INFO
{
public:
  int 	id;		// class id
  ID_VECTOR		memops;
	TREE_SET		types;
  long double 	instruction_count;  // instruction count
  long long 	address_count;  // memory accessed count
  bool	strided;

  ALIAS_CLASS_INFO() :
    strided (true),
    memops(),
    id(0),
    instruction_count(0),
    address_count(0)
  {}
};


typedef std::map<int, ALIAS_CLASS_INFO> ALIAS_CLASS_MAP;
extern ALIAS_CLASS_MAP alias_class_map;


/* formal parameters that are being analyzed for loop-invariant definitions */
extern bitmap	formal_checking;

/* formal variables that have been analyzed for loop-invariant definitions.
   The analyzed results are stored in FORMAL_WIRTE */
extern bitmap	formal_checked;

extern htab_t	slice_hash;
extern htab_t	dr_hash;


extern data_dependence_set   loop_carried_dependencies;
extern data_dependence_set   loop_independent_dependencies;
extern data_dependence_set   must_loop_carried_dependencies;
extern data_dependence_set   must_loop_independent_dependencies;

/* Compares database elements E1 and E2.  */

int eq_slice_id(const void *e1, const void *e2);
hashval_t hash_slice_id (const void *elt);
void hash_slice_del (void *e);

int eq_dr_id(const void *e1, const void *e2);
hashval_t hash_dr_id (const void *elt);
void hash_dr_del (void *e);

typedef void (*ProfileMalloc) (gimple_stmt_iterator *, tree, tree, tree, int);
typedef void (*ProfileMemop) (gimple_stmt_iterator *, tree, int, int);
typedef void (*ProfileStmt) (gimple_stmt_iterator *);
typedef void (*ProfileLoopedge) (int, int, edge);
typedef void (*ProfileExit) (gimple_stmt_iterator *);
typedef void (*ProfileLoopheader) (int, gimple_stmt_iterator *);
typedef void (*ProfileBB) (basic_block bb);

void profile_malloc (struct cgraph_node *node, ProfileMalloc func);
void profile_memop (struct cgraph_node *node, ProfileMemop func);
void profile_stmt (struct cgraph_node *node, ProfileStmt func);
void profile_edge (struct cgraph_node *node, ProfileLoopedge func);
void profile_exit (struct cgraph_node *node, ProfileExit func);
void profile_bb (struct cgraph_node *node, ProfileBB func);
void profile_begin(struct cgraph_node *node, ProfileExit func);
void profile_end(struct cgraph_node *node, ProfileExit func);

HOST_WIDE_INT field_byte_offset (const_tree decl);
tree tree_build_address_expr (tree mem);
tree	get_base_pointer (tree ref);
tree  dr_var_name (ipa_data_reference * dr, const char *prefix);
tree  ipa_add_new_external_global (tree type, tree name);

extern htab_t loop_hash;
extern loop_p ipa_get_loop(int loop_uid);



inline const char* 
fn_decl_name(tree decl)
{
  switch_to_context (decl);
  const char *name = lang_hooks.decl_printable_name (decl, 2);
  switch_off_context();
	return name;
}


/* This function inserts NEW_DECL to varpool.  */

inline void
insert_global_to_varpool (tree new_decl)
{
  struct varpool_node *new_node;

  new_node = varpool_node (new_decl);
  notice_global_symbol (new_decl);
  varpool_mark_needed_node (new_node);
  varpool_finalize_decl (new_decl);
}

/* If X has a smaller reverse topological sort number than Y, returns -1;
   if greater, returns 1.  */
static inline int
bb_post_order_comparator (const void *x, const void *y)
{
  basic_block bb1 = *(const basic_block *) x;
  basic_block bb2 = *(const basic_block *) y;

  gcc_assert (bb1 == bb2
	      || bb_extra_info (bb1)->postorder
	      != bb_extra_info (bb2)->postorder);

  /* It's a reverse topological order in REV_TOP_ORDER_INDEX, so
     bbs with less number should go earlier.  */
  if (bb_extra_info (bb1)->postorder < bb_extra_info (bb2)->postorder)
    return -1;
  else
    return 1;
}

void ipa_dump_data_references_in_stmt (gimple stmt, FILE * fp, int indent=0);

extern void check_expose_refs_for_function( cgraph_node_ptr node );
extern void union_expose_references_for_recursion(std::set<cgraph_node_ptr> scc, exposed_ref_summary &summary);
extern bool ipa_ref_may_alias_p (tree ref1, tree ref2);
extern bool ipa_ref_contains_symbols_defined_in_loop (struct loop *nest, tree ref, gimple stmt);
extern bool ipa_object_address_invariant_in_loop_p ( struct loop *loop, tree obj, gimple stmt);
extern void enter_loop_independent_dependency( int loop, ipa_data_reference *a, ipa_data_reference *b, bool must);
void ipa_add_abnormal_goto_call_edges (gimple_stmt_iterator gsi);
void	ipa_partition_memory_operations();
void ipa_partition_dependence_edges(FILE *fp);
void ipa_statistic_dependence_edges();
void ipa_collect_data_dependencies (bool check);

extern void	ipa_dump_data_references (FILE *fp);
extern tree	ipa_create_data_references (FILE *fp);
void ipa_dump_basic_blocks (FILE * fp);
void ipa_dump_call_sites (FILE * fp);
void ipa_dump_source_info (FILE * fp);
unsigned int ipa_ssa_loop_init_node (FILE * fp, cgraph_node_ptr node);
extern unsigned int ipa_ssa_loop_init (FILE *fp);
extern unsigned int ipa_ssa_loop_done (void);
extern unsigned int ipa_gimple_id_assignment (FILE *fp);
void ipa_update_bb_info ();
void ipa_update_loop_info ();
void update_new_call_edge (struct cgraph_node *node);

extern void ipa_collect_data_references_for_loop (struct loop *loop, FILE *fp);
extern void ipa_update_alias_class_for_loop(struct loop *loop, FILE *fp);
void dump_data_references_for_loop (struct loop *loop, FILE *fp);

extern void cgraph_resolve_indirect_edges ();
extern void annotate_inside_loops();
extern unsigned int ipa_mod_ref_analysis ();
extern unsigned int ipa_expose_reference_analysis(const	std::set<cgraph_node_ptr> &scc, FILE *fp );
extern void collect_data_references_to_be_profiled (const std::set<cgraph_node_ptr> &scc );
extern void release_resouces_of_callees(const std::set<cgraph_node_ptr> &scc );
extern void release_resouces_of_callees(const cgraph_node_ptr node );
extern int compute_access_pattern (void **slot, void *data ATTRIBUTE_UNUSED);
extern int print_dr (void **slot, void *data);
int count_profile_candidate(void **slot, void *data);
extern tree gimple_build_addr_expr (gimple_stmt_iterator * gsi, tree mem);
bool profiling_dr_not_alias_p (const ipa_data_reference *a, const ipa_data_reference *b);
void  build_alias_oracle();
void  read_count_information();
void  read_count_infor_and_remove_bb();
void  read_count_infor_and_invalidate_dr();
void	ipa_classify_memory_operations(FILE *fp);
int compute_reference_set (void **slot, void *data ATTRIBUTE_UNUSED);
void recognize_mem_allocate_functions(FILE *fp);
void read_mem_allocate_func( const char *name);
struct ptr_info_def * get_reference_set (tree ref);
void get_reference_set (tree mem, struct ptr_info_def *pi);
void find_library_calls(FILE *fp);
bool is_shadow_profile_candidate(ipa_data_reference_p dr, tree ref);
bool is_profile_candidate(ipa_data_reference_p dr);
void specify_loops_to_profile();
void  find_candidate_loops(std::set<int> &loop_set);
void canonicalize_gimple_call();
void cgraph_remove_invalid_edge (struct cgraph_node *node);
void ipa_collect_data_references_in_stmt (gimple stmt, bool analyzable, FILE * fp);
unsigned int ipa_flatten_structural_assignments ();
void Get_type_name (tree t, std::string &name);
void collect_callees(cgraph_node_ptr node, IPA_NODE_SET &callee_set);
void collect_loop_callees(struct loop *loop, IPA_NODE_SET &callee_set);
void mark_min_loop_level (loop_p loop, IPA_NODE_INT_MAP &node_map);
void ipa_convert_icall();
void gimple_duplicate_extra_info(struct function *nfun, gimple nstmt, struct function *ofun, gimple ostmt);

extern unsigned int remove_upp_directives();
extern int loop_dr_count;
extern std::set < int > loop_set;


extern "C"
{
  void  print_var_set (bitmap vars, FILE *fp);
  void  translate_var_set (bitmap vars, bitmap global);
  bitmap look_up_points_to_set (tree p);
  void	ipa_points_to_analysis (void);
  void delete_points_to_sets (void);
  int look_up_var (tree p);
  void 	look_up_fields (tree t, struct ptr_info_def *struct_pi, struct ptr_info_def *pi);
  VEC(tree, heap) *ipa_get_vector_of_formal_parms (tree fndecl);
	void dump_function_to_file (tree fn, FILE *file, int flags);
	void 	cgraph_materialize_clone (struct cgraph_node *node);
	bool 	copy_edges_for_bb (basic_block bb, gcov_type count_scale, basic_block ret_bb);	
	void initialize_cfun (tree new_fndecl, tree callee_fndecl, gcov_type count);	
	tree array_bottom_element_type (tree exp);
}




inline void
dump_all_nodes( const char *fpname )
{
  FILE *fp = fopen(fpname, "w");
  for (cgraph_node_ptr node = cgraph_nodes; node; node = node->next)
  {
    if (!valid_function_node_p (node))
      continue;
    switch_to_context(node->decl);    
    dump_function_to_file (node->decl, fp, TDF_VOPS);
    switch_off_context();
  }
  fclose(fp);
}


inline void
update_cgraph()
{
  /* Cleanup. */
  for (struct cgraph_node *node = cgraph_nodes; node; node = node->next)
  {
    if (!valid_function_node_p (node))
      continue;
    switch_to_context(node->decl);    
		
		rewrite_into_loop_closed_ssa (NULL, TODO_update_ssa);
    update_ssa (TODO_update_ssa);
    cgraph_remove_invalid_edge (node);
    update_new_call_edge (node);
		ipa_update_bb_info ();
			
    verify_cgraph_node (node);    
    verify_flow_info ();
    verify_dominators (CDI_DOMINATORS);
    verify_loop_structure ();
    verify_ssa (true);
		
    switch_off_context();
  }
}


inline void
cleanup_cgraph()
{
  /* Cleanup. */
  for (struct cgraph_node *node = cgraph_nodes; node; node = node->next)
  {
    /* Get rid of the aux information.      */
    if (node->aux)
    {
      free (node->aux);
      node->aux = NULL;
    }
  }
}

static inline void
ipa_add_local_decl (struct function *fun, tree d)
{
	switch_to_context (fun->decl);
  //VEC_safe_push (tree, gc, fun->local_decls, d);  
	gimple_add_tmp_var (d);
	switch_off_context ();
}


#endif

