#include <string.h>
#include "instr-sourcec.h"

#define NONE_INSTR 0
#define INSTRED 1
#define MAXLEVEL 1000
seedlist criteria = NULL;

int funcnum = 0;
int funcID[10000]; //function id, currently assume there are no more than 10000 functions
 
int paranum = 0;
int writenum = 0;
int instrlevel = 0; //default all level 
struct Instr_Stat instr_stat;

static GTY (()) tree function_entry_fn_type;
static GTY (()) tree function_return_fn_type;
static GTY (()) tree atomic_start_fn_type;
static GTY (()) tree atomic_stop_fn_type;
static GTY (()) tree gv_update_fn_type;
static GTY (()) tree gv_reset_fn_type;
static GTY (()) tree record_memory_fn_type;
static GTY (()) tree inc_counter_fn_type;
static GTY (()) tree check_ipt_fn_type;
static GTY (()) tree read_memory_fn_type;

static const char *function_entry_instr = "__write_function_entry";
static const char *function_return_instr = "__write_function_return";
static const char *atomic_start_instr = "__start_atomic_control";
static const char *atomic_stop_instr = "__stop_atomic_control";
static const char *gv_update_instr = "__gv_update";
static const char *gv_reset_instr = "__gv_reset";
static const char *record_memory = "__record_memory";

//for replay only
static const char *inc_counter = "__inc_counter"; //increase the function counter
static const char *check_ipt = "__check_ipt"; // check whether an interrupt should be called
static const char *read_memory = "__read_memory";//read values from memory

static char * get_fname(int fid);

static void
initialize_instr_prototype ()
{

  function_entry_fn_type = build_function_type_list (void_type_node,/* return type */
						     integer_type_node,/* int fun_id */
						     NULL_TREE);
  function_return_fn_type = build_function_type_list (void_type_node, /* return type */
						      integer_type_node, /* int fun_id */
						      NULL_TREE);
  atomic_start_fn_type = build_function_type_list (void_type_node, /* return type*/
						   NULL_TREE);
  atomic_stop_fn_type = build_function_type_list (void_type_node, /* return type*/
						  NULL_TREE);
  gv_update_fn_type = build_function_type_list (void_type_node, /* return type*/
						NULL_TREE);
  gv_reset_fn_type = build_function_type_list (void_type_node, /* return type*/
					       NULL_TREE);

  record_memory_fn_type = build_function_type_list(void_type_node,
                                                   integer_type_node, /* memory address*/
						   ptr_type_node,     /* data */
						   integer_type_node, /* how many bytes*/
						   NULL_TREE);

  inc_counter_fn_type = build_function_type_list (void_type_node, /*return type*/
                                                  integer_type_node, /* int fn_counter */
                                                  NULL_TREE);

  check_ipt_fn_type = build_function_type_list (void_type_node, /*return type*/
                                                integer_type_node, /* int counter*/
                                                integer_type_node, /* int gv */
						integer_type_node, /* int ipt_code */
						NULL_TREE);
  read_memory_fn_type = build_function_type_list (ptr_type_node, /* return type*/
						  integer_type_node, /* memory address*/
						  integer_type_node, /* how many bytes*/
						  NULL_TREE);
}

static 
void check_bb_order( basic_block bb, FILE *fp){

  basic_block bb_next;
  edge e;
  edge_iterator ei;

   
  fprintf(fp, " < "); 
  FOR_EACH_EDGE(e, ei, bb->succs){

    bb_next = e->dest;
    
    struct Basic_Block_Extra *info = bb_extra_info(bb_next);
    if (info->visited) continue;

    info->visited = 1;
    fprintf(fp, "%d,",  bb_next->index);
    check_bb_order(bb_next, fp);
 
  }
  fprintf(fp, " > ");
  
  
}

static
void print_bb_order( tree func_decl){

  FILE *fp = fopen("bb_order.txt", "w");

  /*clear bb -> visited*/
  switch_to_context_new(func_decl);
  struct Basic_Block_Extra *info;
  basic_block bb;
  FOR_EACH_BB(bb){
    info = XNEWVEC(struct Basic_Block_Extra, 1);
    info->visited = 0;
    bb->extra_info = info;
  }

  switch_off_context_new();
  /* enter bb, always index = 0*/
  bb = ENTRY_BLOCK_PTR_FOR_FUNCTION(DECL_STRUCT_FUNCTION(func_decl)); 
  info = XNEWVEC(struct Basic_Block_Extra, 1);                          
  info->visited = 0;
  bb->extra_info = info;
  /* exit bb, always index = 1*/
  bb = EXIT_BLOCK_PTR_FOR_FUNCTION(DECL_STRUCT_FUNCTION(func_decl));
  info = XNEWVEC(struct Basic_Block_Extra, 1);
  info->visited = 0;
  bb->extra_info = info;

  check_bb_order(ENTRY_BLOCK_PTR_FOR_FUNCTION(DECL_STRUCT_FUNCTION(func_decl)), fp);

  fclose(fp);

}





/* the criteria is like:
   ___SLICE = ***;
 */
static
bool isSeed(gimple stmt){
  tree lhs;
  if (gimple_code(stmt) != GIMPLE_ASSIGN) return false;
  lhs = gimple_assign_lhs(stmt);
  if (TREE_CODE(lhs) != VAR_DECL && TREE_CODE(lhs) != SSA_NAME) return false; 
  //    && ( TREE_CODE(lhs) == SSA_NAME && TREE_CODE(SSA_NAME_VAR(lhs))!= VAR_DECL)) return false;
  
  if (TREE_CODE(lhs) == SSA_NAME) lhs = SSA_NAME_VAR(lhs); 
  if (!DECL_NAME(lhs)) return false; 
  if (strcmp(IDENTIFIER_POINTER(DECL_NAME(lhs)), "___SLICE")) return false;
  return true;
}

static 
void get_slice_criteria() {
  struct cgraph_node *node;
  seedlist newseed;
  for (node = cgraph_nodes; node; node = node->next)
    {
      switch_to_context_new (node->decl);
      if (cfun == NULL) continue;
      basic_block bb;
      gimple_stmt_iterator gsi;
      gimple stmt;


      FOR_EACH_BB (bb){
	
	for (gsi = gsi_start_bb (bb); !gsi_end_p (gsi); gsi_next (&gsi))
	  {
	    stmt = gsi_stmt (gsi);
	    if (isSeed(stmt)) {
	      newseed = XNEWVEC(struct Seed_List, 1);
	      newseed->id = stmt;
	      newseed->next = criteria;
	      criteria = newseed;
	    }
	    
	  }
      }
      switch_off_context_new();
    }    

}

 

/*
  1. assign each function an id for function instrumentation,
  each id is an integer
  2 dump a func.info file which contains the matching info
*/
static void function_id_assignment(FILE *fp){
  
  struct cgraph_node *node;

  funcnum = 0;
  for (node = cgraph_nodes; node; node = node->next){
    if (!valid_function(node)) continue;
    fprintf(fp, "UID %d\tFUNCTION %s\n", node->uid, cgraph_node_name(node));
    //funcID[node->uid] = NONE_INSTR;
    if (flag_instr_baseline) funcID[node->uid] = INSTRED; 
    else
      funcID[node->uid] = NONE_INSTR;
    funcnum++;
  }
  fprintf( fp, "TOTAL: %d \n", funcnum);
}



/* return the varible node with its id */
static
tree find_var_with_id(unsigned int id){ 

  struct varpool_node *node;
  tree decl;

  for (node = varpool_nodes; node; node = node->next) {
    decl = node->decl;
    if (id == DECL_UID(decl)) return decl;
  }

  return NULL;
}

/* wheather var is a global varible*/
bool is_global_1(tree var){

  
  struct varpool_node *node;
  tree decl;

  for (node = varpool_nodes; node; node = node->next) {
    if (var == DECL_NAME(node->decl)) return true;
  }

  return false;
  

}

bool is_global(tree var) {
  
  tree node; 

  if (var == NULL ) return false;
  switch (TREE_CODE(var)){
  case IDENTIFIER_NODE:
    if (is_global_1(var)) return true;
    break;

  case VAR_DECL:
    if (is_global_1(DECL_NAME(var))) return true;
    break;

  case SSA_NAME:
    node =  SSA_NAME_VAR(var);
    if (is_global(DECL_NAME(node))) return true;
    break;

  case PARM_DECL:
    return false;

  case ARRAY_REF: //a[i]
  case COMPONENT_REF: //a.b
    node = gimple_op(var,0);
    if (is_global(node)) return true;
    break;

  case INDIRECT_REF:
  case MEM_REF: //a->b, &a, *a
    return true;
    
  default:
    break;
  }

  return false;
}

static
bool is_global_op(gimple stmt){

  unsigned n;
  tree var;
  /* check the oprand one by one, if it contains global varibles, return true. 
   * Otherwise, return false
  */
  for ( n = 0; n < gimple_num_ops(stmt); n++) {
    tree oprand =  gimple_op( stmt, n);
    if (oprand == NULL) continue;
    switch (TREE_CODE(oprand)){
    case VAR_DECL:
      if (is_global(DECL_NAME(oprand))) return true;
      break;
    case SSA_NAME:
      var =  DECL_NAME(SSA_NAME_VAR(oprand));
      if (is_global(var)) return true;
      break;
    case PARM_DECL:
      return true;
    defalt:
      continue;
    }
  }
  return false;
}

/*decide whether volatile cast should be output*/
static int
if_volatile_cast(gimple gs){

  tree type = TREE_TYPE(gimple_assign_lhs(gs));
  if (TREE_CODE(type) == POINTER_TYPE) type = TREE_TYPE(type);
  unsigned int quals = TYPE_QUALS (type);
  if (!(quals & TYPE_QUAL_VOLATILE))
    return 0;

  if (gimple_num_ops(gs) != 2) return 0;
  tree rhs = gimple_assign_rhs1(gs);
  if (TREE_CODE(rhs) != INTEGER_CST) return 0;
  return 1;
}

static char *
get_type_name(tree type){

  enum tree_code_class tclass = TREE_CODE_CLASS (TREE_CODE (type));
  if (tclass != tcc_type) return NULL;

  if (TYPE_NAME (type)){
    if (TREE_CODE (TYPE_NAME (type)) == IDENTIFIER_NODE)
      return TYPE_NAME(type)->identifier.id.str;
    //pp_tree_identifier_c (buffer, TYPE_NAME (node));                                                                              
    else if (TREE_CODE (TYPE_NAME (type)) == TYPE_DECL
             && DECL_NAME (TYPE_NAME (type)))
      return DECL_NAME(TYPE_NAME(type))->identifier.id.str;
    //dump_decl_name (buffer, TYPE_NAME (type), flags);                                                                         
    else
      return NULL;
  }
  return NULL;
}

static int
read_from_memory (gimple stmt1, gimple stmt2){
  if (gimple_code(stmt1) != GIMPLE_ASSIGN) return 0;
  if (! if_volatile_cast(stmt1)) return 0;
  if (stmt2 == NULL || gimple_code(stmt2) != GIMPLE_ASSIGN) return 0;

  if (gimple_num_ops(stmt1) != 2 || gimple_num_ops(stmt2) != 2) return 0;

  //check lhs(stmt1) type                                                                                                           
  tree lhs, rhs, type;
  lhs = gimple_assign_lhs(stmt1);
  rhs = gimple_assign_rhs1(stmt1);
  type = TREE_TYPE(lhs);
  if (TREE_CODE(type) != POINTER_TYPE || TREE_CODE(TREE_TYPE(type)) != INTEGER_TYPE) return 0;
  if (strcmp(get_type_name(TREE_TYPE(type)), "uint16_t") != 0) return 0;
  //check rhs(stmt1)
                                                                                                              
  if (TREE_CODE(rhs) != INTEGER_CST) return 0;

  //check rhs(stmt2) should be lhs(stmt1)                                                                                          
 
  if (!gimple_has_volatile_ops(stmt2)) return 0;
  rhs = gimple_assign_rhs1(stmt2);
  if (TREE_CODE(rhs) != MEM_REF) return 0;

                                                                                                          
  return 1;
}


static 
void get_read_info(gimple stmt, gimple next_stmt, unsigned int *addr, int *size){
  tree lhs;
  tree rhs;

  lhs = gimple_assign_lhs(next_stmt);
  rhs = gimple_assign_rhs1(stmt);
  
  *size = TYPE_PRECISION(TREE_TYPE(lhs)) / 8;
  if (TREE_CODE(rhs) != INTEGER_CST) *addr = 0;
  else *addr = TREE_INT_CST_LOW(rhs);
  

}
 

/* record the value read directly from a memory address
 * eg. D_1 = (volatile uint16_t *) integer;
       D_2 = *D_1; 
*/
static 
void instr_reading_mem_directly(){
  gimple_stmt_iterator gsi;
  gimple stmt, next_stmt;
  basic_block bb;
  tree read_memory_fn = build_fn_decl(read_memory, read_memory_fn_type);
  tree record_memory_fn = build_fn_decl(record_memory, record_memory_fn_type);

  FOR_EACH_BB(bb) {

    for (gsi = gsi_start_bb(bb); !gsi_end_p(gsi); gsi_next(&gsi)){
      stmt = gsi_stmt(gsi);
      if (!gsi_one_before_end_p(gsi)) {
	gsi_next(&gsi);
	next_stmt = gsi_stmt(gsi);
	if (read_from_memory(stmt, next_stmt)){
	  //TODO write or read the value;
	  if (flag_instr_replay){
	    //for replay part, using reading stmt to replace those two stmts
	    tree func_params;
	    tree lhs;
	    tree tmp_var;
	    gimple new_stmt,call;
	    unsigned int m_addr; // the address of the memory, from rhs of stmt
	    int m_size; // how many bytes should be read, from lhs of stmt
	    //step1: get parameters for function __read_memory
	    get_read_info(stmt, next_stmt, &m_addr, &m_size);

	    //step2: remove the first assignment D_1 = (volatile uint16_t *) integer;
	    gsi_prev(&gsi); // now gsi points to stmt
	    gsi_remove(&gsi, true); //now gsi points to next_stmt
	    //step3: replace next_stmt as D_2 = __read_memory(addr, size);
	    func_params = tree_cons(NULL_TREE,build_int_cst(integer_type_node, 1),
                                    tree_cons(NULL_TREE,build_int_cst(integer_type_node, 2),NULL_TREE));
	    call = gimple_build_call(read_memory_fn, 2, 
				     build_int_cst(integer_type_node, m_addr), 
				     build_int_cst(integer_type_node, m_size), 
				     NULL_TREE);

	    lhs = gimple_assign_lhs(next_stmt);
	    gimple_call_set_lhs (call, lhs);
	    gsi_insert_before(&gsi, call, GSI_SAME_STMT);
	    gsi_remove(&gsi,true);
	    if (gsi_end_p(gsi)) break;
	  }else {
	    //for recording part, adding writing part.
	    gimple call;
	    tree rhs = gimple_assign_rhs1(stmt);
	    
	    call = gimple_build_call(record_memory_fn, 3, 
				     build_int_cst(integer_type_node, TREE_INT_CST_LOW(rhs)), 
				     gimple_assign_lhs(next_stmt), 
				     build_int_cst(integer_type_node,2), NULL_TREE);
	    gsi_insert_after(&gsi, call, GSI_NEW_STMT);
          }
	}else gsi_prev(&gsi);
      } //end of "if (!gsi..."
    }
      
  }
  
}

static
void instr_writing_replacement(){
  gimple_stmt_iterator gsi;
  gimple stmt, next_stmt;
  basic_block bb;
  
  tree record_memory_fn = build_fn_decl(record_memory, record_memory_fn_type);
  
  FOR_EACH_BB(bb){
    for (gsi = gsi_start_bb(bb); !gsi_end_p(gsi); gsi_next(&gsi)){
      stmt = gsi_stmt(gsi);
      if (gimple_code(stmt) != GIMPLE_ASSIGN) continue;
      tree lhs,arg0;
      lhs = gimple_assign_lhs(stmt);
      //arg0 = TREE_OPERAND(lhs, 0);
      if (TREE_CODE(lhs) != MEM_REF) continue;
      if (TREE_CODE(TREE_TYPE(lhs)) != INTEGER_TYPE) continue;
      arg0 = TREE_OPERAND(lhs, 0);
      if (TREE_CODE(arg0) != SSA_NAME) continue;  

      gimple rstmt = SSA_NAME_DEF_STMT(arg0);
      if (gimple_code(rstmt) != GIMPLE_ASSIGN) continue; 

      tree rhs = gimple_assign_rhs1(rstmt);
      if (TREE_CODE(rhs) != INTEGER_CST) continue;
      int index = TREE_INT_CST_LOW(rhs);
      gimple call;
      call = gimple_build_call(record_memory_fn, 3, build_int_cst(integer_type_node, index), 
			       gimple_assign_rhs1(stmt), build_int_cst(integer_type_node, 2), NULL_TREE);
      gsi_insert_before(&gsi, call, GSI_SAME_STMT);
      gsi_remove(&gsi, true);
    }//end of gsi iteration
  }

}

/*
In each function cfun, find out statement involves memory load/write, instr #gv_ref
e.g:
 a = 5;
=>
__atomic_start();
a = 5;
__gv_update();
__atomic_stop();   
*/
static 
void instr_mem_operations(){

  tree atomic_start_fn = build_fn_decl(atomic_start_instr, atomic_start_fn_type);
  tree atomic_stop_fn = build_fn_decl(atomic_stop_instr, atomic_stop_fn_type);
  tree gv_update_fn = build_fn_decl(gv_update_instr, gv_update_fn_type);
  gimple_stmt_iterator gsi;
  basic_block bb;
  FOR_EACH_BB(bb) {

    for (gsi = gsi_start_bb (bb); !gsi_end_p (gsi); gsi_next (&gsi))
      {
	gimple stmt = gsi_stmt (gsi);
	/* check is there any mem operation*/
	if (is_global_op( stmt )) {
	  //instr code
	  gimple call;
	  if (flag_instr_source || flag_instr_baseline || flag_instr_non_inline) {
	    // atomic control is only needed in record
	    call =  gimple_build_call(atomic_start_fn, 0, NULL_TREE);
	    gsi_insert_before(&gsi, call, GSI_SAME_STMT);
	  }
	  
	  call = gimple_build_call(gv_update_fn, 0, NULL_TREE);
	  gsi_insert_after(&gsi, call, GSI_NEW_STMT);

	  if (flag_instr_source || flag_instr_baseline || flag_instr_non_inline) {
	    // atomic control is only needed in record 
	    call = gimple_build_call(atomic_stop_fn, 0, NULL_TREE);
	    gsi_insert_after(&gsi, call, GSI_NEW_STMT);
	  }

	}
      }
  }
}

#if 0
bool 
def_handle (tree node, gimple stmt, void *data){

  return false;
}
 
/* 
 * find the use-def chain starting from node
*/
static 
void trace_use_def(tree node, int level){
  if (TREE_CODE(node) == SSA_NAME) {
    walk_use_def_chains( node, def_handle, NULL, true);
  }
}


static 
void start_instrumentation(int level){
  seedlist head = criteria;

  while (head != NULL) {
    /*for each SSA variable, find out all its use-def chain*/
    gimple stmt =  head->id;
    
    /* seed is ___SLICE = ***, it should be GIMPLE_ASSIGN*/
    if (gimple_code(stmt) != GIMPLE_ASSIGN){
      printf("illegle seed\n");
      continue;
    }
    
    switch (gimple_num_ops(stmt)) {
    case 2:
      trace_use_def(gimple_assign_rhs1(stmt), level);
      break;
    case 3:
      trace_use_def(gimple_assign_rhs1(stmt), level);
      trace_use_def(gimple_assign_rhs2(stmt), level);
      break;
    case 4:
      trace_use_def(gimple_assign_rhs1(stmt), level);
      trace_use_def(gimple_assign_rhs2(stmt), level);
      trace_use_def(gimple_assign_rhs3(stmt), level);
      break;
    default:
      break;
    }
    
    head = head->next;
  }

}
#endif

//dump the name of all instred functions, for replay use later
static void dump_instrued_function(){
  struct cgraph_node *node;
  int n = 0;
  FILE *fp = fopen("instrued_function.txt", "w");
  for (node = cgraph_nodes; node; node = node->next){
    if (func_mod_ref_info(node)->isInstr != NONE_INSTR){
      fprintf(fp, "%s\n", get_fname(node->uid));
      fprintf(fp, "%d\n", node->uid);
      n++;
    }
  }
  fprintf(fp, "##\n");
  fprintf(fp, "%d\n", n);
  fclose(fp);
}

// for replay, mark the functions for instrumenation
static void mark_instrued_function(){
  FILE *fp;
  struct cgraph_node *node;
  char name[50];
  int id;

  //initialize extra_info
  for (node = cgraph_nodes; node ; node =  node->next){
    struct Function_Mod_Ref_Info *info = XNEWVEC(struct Function_Mod_Ref_Info, 1);
    info->isInstr = NONE_INSTR;
    node->extra_info = info;
  }

  if (!(fp = fopen("instrued_function.txt","r"))) {
    //default is all functions are instrumented
    for (node = cgraph_nodes; node; node = node->next)
      func_mod_ref_info(node)->isInstr = INSTRED;
    return;
  }
  
  fscanf(fp, "%s\n", name);
  fscanf(fp, "%d\n", &id);
  while (name[0] != '#'){
    for (node = cgraph_nodes; node; node = node->next){
      if (!valid_function(node)) continue;
      if (strcmp(cgraph_node_name(node), name) == 0){
	func_mod_ref_info(node)->isInstr = INSTRED;
	funcID[node->uid] = id;
	break;
      }
    }
    fscanf(fp, "%s\n", name);
    fscanf(fp, "%d\n", &id);
  }
  fclose(fp);
}

//instrument function entry and return
static void instr_function_entry_exit () {
  
  struct cgraph_node *node;
  for (node = cgraph_nodes; node; node = node->next)
    {
      if (!valid_function(node)) continue;
      if (func_mod_ref_info(node)->isInstr == NONE_INSTR)
	continue;
      switch_to_context_new (node->decl);
      if ( cfun == NULL ) 
	continue;

      //instr_stat.instr_num_of_func ++;
      instr_mem_operations();
      
      //instr memory reading
      instr_reading_mem_directly();

      //instr memory writing for replay. 
      //In replay, writing to a memeory address will be replaced by writing to a variable
      if (flag_instr_replay)
	instr_writing_replacement();

      basic_block bb;
      gimple_stmt_iterator gsi;
      gimple stmt, stmt1;
      tree tree_write_function_entry_fn = build_fn_decl(function_entry_instr, function_entry_fn_type);
      tree tree_write_function_return_fn = build_fn_decl(function_return_instr, function_return_fn_type);
      tree tree_gv_reset_fn = build_fn_decl(gv_reset_instr, gv_reset_fn_type);
      gimple call;

      /* Entry
	1. set ___GLOBAL_COUNTER = 0, 
	2. record function entry
       */
      bb = cfun->cfg->x_entry_block_ptr->next_bb;
      gsi = gsi_start_bb (bb);

      if (flag_instr_source){
	call = gimple_build_call(tree_write_function_entry_fn, 1, build_int_cst(integer_type_node,node->uid), NULL_TREE);
	gsi_insert_before(&gsi, call, GSI_NEW_STMT);
      }
      
      if (flag_instr_replay){
	tree tree_inc_counter_fn = build_fn_decl(inc_counter, inc_counter_fn_type);
	call = gimple_build_call(tree_inc_counter_fn, 1, build_int_cst(integer_type_node, funcID[node->uid]), NULL_TREE);
	gsi_insert_before(&gsi, call, GSI_NEW_STMT);
      }

      call = gimple_build_call(tree_gv_reset_fn, 0, NULL_TREE);
      gsi_insert_before(&gsi, call, GSI_NEW_STMT);

      /* Exit
	 for each return statement, set ___GLOBAL_COUNTER = 0,
	 record function return 
       */
      FOR_EACH_BB(bb) {
	
        for (gsi = gsi_start_bb (bb); !gsi_end_p (gsi); gsi_next (&gsi))
          {
            stmt = gsi_stmt (gsi);
	    if (gimple_code(stmt) == GIMPLE_RETURN) {
	      call =  gimple_build_call(tree_gv_reset_fn, 0, NULL_TREE);
	      gsi_insert_before(&gsi, call, GSI_SAME_STMT);
	      if (flag_instr_source){
		call = gimple_build_call(tree_write_function_return_fn, 1, build_int_cst(integer_type_node, node->uid), NULL_TREE);
		gsi_insert_before(&gsi, call, GSI_SAME_STMT);
	      }
	    }

          }
      }

      //instr_mem_operations();
      switch_off_context_new();
    }
}


static 
void collect_def_use (tree op, bitmap mymap){

  switch (TREE_CODE(op)){
  case MEM_REF:
  case ARRAY_REF:
  case COMPONENT_REF:
    {
      break;
    }
    
  case VAR_DECL:
    bitmap_set_bit (mymap, DECL_UID (op));
    break;
    
  case SSA_NAME:
  default:
    break;
    
  }
}

static 
void collect_global_ref_in_stmt(gimple stmt, bitmap var_read, bitmap var_write){
  
  tree op0, op1, op2;

  if (stmt == NULL ) return ;

  switch( gimple_code (stmt) ){
  case GIMPLE_ASSIGN:
    {
      op0 = gimple_assign_lhs (stmt);
      op1 = gimple_assign_rhs1 (stmt);
      op2 = gimple_assign_rhs2 (stmt);
      collect_def_use (op0, var_write);
      collect_def_use (op1, var_read);
      if (op2)
	collect_def_use (op2, var_read);
      break;
    }

  case GIMPLE_COND:
    {
      op1 = gimple_cond_lhs (stmt);
      op2 = gimple_cond_rhs (stmt);
      collect_def_use (op1, var_read);
      collect_def_use (op2, var_read);
      break;
    }

  case GIMPLE_CALL:
    {
      int i;
      int n = gimple_call_num_args (stmt);

      for (i = 0; i < n; i++)
	{
	  op0 = gimple_call_arg (stmt, i);
	  collect_def_use (op0, var_read);
	}

      /* return value */
      op0 = gimple_call_lhs (stmt);
      if (op0)
	collect_def_use (op0, var_write);

      break;
    }

  default:
    break;
  }
 
}

/* collect ref for each function*/
static 
void collect_global_ref(cgraph_node_ptr node){

  basic_block bb;
  
  switch_to_context_new (node->decl);

  func_mod_ref_info(node)->var_read = BITMAP_ALLOC (NULL);
  func_mod_ref_info(node)->var_write = BITMAP_ALLOC (NULL);
  bitmap var_read = func_mod_ref_info(node)->var_read;
  bitmap var_write = func_mod_ref_info(node)->var_write;

  FOR_EACH_BB (bb)
  {
    gimple_stmt_iterator gsi;
    for (gsi = gsi_start_bb (bb); !gsi_end_p (gsi); gsi_next (&gsi))
      {
	gimple stmt = gsi_stmt(gsi);
	
	collect_global_ref_in_stmt(stmt, var_read, var_write);
      }
  }

  switch_off_context_new ();
  
}

/* collect the mod_ref info for each function. The result is stored into 
cgraph_node->extra_info
*/
static
void mod_ref_analysis(){
  struct cgraph_node *node;

  for (node = cgraph_nodes; node; node = node->next){
    if (!valid_function(node)) continue;
    collect_global_ref(node);
  }

  /* buttom up propagation */
  struct cgraph_node **order =
    XCNEWVEC (struct cgraph_node *, cgraph_n_nodes);
  struct cgraph_edge *e, *ie;
  int order_pos = ipa_utils_reduced_inorder (order, true, true, NULL);
  int i;

  for (i = 0; i < order_pos; i++){
    funcset set = NULL;
    funcset head = NULL;
    node = order[i];
    int num = add_func2set(set, node);
    if (!valid_function(node)) continue;
    //collect each  ssc
    struct ipa_dfs_info *n_info = (struct ipa_dfs_info *) node->aux;
    struct cgraph_node *next = n_info->next_cycle;
    while (next){
      n_info = (struct ipa_dfs_info *) next->aux;
      if (valid_function(next)) num = add_func2set(set, next);
      next = n_info->next_cycle;
    }

    if (num <= 1) {
      //there is at most 1 valid function in the scc, 
      //therefore, just merge mod_ref info of all its callees
      struct cgraph_edge *e;
      for (e = node->callees; e; e = e->next_callee)
        {
          if ( valid_function(e->callee) )
	    {
	      bitmap_ior_into (func_mod_ref_info (node)->var_read, func_mod_ref_info (e->callee)->var_read);
	      bitmap_ior_into (func_mod_ref_info (node)->var_write, func_mod_ref_info (e->callee)->var_write);          
	    }
        }

      for (e = node->indirect_calls; e; e = e->next_callee)
        {
          struct nodeList * callees = e->indirect_info->callees;
          while (callees)
	    {
	      if ( valid_function(callees->node) )
		{
		  bitmap_ior_into (func_mod_ref_info (node)->var_read, func_mod_ref_info (callees->node)->var_read);
		  bitmap_ior_into (func_mod_ref_info (node)->var_write, func_mod_ref_info (callees->node)->var_write);          
		}
	      callees = callees->next;
	    }
        }

      
    } // end of <<if (num<=1)>> 
    else {
      // num > 1, recursively combine mod_ref info
      bitmap scc_read = BITMAP_ALLOC (NULL);
      bitmap scc_write = BITMAP_ALLOC (NULL);

      /* Merge all callees out of scc */
      head = set;
      int i;
      for (i = 0; i < num; i++)	{
	struct cgraph_node *cur = head;
	for (e = cur->callees; e; e = e->next_callee) {
	  if ( !valid_function(e->callee) )
	    continue;
	  
	  bitmap_ior_into (scc_read,
			   func_mod_ref_info (e->callee)->var_read);
	  bitmap_ior_into (scc_write,
			   func_mod_ref_info (e->callee)->var_write);
	}
	head = head->next;
      }


      /* Merge all nodes inside scc */
      head = set;
      for (i = 0; i < num; i++)	{
	struct cgraph_node *cur = head;
	bitmap_ior_into (scc_read, 
			 func_mod_ref_info (cur)->var_read);
	bitmap_ior_into (scc_write,
			 func_mod_ref_info (cur)->var_write);
	head = head->next;
      }


      /* Merge the whole result to current */
      head = set;
      for (i = 0; i < num; i++){
	struct cgraph_node *cur = head;
	bitmap_ior_into (func_mod_ref_info (cur)->var_read, scc_read);
	bitmap_ior_into (func_mod_ref_info (cur)->var_write, scc_write);
	head = head->next;
      }
    } // end of << num > 1>>
    
    
  }
}


/* var is defined in stmt, if it is not ssa, check its ssa expression before end_stmt*/
static 
tree find_ssa_of_var( tree var, gimple stmt, gimple end_stmt){
  
  if (var == NULL) {
    printf("something is wrong in find_ssa_of_var\n");
    return NULL;
  }

  if (TREE_CODE(var) == SSA_NAME) return var;

  if (stmt == NULL) return NULL; //end of the program
 
  /*VAR_DECL*/
  basic_block bb, bb_next;
  gimple new_stmt = NULL;
  gimple_stmt_iterator gsi;
  edge e;
  edge_iterator ei;
  tree decl;
  struct Basic_Block_Extra *info;
  bb = gimple_bb(stmt);
  
  FOR_EACH_EDGE(e, ei, bb->succs){

    bb_next = e->dest;
    
    info = bb_extra_info(bb_next);
    if (info->visited) continue;
    info->visited = 1;

    for (gsi = gsi_start_bb(bb_next); !gsi_end_p(gsi); gsi_next(&gsi)) {
      new_stmt = gsi_stmt(gsi);
      if (gimple_code(new_stmt) != GIMPLE_ASSIGN) continue;
      if (gimple_num_ops(new_stmt) == 2 && gimple_assign_rhs1(new_stmt) == var) 
	return gimple_assign_lhs(new_stmt);
    }
    
    decl = find_ssa_of_var(var, new_stmt, end_stmt);
    if (decl) return decl;
    
  }
  return NULL;
}
  
/* decl is defined in b, check whether it is used in a*/
/* forward analysis*/
static 
int def_is_used( tree decl, gimple a, gimple b, struct cgraph_node *func){
  
  gimple stmt;
  imm_use_iterator uit;
  tree def;
  tree ssa_decl;

  /*if decl is not a ssa, it should be a global varible, and find its ssa def first */
  initial_bb_visit(func->decl);
  ssa_decl = find_ssa_of_var(decl, b, a);

  if ( ssa_decl == NULL) return 0;
    
  FOR_EACH_IMM_USE_STMT(stmt, uit, ssa_decl) {
    if (stmt == a) return 1;
    /*
    switch (gimple_code(stmt)){
    case GIMPLE_ASSIGN:
      def =  gimple_assign_lhs(stmt);
      if (def_is_used(def, a, stmt, func)) return 1;
    case GIMPLE_CALL:
      break;
    case GIMPLE_COND:
      break;
    default:
      break;
      }*/
  }
  return 0;

}

static
int check_data_dependence_1(tree decl, gimple a, gimple b, struct cgraph_node *func){

 cond_check_1:
  switch(TREE_CODE(decl)){
  case ADDR_EXPR:
    decl = TREE_OPERAND(decl, 0);
    goto cond_check_1;
    break;
  case VAR_DECL:
    if (def_is_used(decl, a, b, func)) return 1;
    break;
  default:
    break;
  }
  return 0;
}

/* check:
 * in function func, whether statement a is dependent on statement b
 * both a and b are GIMPLE_CALL
 */
static
void check_data_dependence(gimple a, gimple b, struct cgraph_node *func){
  
  if (a == b) return;

  struct cgraph_node *na, *nb;
  na = cgraph_get_node(gimple_call_fndecl(a));
  nb = cgraph_get_node(gimple_call_fndecl(b));
  struct Function_Mod_Ref_Info *info = func_mod_ref_info( na );
  if (info->dep_chain[nb->uid] == 1) return;

  bitmap mywrite = func_mod_ref_info(cgraph_get_node(gimple_call_fndecl(b)))->var_write;
  basic_block bb_a, bb_b;
  basic_block bb;

  gimple_stmt_iterator gsi;

  bb_a = gimple_bb(a);
  for (gsi = gsi_last_bb( bb_a ); !gsi_end_p( gsi); gsi_prev( &gsi ))
    if ( gsi_stmt( gsi ) == a ) break;
  
  
  gimple stmt, stmt_cur;
  imm_use_iterator uit;
  tree var;
  int i;
  tree decl;
    

  /* step1: for each def in b, check if it is used in a*/
  const bitmap_element *ptr;
  int flag = 0;
  if (mywrite) {
    for (ptr = mywrite->first; ptr && !flag; ptr = ptr->next){
      unsigned int i, j, id;
      for (i = 0; i < BITMAP_ELEMENT_WORDS && !flag; i++)
	for (j = 0; j < BITMAP_WORD_BITS && !flag; j++)
	  if ( ( ptr->bits[i] >> j) & 1) {
	    
	    id = ptr->indx * BITMAP_ELEMENT_ALL_BITS 
	      + i * BITMAP_WORD_BITS + j;
	    
	    decl = find_var_with_id( id );
	    
	    if (def_is_used(decl, a, b, func)){ 
	      /* add data-dependence chain*/
	      info->dep_chain[ nb->uid ] = 1;
	      flag = 1;
	      break;
	    }
	  }
    } //< end of for ptr>
  }

  if (flag) return;
  
  /* step 2: check return value of b*/
  decl = gimple_call_lhs( b );
  if (decl != NULL && def_is_used(decl, a, b, func)){
    info->dep_chain[nb->uid] = 1;
    flag = 1;
    return;
  }

 /* step 3: check parameters of b */
  for (i = 0; i < gimple_call_num_args(b); i++){

    decl = gimple_call_arg(b, i);

    //cond_check_1:
    switch(TREE_CODE(decl)){
    case ADDR_EXPR:
      if (check_data_dependence_1(TREE_OPERAND(decl, 0), a, b, func)){
	info->dep_chain[ nb->uid ] = 1;
	return;
      }
      break;
    case VAR_DECL:
        break;
    case SSA_NAME:
      break;
    default:
      break;
    }
    
  }
    

}
   

/* if stmt is a function call, whether fn is the callee*/
static 
int is_same_fn(gimple stmt, tree fn) {

  if (gimple_code(stmt) != GIMPLE_CALL) return 0;
  
  tree fn1 = gimple_call_fn(stmt);
  
  if ( TREE_CODE(fn) == NON_LVALUE_EXPR) 
    fn1 = TREE_OPERAND(fn1, 0);
  
 cont_1:
  switch (TREE_CODE (fn1)){
  case FUNCTION_DECL:
    goto cont_2;
    break;
    
  case ADDR_EXPR:
  case INDIRECT_REF:
  case NOP_EXPR:
    fn1 = TREE_OPERAND (fn1, 0);
    goto cont_1;
  case VAR_DECL:
  case PARM_DECL:
  default:
    break;
  }
  return 0;
 cont_2:
  if (fn1 == fn) return 1;
  return 0;
}
  
static 
void find_data_dependence_edge(struct cgraph_node *callee, struct cgraph_node *caller){
  
  basic_block bb, bb2;
  gimple_stmt_iterator gsi, gsi2;
  gimple stmt_cur, stmt_callsite;
  
  if (caller->decl == NULL) return 0;
  
  switch_to_context_new (caller->decl);
  FOR_EACH_BB (bb){
    
    for (gsi = gsi_start_bb (bb); !gsi_end_p (gsi); gsi_next (&gsi)){
      
      stmt_cur = gsi_stmt (gsi);
      if (gimple_code(stmt_cur) != GIMPLE_CALL) continue;
      
      tree fn = gimple_call_fndecl(stmt_cur);
      
      if ( is_same_fn(stmt_cur, callee->decl)) {
	FOR_EACH_BB(bb2){
	  
	  for (gsi2 = gsi_start_bb(bb2); !gsi_end_p(gsi2); gsi_next (&gsi2)){
	    stmt_callsite = gsi_stmt(gsi2);
	    if (gimple_code(stmt_callsite) != GIMPLE_CALL) continue;
	    check_data_dependence(stmt_cur, stmt_callsite, caller);
	  }
	  
	}
      } // end of <if>
      
      
    }// end of <for gsi>
    
    
  }
  switch_off_context_new();
  return ;
}

static
void func_data_dependence(struct cgraph_node *node) {
  struct cgraph_edge *edge;

  for (edge = node->callers; edge; edge = edge->next_caller){
    struct cgraph_node *w = edge->caller;
    if (w!=NULL){
      find_data_dependence_edge(node, w);

    }
    
  }
}

static
void build_PFDG(){

  FILE *fp;
  struct cgraph_edge *edge;
  fp = fopen("dump_cgraph","w");
  dump_cgraph(fp);
  fclose(fp);
  /* initialize Func_Mod_Ref_Info, that is node->extra_info*/
  struct cgraph_node *node;
  for (node = cgraph_nodes; node ; node =  node->next){
    struct Function_Mod_Ref_Info *info = XNEWVEC(struct Function_Mod_Ref_Info, 1);
    info->isInstr = NONE_INSTR;
    node->extra_info = info;
    /*For some special functions, such as interrupt handler,
     * e.g. with attributes wakeup , interrupt
     *based on predefined by users, it should be always instruted
     */
    if (!DECL_ATTRIBUTES(node->decl)) continue;
    if (lookup_attribute ("wakeup", DECL_ATTRIBUTES(node->decl))&&
	lookup_attribute ("interrupt", DECL_ATTRIBUTES(node->decl)))
      info->isInstr = INSTRED;

  }

  /*collect mod-ref info for each function*/
  mod_ref_analysis();

  /*create PFDG. */
  /* step 1 : add caller-callee edge */
  for (node = cgraph_nodes; node ; node = node->next){
    struct Function_Mod_Ref_Info *info = func_mod_ref_info( node );
    for (edge = node->callers; edge; edge = edge->next_caller){
      struct cgraph_node *w = edge->caller;
      info->dep_chain[w->uid] = 1;
    }
    if (valid_function(node))
      print_bb_order(node->decl);// for testing
  }

  /* step 2 : Start from the Inv_func, adding data-dependence edges */
  for (node = cgraph_nodes; node; node = node->next){
    if (! strcmp(cgraph_node_name(node), "Inv_fun")) break;
  }

  if (node == NULL) {
    printf("can't find out the inv_fun\n!");
    return;
  }

  func_data_dependence(node);

}


static
struct cgraph_node* find_func_with_id(int id) {
  struct cgraph_node *node;
  for (node = cgraph_nodes; node; node = node->next)
    if (node->uid == id) return node;
  return NULL;
}
 
static
void depend_DFS(int l, struct cgraph_node *node){
  
  if (l == 0) return;
  struct Function_Mod_Ref_Info *info = func_mod_ref_info(node);
  info->isInstr = INSTRED;

  int i;
  for (i = 0; i < funcnum; i++){
    if (info->dep_chain[i] == 1)
      depend_DFS(l-1, find_func_with_id(i));
  }

}
/* based on PFDG, decide which funcs should be instrumented */
static
void instr_decision(int level){
  
  struct cgraph_node *node;
  for (node = cgraph_nodes; node; node = node->next) {
    if (! strcmp(cgraph_node_name(node), "Inv_fun")) break;
  }
  
  if (node == NULL) {
    printf("can't find out the inv_fun\n");
    return ;
  }

  /* start from Inv_fun*/
  depend_DFS(level, node); 
}


/*get the function name based on the id*/
static
char * get_fname(int fid){
  FILE *fp;
  char line[255];
  fp = fopen("func.info", "r");
  while(1){
    fgets(line, 255, fp);
    if (line[0] == 'T' ) break;
    int j = 4;
    int id = 0;
    int i;
    line[strlen(line)-1]='\0';
    while (line[j] >= '0' && line[j] <='9') {
      id = id*10 + (line[j]-'0');
      j++;
    }
    if (id != fid) continue;
    /* skip white spaces*/
    while (line[j] != 'F')
      j++;
    /* skip "FUNCTION"*/
    while (line[j] != ' ')
      j++;
    /* start reading the function name*/
    j++;
    
    //char *ret = XNEWVEC(char, (strlen(line)-j));
    char ret[100];// =(char *)xmalloc(sizeof(char)* (strlen(line)-j));
    for (i = j; i < strlen(line); i++)
      ret[i-j] = line[i];
    ret[i-j]='\0';
    fclose(fp);
    return ret;
  }
  
  fclose(fp);
  return NULL;
}

static
void instr_interrupt_call(char *file){

  char *iterruptname; //interrupt name
  int i;
  tree tree_inc_counter_fn = build_fn_decl(inc_counter, inc_counter_fn_type);
  tree check_ipt_fn = build_fn_decl(check_ipt, check_ipt_fn_type);
  /* file is the interrupt table, the format is:
     number  // how many functions
     fn1
     n1  // in fn1, n1 interrupts were called
     ins1 gv1 irt1 // if fun_COUNTER == ins1 and gv == gv1, irt 1 should be called here.
     ins2 gv2 irt2
     ...

   */
  FILE *fp = fopen(file, "r");
  int fn_num;
  fscanf (fp, "%d\n", &fn_num);
  for (i = 0 ; i < fn_num; i++) {
    int num, fid;
    //fscanf(fp, "%s\n", fname);
    fscanf(fp, "%d\n", &fid);
    fscanf(fp, "%d\n", &num);
    int *insno; //function instance number
    int *gv; //gv reference
    int *irpt;
    int j;
    // read all interrupts
    insno = XNEWVEC(int, num);
    gv = XNEWVEC(int, num);
    irpt = XNEWVEC(int, num);
    for (j = 0; j < num; j++)
      fscanf(fp, "%d %d %d\n", &insno[j], &gv[j], &irpt[j]);

    // find the function for instrument
    struct cgraph_node *node;
    char *name;
    name = get_fname(fid);
    for (node = cgraph_nodes; node; node = node->next) {
            if (! strcmp(cgraph_node_name(node), name)) break;
    }
    if (node == NULL) {
      printf("error\n");
      return;
    }
    
    switch_to_context_new (node->decl);


    // after each global varible operation, insert code to decide whether an interrupt 
    //should be called
    basic_block bb;
    gimple_stmt_iterator gsi;
    gimple stmt,call;
    int k;
    tree fn;

    bb = cfun->cfg->x_entry_block_ptr->next_bb;
    gsi = gsi_start_bb (bb);

    // if gv ==0, just instr the interruput call at the very beginning of the fuction
    for (k = 0; k < num; k++) {
      if (gv[k] == 0){
	call = gimple_build_call(check_ipt_fn, 3,
				 build_int_cst(integer_type_node,insno[k]),
				 build_int_cst(integer_type_node,gv[k]),
				 build_int_cst(integer_type_node,irpt[k]), NULL_TREE);
	gsi_insert_after(&gsi, call, GSI_NEW_STMT);
      }
    }

    FOR_EACH_BB (bb){
      
      for (gsi = gsi_start_bb (bb); !gsi_end_p (gsi); gsi_next (&gsi))
	{
	  stmt = gsi_stmt (gsi);
	  if (gimple_code (stmt) != GIMPLE_CALL) continue;
	  fn = gimple_call_fndecl(stmt);
	  
	  if (strcmp(IDENTIFIER_POINTER(DECL_NAME(fn)),"__gv_update")) continue;
	  
	  for (k = 0; k < num; k++) {
	    if (gv[k] != 0){
	    call = gimple_build_call(check_ipt_fn, 3, 
				     build_int_cst(integer_type_node,insno[k]), 
				     build_int_cst(integer_type_node,gv[k]), 
				     build_int_cst(integer_type_node,irpt[k]), NULL_TREE);
	    gsi_insert_after(&gsi, call, GSI_NEW_STMT);
	    }
	  }

	}          
    }
    switch_off_context_new();

    
  }
  fclose(fp);

}


static 
void instr_sourcec_main(){
  
  
  if (flag_instr_source) 
    printf("to do: instrumenting source code...\n");
  else if (flag_instr_replay)
    printf("to do: instrumenting replay code...\n");
  else if (flag_instr_baseline)
    printf("to do: instrumenting baseline code...\n");
  else if (flag_instr_non_inline)
    printf ("to do: instrumenting code without considering inline...\n");
  else return;

  initialize_instr_prototype();
  
  //assign function id
  if (flag_instr_source || flag_instr_baseline || flag_instr_non_inline)
  {
    FILE *fp = fopen("func.info","w");
    function_id_assignment(fp);
    close(fp);
  }

  // step 1: find out the slicing criteria, saved in criteria
  if (flag_instr_source)
  get_slice_criteria(); // not necessary at this time
  
  
  // step 2: add a global variable int __global_count, which is used to record
  //         the number of global write
  tree new_decl,init;
  struct varpool_node *new_node;
  new_decl = create_tmp_var_raw(integer_type_node, NULL);
  DECL_NAME (new_decl) = get_identifier("___GLOBAL_COUNTER");
  TREE_READONLY (new_decl) = 0;
  TREE_STATIC (new_decl) = 1;
  TREE_USED (new_decl) = 1;
  DECL_CONTEXT (new_decl) = NULL_TREE;
  DECL_ABSTRACT (new_decl) = 0;
  init = build_int_cst(NULL, 0);
  DECL_INITIAL (new_decl) = init;
  lang_hooks.dup_lang_specific_decl (new_decl);
  create_var_ann (new_decl);
  new_node = varpool_node (new_decl);
  varpool_mark_needed_node (new_node);
  varpool_finalize_decl (new_decl);
  
  // step 3: do multi-level slicing from the criteria and instrument
  if (flag_instr_baseline){
    //instr all functions under this situation
    instr_function_entry_exit();
  }


  if (flag_instr_source || flag_instr_non_inline) { 
    // instr recording code, w/o inlining
    build_PFDG();
    instr_decision(MAXLEVEL);
    instr_function_entry_exit();
    dump_instrued_function();
  }

  if (flag_instr_replay) { // instr replay code
    mark_instrued_function();
    instr_function_entry_exit();
    
    //read interrupt record && instr interrupt caller
    instr_interrupt_call("inter_table");
    
  }

  printf("==================finish instrumenting====================\n\n");

}

struct simple_ipa_opt_pass pass_instr_sourcec =
  {
    {
      SIMPLE_IPA_PASS,
      "instr-sourcec",               /* name */
      instr_sourcec_main,             /* gate */
      NULL, /* sub */
      NULL, /* next */
      0, /* static_pass_number */
      0, /* tv_id */
      PROP_ssa, /* properties_required */
      0, /* properties_provided */
      0, /* properties_destroyed */
      0, /* todo_flags_start */
    0
    }
  };




