/* Pretty formatting of GIMPLE statements and expressions.
   Copyright (C) 2001, 2002, 2003, 2004, 2005, 2006, 2007, 2008, 2009, 2010
   Free Software Foundation, Inc.
   Contributed by Aldy Hernandez <aldyh@redhat.com> and
   Diego Novillo <dnovillo@google.com>

This file is part of GCC.

GCC is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free
Software Foundation; either version 3, or (at your option) any later
version.

GCC is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License
along with GCC; see the file COPYING3.  If not see
<http://www.gnu.org/licenses/>.  */

/* this version is modified from gimple-pretty-print.c, 
   used for dumping compilerable c source code
*/
#include "config.h"
#include "system.h"
#include "coretypes.h"
#include "tm.h"
#include "tree.h"
#include "langhooks.h"
#include "diagnostic.h"
#include "tree-pretty-print.h"
#include "gimple-pretty-print.h"
#include "hashtab.h"
#include "tree-flow.h"
#include "tree-pass.h"
#include "gimple.h"
#include "value-prof.h"


#define INDENT(SPACE)							\
  do { int i; for (i = 0; i < SPACE; i++) pp_space (buffer); } while (0)

static pretty_printer buffer;
static bool initialized = false;
static int lablestack[100]; //first assume in each function, the # of bb is less than 100  
static int lable = 0; //global index of each lable;
 
#define GIMPLE_NIY do_niy (buffer,gs)


static int
is_in_lable_stack (int);

static int
isPrintf(gimple);

/* Try to print on BUFFER a default message for the unrecognized
   gimple statement GS.  */

static void
do_niy (pretty_printer *buffer, gimple gs)
{
  pp_printf (buffer, "<<< Unknown GIMPLE statement: %s >>>\n",
	     gimple_code_name[(int) gimple_code (gs)]);
}


/* Initialize the pretty printer on FILE if needed.  */

static void
maybe_init_pretty_print (FILE *file)
{
  if (!initialized)
    {
      pp_construct (&buffer, NULL, 0);
      pp_needs_newline (&buffer) = true;
      initialized = true;
    }

  buffer.buffer->stream = file;
}


/* Emit a newline and SPC indentantion spaces to BUFFER.  */

static void
newline_and_indent (pretty_printer *buffer, int spc)
{
  pp_newline (buffer);
  INDENT (spc);
}


/* Print the GIMPLE statement GS on stderr.  */

DEBUG_FUNCTION void
debug_gimple_stmt_c (gimple gs)
{
  print_gimple_stmt_c (stderr, gs, 0, TDF_VOPS|TDF_MEMSYMS);
  fprintf (stderr, "\n");
}


/* Dump GIMPLE statement G to FILE using SPC indentantion spaces and
   FLAGS as in dump_gimple_stmt_c.  */

void
print_gimple_stmt_c (FILE *file, gimple g, int spc, int flags)
{
  maybe_init_pretty_print (file);
  dump_gimple_stmt_c (&buffer, g, spc, flags);
  pp_flush (&buffer);
}


/* Dump GIMPLE statement G to FILE using SPC indentantion spaces and
   FLAGS as in dump_gimple_stmt_c.  Print only the right-hand side
   of the statement.  */

void
print_gimple_expr_c (FILE *file, gimple g, int spc, int flags)
{
  flags |= TDF_RHS_ONLY;
  maybe_init_pretty_print (file);
  dump_gimple_stmt_c (&buffer, g, spc, flags);
}


/* Print the GIMPLE sequence SEQ on BUFFER using SPC indentantion
   spaces and FLAGS as in dump_gimple_stmt_c.  */

static void
dump_gimple_seq (pretty_printer *buffer, gimple_seq seq, int spc, int flags)
{
  gimple_stmt_iterator i;

  for (i = gsi_start (seq); !gsi_end_p (i); gsi_next (&i))
    {
      gimple gs = gsi_stmt (i);
      INDENT (spc);
      dump_gimple_stmt_c (buffer, gs, spc, flags);
      if (!gsi_one_before_end_p (i))
	pp_newline (buffer);
    }
}


/* Dump GIMPLE sequence SEQ to FILE using SPC indentantion spaces and
   FLAGS as in dump_gimple_stmt_c.  */

void
print_gimple_seq_c (FILE *file, gimple_seq seq, int spc, int flags)
{
  maybe_init_pretty_print (file);
  dump_gimple_seq (&buffer, seq, spc, flags);
  pp_flush (&buffer);
}


/* Print the GIMPLE sequence SEQ on stderr.  */

DEBUG_FUNCTION void
debug_gimple_seq_c (gimple_seq seq)
{
  print_gimple_seq_c (stderr, seq, 0, TDF_VOPS|TDF_MEMSYMS);
}


/* A simple helper to pretty-print some of the gimple tuples in the printf
   style. The format modifiers are preceeded by '%' and are:
     'G' - outputs a string corresponding to the code of the given gimple,
     'S' - outputs a gimple_seq with indent of spc + 2,
     'T' - outputs the tree t,
     'd' - outputs an int as a decimal,
     's' - outputs a string,
     'n' - outputs a newline,
     '+' - increases indent by 2 then outputs a newline,
     '-' - decreases indent by 2 then outputs a newline.   */

static void
dump_gimple_fmt (pretty_printer *buffer, int spc, int flags,
                 const char *fmt, ...)
{
  va_list args;
  const char *c;
  const char *tmp;

  va_start (args, fmt);
  for (c = fmt; *c; c++)
    {
      if (*c == '%')
        {
          gimple_seq seq;
          tree t;
          gimple g;
          switch (*++c)
            {
              case 'G':
                g = va_arg (args, gimple);
                tmp = gimple_code_name[gimple_code (g)];
                pp_string (buffer, tmp);
                break;

              case 'S':
                seq = va_arg (args, gimple_seq);
                pp_newline (buffer);
                dump_gimple_seq (buffer, seq, spc + 2, flags);
                newline_and_indent (buffer, spc);
                break;

              case 'T':
                t = va_arg (args, tree);
                if (t == NULL_TREE)
                  pp_string (buffer, "NULL");
                else
                  dump_generic_node_c (buffer, t, spc, flags, false);
                break;

              case 'd':
                pp_decimal_int (buffer, va_arg (args, int));
                break;

              case 's':
                pp_string (buffer, va_arg (args, char *));
                break;

              case 'n':
                newline_and_indent (buffer, spc);
                break;

              case '+':
                spc += 2;
                newline_and_indent (buffer, spc);
                break;

              case '-':
                spc -= 2;
                newline_and_indent (buffer, spc);
                break;

              default:
                gcc_unreachable ();
            }
        }
      else
        pp_character (buffer, *c);
    }
  va_end (args);
}


/* Helper for dump_gimple_assign.  Print the unary RHS of the
   assignment GS.  BUFFER, SPC and FLAGS are as in dump_gimple_stmt_c.  */

static void
dump_unary_rhs (pretty_printer *buffer, gimple gs, int spc, int flags)
{
  enum tree_code rhs_code = gimple_assign_rhs_code (gs);
  tree lhs = gimple_assign_lhs (gs);
  tree rhs = gimple_assign_rhs1 (gs);

  /*handle the BIT_FIELD_REF case*/
  if (TREE_CODE(rhs) == BIT_FIELD_REF) {
    tree op1 = TREE_OPERAND(rhs, 0);
    tree op2 = TREE_OPERAND(rhs, 2); //number of bits, not byte
    tree mytype = TREE_TYPE(lhs);
    char symbol='&';
    unsigned HOST_WIDE_INT low = (TREE_INT_CST_LOW (op2))/8;
    
    if (op1
	&& (TREE_CODE (op1) == INDIRECT_REF
	    || (TREE_CODE (op1) == MEM_REF
		&& TREE_CODE (TREE_OPERAND (op1, 1)) != ADDR_EXPR
		&& integer_zerop (TREE_OPERAND (op1, 1))
		/* Dump the types of INTEGER_CSTs explicitly, for we
                     can't infer them and MEM_ATTR caching will share                                                              
                     MEM_REFs with differently-typed op0s.  */
		&& TREE_CODE (TREE_OPERAND (op1, 0)) != INTEGER_CST
		/* Same pointer types, but ignoring POINTER_TYPE vs.                                                              
		   REFERENCE_TYPE.  */
		&& (TREE_TYPE (TREE_TYPE (TREE_OPERAND (op1, 0)))
		    == TREE_TYPE (TREE_TYPE (TREE_OPERAND (op1, 1))))
		&& (TYPE_MODE (TREE_TYPE (TREE_OPERAND (op1, 0)))
		    == TYPE_MODE (TREE_TYPE (TREE_OPERAND (op1, 1))))
		&& (TYPE_REF_CAN_ALIAS_ALL (TREE_TYPE (TREE_OPERAND (op1, 0)))
		    == TYPE_REF_CAN_ALIAS_ALL (TREE_TYPE (TREE_OPERAND (op1, 1))))
		/* Same value types ignoring qualifiers.  */
		&& (TYPE_MAIN_VARIANT (TREE_TYPE (op1))
                      == TYPE_MAIN_VARIANT
		    (TREE_TYPE (TREE_TYPE (TREE_OPERAND (op1, 1))))))))
    {
      op1 = TREE_OPERAND (op1, 0);
      symbol = ' ';
    }

    pp_string(buffer, " *(");
    dump_generic_node_c(buffer, mytype, spc, flags, false);
    pp_string(buffer, " *)((void *)");
    pp_character(buffer, symbol);
    dump_generic_node_c(buffer, op1, spc, flags, false);
    pp_string(buffer, " + ");
    pp_decimal_int(buffer, low);
    pp_string(buffer, " )");
    
    return;
  }
  switch (rhs_code)
    {
    case VIEW_CONVERT_EXPR:
    case ASSERT_EXPR:
      dump_generic_node_c (buffer, rhs, spc, flags, false);
      break;

    case FIXED_CONVERT_EXPR:
    case ADDR_SPACE_CONVERT_EXPR:
    case FIX_TRUNC_EXPR:
    case FLOAT_EXPR:
    CASE_CONVERT:
      pp_character (buffer, '(');
      dump_generic_node_c (buffer, TREE_TYPE (lhs), spc, flags, false);
      pp_string (buffer, ") ");
      if (op_prio (rhs) < op_code_prio (rhs_code))
	{
	  pp_character (buffer, '(');
	  dump_generic_node_c (buffer, rhs, spc, flags, false);
	  pp_character (buffer, ')');
	}
      else
	dump_generic_node_c (buffer, rhs, spc, flags, false);
      break;

    case PAREN_EXPR:
      pp_string (buffer, "((");
      dump_generic_node_c (buffer, rhs, spc, flags, false);
      pp_string (buffer, "))");
      break;

    case ABS_EXPR:
      pp_string (buffer, "ABS_EXPR <");
      dump_generic_node_c (buffer, rhs, spc, flags, false);
      pp_character (buffer, '>');
      break;

    case POINTER_PLUS_EXPR:
      pp_string (buffer, "<pointer_assign>");
    default:
      if (TREE_CODE_CLASS (rhs_code) == tcc_declaration
	  || TREE_CODE_CLASS (rhs_code) == tcc_constant
	  || TREE_CODE_CLASS (rhs_code) == tcc_reference
	  || rhs_code == SSA_NAME
	  || rhs_code == ADDR_EXPR
	  || rhs_code == CONSTRUCTOR)
	{
	  dump_generic_node_c (buffer, rhs, spc, flags, false);
	  break;
	}
      else if (rhs_code == BIT_NOT_EXPR)
	pp_character (buffer, '~');
      else if (rhs_code == TRUTH_NOT_EXPR)
	pp_character (buffer, '!');
      else if (rhs_code == NEGATE_EXPR)
	pp_character (buffer, '-');
      else
	{
	  pp_character (buffer, '[');
	  pp_string (buffer, tree_code_name [rhs_code]);
	  pp_string (buffer, "] ");
	}

      if (op_prio (rhs) < op_code_prio (rhs_code))
	{
	  pp_character (buffer, '(');
	  dump_generic_node_c (buffer, rhs, spc, flags, false);
	  pp_character (buffer, ')');
	}
      else
	dump_generic_node_c (buffer, rhs, spc, flags, false);
      break;
    }
}


/* Helper for dump_gimple_assign.  Print the binary RHS of the
   assignment GS.  BUFFER, SPC and FLAGS are as in dump_gimple_stmt_c.  */

static void
dump_binary_rhs (pretty_printer *buffer, gimple gs, int spc, int flags)
{
  const char *p;
  enum tree_code code = gimple_assign_rhs_code (gs);

  if (code == POINTER_PLUS_EXPR) {   
    // e.g. *x = &sx1[3] will be translated as *x = POINTER_PLUS_EXPR(&x1,3) , which should be *x = &(*(x1+3))
    tree op1 = gimple_assign_rhs1 (gs);
    tree op2 = gimple_assign_rhs2 (gs);
    if (TREE_CODE(op1) == ADDR_EXPR) {
      pp_string(buffer, "&(*( ");
      if (op_prio (op1) <= op_code_prio (code))
	{
	  pp_character (buffer, '(');
	  dump_generic_node_c (buffer, TREE_OPERAND(op1, 0), spc, flags,
			       false);
	  pp_character (buffer, ')');
	}
      else
	dump_generic_node_c (buffer, TREE_OPERAND(op1, 0), spc, flags, false);
      pp_string (buffer, " + ");
      //convert the offset to index
      if (TREE_CODE(TREE_TYPE(TREE_OPERAND(op1, 0))) == ARRAY_TYPE) {
	
	gimple_stmt_iterator temp_gsi = gsi_for_stmt(gs);
	gsi_prev(&temp_gsi);
	gimple gs_1 = gsi_stmt(temp_gsi);
	gsi_prev(&temp_gsi);
	gimple gs_2 = gsi_stmt(temp_gsi);
	if (gimple_code(gs_1) == GIMPLE_ASSIGN &&
	    gimple_code(gs_2) == GIMPLE_ASSIGN &&
	    op2 == gimple_assign_lhs(gs_1))
	  op2 = gimple_assign_rhs1(gs_2);
      }
      if (op_prio (op2) <= op_code_prio (code))
	{
	  pp_character (buffer, '(');
	  dump_generic_node_c (buffer, op2, spc, flags,
			       false);
	  pp_character (buffer, ')');
	}
      else
	dump_generic_node_c (buffer, op2, spc, flags, false);
      pp_string(buffer, "))");
      return;
     }

    if (TREE_CODE(op2) == INTEGER_CST ||
	TREE_CODE(TREE_TYPE(op2))==INTEGER_TYPE) {
      //handle pointer plus                                                                                                   
      //int value =  TREE_INT_CST_LOW(op1);
      tree type;
      int size;
      type = TREE_TYPE(op1);
      //while (TREE_CODE(type) == POINTER_TYPE) type = TREE_TYPE(type);
      if (TREE_CODE(type) == POINTER_TYPE) type = TREE_TYPE(type);
      size = TYPE_PRECISION(type);
      if (size == 0) pp_string(buffer, "(uint8_t *)"); 
      dump_generic_node_c (buffer, op1, spc, flags, false);
      pp_string (buffer, " + ");
      dump_generic_node_c(buffer, op2, spc, flags, false);
      if (size == 0) pp_string(buffer,";//");
      pp_character(buffer,'/');
      pp_decimal_int (buffer, TYPE_PRECISION (type)/8);
      
      return;
    } 
  }
  
  switch (code)
    {
    case COMPLEX_EXPR:
    case MIN_EXPR:
      dump_generic_node_c (buffer, gimple_assign_rhs1 (gs), spc, flags, false);
      pp_string (buffer, " < ");
      dump_generic_node_c (buffer, gimple_assign_rhs2 (gs), spc, flags, false);
      pp_string (buffer, " ? ");
      dump_generic_node_c (buffer, gimple_assign_rhs1 (gs), spc, flags, false);
      pp_string (buffer, ": ");
      dump_generic_node_c (buffer, gimple_assign_rhs2 (gs), spc, flags, false);
      break;
    case MAX_EXPR:
      dump_generic_node_c (buffer, gimple_assign_rhs1 (gs), spc, flags, false);
      pp_string (buffer, " > ");
      dump_generic_node_c (buffer, gimple_assign_rhs2 (gs), spc, flags, false);
      pp_string (buffer, " ? ");
      dump_generic_node_c (buffer, gimple_assign_rhs1 (gs), spc, flags, false);
      pp_string (buffer, " : ");
      dump_generic_node_c (buffer, gimple_assign_rhs2 (gs), spc, flags, false);
      break;
    case VEC_WIDEN_MULT_HI_EXPR:
    case VEC_WIDEN_MULT_LO_EXPR:
    case VEC_PACK_TRUNC_EXPR:
    case VEC_PACK_SAT_EXPR:
    case VEC_PACK_FIX_TRUNC_EXPR:
    case VEC_EXTRACT_EVEN_EXPR:
    case VEC_EXTRACT_ODD_EXPR:
    case VEC_INTERLEAVE_HIGH_EXPR:
    case VEC_INTERLEAVE_LOW_EXPR:
      for (p = tree_code_name [(int) code]; *p; p++)
	pp_character (buffer, TOUPPER (*p));
      pp_string (buffer, " <");
      dump_generic_node_c (buffer, gimple_assign_rhs1 (gs), spc, flags, false);
      pp_string (buffer, ", ");
      dump_generic_node_c (buffer, gimple_assign_rhs2 (gs), spc, flags, false);
      pp_character (buffer, '>');
      break;

    default:
      if (op_prio (gimple_assign_rhs1 (gs)) <= op_code_prio (code))
	{
	  pp_character (buffer, '(');
	  dump_generic_node_c (buffer, gimple_assign_rhs1 (gs), spc, flags,
			     false);
	  pp_character (buffer, ')');
	}
      else
	dump_generic_node_c (buffer, gimple_assign_rhs1 (gs), spc, flags, false);
      pp_space (buffer);
      pp_string (buffer, op_symbol_code (gimple_assign_rhs_code (gs)));
      pp_space (buffer);
      if (op_prio (gimple_assign_rhs2 (gs)) <= op_code_prio (code))
	{
	  pp_character (buffer, '(');
	  dump_generic_node_c (buffer, gimple_assign_rhs2 (gs), spc, flags,
			     false);
	  pp_character (buffer, ')');
	}
      else
	dump_generic_node_c (buffer, gimple_assign_rhs2 (gs), spc, flags, false);
    }
}

/* Helper for dump_gimple_assign.  Print the ternary RHS of the
   assignment GS.  BUFFER, SPC and FLAGS are as in dump_gimple_stmt_c.  */

static void
dump_ternary_rhs (pretty_printer *buffer, gimple gs, int spc, int flags)
{
  const char *p;
  enum tree_code code = gimple_assign_rhs_code (gs);
  switch (code)
    {
    case WIDEN_MULT_PLUS_EXPR:
    case WIDEN_MULT_MINUS_EXPR:
      for (p = tree_code_name [(int) code]; *p; p++)
	pp_character (buffer, TOUPPER (*p));
      pp_string (buffer, " <");
      dump_generic_node_c (buffer, gimple_assign_rhs1 (gs), spc, flags, false);
      pp_string (buffer, ", ");
      dump_generic_node_c (buffer, gimple_assign_rhs2 (gs), spc, flags, false);
      pp_string (buffer, ", ");
      dump_generic_node_c (buffer, gimple_assign_rhs3 (gs), spc, flags, false);
      pp_character (buffer, '>');
      break;

    case FMA_EXPR:
      dump_generic_node_c (buffer, gimple_assign_rhs1 (gs), spc, flags, false);
      pp_string (buffer, " * ");
      dump_generic_node_c (buffer, gimple_assign_rhs2 (gs), spc, flags, false);
      pp_string (buffer, " + ");
      dump_generic_node_c (buffer, gimple_assign_rhs3 (gs), spc, flags, false);
      break;

    case DOT_PROD_EXPR:
      pp_string (buffer, "DOT_PROD_EXPR <");
      dump_generic_node_c (buffer, gimple_assign_rhs1 (gs), spc, flags, false);
      pp_string (buffer, ", ");
      dump_generic_node_c (buffer, gimple_assign_rhs2 (gs), spc, flags, false);
      pp_string (buffer, ", ");
      dump_generic_node_c (buffer, gimple_assign_rhs3 (gs), spc, flags, false);
      pp_string (buffer, ">");
      break;

    case REALIGN_LOAD_EXPR:
      pp_string (buffer, "REALIGN_LOAD <");
      dump_generic_node_c (buffer, gimple_assign_rhs1 (gs), spc, flags, false);
      pp_string (buffer, ", ");
      dump_generic_node_c (buffer, gimple_assign_rhs2 (gs), spc, flags, false);
      pp_string (buffer, ", ");
      dump_generic_node_c (buffer, gimple_assign_rhs3 (gs), spc, flags, false);
      pp_string (buffer, ">");
      break;

    default:
      gcc_unreachable ();
    }
}


/*decide whether volatile cast should be output*/
static int
if_volatile_cast(gimple gs){
  
  tree type = TREE_TYPE(gimple_assign_lhs(gs));
  if (TREE_CODE(type) == POINTER_TYPE) type = TREE_TYPE(type);
  unsigned int quals = TYPE_QUALS (type);
  if (!(quals & TYPE_QUAL_VOLATILE))
    return 0;
  
  if (gimple_num_ops(gs) != 2) return 0;
  tree rhs = gimple_assign_rhs1(gs);
  if (TREE_CODE(rhs) != INTEGER_CST) return 0;
  return 1;
}

/* Dump the gimple assignment GS.  BUFFER, SPC and FLAGS are as in
   dump_gimple_stmt_c.  */

static void
dump_gimple_assign (pretty_printer *buffer, gimple gs, int spc, int flags )
{
 
  /* handle this case: 
   * struct a{
     int a1;
     int a2;
     struct b a3;
     } c = {.a1 =10, .a2=11};
    will output an extra line: c={}; this should be removed.
    if there is no "struct b a3", there is no c={} 
*/
  if (gimple_num_ops(gs) == 2 && TREE_CODE(gimple_assign_rhs1(gs)) == CONSTRUCTOR) {
    tree rhs = gimple_assign_rhs1(gs);
    int l = VEC_length (constructor_elt, CONSTRUCTOR_ELTS(rhs));
    if (l<=0) return;
  }
  
  dump_generic_node_c (buffer, gimple_assign_lhs (gs), spc, flags, false);
  pp_space (buffer);
  pp_character (buffer, '=');
  
  if (gimple_assign_nontemporal_move_p (gs))
    pp_string (buffer, "{nt}");
  
  if (if_volatile_cast(gs)) {

    pp_string (buffer, "( ");
    //print out the type of lhs
    tree lhs = gimple_assign_lhs(gs);
    dump_generic_node_c (buffer, TREE_TYPE (lhs), spc, flags, false);
    pp_string (buffer, " )"); 
  }

  pp_space (buffer);
  
  if (gimple_num_ops (gs) == 2)
    dump_unary_rhs (buffer, gs, spc, flags);
  else if (gimple_num_ops (gs) == 3)
    dump_binary_rhs (buffer, gs, spc, flags);
  else if (gimple_num_ops (gs) == 4)
    dump_ternary_rhs (buffer, gs, spc, flags);
  else
    gcc_unreachable ();
  if (!(flags & TDF_RHS_ONLY))
    pp_semicolon(buffer);
    
}


/* Dump the return statement GS.  BUFFER, SPC and FLAGS are as in
   dump_gimple_stmt_c.  */

static void
dump_gimple_return (pretty_printer *buffer, gimple gs, int spc, int flags)
{
  tree t;

  t = gimple_return_retval (gs);
  if (flags & TDF_RAW)
    dump_gimple_fmt (buffer, spc, flags, "%G <%T>", gs, t);
  else
    {
      pp_string (buffer, "return");
      if (t)
	{
	  pp_space (buffer);
	  dump_generic_node_c (buffer, t, spc, flags, false);
	}
      pp_semicolon (buffer);
    }
}


/* Dump the call arguments for a gimple call. BUFFER, FLAGS are as in
   dump_gimple_call.  */

static void
dump_gimple_call_args (pretty_printer *buffer, gimple gs, int flags)
{
  size_t i;

  for (i = 0; i < gimple_call_num_args (gs); i++)
    {
      
      dump_generic_node_c (buffer, gimple_call_arg (gs, i), 0, flags, false);
      
      if ((flags & TDF_PRINT) && i == 0) flags = flags & (~(TDF_PRINT));

      if (i < gimple_call_num_args (gs) - 1)
	pp_string (buffer, ", ");
    }

  if (gimple_call_va_arg_pack_p (gs))
    {
      if (gimple_call_num_args (gs) > 0)
        {
          pp_character (buffer, ',');
          pp_space (buffer);
        }

      pp_string (buffer, "__builtin_va_arg_pack ()");
    }
}

/* Dump the points-to solution *PT to BUFFER.  */

static void
pp_points_to_solution (pretty_printer *buffer, struct pt_solution *pt)
{
  if (pt->anything)
    {
      pp_string (buffer, "anything ");
      return;
    }
  if (pt->nonlocal)
    pp_string (buffer, "nonlocal ");
  if (pt->escaped)
    pp_string (buffer, "escaped ");
  if (pt->ipa_escaped)
    pp_string (buffer, "unit-escaped ");
  if (pt->null)
    pp_string (buffer, "null ");
  if (pt->vars
      && !bitmap_empty_p (pt->vars))
    {
      bitmap_iterator bi;
      unsigned i;
      pp_string (buffer, "{ ");
      EXECUTE_IF_SET_IN_BITMAP (pt->vars, 0, i, bi)
	{
	  tree var = referenced_var_lookup (cfun, i);
	  if (var)
	    {
	      dump_generic_node_c (buffer, var, 0, dump_flags, false);
	      if (DECL_PT_UID (var) != DECL_UID (var))
		{
		  pp_string (buffer, "ptD_");
		  pp_decimal_int (buffer, DECL_PT_UID (var));
		}
	    }
	  else
	    {
	      pp_string (buffer, "D_");
	      pp_decimal_int (buffer, i);
	    }
	  pp_character (buffer, ' ');
	}
      pp_character (buffer, '}');
      if (pt->vars_contains_global)
	pp_string (buffer, " (glob)");
      if (pt->vars_contains_restrict)
	pp_string (buffer, " (restr)");
    }
}

/* Dump the call statement GS.  BUFFER, SPC and FLAGS are as in
   dump_gimple_stmt_c.  */

static void
dump_gimple_call (pretty_printer *buffer, gimple gs, int spc, int flags)
{
  tree lhs = gimple_call_lhs (gs);

  if (flags & TDF_ALIAS)
    {
      struct pt_solution *pt;
      pt = gimple_call_use_set (gs);
      if (!pt_solution_empty_p (pt))
	{
	  pp_string (buffer, "# USE = ");
	  pp_points_to_solution (buffer, pt);
	  newline_and_indent (buffer, spc);
	}
      pt = gimple_call_clobber_set (gs);
      if (!pt_solution_empty_p (pt))
	{
	  pp_string (buffer, "# CLB = ");
	  pp_points_to_solution (buffer, pt);
	  newline_and_indent (buffer, spc);
	}
    }

  if (flags & TDF_RAW)
    {
      dump_gimple_fmt (buffer, spc, flags, "%G <%T, %T",
                     gs, gimple_call_fn (gs), lhs);
      if (gimple_call_num_args (gs) > 0)
        {
          pp_string (buffer, ", ");
          dump_gimple_call_args (buffer, gs, flags);
        }
      pp_character (buffer, '>');
    }
  else
    {
      if (lhs && !(flags & TDF_RHS_ONLY))
        {
          dump_generic_node_c (buffer, lhs, spc, flags, false);
          pp_string (buffer, " =");

	  if (gimple_has_volatile_ops (gs))
	    pp_string (buffer, "{v}");

	  pp_space (buffer);
        }
      print_call_name_c (buffer, gimple_call_fn (gs), spc, flags);
      pp_string (buffer, " (");
      dump_gimple_call_args (buffer, gs, flags);
      pp_character (buffer, ')');
      if (!(flags & TDF_RHS_ONLY))
	pp_semicolon (buffer);
    }

  if (gimple_call_chain (gs))
    {
      pp_string (buffer, " [static-chain: ");
      dump_generic_node_c (buffer, gimple_call_chain (gs), spc, flags, false);
      pp_character (buffer, ']');
    }

  if (gimple_call_return_slot_opt_p (gs))
    pp_string (buffer, " [return slot optimization]");

  if (gimple_call_tail_p (gs))
    pp_string (buffer, " [tail call]");
}


/* Dump the switch statement GS.  BUFFER, SPC and FLAGS are as in
   dump_gimple_stmt_c.  */

static void
dump_gimple_switch (pretty_printer *buffer, gimple gs, int spc, int flags)
{
  unsigned int i;

  GIMPLE_CHECK (gs, GIMPLE_SWITCH);
  if (flags & TDF_RAW)
    dump_gimple_fmt (buffer, spc, flags, "%G <%T, ", gs,
                   gimple_switch_index (gs));
  else
    {
      pp_string (buffer, "switch (");
      dump_generic_node_c (buffer, gimple_switch_index (gs), spc, flags, true);
      pp_string (buffer, ") {");
    }

  /* print "default"(gimple_switch_label(gs,0)) after all cases*/
  for (i = 1; i < gimple_switch_num_labels (gs); i++)
    {
      tree case_label = gimple_switch_label (gs, i);
      if (case_label == NULL_TREE)
	continue;

      dump_generic_node_c (buffer, case_label, spc, flags, false);
      pp_string (buffer, " goto ");
      dump_generic_node_c (buffer, CASE_LABEL (case_label), spc, flags, false);
      //if (i < gimple_switch_num_labels (gs) - 1)
        pp_string (buffer, "; ");
    }
  /* print "default" part*/
  tree case_label = gimple_switch_label (gs, 0);
  if (case_label != NULL_TREE) {
    dump_generic_node_c (buffer, case_label, spc, flags, false);
    pp_string (buffer, " goto ");
    dump_generic_node_c (buffer, CASE_LABEL (case_label), spc, flags, false);
    pp_string (buffer, "; ");
  }

  pp_character (buffer, '}');
}


/* Dump the gimple conditional GS.  BUFFER, SPC and FLAGS are as in
   dump_gimple_stmt_c.  */

static void
dump_gimple_cond (pretty_printer *buffer, gimple gs, int spc, int flags)
{
  if (flags & TDF_RAW)
    dump_gimple_fmt (buffer, spc, flags, "%G <%s, %T, %T, %T, %T>", gs,
                   tree_code_name [gimple_cond_code (gs)],
                   gimple_cond_lhs (gs), gimple_cond_rhs (gs),
                   gimple_cond_true_label (gs), gimple_cond_false_label (gs));
  else
    {
      if (!(flags & TDF_RHS_ONLY))
	pp_string (buffer, "if (");
      dump_generic_node_c (buffer, gimple_cond_lhs (gs), spc, flags, false);
      pp_space (buffer);
      pp_string (buffer, op_symbol_code (gimple_cond_code (gs)));
      pp_space (buffer);
      dump_generic_node_c (buffer, gimple_cond_rhs (gs), spc, flags, false);
      if (!(flags & TDF_RHS_ONLY))
	{
	  pp_character (buffer, ')');

	  if (gimple_cond_true_label (gs))
	    {
	      pp_string (buffer, " goto ");
	      dump_generic_node_c (buffer, gimple_cond_true_label (gs),
				 spc, flags, false);
	      pp_semicolon (buffer);
	    }
	  if (gimple_cond_false_label (gs))
	    {
	      pp_string (buffer, " else goto ");
	      dump_generic_node_c (buffer, gimple_cond_false_label (gs),
				 spc, flags, false);
	      pp_semicolon (buffer);
	    }
	}
    }
}


/* Dump a GIMPLE_LABEL tuple on the pretty_printer BUFFER, SPC
   spaces of indent.  FLAGS specifies details to show in the dump (see
   TDF_* in tree-pass.h).  */

static void
dump_gimple_label (pretty_printer *buffer, gimple gs, int spc, int flags)
{
  tree label = gimple_label_label (gs);
  if (flags & TDF_RAW)
      dump_gimple_fmt (buffer, spc, flags, "%G <%T>", gs, label);
  else
    {
      dump_generic_node_c (buffer, label, spc, flags, false);
      pp_character (buffer, ':');
    }
  if (DECL_NONLOCAL (label))
    pp_string (buffer, " [non-local]");
  if ((flags & TDF_EH) && EH_LANDING_PAD_NR (label))
    pp_printf (buffer, " [LP %d]", EH_LANDING_PAD_NR (label));
}

/* Dump a GIMPLE_GOTO tuple on the pretty_printer BUFFER, SPC
   spaces of indent.  FLAGS specifies details to show in the dump (see
   TDF_* in tree-pass.h).  */

static void
dump_gimple_goto (pretty_printer *buffer, gimple gs, int spc, int flags)
{
  tree label = gimple_goto_dest (gs);
  //  if (flags & TDF_RAW)
  //  dump_gimple_fmt (buffer, spc, flags, "%G <%T>", gs, label);
  // else
    dump_gimple_fmt (buffer, spc, flags, "goto %T;", label);
}


/* Dump a GIMPLE_BIND tuple on the pretty_printer BUFFER, SPC
   spaces of indent.  FLAGS specifies details to show in the dump (see
   TDF_* in tree-pass.h).  */

static void
dump_gimple_bind (pretty_printer *buffer, gimple gs, int spc, int flags)
{
  if (flags & TDF_RAW)
    dump_gimple_fmt (buffer, spc, flags, "%G <", gs);
  else
    pp_character (buffer, '{');
  if (!(flags & TDF_SLIM))
    {
      tree var;

      for (var = gimple_bind_vars (gs); var; var = DECL_CHAIN (var))
	{
          newline_and_indent (buffer, 2);
	  print_declaration (buffer, var, spc, flags);
	}
      if (gimple_bind_vars (gs))
	pp_newline (buffer);
    }
  pp_newline (buffer);
  dump_gimple_seq (buffer, gimple_bind_body (gs), spc + 2, flags);
  newline_and_indent (buffer, spc);
  //if (flags & TDF_RAW)
  //  pp_character (buffer, '>');
  //else
    pp_character (buffer, '}');
}


/* Dump a GIMPLE_TRY tuple on the pretty_printer BUFFER, SPC spaces of
   indent.  FLAGS specifies details to show in the dump (see TDF_* in
   tree-pass.h).  */

static void
dump_gimple_try (pretty_printer *buffer, gimple gs, int spc, int flags)
{
  if (flags & TDF_RAW)
    {
      /* const char *type;
      if (gimple_try_kind (gs) == GIMPLE_TRY_CATCH)
        type = "GIMPLE_TRY_CATCH";
      else if (gimple_try_kind (gs) == GIMPLE_TRY_FINALLY)
        type = "GIMPLE_TRY_FINALLY";
      else
        type = "UNKNOWN GIMPLE_TRY";
      dump_gimple_fmt (buffer, spc, flags,
                       "%G <%s,%+EVAL <%S>%nCLEANUP <%S>%->", gs, type,
                       gimple_try_eval (gs), gimple_try_cleanup (gs));
      */
    }
  else
    {
      pp_string (buffer, "try");
      newline_and_indent (buffer, spc + 2);
      pp_character (buffer, '{');
      pp_newline (buffer);

      dump_gimple_seq (buffer, gimple_try_eval (gs), spc + 4, flags);
      newline_and_indent (buffer, spc + 2);
      pp_character (buffer, '}');

      if (gimple_try_kind (gs) == GIMPLE_TRY_CATCH)
	{
	  newline_and_indent (buffer, spc);
	  pp_string (buffer, "catch");
	  newline_and_indent (buffer, spc + 2);
	  pp_character (buffer, '{');
	}
      else if (gimple_try_kind (gs) == GIMPLE_TRY_FINALLY)
	{
	  newline_and_indent (buffer, spc);
	  pp_string (buffer, "finally");
	  newline_and_indent (buffer, spc + 2);
	  pp_character (buffer, '{');
	}
      else
	pp_string (buffer, " <UNKNOWN GIMPLE_TRY> {");

      pp_newline (buffer);
      dump_gimple_seq (buffer, gimple_try_cleanup (gs), spc + 4, flags);
      newline_and_indent (buffer, spc + 2);
      pp_character (buffer, '}');
    }
}


/* Dump a GIMPLE_CATCH tuple on the pretty_printer BUFFER, SPC spaces of
   indent.  FLAGS specifies details to show in the dump (see TDF_* in
   tree-pass.h).  */

static void
dump_gimple_catch (pretty_printer *buffer, gimple gs, int spc, int flags)
{
  //if (flags & TDF_RAW)
  //  dump_gimple_fmt (buffer, spc, flags, "%G <%T, %+CATCH <%S>%->", gs,
  //                     gimple_catch_types (gs), gimple_catch_handler (gs));
  //else
      dump_gimple_fmt (buffer, spc, flags, "catch (%T)%+{%S}",
                       gimple_catch_types (gs), gimple_catch_handler (gs));
}


/* Dump a GIMPLE_EH_FILTER tuple on the pretty_printer BUFFER, SPC spaces of
   indent.  FLAGS specifies details to show in the dump (see TDF_* in
   tree-pass.h).  */

static void
dump_gimple_eh_filter (pretty_printer *buffer, gimple gs, int spc, int flags)
{
  //if (flags & TDF_RAW)
  // dump_gimple_fmt (buffer, spc, flags, "%G <%T, %+FAILURE <%S>%->", gs,
  //                   gimple_eh_filter_types (gs),
  //                   gimple_eh_filter_failure (gs));
  //else
    dump_gimple_fmt (buffer, spc, flags, "<<<eh_filter (%T)>>>%+{%+%S%-}",
                     gimple_eh_filter_types (gs),
                     gimple_eh_filter_failure (gs));
}


/* Dump a GIMPLE_EH_MUST_NOT_THROW tuple.  */

static void
dump_gimple_eh_must_not_throw (pretty_printer *buffer, gimple gs,
			       int spc, int flags)
{
  //if (flags & TDF_RAW)
  //  dump_gimple_fmt (buffer, spc, flags, "%G <%T>", gs,
  //		     gimple_eh_must_not_throw_fndecl (gs));
  //else
    dump_gimple_fmt (buffer, spc, flags, "<<<eh_must_not_throw (%T)>>>",
		     gimple_eh_must_not_throw_fndecl (gs));
}


/* Dump a GIMPLE_RESX tuple on the pretty_printer BUFFER, SPC spaces of
   indent.  FLAGS specifies details to show in the dump (see TDF_* in
   tree-pass.h).  */

static void
dump_gimple_resx (pretty_printer *buffer, gimple gs, int spc, int flags)
{
  //if (flags & TDF_RAW)
  // dump_gimple_fmt (buffer, spc, flags, "%G <%d>", gs,
  //	     gimple_resx_region (gs));
  //else
    dump_gimple_fmt (buffer, spc, flags, "resx %d", gimple_resx_region (gs));
}

/* Dump a GIMPLE_EH_DISPATCH tuple on the pretty_printer BUFFER.  */

static void
dump_gimple_eh_dispatch (pretty_printer *buffer, gimple gs, int spc, int flags)
{
  //if (flags & TDF_RAW)
  //  dump_gimple_fmt (buffer, spc, flags, "%G <%d>", gs,
  //		     gimple_eh_dispatch_region (gs));
  //else
    dump_gimple_fmt (buffer, spc, flags, "eh_dispatch %d",
		     gimple_eh_dispatch_region (gs));
}

/* Dump a GIMPLE_DEBUG tuple on the pretty_printer BUFFER, SPC spaces
   of indent.  FLAGS specifies details to show in the dump (see TDF_*
   in tree-pass.h).  */

static void
dump_gimple_debug (pretty_printer *buffer, gimple gs, int spc, int flags)
{
  switch (gs->gsbase.subcode)
    {
    case GIMPLE_DEBUG_BIND:
      if (flags & TDF_RAW)
	dump_gimple_fmt (buffer, spc, flags, "%G BIND <%T, %T>", gs,
			 gimple_debug_bind_get_var (gs),
			 gimple_debug_bind_get_value (gs));
      else
	dump_gimple_fmt (buffer, spc, flags, "# DEBUG %T => %T",
			 gimple_debug_bind_get_var (gs),
			 gimple_debug_bind_get_value (gs));
      break;

    default:
      gcc_unreachable ();
    }
}

/* Dump a GIMPLE_OMP_FOR tuple on the pretty_printer BUFFER.  */
static void
dump_gimple_omp_for (pretty_printer *buffer, gimple gs, int spc, int flags)
{
  size_t i;

  if (flags & TDF_RAW)
    {
      dump_gimple_fmt (buffer, spc, flags, "%G <%+BODY <%S>%nCLAUSES <", gs,
                       gimple_omp_body (gs));
      dump_omp_clauses (buffer, gimple_omp_for_clauses (gs), spc, flags);
      dump_gimple_fmt (buffer, spc, flags, " >,");
      for (i = 0; i < gimple_omp_for_collapse (gs); i++)
	dump_gimple_fmt (buffer, spc, flags,
			 "%+%T, %T, %T, %s, %T,%n",
			 gimple_omp_for_index (gs, i),
			 gimple_omp_for_initial (gs, i),
			 gimple_omp_for_final (gs, i),
			 tree_code_name[gimple_omp_for_cond (gs, i)],
			 gimple_omp_for_incr (gs, i));
      dump_gimple_fmt (buffer, spc, flags, "PRE_BODY <%S>%->",
		       gimple_omp_for_pre_body (gs));
    }
  else
    {
      pp_string (buffer, "#pragma omp for");
      dump_omp_clauses (buffer, gimple_omp_for_clauses (gs), spc, flags);
      for (i = 0; i < gimple_omp_for_collapse (gs); i++)
	{
	  if (i)
	    spc += 2;
	  newline_and_indent (buffer, spc);
	  pp_string (buffer, "for (");
	  dump_generic_node_c (buffer, gimple_omp_for_index (gs, i), spc,
			     flags, false);
	  pp_string (buffer, " = ");
	  dump_generic_node_c (buffer, gimple_omp_for_initial (gs, i), spc,
			     flags, false);
	  pp_string (buffer, "; ");

	  dump_generic_node_c (buffer, gimple_omp_for_index (gs, i), spc,
			     flags, false);
	  pp_space (buffer);
	  switch (gimple_omp_for_cond (gs, i))
	    {
	    case LT_EXPR:
	      pp_character (buffer, '<');
	      break;
	    case GT_EXPR:
	      pp_character (buffer, '>');
	      break;
	    case LE_EXPR:
	      pp_string (buffer, "<=");
	      break;
	    case GE_EXPR:
	      pp_string (buffer, ">=");
	      break;
	    default:
	      gcc_unreachable ();
	    }
	  pp_space (buffer);
	  dump_generic_node_c (buffer, gimple_omp_for_final (gs, i), spc,
			     flags, false);
	  pp_string (buffer, "; ");

	  dump_generic_node_c (buffer, gimple_omp_for_index (gs, i), spc,
			     flags, false);
	  pp_string (buffer, " = ");
	  dump_generic_node_c (buffer, gimple_omp_for_incr (gs, i), spc,
			     flags, false);
	  pp_character (buffer, ')');
	}

      if (!gimple_seq_empty_p (gimple_omp_body (gs)))
	{
	  newline_and_indent (buffer, spc + 2);
	  pp_character (buffer, '{');
	  pp_newline (buffer);
	  dump_gimple_seq (buffer, gimple_omp_body (gs), spc + 4, flags);
	  newline_and_indent (buffer, spc + 2);
	  pp_character (buffer, '}');
	}
    }
}

/* Dump a GIMPLE_OMP_CONTINUE tuple on the pretty_printer BUFFER.  */

static void
dump_gimple_omp_continue (pretty_printer *buffer, gimple gs, int spc, int flags)
{
  if (flags & TDF_RAW)
    {
      dump_gimple_fmt (buffer, spc, flags, "%G <%T, %T>", gs,
                       gimple_omp_continue_control_def (gs),
                       gimple_omp_continue_control_use (gs));
    }
  else
    {
      pp_string (buffer, "#pragma omp continue (");
      dump_generic_node_c (buffer, gimple_omp_continue_control_def (gs),
	  		 spc, flags, false);
      pp_character (buffer, ',');
      pp_space (buffer);
      dump_generic_node_c (buffer, gimple_omp_continue_control_use (gs),
	  		 spc, flags, false);
      pp_character (buffer, ')');
    }
}

/* Dump a GIMPLE_OMP_SINGLE tuple on the pretty_printer BUFFER.  */

static void
dump_gimple_omp_single (pretty_printer *buffer, gimple gs, int spc, int flags)
{
  if (flags & TDF_RAW)
    {
      dump_gimple_fmt (buffer, spc, flags, "%G <%+BODY <%S>%nCLAUSES <", gs,
		       gimple_omp_body (gs));
      dump_omp_clauses (buffer, gimple_omp_single_clauses (gs), spc, flags);
      dump_gimple_fmt (buffer, spc, flags, " >");
    }
  else
    {
      pp_string (buffer, "#pragma omp single");
      dump_omp_clauses (buffer, gimple_omp_single_clauses (gs), spc, flags);
      if (!gimple_seq_empty_p (gimple_omp_body (gs)))
	{
	  newline_and_indent (buffer, spc + 2);
	  pp_character (buffer, '{');
	  pp_newline (buffer);
	  dump_gimple_seq (buffer, gimple_omp_body (gs), spc + 4, flags);
	  newline_and_indent (buffer, spc + 2);
	  pp_character (buffer, '}');
	}
    }
}

/* Dump a GIMPLE_OMP_SECTIONS tuple on the pretty_printer BUFFER.  */

static void
dump_gimple_omp_sections (pretty_printer *buffer, gimple gs, int spc,
			  int flags)
{
  if (flags & TDF_RAW)
    {
      dump_gimple_fmt (buffer, spc, flags, "%G <%+BODY <%S>%nCLAUSES <", gs,
		       gimple_omp_body (gs));
      dump_omp_clauses (buffer, gimple_omp_sections_clauses (gs), spc, flags);
      dump_gimple_fmt (buffer, spc, flags, " >");
    }
  else
    {
      pp_string (buffer, "#pragma omp sections");
      if (gimple_omp_sections_control (gs))
	{
	  pp_string (buffer, " <");
	  dump_generic_node_c (buffer, gimple_omp_sections_control (gs), spc,
			     flags, false);
	  pp_character (buffer, '>');
	}
      dump_omp_clauses (buffer, gimple_omp_sections_clauses (gs), spc, flags);
      if (!gimple_seq_empty_p (gimple_omp_body (gs)))
	{
	  newline_and_indent (buffer, spc + 2);
	  pp_character (buffer, '{');
	  pp_newline (buffer);
	  dump_gimple_seq (buffer, gimple_omp_body (gs), spc + 4, flags);
	  newline_and_indent (buffer, spc + 2);
	  pp_character (buffer, '}');
	}
    }
}

/* Dump a GIMPLE_OMP_{MASTER,ORDERED,SECTION} tuple on the pretty_printer
   BUFFER.  */

static void
dump_gimple_omp_block (pretty_printer *buffer, gimple gs, int spc, int flags)
{
  if (flags & TDF_RAW)
    dump_gimple_fmt (buffer, spc, flags, "%G <%+BODY <%S> >", gs,
		     gimple_omp_body (gs));
  else
    {
      switch (gimple_code (gs))
	{
	case GIMPLE_OMP_MASTER:
	  pp_string (buffer, "#pragma omp master");
	  break;
	case GIMPLE_OMP_ORDERED:
	  pp_string (buffer, "#pragma omp ordered");
	  break;
	case GIMPLE_OMP_SECTION:
	  pp_string (buffer, "#pragma omp section");
	  break;
	default:
	  gcc_unreachable ();
	}
      if (!gimple_seq_empty_p (gimple_omp_body (gs)))
	{
	  newline_and_indent (buffer, spc + 2);
	  pp_character (buffer, '{');
	  pp_newline (buffer);
	  dump_gimple_seq (buffer, gimple_omp_body (gs), spc + 4, flags);
	  newline_and_indent (buffer, spc + 2);
	  pp_character (buffer, '}');
	}
    }
}

/* Dump a GIMPLE_OMP_CRITICAL tuple on the pretty_printer BUFFER.  */

static void
dump_gimple_omp_critical (pretty_printer *buffer, gimple gs, int spc,
			  int flags)
{
  if (flags & TDF_RAW)
    dump_gimple_fmt (buffer, spc, flags, "%G <%+BODY <%S> >", gs,
		     gimple_omp_body (gs));
  else
    {
      pp_string (buffer, "#pragma omp critical");
      if (gimple_omp_critical_name (gs))
	{
	  pp_string (buffer, " (");
	  dump_generic_node_c (buffer, gimple_omp_critical_name (gs), spc,
			     flags, false);
	  pp_character (buffer, ')');
	}
      if (!gimple_seq_empty_p (gimple_omp_body (gs)))
	{
	  newline_and_indent (buffer, spc + 2);
	  pp_character (buffer, '{');
	  pp_newline (buffer);
	  dump_gimple_seq (buffer, gimple_omp_body (gs), spc + 4, flags);
	  newline_and_indent (buffer, spc + 2);
	  pp_character (buffer, '}');
	}
    }
}

/* Dump a GIMPLE_OMP_RETURN tuple on the pretty_printer BUFFER.  */

static void
dump_gimple_omp_return (pretty_printer *buffer, gimple gs, int spc, int flags)
{
  if (flags & TDF_RAW)
    {
      dump_gimple_fmt (buffer, spc, flags, "%G <nowait=%d>", gs,
                       (int) gimple_omp_return_nowait_p (gs));
    }
  else
    {
      pp_string (buffer, "#pragma omp return");
      if (gimple_omp_return_nowait_p (gs))
	pp_string (buffer, "(nowait)");
    }
}

/* Dump a GIMPLE_ASM tuple on the pretty_printer BUFFER, SPC spaces of
   indent.  FLAGS specifies details to show in the dump (see TDF_* in
   tree-pass.h).  */

static void
dump_gimple_asm (pretty_printer *buffer, gimple gs, int spc, int flags)
{
  unsigned int i, n, f, fields;

  if (flags & TDF_RAW)
    {
      dump_gimple_fmt (buffer, spc, flags, "%G <%+STRING <%n%s%n>", gs,
                       gimple_asm_string (gs));

      n = gimple_asm_noutputs (gs);
      if (n)
	{
	  newline_and_indent (buffer, spc + 2);
	  pp_string (buffer, "OUTPUT: ");
	  for (i = 0; i < n; i++)
	    {
	      dump_generic_node_c (buffer, gimple_asm_output_op (gs, i),
				 spc, flags, false);
	      if (i < n - 1)
		pp_string (buffer, ", ");
	    }
	}

      n = gimple_asm_ninputs (gs);
      if (n)
	{
	  newline_and_indent (buffer, spc + 2);
	  pp_string (buffer, "INPUT: ");
	  for (i = 0; i < n; i++)
	    {
	      dump_generic_node_c (buffer, gimple_asm_input_op (gs, i),
				 spc, flags, false);
	      if (i < n - 1)
		pp_string (buffer, ", ");
	    }
	}

      n = gimple_asm_nclobbers (gs);
      if (n)
	{
	  newline_and_indent (buffer, spc + 2);
	  pp_string (buffer, "CLOBBER: ");
	  for (i = 0; i < n; i++)
	    {
	      dump_generic_node_c (buffer, gimple_asm_clobber_op (gs, i),
				 spc, flags, false);
	      if (i < n - 1)
		pp_string (buffer, ", ");
	    }
	}

      n = gimple_asm_nlabels (gs);
      if (n)
	{
	  newline_and_indent (buffer, spc + 2);
	  pp_string (buffer, "LABEL: ");
	  for (i = 0; i < n; i++)
	    {
	      dump_generic_node_c (buffer, gimple_asm_label_op (gs, i),
				 spc, flags, false);
	      if (i < n - 1)
		pp_string (buffer, ", ");
	    }
	}

      newline_and_indent (buffer, spc);
      pp_character (buffer, '>');
    }
  else
    {
      pp_string (buffer, "__asm");
      if (gimple_asm_volatile_p (gs))
	pp_string (buffer, " volatile");
      if (gimple_asm_nlabels (gs))
	pp_string (buffer, " goto");
      pp_string (buffer, "(\"");
      pp_string (buffer, gimple_asm_string (gs));
      pp_string (buffer, "\"");

      if (gimple_asm_nlabels (gs))
	fields = 4;
      else if (gimple_asm_nclobbers (gs))
	fields = 3;
      else if (gimple_asm_ninputs (gs))
	fields = 2;
      else if (gimple_asm_noutputs (gs))
	fields = 1;
      else
	fields = 0;
      
      flags = flags | TDF_ASM;
      for (f = 0; f < fields; ++f)
	{
	  pp_string (buffer, " : ");
	  // for different field, the IR is defferent. for clobbers parts, there is no "PURPOSE_NODE". 
	  // for input and output part, there is "PURPOSE_NODE". e.g "r=" is an purpose_node in a tree_list
	  if (f >= 2 ) flags = flags & (!TDF_ASM); 
	  switch (f)
	    {
	    case 0:
	      n = gimple_asm_noutputs (gs);
	      for (i = 0; i < n; i++)
		{
		  dump_generic_node_c (buffer, gimple_asm_output_op (gs, i),
				     spc, flags, false);
		  if (i < n - 1)
		    pp_string (buffer, ", ");
		}
	      break;

	    case 1:
	      n = gimple_asm_ninputs (gs);
	      for (i = 0; i < n; i++)
		{
		  dump_generic_node_c (buffer, gimple_asm_input_op (gs, i),
				     spc, flags, false);
		  if (i < n - 1)
		    pp_string (buffer, ", ");
		}
	      break;

	    case 2:
	      n = gimple_asm_nclobbers (gs);
	      for (i = 0; i < n; i++)
		{
		  dump_generic_node_c (buffer, gimple_asm_clobber_op (gs, i),
				     spc, flags, false);
		  if (i < n - 1)
		    pp_string (buffer, ", ");
		}
	      break;

	    case 3:
	      n = gimple_asm_nlabels (gs);
	      for (i = 0; i < n; i++)
		{
		  dump_generic_node_c (buffer, gimple_asm_label_op (gs, i),
				     spc, flags, false);
		  if (i < n - 1)
		    pp_string (buffer, ", ");
		}
	      break;

	    default:
	      gcc_unreachable ();
	    }
	}

      pp_string (buffer, ");");
    }
}


/* Dump a PHI node PHI.  BUFFER, SPC and FLAGS are as in
   dump_gimple_stmt_c.  */

static void
dump_gimple_phi (pretty_printer *buffer, gimple phi, int spc, int flags)
{
  size_t i;
  tree lhs = gimple_phi_result (phi);

  if (flags & TDF_ALIAS
      && POINTER_TYPE_P (TREE_TYPE (lhs))
      && SSA_NAME_PTR_INFO (lhs))
    {
      struct ptr_info_def *pi = SSA_NAME_PTR_INFO (lhs);
      pp_string (buffer, "PT = ");
      pp_points_to_solution (buffer, &pi->pt);
      newline_and_indent (buffer, spc);
      if (pi->align != 1)
	pp_printf (buffer, "# ALIGN = %u, MISALIGN = %u",
		   pi->align, pi->misalign);
      newline_and_indent (buffer, spc);
      pp_string (buffer, "# ");
    }

  if (flags & TDF_RAW)
      dump_gimple_fmt (buffer, spc, flags, "%G <%T, ", phi,
                       gimple_phi_result (phi));
  else
    {
      dump_generic_node_c (buffer, lhs, spc, flags, false);
      pp_string (buffer, " = PHI <");
    }
  for (i = 0; i < gimple_phi_num_args (phi); i++)
    {
      if ((flags & TDF_LINENO) && gimple_phi_arg_has_location (phi, i))
        {
	  expanded_location xloc;

	  xloc = expand_location (gimple_phi_arg_location (phi, i));
	  pp_character (buffer, '[');
	  if (xloc.file)
	    {
	      pp_string (buffer, xloc.file);
	      pp_string (buffer, " : ");
	    }
	  pp_decimal_int (buffer, xloc.line);
	  pp_string (buffer, ":");
	  pp_decimal_int (buffer, xloc.column);
	  pp_string (buffer, "] ");
	}
      dump_generic_node_c (buffer, gimple_phi_arg_def (phi, i), spc, flags,
			 false);
      pp_character (buffer, '(');
      pp_decimal_int (buffer, gimple_phi_arg_edge (phi, i)->src->index);
      pp_character (buffer, ')');
      if (i < gimple_phi_num_args (phi) - 1)
	pp_string (buffer, ", ");
    }
  pp_character (buffer, '>');
}


/* Dump a GIMPLE_OMP_PARALLEL tuple on the pretty_printer BUFFER, SPC spaces
   of indent.  FLAGS specifies details to show in the dump (see TDF_* in
   tree-pass.h).  */

static void
dump_gimple_omp_parallel (pretty_printer *buffer, gimple gs, int spc,
                          int flags)
{
  if (flags & TDF_RAW)
    {
      dump_gimple_fmt (buffer, spc, flags, "%G <%+BODY <%S>%nCLAUSES <", gs,
                       gimple_omp_body (gs));
      dump_omp_clauses (buffer, gimple_omp_parallel_clauses (gs), spc, flags);
      dump_gimple_fmt (buffer, spc, flags, " >, %T, %T%n>",
                       gimple_omp_parallel_child_fn (gs),
                       gimple_omp_parallel_data_arg (gs));
    }
  else
    {
      gimple_seq body;
      pp_string (buffer, "#pragma omp parallel");
      dump_omp_clauses (buffer, gimple_omp_parallel_clauses (gs), spc, flags);
      if (gimple_omp_parallel_child_fn (gs))
	{
	  pp_string (buffer, " [child fn: ");
	  dump_generic_node_c (buffer, gimple_omp_parallel_child_fn (gs),
			     spc, flags, false);
	  pp_string (buffer, " (");
	  if (gimple_omp_parallel_data_arg (gs))
	    dump_generic_node_c (buffer, gimple_omp_parallel_data_arg (gs),
			       spc, flags, false);
	  else
	    pp_string (buffer, "???");
	  pp_string (buffer, ")]");
	}
      body = gimple_omp_body (gs);
      if (body && gimple_code (gimple_seq_first_stmt (body)) != GIMPLE_BIND)
	{
	  newline_and_indent (buffer, spc + 2);
	  pp_character (buffer, '{');
	  pp_newline (buffer);
	  dump_gimple_seq (buffer, body, spc + 4, flags);
	  newline_and_indent (buffer, spc + 2);
	  pp_character (buffer, '}');
	}
      else if (body)
	{
	  pp_newline (buffer);
	  dump_gimple_seq (buffer, body, spc + 2, flags);
	}
    }
}


/* Dump a GIMPLE_OMP_TASK tuple on the pretty_printer BUFFER, SPC spaces
   of indent.  FLAGS specifies details to show in the dump (see TDF_* in
   tree-pass.h).  */

static void
dump_gimple_omp_task (pretty_printer *buffer, gimple gs, int spc,
		      int flags)
{
  if (flags & TDF_RAW)
    {
      dump_gimple_fmt (buffer, spc, flags, "%G <%+BODY <%S>%nCLAUSES <", gs,
                       gimple_omp_body (gs));
      dump_omp_clauses (buffer, gimple_omp_task_clauses (gs), spc, flags);
      dump_gimple_fmt (buffer, spc, flags, " >, %T, %T, %T, %T, %T%n>",
                       gimple_omp_task_child_fn (gs),
                       gimple_omp_task_data_arg (gs),
		       gimple_omp_task_copy_fn (gs),
		       gimple_omp_task_arg_size (gs),
		       gimple_omp_task_arg_size (gs));
    }
  else
    {
      gimple_seq body;
      pp_string (buffer, "#pragma omp task");
      dump_omp_clauses (buffer, gimple_omp_task_clauses (gs), spc, flags);
      if (gimple_omp_task_child_fn (gs))
	{
	  pp_string (buffer, " [child fn: ");
	  dump_generic_node_c (buffer, gimple_omp_task_child_fn (gs),
			     spc, flags, false);
	  pp_string (buffer, " (");
	  if (gimple_omp_task_data_arg (gs))
	    dump_generic_node_c (buffer, gimple_omp_task_data_arg (gs),
			       spc, flags, false);
	  else
	    pp_string (buffer, "???");
	  pp_string (buffer, ")]");
	}
      body = gimple_omp_body (gs);
      if (body && gimple_code (gimple_seq_first_stmt (body)) != GIMPLE_BIND)
	{
	  newline_and_indent (buffer, spc + 2);
	  pp_character (buffer, '{');
	  pp_newline (buffer);
	  dump_gimple_seq (buffer, body, spc + 4, flags);
	  newline_and_indent (buffer, spc + 2);
	  pp_character (buffer, '}');
	}
      else if (body)
	{
	  pp_newline (buffer);
	  dump_gimple_seq (buffer, body, spc + 2, flags);
	}
    }
}


/* Dump a GIMPLE_OMP_ATOMIC_LOAD tuple on the pretty_printer BUFFER, SPC
   spaces of indent.  FLAGS specifies details to show in the dump (see TDF_*
   in tree-pass.h).  */

static void
dump_gimple_omp_atomic_load (pretty_printer *buffer, gimple gs, int spc,
                             int flags)
{
  if (flags & TDF_RAW)
    {
      dump_gimple_fmt (buffer, spc, flags, "%G <%T, %T>", gs,
                       gimple_omp_atomic_load_lhs (gs),
                       gimple_omp_atomic_load_rhs (gs));
    }
  else
    {
      pp_string (buffer, "#pragma omp atomic_load");
      newline_and_indent (buffer, spc + 2);
      dump_generic_node_c (buffer, gimple_omp_atomic_load_lhs (gs),
	  		 spc, flags, false);
      pp_space (buffer);
      pp_character (buffer, '=');
      pp_space (buffer);
      pp_character (buffer, '*');
      dump_generic_node_c (buffer, gimple_omp_atomic_load_rhs (gs),
	  		 spc, flags, false);
    }
}

/* Dump a GIMPLE_OMP_ATOMIC_STORE tuple on the pretty_printer BUFFER, SPC
   spaces of indent.  FLAGS specifies details to show in the dump (see TDF_*
   in tree-pass.h).  */

static void
dump_gimple_omp_atomic_store (pretty_printer *buffer, gimple gs, int spc,
                             int flags)
{
  if (flags & TDF_RAW)
    {
      dump_gimple_fmt (buffer, spc, flags, "%G <%T>", gs,
                       gimple_omp_atomic_store_val (gs));
    }
  else
    {
      pp_string (buffer, "#pragma omp atomic_store (");
      dump_generic_node_c (buffer, gimple_omp_atomic_store_val (gs),
	  		 spc, flags, false);
      pp_character (buffer, ')');
    }
}


/* Dump all the memory operands for statement GS.  BUFFER, SPC and
   FLAGS are as in dump_gimple_stmt_c.  */

static void
dump_gimple_mem_ops (pretty_printer *buffer, gimple gs, int spc, int flags)
{
  tree vdef = gimple_vdef (gs);
  tree vuse = gimple_vuse (gs);

  if (!ssa_operands_active () || !gimple_references_memory_p (gs))
    return;

  if (vdef != NULL_TREE)
    {
      pp_string (buffer, "# ");
      dump_generic_node_c (buffer, vdef, spc + 2, flags, false);
      pp_string (buffer, " = VDEF <");
      dump_generic_node_c (buffer, vuse, spc + 2, flags, false);
      pp_character (buffer, '>');
      newline_and_indent (buffer, spc);
    }
  else if (vuse != NULL_TREE)
    {
      pp_string (buffer, "# VUSE <");
      dump_generic_node_c (buffer, vuse, spc + 2, flags, false);
      pp_character (buffer, '>');
      newline_and_indent (buffer, spc);
    }
}


/* Dump the gimple statement GS on the pretty printer BUFFER, SPC
   spaces of indent.  FLAGS specifies details to show in the dump (see
   TDF_* in tree-pass.h).  */

void
dump_gimple_stmt_c (pretty_printer *buffer, gimple gs, int spc, int flags)
{
  if (!gs)
    return;

  if (flags & TDF_STMTADDR)
    pp_printf (buffer, "<&%p> ", (void *) gs);

  if ((flags & TDF_LINENO) && gimple_has_location (gs))
    {
      expanded_location xloc = expand_location (gimple_location (gs));
      pp_character (buffer, '[');
      if (xloc.file)
	{
	  pp_string (buffer, xloc.file);
	  pp_string (buffer, " : ");
	}
      pp_decimal_int (buffer, xloc.line);
      pp_string (buffer, ":");
      pp_decimal_int (buffer, xloc.column);
      pp_string (buffer, "] ");
    }

  if (flags & TDF_EH)
    {
      int lp_nr = lookup_stmt_eh_lp (gs);
      if (lp_nr > 0)
	pp_printf (buffer, "[LP %d] ", lp_nr);
      else if (lp_nr < 0)
	pp_printf (buffer, "[MNT %d] ", -lp_nr);
    }

  if ((flags & (TDF_VOPS|TDF_MEMSYMS))
      && gimple_has_mem_ops (gs))
    dump_gimple_mem_ops (buffer, gs, spc, flags);

  if ((flags & TDF_ALIAS)
      && gimple_has_lhs (gs))
    {
      tree lhs = gimple_get_lhs (gs);
      if (TREE_CODE (lhs) == SSA_NAME
	  && POINTER_TYPE_P (TREE_TYPE (lhs))
	  && SSA_NAME_PTR_INFO (lhs))
	{
	  struct ptr_info_def *pi = SSA_NAME_PTR_INFO (lhs);
	  pp_string (buffer, "# PT = ");
	  pp_points_to_solution (buffer, &pi->pt);
	  newline_and_indent (buffer, spc);
	  if (pi->align != 1)
	    {
	      pp_printf (buffer, "# ALIGN = %u, MISALIGN = %u",
			 pi->align, pi->misalign);
	      newline_and_indent (buffer, spc);
	    }
	}
    }

  switch (gimple_code (gs))
    {
    case GIMPLE_ASM:
      dump_gimple_asm (buffer, gs, spc, flags);
      break;

    case GIMPLE_ASSIGN:
      dump_gimple_assign (buffer, gs, spc, flags);
      break;

    case GIMPLE_BIND:
      dump_gimple_bind (buffer, gs, spc, flags);
      break;

    case GIMPLE_CALL:
      dump_gimple_call (buffer, gs, spc, flags);
      break;

    case GIMPLE_COND:
      dump_gimple_cond (buffer, gs, spc, flags);
      break;

    case GIMPLE_LABEL:
      dump_gimple_label (buffer, gs, spc, flags);
      break;

    case GIMPLE_GOTO:
      dump_gimple_goto (buffer, gs, spc, flags);
      break;

    case GIMPLE_NOP:
      pp_string (buffer, "GIMPLE_NOP");
      break;

    case GIMPLE_RETURN:
      dump_gimple_return (buffer, gs, spc, flags);
      break;

    case GIMPLE_SWITCH:
      dump_gimple_switch (buffer, gs, spc, flags);
      break;

    case GIMPLE_TRY:
      dump_gimple_try (buffer, gs, spc, flags);
      break;

    case GIMPLE_PHI:
      dump_gimple_phi (buffer, gs, spc, flags);
      break;

    case GIMPLE_OMP_PARALLEL:
      dump_gimple_omp_parallel (buffer, gs, spc, flags);
      break;

    case GIMPLE_OMP_TASK:
      dump_gimple_omp_task (buffer, gs, spc, flags);
      break;

    case GIMPLE_OMP_ATOMIC_LOAD:
      dump_gimple_omp_atomic_load (buffer, gs, spc, flags);

      break;

    case GIMPLE_OMP_ATOMIC_STORE:
      dump_gimple_omp_atomic_store (buffer, gs, spc, flags);
      break;

    case GIMPLE_OMP_FOR:
      dump_gimple_omp_for (buffer, gs, spc, flags);
      break;

    case GIMPLE_OMP_CONTINUE:
      dump_gimple_omp_continue (buffer, gs, spc, flags);
      break;

    case GIMPLE_OMP_SINGLE:
      dump_gimple_omp_single (buffer, gs, spc, flags);
      break;

    case GIMPLE_OMP_RETURN:
      dump_gimple_omp_return (buffer, gs, spc, flags);
      break;

    case GIMPLE_OMP_SECTIONS:
      dump_gimple_omp_sections (buffer, gs, spc, flags);
      break;

    case GIMPLE_OMP_SECTIONS_SWITCH:
      pp_string (buffer, "GIMPLE_SECTIONS_SWITCH");
      break;

    case GIMPLE_OMP_MASTER:
    case GIMPLE_OMP_ORDERED:
    case GIMPLE_OMP_SECTION:
      dump_gimple_omp_block (buffer, gs, spc, flags);
      break;

    case GIMPLE_OMP_CRITICAL:
      dump_gimple_omp_critical (buffer, gs, spc, flags);
      break;

		case GIMPLE_UPP_LOOP:
			pp_string (buffer, "#pragma upp loop");
			break;

    case GIMPLE_CATCH:
      dump_gimple_catch (buffer, gs, spc, flags);
      break;

    case GIMPLE_EH_FILTER:
      dump_gimple_eh_filter (buffer, gs, spc, flags);
      break;

    case GIMPLE_EH_MUST_NOT_THROW:
      dump_gimple_eh_must_not_throw (buffer, gs, spc, flags);
      break;

    case GIMPLE_RESX:
      dump_gimple_resx (buffer, gs, spc, flags);
      break;

    case GIMPLE_EH_DISPATCH:
      dump_gimple_eh_dispatch (buffer, gs, spc, flags);
      break;

    case GIMPLE_DEBUG:
      dump_gimple_debug (buffer, gs, spc, flags);
      break;

    case GIMPLE_PREDICT:
      pp_string (buffer, "// predicted ");
      if (gimple_predict_outcome (gs))
	pp_string (buffer, "likely by ");
      else
	pp_string (buffer, "unlikely by ");
      pp_string (buffer, predictor_name (gimple_predict_predictor (gs)));
      pp_string (buffer, " predictor.");
      break;

    default:
      GIMPLE_NIY;
    }

  pp_newline(buffer);
  if (!(flags & TDF_DIAGNOSTIC))
    pp_write_text_to_stream (buffer);
}


/* Dumps header of basic block BB to buffer BUFFER indented by INDENT
   spaces and details described by flags.  */
static void
dump_bb_header (pretty_printer *buffer, basic_block bb, int indent, int flags)
{
  edge e;
  gimple stmt;
  edge_iterator ei;

  if (flags & TDF_BLOCKS)
    {
      INDENT (indent);
      pp_string (buffer, "# BLOCK ");
      pp_decimal_int (buffer, bb->index);
      if (bb->frequency)
	{
          pp_string (buffer, " freq:");
          pp_decimal_int (buffer, bb->frequency);
	}
      if (bb->count)
	{
          pp_string (buffer, " count:");
          pp_widest_integer (buffer, bb->count);
	}

      if (flags & TDF_LINENO)
	{
	  gimple_stmt_iterator gsi;

	  for (gsi = gsi_start_bb (bb); !gsi_end_p (gsi); gsi_next (&gsi))
	    if (!is_gimple_debug (gsi_stmt (gsi))
		&& get_lineno (gsi_stmt (gsi)) != UNKNOWN_LOCATION)
	      {
		pp_string (buffer, ", starting at line ");
		pp_decimal_int (buffer, get_lineno (gsi_stmt (gsi)));
		break;
	      }

          if (bb->discriminator)
            {
              pp_string (buffer, ", discriminator ");
	      pp_decimal_int (buffer, bb->discriminator);
            }
	}
      newline_and_indent (buffer, indent);

      pp_string (buffer, "# PRED:");
      pp_write_text_to_stream (buffer);
      FOR_EACH_EDGE (e, ei, bb->preds)
	if (flags & TDF_SLIM)
	  {
	    pp_character (buffer, ' ');
	    if (e->src == ENTRY_BLOCK_PTR)
	      pp_string (buffer, "ENTRY");
	    else
	      pp_decimal_int (buffer, e->src->index);
	  }
	else
	  dump_edge_info (buffer->buffer->stream, e, 0);
      pp_newline (buffer);
    }
  else
    {
      stmt = first_stmt (bb);
      if (!stmt || gimple_code (stmt) != GIMPLE_LABEL)
	{
	  int temp = is_in_lable_stack(bb->index);
	  if (temp >= 0){
	    INDENT (indent - 2);
	    pp_string (buffer, "BB_");
	    pp_decimal_int (buffer, temp);
	    pp_string (buffer, ":");
	    pp_newline (buffer);
	  }
	}
    }
  pp_write_text_to_stream (buffer);
  if (cfun)
    check_bb_profile (bb, buffer->buffer->stream);
}


/* Dumps end of basic block BB to buffer BUFFER indented by INDENT
   spaces.  */

static void
dump_bb_end (pretty_printer *buffer, basic_block bb, int indent, int flags)
{
  edge e;
  edge_iterator ei;

  INDENT (indent);
  pp_string (buffer, "# SUCC:");
  pp_write_text_to_stream (buffer);
  FOR_EACH_EDGE (e, ei, bb->succs)
    if (flags & TDF_SLIM)
      {
	pp_character (buffer, ' ');
	if (e->dest == EXIT_BLOCK_PTR)
	  pp_string (buffer, "EXIT");
	else
	  pp_decimal_int (buffer, e->dest->index);
      }
    else
      dump_edge_info (buffer->buffer->stream, e, 1);
  pp_newline (buffer);
}


/* Dump PHI nodes of basic block BB to BUFFER with details described
   by FLAGS and indented by INDENT spaces.  */

static void
dump_phi_nodes (pretty_printer *buffer, basic_block bb, int indent, int flags)
{
  gimple_stmt_iterator i;

  for (i = gsi_start_phis (bb); !gsi_end_p (i); gsi_next (&i))
    {
      gimple phi = gsi_stmt (i);
      if (is_gimple_reg (gimple_phi_result (phi)) || (flags & TDF_VOPS))
        {
          INDENT (indent);
          pp_string (buffer, "# ");
          dump_gimple_phi (buffer, phi, indent, flags);
          pp_newline (buffer);
        }
    }
}


/* Dump jump to basic block BB that is represented implicitly in the cfg
   to BUFFER.  */

static void
pp_cfg_jump (pretty_printer *buffer, basic_block bb)
{
  gimple stmt;

  stmt = first_stmt (bb);

  if (stmt && gimple_code (stmt) == GIMPLE_LABEL){
    pp_string(buffer, "goto ");
    dump_generic_node_c(buffer, gimple_label_label(stmt), 0, 0, false);
  }else {
    pp_string (buffer, "goto BB_");
    pp_decimal_int (buffer, is_in_lable_stack(bb->index));
  }
  pp_semicolon (buffer);

}


/* record the labled should be printed*/
static void push_lable (int bbindex){
  lablestack[bbindex] = lable;
  lable++;
}

/* initial the lable stack*/
static void initialize_lable(){
  int i;
  for (i = 0; i < 100; i++)
    lablestack[i] = -1;
    
}

/* check if the lable should be printed*/
static int is_in_lable_stack(int bbindex){
  return lablestack[bbindex];
}


/* to decide which label should be printed */
void pre_check_bb_edge (tree decl, FILE *dumpfile){

  basic_block bb;
  initialize_lable();
  FOR_EACH_BB(bb)
  {
    edge e;
    gimple stmt;
    stmt = last_stmt(bb);
    if (stmt && gimple_code (stmt) == GIMPLE_COND) {
      edge true_edge, false_edge;
      extract_true_false_edges_from_block (bb, &true_edge, &false_edge);
      push_lable(true_edge->dest->index);
      push_lable(false_edge->dest->index);
    }

    e = find_fallthru_edge (bb->succs);

    if (e && e->dest != bb->next_bb)
      	push_lable( e->dest->index);

  }


}

/* Dump edges-presented implicitly in basic block BB to BUFFER, indented
   by INDENT spaces, with details given by FLAGS.  */

static void
dump_implicit_edges (pretty_printer *buffer, basic_block bb, int indent,
		     int flags)
{
  edge e;
  gimple stmt;

  stmt = last_stmt (bb);

  if (stmt && gimple_code (stmt) == GIMPLE_COND)
    {
      edge true_edge, false_edge;

      /* When we are emitting the code or changing CFG, it is possible that
	 the edges are not yet created.  When we are using debug_bb in such
	 a situation, we do not want it to crash.  */
      if (EDGE_COUNT (bb->succs) != 2)
	return;
      extract_true_false_edges_from_block (bb, &true_edge, &false_edge);

      INDENT (indent + 2);
      pp_cfg_jump (buffer, true_edge->dest);
      newline_and_indent (buffer, indent);
      pp_string (buffer, "else");
      newline_and_indent (buffer, indent + 2);
      pp_cfg_jump (buffer, false_edge->dest);
      pp_newline (buffer);
      return;
    }

  /* If there is a fallthru edge, we may need to add an artificial
     goto to the dump.  */
  e = find_fallthru_edge (bb->succs);

  if (e && e->dest != bb->next_bb)
    {
      INDENT (indent);

      if ((flags & TDF_LINENO)
	  && e->goto_locus != UNKNOWN_LOCATION
	  )
	{
	  expanded_location goto_xloc;
	  goto_xloc = expand_location (e->goto_locus);
	  pp_character (buffer, '[');
	  if (goto_xloc.file)
	    {
	      pp_string (buffer, goto_xloc.file);
	      pp_string (buffer, " : ");
	    }
	  pp_decimal_int (buffer, goto_xloc.line);
	  pp_string (buffer, " : ");
	  pp_decimal_int (buffer, goto_xloc.column);
	  pp_string (buffer, "] ");
	}

      pp_cfg_jump (buffer, e->dest);
      pp_newline (buffer);
    }
}


/*********************/
static void 
dump_gimple_statement(pretty_printer *buffer, gimple stmt){
  /*get the gimple code*/
  enum gimple_code stmt_code = gimple_code (stmt);  
  
  dump_gimple_stmt_c (buffer, stmt, 0, 0); //added in gimple-pretty-print-c.c

  
}

/* get the function name from a function_decl tree node*/
static char *
get_function_name(tree fn){
  
 continue_gfn_1:
  switch (TREE_CODE(fn)){
  case VAR_DECL:
  case PARM_DECL:
  case FUNCTION_DECL:
    goto continue_gfn_2;
    break;
  case ADDR_EXPR:
  case INDIRECT_REF:
  case NOP_EXPR:
    fn = TREE_OPERAND (fn, 0);
    goto continue_gfn_1;
  default:
    printf("error\n");
    return NULL;
    break;
  }
 continue_gfn_2:;
  char *ret =  lang_hooks.decl_printable_name (fn, 1);
  return ret ;
}

/*if stmt is a "printf" function call */
static int 
isPrintf(gimple stmt){
  enum gimple_code stmt_code = gimple_code (stmt);
  if (stmt_code != GIMPLE_CALL) return 0;
  
  tree fn = gimple_call_fn(stmt);
  if (TREE_CODE (fn) == NON_LVALUE_EXPR)
    fn = TREE_OPERAND (fn, 0);
  if (strcmp(get_function_name(fn), "printf")) return 0;
  return 1;

}

/* handle D.*="";
   printf(D.*,...);
*/
static void 
dump_gimple_statement_1(pretty_printer *buffer, gimple stmt, gimple stmt1){
  /*get the gimple code*/
  int flags;
  enum gimple_code stmt_code = gimple_code (stmt);
  
  dump_gimple_stmt_c(buffer, stmt1, 0, 0);
  dump_gimple_stmt_c(buffer, stmt, 0, 0);
  return;

  /* if #include<stdio.h> is in the input file, should handle:
     D.*="";
     printf(D.*,***);
   */
  if (!isPrintf(stmt)){
    dump_gimple_stmt_c(buffer, stmt1, 0, 0);
    pp_newline(buffer);
    flags = 0;
  }else 

  /* if this is a printf call, the first parameter should be builted in*/
  flags = TDF_PRINT;
  dump_gimple_stmt_c (buffer, stmt, 0, flags); //added in gimple-pretty-print-c.c
  flags = 0;
}

/* return the type name  */
static char *
get_type_name(tree type){
  
  enum tree_code_class tclass = TREE_CODE_CLASS (TREE_CODE (type));
  if (tclass != tcc_type) return NULL;
  
  if (TYPE_NAME (type)){
    if (TREE_CODE (TYPE_NAME (type)) == IDENTIFIER_NODE)
      return TYPE_NAME(type)->identifier.id.str;
    //pp_tree_identifier_c (buffer, TYPE_NAME (node));
    else if (TREE_CODE (TYPE_NAME (type)) == TYPE_DECL
	     && DECL_NAME (TYPE_NAME (type)))
      return DECL_NAME(TYPE_NAME(type))->identifier.id.str;
	//dump_decl_name (buffer, TYPE_NAME (type), flags);
    else
      return NULL;
  }
  return NULL;
}

/* return 1 if 
 *  stmt1: D_1 = (volatile uint16_t *) intereger;                                                                                  
 *  stmt2: D_2 = * D_1;
 * otherwise return 0   
 */
static int
read_from_memory (gimple stmt1, gimple stmt2){
  if (gimple_code(stmt1) != GIMPLE_ASSIGN) return 0;
  if (! if_volatile_cast(stmt1)) return 0; 
  if (stmt2 == NULL || gimple_code(stmt2) != GIMPLE_ASSIGN) return 0;

  if (gimple_num_ops(stmt1) != 2 || gimple_num_ops(stmt2) != 2) return 0;

  //check lhs(stmt1) type
  tree lhs, rhs, type;
  lhs = gimple_assign_lhs(stmt1);
  rhs = gimple_assign_rhs1(stmt1);
  type = TREE_TYPE(lhs);
  if (TREE_CODE(type) != POINTER_TYPE || TREE_CODE(TREE_TYPE(type)) != INTEGER_TYPE) return 0;
  if (strcmp(get_type_name(TREE_TYPE(type)), "uint16_t") != 0) return 0;

  //check rhs(stmt1) 
  if (TREE_CODE(rhs) != INTEGER_CST) return 0;

  //check rhs(stmt2) should be lhs(stmt1)
  if (!gimple_has_volatile_ops(stmt2)) return 0;
  rhs = gimple_assign_rhs1(stmt2);
  if (TREE_CODE(rhs) != MEM_REF) return 0;
   

  return 1;
}

/**********************/
/* Dumps basic block BB to buffer BUFFER with details described by FLAGS and
   indented by INDENT spaces.  */

static void
gimple_dump_bb_buff (pretty_printer *buffer, basic_block bb, int indent,
		     int flags)
{
  gimple_stmt_iterator gsi;
  gimple stmt,tempstmt,next_stmt;
  int flag = 0;
  int label_indent = indent - 2;
  enum gimple_code stmt_code;

  if (label_indent < 0)
    label_indent = 0;

  dump_bb_header (buffer, bb, indent, flags);


  
  for (gsi = gsi_start_bb (bb); !gsi_end_p (gsi); gsi_next (&gsi))
    {
      
      int curr_indent;

      stmt = gsi_stmt (gsi);
      
      curr_indent = gimple_code (stmt) == GIMPLE_LABEL ? label_indent : indent;

      INDENT (curr_indent);
      
      /* check if the statements represents reading from a memory address
       * in current situation , the pattern is:
       *  D_1 = (volatile uint16_t *) intereger;
       *  D_2 = * D_1;
       * therefore, the value of D_2 should be recorded for later replay
       */
      stmt_code = gimple_code(stmt);
      if (!gsi_one_before_end_p(gsi)){
	gsi_next(&gsi);
      	next_stmt = gsi_stmt(gsi);
	if (read_from_memory(stmt, next_stmt)) {
	  pp_string(buffer, "/* marked by wang80 */");
	  //TO DO rewrite the output as WRITE or READ
	}
        gsi_prev(&gsi);      
      }
      
      if (gsi.ptr->next == NULL) {
	if (flag ==1 ) dump_gimple_statement_1(buffer, stmt, tempstmt);
	else dump_gimple_statement(buffer, stmt);
	flag = 0;
	continue;
      }

      if (flag ==0 && stmt_code == GIMPLE_ASSIGN) {
        tempstmt = stmt;
        flag = 1;
        continue;
      }

      if (flag ==0 && stmt_code != GIMPLE_ASSIGN){
        dump_gimple_statement(buffer, stmt);
        continue;
      }

      if (flag == 1 && stmt_code != GIMPLE_ASSIGN) {
        dump_gimple_statement_1(buffer, stmt,tempstmt);
        flag = 0;
      } else {
        dump_gimple_statement(buffer, tempstmt);
        tempstmt = stmt;
      }
      
      pp_newline(buffer);
      dump_histograms_for_stmt(cfun,buffer->buffer->stream,stmt);
    }

  dump_implicit_edges (buffer, bb, indent, flags);

  if (flags & TDF_BLOCKS)
    dump_bb_end (buffer, bb, indent, flags);
}


/* Dumps basic block BB to FILE with details described by FLAGS and
   indented by INDENT spaces.  */

void
gimple_dump_bb_c (basic_block bb, FILE *file, int indent, int flags)
{
  maybe_init_pretty_print (file);
  gimple_dump_bb_buff (&buffer, bb, indent, flags);
  pp_flush (&buffer);
}


/*dump global variables*/
void dump_varpool_c(FILE *dumpfile){
  struct varpool_node *node;
  tree decl;
  fprintf (dumpfile, "/*global variable:*/\n\n");

  for (node = varpool_nodes; node; node = node->next) {
    
    decl = node->decl;
    print_generic_decl_c (dumpfile,decl,TDF_GLOBAL);
    fprintf(dumpfile,"\n");
    
  }

}

/* check this type has been printed or not*/
static bool isPrinted(tree ttype){

  struct Type_List *list;
  list = printedtypelist;
  while (list!= NULL){
    if ((unsigned int)TYPE_UID(ttype) == list->uid) return true;
    list = list->next;
  }
  struct Type_List *newNode = XNEWVEC(struct Type_List,1);
  newNode->uid = TYPE_UID(ttype);
  newNode->next = printedtypelist;
  printedtypelist = newNode;
  return false;

}


static void gimple_dump_struct_decl_c(pretty_printer *buffer, tree decl, int spc, int flags, bool isStmt){
  
  switch (TREE_CODE(decl)){
  case RECORD_TYPE:
  case UNION_TYPE:
  case QUAL_UNION_TYPE:
    {
      if (isPrinted(decl)) break;
      tree context = TYPE_CONTEXT(decl);
      if ((flags & TDF_LOCAL) && TYPE_FILE_SCOPE_P(decl)) break;
      dump_struct_node_c(buffer, decl, spc, flags, true);
      break;
    }
  case POINTER_TYPE:
  case REFERENCE_TYPE:
    {
      if (TREE_TYPE (decl) == NULL) return;
      gimple_dump_struct_decl_c(buffer, TREE_TYPE(decl), 0, flags, true);
      break;
    }
  default:
    break;
  }

}


static
void remove_unnecessary_decl(tree decl, struct Remove_List *removedlist){
  
  basic_block bb;
  gimple_stmt_iterator gsi;
  gimple stmt;
  tree arg, var;
  struct Remove_List *newNode;

  FOR_EACH_BB(bb)
  for (gsi = gsi_start_bb (bb); !gsi_end_p (gsi); gsi_next (&gsi))
    {
      stmt = gsi_stmt (gsi);
      if (isPrintf(stmt)){
	// record the var name
	arg = gimple_call_arg (stmt, 0);
	if (TREE_CODE(arg) != SSA_NAME) continue;
	var = SSA_NAME_VAR(arg);
	
	newNode = XNEWVEC(struct Remove_List,1);
	newNode->id = var;
	newNode->next = removedlist->next;
	removedlist->next = newNode;
		
      }
      
    }
}


static 
void dump_attributes_c( tree fn, FILE *dumpfile){
  
  tree a,value;
  if (lookup_attribute("wakeup", DECL_ATTRIBUTES(fn)))
    fprintf(dumpfile, "__attribute((wakeup)) ");

  a = lookup_attribute("interrupt",DECL_ATTRIBUTES(fn));
  
  if (a != NULL_TREE){
    if (TREE_CODE(a) == TREE_LIST) value = TREE_VALUE(TREE_VALUE(a));
    HOST_WIDE_INT code = TREE_INT_CST_LOW (value); 
    fprintf(dumpfile, "__attribute((interrupt(%lld))) ", code);
  }
}


static 
int is_irpt_fun(char *name){
  int len = strlen(name);
  if (len < 4) return 0;
  if (name[0] == 's' &&
      name[1] == 'i' &&
      name[2] == 'g' &&
      name[3] == '_')
    return 1;
  return 0;
}

/* Dump the function declaration*/
void
dump_function_decl_c(tree fn, FILE *dumpfile){
  tree args;
  int flag;
  struct Remove_List *removedlist = XNEWVEC(struct Remove_List,1);
  removedlist->id = NULL;
  removedlist->next = NULL;
  /* handle interrupt vector function separately */
  char *name = get_function_name(fn);
  if (is_irpt_fun(name))
    flag = print_function_header_c_irpt(dumpfile, fn, 0);
  else {
    /* handle non-interrupt vector function*/
    flag = print_function_header_c(dumpfile, fn, 0);
    if (flag)
      if (DECL_ATTRIBUTES(fn) != NULL_TREE)
	// output the attributs of this function
	dump_attributes_c(fn, dumpfile);
  }

  
  if (flag){
    fprintf (dumpfile, "{\n");


    /*print local struct type declaration*/
    struct function *dsf = DECL_STRUCT_FUNCTION (fn);
    tree var;
    bool ignore_topmost_bind = false, any_var = false;
    int flags = TDF_LOCAL;
    maybe_init_pretty_print (dumpfile);
    if (dsf && dsf->decl == fn && !VEC_empty (tree, dsf->local_decls))
      {
	unsigned ix;
	ignore_topmost_bind = true;

	FOR_EACH_LOCAL_DECL (dsf, ix, var)
	  {
	    gimple_dump_struct_decl_c(&buffer, TREE_TYPE(var),2, flags,true);

	  }
      }
    pp_flush(&buffer);

    print_function_local_decl_c(dumpfile,fn, removedlist, flags);
  }


};

/* dump declaration*/
void dump_struct_declaration_c(FILE *dumpfile){
  
  struct varpool_node *node;
  struct cgraph_node *cnode;
  tree decl;
  int flags = TDF_GLOBAL;
  
  maybe_init_pretty_print (dumpfile);

  // check each global variabl
  for (node = varpool_nodes; node; node = node->next){
    decl = TREE_TYPE(node->decl);
    gimple_dump_struct_decl_c(&buffer, decl, 0, flags, true);
  }
   
    
  // check each function parameter
  for (cnode = cgraph_nodes; cnode; cnode = cnode->next){
    decl = cnode->decl;
    tree arg;
    tree arg_type;
    arg_type = TYPE_ARG_TYPES(TREE_TYPE(decl));
    
    for (arg = DECL_ARGUMENTS (decl); arg ; arg = DECL_CHAIN (arg))
      {
	if (TREE_CODE(arg)== PARM_DECL)  {
	  tree ttype = TREE_VALUE(arg_type);
	  gimple_dump_struct_decl_c(&buffer, ttype, 0, flags, true);
	}
	arg_type = TREE_CHAIN (arg_type);
      }
    
  }
  
  pp_flush(&buffer);

}

#ifdef NONE_USED_SYMBOL
static void 
dump_gimple_statement(FILE *dumpfile, gimple stmt){
  /*get the gimple code*/
  enum gimple_code stmt_code = gimple_code (stmt);  
  
  /*print the statement based on different type*/
  //fprintf (dumpfile, "LINE : %d", LOCATION_LINE (gimple_location (stmt)));

  //fprintf (dumpfile, " STMT %lld: ", gimple_uid(stmt));
  print_gimple_stmt_c (dumpfile, stmt, 0, 0); //added in gimple-pretty-print-c.c
  fprintf(dumpfile,"\n");

  
}


/*if stmt is a "printf" function call */
static int 
isPrintf(gimple stmt){
  enum gimple_code stmt_code = gimple_code (stmt);
  if (stmt_code != GIMPLE_CALL) return 0;
  
  tree fn = gimple_call_fn(stmt);
  if (TREE_CODE (fn) == NON_LVALUE_EXPR)
    fn = TREE_OPERAND (fn, 0);

 continue_1:
  switch (TREE_CODE (fn))
    {
    case VAR_DECL:
    case PARM_DECL:
    case FUNCTION_DECL:
      goto continue_2;
      break;
      
    case ADDR_EXPR:
    case INDIRECT_REF:
    case NOP_EXPR:
      fn = TREE_OPERAND (fn, 0);
      goto continue_1;
    default:
      break;
    }
  return 0;
 continue_2:
  printf("%s\n",lang_hooks.decl_printable_name (fn, 1));
  if (strcmp(lang_hooks.decl_printable_name (fn, 1),"printf")) return 0; 
  return 1;
}

/* handle D.*="";
   printf(D.*,...);
*/
static void 
dump_gimple_statement_1(FILE *dumpfile, gimple stmt, gimple stmt1){
  /*get the gimple code*/
  int flags;
  enum gimple_code stmt_code = gimple_code (stmt);
  
  if (!isPrintf(stmt)){
    print_gimple_stmt_c(dumpfile, stmt1, 0, 0);
    flags = 0;

  } else
    flags = TDF_PRINT;
  print_gimple_stmt_c (dumpfile, stmt, 0, flags); //added in gimple-pretty-print-c.c
                                              
  fprintf(dumpfile,"\n");

}

void dump_bb_new(basic_block bb, FILE *dumpfile){
  gimple_stmt_iterator bsi; //iterator object for gimple statement sequence
  
  gimple tempstmt;
  int flag = 0;
  enum gimple_code stmt_code;
  for (bsi = gsi_start_bb (bb); !gsi_end_p (bsi); gsi_next (&bsi))
    {
      gimple stmt = gsi_stmt (bsi);
      stmt_code = gimple_code(stmt);
      if (flag ==0 && stmt_code == GIMPLE_ASSIGN) {
	tempstmt = stmt;
	flag = 1;
	continue;
      }
     
      if (flag ==0 && stmt_code != GIMPLE_ASSIGN){
	dump_gimple_statement(dumpfile, stmt);
	continue;
      }
	
      if (flag == 1 && stmt_code != GIMPLE_ASSIGN) {
	dump_gimple_statement_1(dumpfile, stmt,tempstmt);
	flag = 0;
      } else {
	dump_gimple_statement(dumpfile, tempstmt);
	tempstmt = stmt;
      }
	
    }

  gimple_seq seq = bb_seq (bb);
  if ( gimple_seq_empty_p(seq) )
    fprintf (dumpfile, "  empty" );

  fprintf (dumpfile, "\n" );
}

#endif
